/* -*- mode: c++; c-basic-offset: 3; -*- */
VCSID("@(#)$Id: frametype.cc 7338 2015-04-14 23:24:47Z ed.maros@LIGO.ORG $");
//////////////////////////////////////////////////////////////////////////
//                                                                      //
// frametype						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

// #define DEBUG
#ifdef DEBUG
#include <vector>
#endif

// ::WRONG:: incorrect frame format if defined!!!
#undef __FILEOFS_4U
//#define __FILEOFS_4U // correct for invalid 4u file offsets in FrameLib

#include "framefast/frametype.hh"
#include <stddef.h>
#include <cstring>
#include <cmath>
#include <string>
#include <iostream>
#include <iomanip>
#include <algorithm>
#include "zlib.h"
#include "constant.hh"

#ifdef __GNU_STDC_OLD
#define showbase ""
#endif


namespace framefast {

   using namespace std;

#define HAVE_LIBZ_COMPRESS2 1

   const int kGZIPlevel = 1;
   const int kDefaultTocSize = 128;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// standard SH/SE dictionary structures			                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   const int kSH_V4 = 9;
   const int kSH_V6 = 9;
   const int kSH_V8 = 9;
   const int kMaxSE = 60;

   struct SH_def_t {
      const char* fName;
      int         fClassNum;
      const char* fComment;
      int         fElementNum;
   };

   struct SE_def_t {
      const char* fName;
      const char* fType;
      const char* fComment;
   };

   const SH_def_t cSH_V4[kSH_V4] = {
   {"FrameH", 3, "Frame Header", 23},
   {"FrDetector", 5, "Detector Data", 12},
   {"FrHistory", 8, "History Data", 4},
   {"FrRawData", 9, "Raw Data", 6},
   {"FrAdcData", 4, "ADC Data", 16},
   {"FrVect", 14, "Vector Data", 13},
   {"FrEndOfFrame", 6, "End of Frame Data", 2},
   {"FrTOC", 20, "Table of Contents", 51},
   {"FrEndOfFile", 17, "End of File", 5}
   };

   const SH_def_t cSH_V6[kSH_V6] = {
   {"FrameH",               3,  "Frame Header Structure",        21},
   {"FrDetector",           5,  "Detector Data Structure",       15},
   {"FrHistory",            9,  "History Structure",              4},
   {"FrRawData",           12,  "Raw Data Structure",             6},
   {"FrAdcData",            4,  "AdcData  Data Structure",       16},
   {"FrVect",              20,  "Vector Data Structure",         13},
   {"FrEndOfFrame",         7,  "End of Frame Data Structure",    4},
   {"FrTOC",               19,  "Table of Contents Structure",   58},
   {"FrEndOfFile",          6,  "End of File Data Structure",     5}
   };

   const SE_def_t cSE_V4[kSH_V4][kMaxSE] = {
      {//-------------------------------  FrameH
	 {"name", "STRING", ""},
	 {"run", "INT_4S", ""},
	 {"frame", "INT_4U", ""},
	 {"dataQuality", "INT_4U", ""},
	 {"GTimeS", "INT_4U", ""},
	 {"GTimeN", "INT_4U", ""},
	 {"ULeapS", "INT_2U", ""},
	 {"localTime", "INT_4S", ""},
	 {"dt", "REAL_8", ""},
	 {"type", "PTR_STRUCT(FrVect *)", ""},
	 {"user", "PTR_STRUCT(FrVect *)", ""},
	 {"detectSim", "PTR_STRUCT(FrDetector *)", ""},
	 {"detectProc", "PTR_STRUCT(FrDetector *)", ""},
	 {"history", "PTR_STRUCT(FrHistory *)", ""},
	 {"rawData", "PTR_STRUCT(FrRawData *)", ""},
	 {"procData", "PTR_STRUCT(FrProcData *)", ""},
	 {"strain", "PTR_STRUCT(FrProcData *)", ""},
	 {"simData", "PTR_STRUCT(FrSimData *)", ""},
	 {"trigData", "PTR_STRUCT(FrTrigData *)", ""},
	 {"simEvent", "PTR_STRUCT(FrSimEvent *)", ""},
	 {"summaryData", "PTR_STRUCT(FrSummary *)", ""},
	 {"auxData", "PTR_STRUCT(FrVect *)", ""},
	 {"auxTable", "PTR_STRUCT(FrTable *)", ""}},
      {//--------------------------------- FrDetector
	 {"name", "STRING", ""},
	 {"longitudeD", "INT_2S", ""},
	 {"longitudeM", "INT_2S", ""},
	 {"longitudeS", "REAL_4", ""},
	 {"latitudeD", "INT_2S", ""},
	 {"latitudeM", "INT_2S", ""},
	 {"latitudeS", "REAL_4", ""},
	 {"elevation", "REAL_4", ""},
	 {"armXazimuth", "REAL_4", ""},
	 {"armYazimuth", "REAL_4", ""},
	 {"more", "PTR_STRUCT(FrVect *)", ""},
	 {"moreTable", "PTR_STRUCT(FrTable *)", ""}},
      {
	 {"name", "STRING", ""},
	 {"time", "INT_4U", ""},
	 {"comment", "STRING", ""},
	 {"next", "PTR_STRUCT(FrHistory *)", ""}},
      {
	 {"name", "STRING", ""},
	 {"firstSer", "PTR_STRUCT(FrSerData *)", ""},
	 {"firstAdc", "PTR_STRUCT(FrAdcData *)", ""},
	 {"firstTable", "PTR_STRUCT(FrTable *)", ""},
	 {"logMsg", "PTR_STRUCT(FrMsg *)", ""},
	 {"more", "PTR_STRUCT(FrVect *)", ""}},
      {
	 {"name", "STRING", ""},
	 {"comment", "STRING", ""},
	 {"channelGroup", "INT_4U", ""},
	 {"channelNumber", "INT_4U", ""},
	 {"nBits", "INT_4U", ""},
	 {"bias", "REAL_4", ""},
	 {"slope", "REAL_4", ""},
	 {"units", "STRING", ""},
	 {"sampleRate", "REAL_8", ""},
	 {"timeOffsetS", "INT_4S", ""},
	 {"timeOffsetN", "INT_4U", ""},
	 {"fShift", "REAL_8", ""},
	 {"dataValid", "INT_2U", ""},
	 {"data", "PTR_STRUCT(FrVect *)", ""},
	 {"aux", "PTR_STRUCT(FrVect *)", ""},
	 {"next", "PTR_STRUCT(FrAdcData *)", ""}},
      {
	 {"name", "STRING", ""},
	 {"compress", "INT_2U", ""},
	 {"type", "INT_2U", ""},
	 {"nData", "INT_4U", ""},
	 {"nBytes", "INT_4U", ""},
	 {"data", "*", ""},
	 {"nDim", "INT_4U", ""},
	 {"nx", "INT_4U *", ""},
	 {"dx", "REAL_8 *", ""},
	 {"startX", "REAL_8 *", ""},
	 {"unitX", "STRING *", ""},
	 {"unitY", "STRING", ""},
	 {"next", "PTR_STRUCT(FrVect *)", ""}},
      {
	 {"run", "INT_4S", ""},
	 {"frame", "INT_4U", ""}},
      {
	 {"ULeapS", "INT_2S", ""},
	 {"localTime", "INT_4S", ""},
	 {"nFrame", "INT_4U", ""},
	 {"GTimeS", "INT_4U *", ""},
	 {"GTimeN", "INT_4U *", ""},
	 {"dt", "REAL_8 *", ""},
	 {"runs", "INT_4S *", ""},
	 {"frame", "INT_4U *", ""},
	 {"positionH", "INT_8U *", ""},
	 {"nFirstADC", "INT_8U *", ""},
	 {"nFirstSer", "INT_8U *", ""},
	 {"nFirstTable", "INT_8U *", ""},
	 {"nFirstMsg", "INT_8U *", ""},
	 {"nSH", "INT_4U", ""},
	 {"SHid", "INT_2U *", ""},
	 {"SHname", "STRING *", ""},
	 {"nStatType", "INT_4U", ""},
	 {"nameStat", "STRING", ""},
	 {"detector", "STRING", ""},
	 {"nStatInstance", "INT_4U", ""},
	 {"tStart", "INT_4U *", ""},
	 {"tEnd", "INT_4U *", ""},
	 {"version", "INT_4U *", ""},
	 {"positionStat", "INT_8U *", ""},
	 {"nADC", "INT_4U", ""},
	 {"name", "STRING *", ""},
	 {"channelID", "INT_4U *", ""},
	 {"groupID", "INT_4U *", ""},
	 {"positionADC", "INT_8U *", ""},
	 {"nProc", "INT_4U", ""},
	 {"nameProc", "STRING *", ""},
	 {"positionProc", "INT_8U *", ""},
	 {"nSim", "INT_4U", ""},
	 {"nameSim", "STRING *", ""},
	 {"positionSim", "INT_8U *", ""},
	 {"nSer", "INT_4U", ""},
	 {"nameSer", "STRING *", ""},
	 {"positionSer", "INT_8U *", ""},
	 {"nSummary", "INT_4U", ""},
	 {"nameSum", "STRING *", ""},
	 {"positionSum", "INT_8U *", ""},
	 {"nTrig", "INT_4U", ""},
	 {"nameTrig", "STRING *", ""},
	 {"GTimeSTrig", "INT_4U *", ""},
	 {"GTimeNTrig", "INT_4U *", ""},
	 {"positionTrig", "INT_8U *", ""},
	 {"nSimEvt", "INT_4U", ""},
	 {"nameSimEvt", "STRING *", ""},
	 {"GTimeSSim", "INT_4U *", ""},
	 {"GTimeNSim", "INT_4U *", ""},
	 {"positionSimEvt", "INT_8U *", ""}},
      {
	 {"nFrames", "INT_4U", ""},
	 {"nBytes", "INT_4U", ""},
	 {"chkFlag", "INT_4U", ""},
	 {"chkSum", "INT_4U", ""},
	 {"seekTOC", "INT_4U", ""}}
   };

   const SE_def_t cSE_V6[kSH_V6][kMaxSE] = {
   {                       
   {"name",            "STRING",       ""},
   {"run",             "INT_4S",       ""},
   {"frame",           "INT_4U",       ""},
   {"dataQuality",     "INT_4U",       ""},
   {"GTimeS",          "INT_4U",       ""},
   {"GTimeN",          "INT_4U",       ""},
   {"ULeapS",          "INT_2U",       ""},
   {"dt",              "REAL_8",       ""},
   {"type",            "PTR_STRUCT(FrVect *)",""},
   {"user",            "PTR_STRUCT(FrVect *)",""},
   {"detectSim",       "PTR_STRUCT(FrDetector *)",""},
   {"detectProc",      "PTR_STRUCT(FrDetector *)",""},
   {"history",         "PTR_STRUCT(FrHistory *)",""},
   {"rawData",         "PTR_STRUCT(FrRawData *)",""},
   {"procData",        "PTR_STRUCT(FrProcData *)",""},
   {"simData",         "PTR_STRUCT(FrSimData *)",""},
   {"event",           "PTR_STRUCT(FrEvent *)",""},
   {"simEvent",        "PTR_STRUCT(FrSimEvent *)",""},
   {"summaryData",     "PTR_STRUCT(FrSummary *)",""},
   {"auxData",         "PTR_STRUCT(FrVect *)",""},
   {"auxTable",        "PTR_STRUCT(FrTable *)",""}},
   {
   {"name",            "STRING",       ""},
   {"prefix",          "CHAR[2]",      ""},
   {"longitude",       "REAL_8",       ""},
   {"latitude",        "REAL_8",       ""},
   {"elevation",       "REAL_4",       ""},
   {"armXazimuth",     "REAL_4",       ""},
   {"armYazimuth",     "REAL_4",       ""},
   {"armXaltitude",    "REAL_4",       ""},
   {"armYaltitude",    "REAL_4",       ""},
   {"armXmidpoint",    "REAL_4",       ""},
   {"armYmidpoint",    "REAL_4",       ""},
   {"localTime",       "INT_4S",       ""},
   {"aux",             "PTR_STRUCT(FrVect *)",""},
   {"table",           "PTR_STRUCT(FrTable *)",""},
   {"next",            "PTR_STRUCT(FrDetector *)",""}},
   {                       
   {"name",            "STRING",       ""},
   {"time",            "INT_4U",       ""},
   {"comment",         "STRING",       ""},
   {"next",            "PTR_STRUCT(FrHistory *)",""}},
   {                       
   {"name",            "STRING",       ""},
   {"firstSer",        "PTR_STRUCT(FrSerData *)",""},
   {"firstAdc",        "PTR_STRUCT(FrAdcData *)",""},
   {"firstTable",      "PTR_STRUCT(FrTable *)",""},
   {"logMsg",          "PTR_STRUCT(FrMsg *)",""},
   {"more",            "PTR_STRUCT(FrVect *)",""}},
   {                       
   {"name",            "STRING",       ""},
   {"comment",         "STRING",       ""},
   {"channelGroup",    "INT_4U",       ""},
   {"channelNumber",   "INT_4U",       ""},
   {"nBits",           "INT_4U",       ""},
   {"bias",            "REAL_4",       ""},
   {"slope",           "REAL_4",       ""},
   {"units",           "STRING",       ""},
   {"sampleRate",      "REAL_8",       ""},
   {"timeOffset",      "REAL_8",       ""},
   {"fShift",          "REAL_8",       ""},
   {"phase",           "REAL_4",       ""},
   {"dataValid",       "INT_2U",       ""},
   {"data",            "PTR_STRUCT(FrVect *)",""},
   {"aux",             "PTR_STRUCT(FrVect *)",""},
   {"next",            "PTR_STRUCT(FrAdcData *)",""}},
   {                
   {"name",            "STRING",       ""},
   {"compress",        "INT_2U",       ""},
   {"type",            "INT_2U",       ""},
   {"nData",           "INT_8U",       ""},
   {"nBytes",          "INT_8U",       ""},
   {"data",            "CHAR[nBytes]", ""},
   {"nDim",            "INT_4U",       ""},
   {"nx",              "INT_8U[nDim]", ""},
   {"dx",              "REAL_8[nDim]", ""},
   {"startX",          "REAL_8[nDim]", ""},
   {"unitX",           "STRING[nDim]", ""},
   {"unitY",           "STRING",       ""},
   {"next",            "PTR_STRUCT(FrVect *)",""}},
   {
   {"run",             "INT_4S",       ""},
   {"frame",           "INT_4U",       ""},
   {"chkType",         "INT_4U",       ""},
   {"chkSum",          "INT_4U",       ""}},
   {
   {"ULeapS",          "INT_2S",       ""},
   {"nFrame",          "INT_4U",       ""},
   {"dataQuality",     "*INT_4U",      ""},
   {"GTimeS",          "*INT_4U",      ""},
   {"GTimeN",          "*INT_4U",      ""},
   {"dt",              "*REAL_8",      ""},
   {"runs",            "*INT_4S",      ""},
   {"frame",           "*INT_4U",      ""},
   {"positionH",       "*INT_8U",      ""},
   {"nFirstADC",       "*INT_8U",      ""},
   {"nFirstSer",       "*INT_8U",      ""},
   {"nFirstTable",     "*INT_8U",      ""},
   {"nFirstMsg",       "*INT_8U",      ""},
   {"nSH",             "INT_4U",       ""},
   {"SHid",            "*INT_2U",      ""},
   {"SHname",          "*STRING",      ""},
   {"nDetector",       "INT_4U",       ""},
   {"nameDetector",    "*STRING",      ""},
   {"positionDetector","*INT_8U",      ""},
   {"nStatType",       "INT_4U",       ""},
   {"nameStat",        "STRING",       ""},
   {"detector",        "STRING",       ""},
   {"nStatInstance",   "INT_4U",       ""},
   {"tStart",          "*INT_4U",      ""},
   {"tEnd",            "*INT_4U",      ""},
   {"version",         "*INT_4U",      ""},
   {"positionStat",    "*INT_8U",      ""},
   {"nADC",            "INT_4U",       ""},
   {"name",            "*STRING",      ""},
   {"channelID",       "*INT_4U",      ""},
   {"groupID",         "*INT_4U",      ""},
   {"positionADC",     "*INT_8U",      ""},
   {"nProc",           "INT_4U",       ""},
   {"nameProc",        "*STRING",      ""},
   {"positionProc",    "*INT_8U",      ""},
   {"nSim",            "INT_4U",       ""},
   {"nameSim",         "*STRING",      ""},
   {"positionSim",     "*INT_8U",      ""},
   {"nSer",            "INT_4U",       ""},
   {"nameSer",         "*STRING",      ""},
   {"positionSer",     "*INT_8U",      ""},
   {"nSummary",        "INT_4U",       ""},
   {"nameSum",         "*STRING",      ""},
   {"positionSum",     "*INT_8U",      ""},
   {"nEventType",      "INT_4U",       ""},
   {"nameEvent",       "*STRING",      ""},
   {"nEvent",          "*INT_4U",      ""},
   {"GTimeSEvent",     "*INT_4U",      ""},
   {"GTimeNEvent",     "*INT_4U",      ""},
   {"amplitudeEvent",  "*REAL_4",      ""},
   {"positionEvent",   "*INT_8U",      ""},
   {"nSimEventType",   "INT_4U",       ""},
   {"nameSimEvent",    "*STRING",      ""},
   {"nSimEvent",       "*INT_4U",      ""},
   {"GTimeSSim",       "*INT_4U",      ""},
   {"GTimeNSim",       "*INT_4U",      ""},
   {"amplitudeSimEvent","*REAL_4",     ""},
   {"positionSimEvent","*INT_8U",      ""}},
   {                     
   {"nFrames",         "INT_4U",       ""},
   {"nBytes",          "INT_8U",       ""},
   {"chkType",         "INT_4U",       ""},
   {"chkSum",          "INT_4U",       ""},
   {"seekTOC",         "INT_8U",       ""}},
   };


   // name                 ID  Comment                      Total
   const SH_def_t cSH_V8[kSH_V8] = {
      {"FrameH",               3,  "Frame Header Structure",        22},
      {"FrDetector",           5,  "Detector Data Structure",       16},
      {"FrHistory",            9,  "History Structure",              5},
      {"FrRawData",           12,  "Raw Data Structure",             7},
      {"FrAdcData",            4,  "AdcData  Data Structure",       17},
      {"FrVect",              20,  "Vector Data Structure",         14},
      {"FrEndOfFrame",         7,  "End of Frame Data Structure",    5},
      {"FrTOC",               19,  "Table of Contents Structure",   58},
      {"FrEndOfFile",          6,  "End of File Data Structure",     6}
   };
   

   const SE_def_t cSE_V8[kSH_V8][kMaxSE] = {
      {//-------------------------------  FrameH (22)
	 {"name",            "STRING",       ""},
	 {"run",             "INT_4S",       ""},
	 {"frame",           "INT_4U",       ""},
	 {"dataQuality",     "INT_4U",       ""},
	 {"GTimeS",          "INT_4U",       ""},
	 {"GTimeN",          "INT_4U",       ""},
	 {"ULeapS",          "INT_2U",       ""},
	 {"dt",              "REAL_8",       ""},
	 {"type",            "PTR_STRUCT(FrVect *)",""},
	 {"user",            "PTR_STRUCT(FrVect *)",""},
	 {"detectSim",       "PTR_STRUCT(FrDetector *)",""},
	 {"detectProc",      "PTR_STRUCT(FrDetector *)",""},
	 {"history",         "PTR_STRUCT(FrHistory *)",""},
	 {"rawData",         "PTR_STRUCT(FrRawData *)",""},
	 {"procData",        "PTR_STRUCT(FrProcData *)",""},
	 {"simData",         "PTR_STRUCT(FrSimData *)",""},
	 {"event",           "PTR_STRUCT(FrEvent *)",""},
	 {"simEvent",        "PTR_STRUCT(FrSimEvent *)",""},
	 {"summaryData",     "PTR_STRUCT(FrSummary *)",""},
	 {"auxData",         "PTR_STRUCT(FrVect *)",""},
	 {"auxTable",        "PTR_STRUCT(FrTable *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrDetector (16)
	 {"name",            "STRING",       ""},
	 {"prefix",          "CHAR[2]",      ""},
	 {"longitude",       "REAL_8",       ""},
	 {"latitude",        "REAL_8",       ""},
	 {"elevation",       "REAL_4",       ""},
	 {"armXazimuth",     "REAL_4",       ""},
	 {"armYazimuth",     "REAL_4",       ""},
	 {"armXaltitude",    "REAL_4",       ""},
	 {"armYaltitude",    "REAL_4",       ""},
	 {"armXmidpoint",    "REAL_4",       ""},
	 {"armYmidpoint",    "REAL_4",       ""},
	 {"localTime",       "INT_4S",       ""},
	 {"aux",             "PTR_STRUCT(FrVect *)",""},
	 {"table",           "PTR_STRUCT(FrTable *)",""},
	 {"next",            "PTR_STRUCT(FrDetector *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrHistory (5)
	 {"name",            "STRING",       ""},
	 {"time",            "INT_4U",       ""},
	 {"comment",         "STRING",       ""},
	 {"next",            "PTR_STRUCT(FrHistory *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrRawData (7)
	 {"name",            "STRING",       ""},
	 {"firstSer",        "PTR_STRUCT(FrSerData *)",""},
	 {"firstAdc",        "PTR_STRUCT(FrAdcData *)",""},
	 {"firstTable",      "PTR_STRUCT(FrTable *)",""},
	 {"logMsg",          "PTR_STRUCT(FrMsg *)",""},
	 {"more",            "PTR_STRUCT(FrVect *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrAdcData (17)
	 {"name",            "STRING",       ""},
	 {"comment",         "STRING",       ""},
	 {"channelGroup",    "INT_4U",       ""},
	 {"channelNumber",   "INT_4U",       ""},
	 {"nBits",           "INT_4U",       ""},
	 {"bias",            "REAL_4",       ""},
	 {"slope",           "REAL_4",       ""},
	 {"units",           "STRING",       ""},
	 {"sampleRate",      "REAL_8",       ""},
	 {"timeOffset",      "REAL_8",       ""},
	 {"fShift",          "REAL_8",       ""},
	 {"phase",           "REAL_4",       ""},
	 {"dataValid",       "INT_2U",       ""},
	 {"data",            "PTR_STRUCT(FrVect *)",""},
	 {"aux",             "PTR_STRUCT(FrVect *)",""},
	 {"next",            "PTR_STRUCT(FrAdcData *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrVect (14)
	 {"name",            "STRING",       ""},
	 {"compress",        "INT_2U",       ""},
	 {"type",            "INT_2U",       ""},
	 {"nData",           "INT_8U",       ""},
	 {"nBytes",          "INT_8U",       ""},
	 {"data",            "CHAR[nBytes]", ""},
	 {"nDim",            "INT_4U",       ""},
	 {"nx",              "INT_8U[nDim]", ""},
	 {"dx",              "REAL_8[nDim]", ""},
	 {"startX",          "REAL_8[nDim]", ""},
	 {"unitX",           "STRING[nDim]", ""},
	 {"unitY",           "STRING",       ""},
	 {"next",            "PTR_STRUCT(FrVect *)",""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrEndFrame (5)
	 {"run",             "INT_4S",       ""},
	 {"frame",           "INT_4U",       ""},
	 {"GTimeS",         "INT_4U",       ""},
	 {"GTimeN",          "INT_4U",       ""},
	 {"chkSum",          "INT_4U",       ""}},
      {//-------------------------------  FrTOC (58)
	 {"ULeapS",          "INT_2S",       ""},
	 {"nFrame",          "INT_4U",       ""},
	 {"dataQuality",     "*INT_4U",      ""},
	 {"GTimeS",          "*INT_4U",      ""},
	 {"GTimeN",          "*INT_4U",      ""},
	 {"dt",              "*REAL_8",      ""},
	 {"runs",            "*INT_4S",      ""},
	 {"frame",           "*INT_4U",      ""},
	 {"positionH",       "*INT_8U",      ""},
	 {"nFirstADC",       "*INT_8U",      ""},
	 {"nFirstSer",       "*INT_8U",      ""},
	 {"nFirstTable",     "*INT_8U",      ""},
	 {"nFirstMsg",       "*INT_8U",      ""},
	 {"nSH",             "INT_4U",       ""},
	 {"SHid",            "*INT_2U",      ""},
	 {"SHname",          "*STRING",      ""},
	 {"nDetector",       "INT_4U",       ""},
	 {"nameDetector",    "*STRING",      ""},
	 {"positionDetector","*INT_8U",      ""},
	 {"nStatType",       "INT_4U",       ""},
	 {"nameStat",        "STRING",       ""},
	 {"detector",        "STRING",       ""},
	 {"nStatInstance",   "INT_4U",       ""},
	 {"tStart",          "*INT_4U",      ""},
	 {"tEnd",            "*INT_4U",      ""},
	 {"version",         "*INT_4U",      ""},
	 {"positionStat",    "*INT_8U",      ""},
	 {"nADC",            "INT_4U",       ""},
	 {"name",            "*STRING",      ""},
	 {"channelID",       "*INT_4U",      ""},
	 {"groupID",         "*INT_4U",      ""},
	 {"positionADC",     "*INT_8U",      ""},
	 {"nProc",           "INT_4U",       ""},
	 {"nameProc",        "*STRING",      ""},
	 {"positionProc",    "*INT_8U",      ""},
	 {"nSim",            "INT_4U",       ""},
	 {"nameSim",         "*STRING",      ""},
	 {"positionSim",     "*INT_8U",      ""},
	 {"nSer",            "INT_4U",       ""},
	 {"nameSer",         "*STRING",      ""},
	 {"positionSer",     "*INT_8U",      ""},
	 {"nSummary",        "INT_4U",       ""},
	 {"nameSum",         "*STRING",      ""},
	 {"positionSum",     "*INT_8U",      ""},
	 {"nEventType",      "INT_4U",       ""},
	 {"nameEvent",       "*STRING",      ""},
	 {"nEvent",          "*INT_4U",      ""},
	 {"GTimeSEvent",     "*INT_4U",      ""},
	 {"GTimeNEvent",     "*INT_4U",      ""},
	 {"amplitudeEvent",  "*REAL_4",      ""},
	 {"positionEvent",   "*INT_8U",      ""},
	 {"nSimEventType",   "INT_4U",       ""},
	 {"nameSimEvent",    "*STRING",      ""},
	 {"nSimEvent",       "*INT_4U",      ""},
	 {"GTimeSSim",       "*INT_4U",      ""},
	 {"GTimeNSim",       "*INT_4U",      ""},
	 {"amplitudeSimEvent","*REAL_4",     ""},
	 {"positionSimEvent","*INT_8U",      ""}},
      {//------------------------------- FrEndFile (6)
	 {"nFrames",         "INT_4U",       ""},
	 {"nBytes",          "INT_8U",       ""},
	 {"seekTOC",         "INT_8U",       ""},
	 {"chkSumFrHeader",  "INT_4U",       ""},
	 {"chkSum",          "INT_4U",       ""},
	 {"chkSumFile",      "INT_4U",       ""}},
   };



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// swapin/out template						        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   template<typename T>
   inline int
   swapin(const char* p, T& out, bool swapit) {
      if (swapit) {
	 for (int i=sizeof(T); i; --i) {
	    reinterpret_cast<char*>(&out)[i] = *p++;
	 }
      } else {
	 memcpy(&out, p, sizeof(T));
      }
      return sizeof(T);
   }

   template<typename T>
   inline int
   swapout(const T& in, char* p, bool swapit) {
      if (swapit) {
	 for (int i=sizeof(T); i; --i) {
	    *p++ = reinterpret_cast<const char*>(&in)[i];
	 }
      } else {
	 memcpy(p, &in, sizeof(T));
      }
      return sizeof(T);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// save_strncpy						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   inline void save_strncpy (char* p, const char* s, int max)
   {
      strncpy (p, s, max-1);
      p[max-1] = 0;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// ptr_struct						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   ptr_struct::ptr_struct()
   {
      fDataClass = 0;
      fDataInstance = 0;
   }

//______________________________________________________________________________
   long ptr_struct::read (int version, const char* p, bool swapit)
   {
      // version 4
      if (version < 6) {
         int_2u_t dataInstance;
         memcpy (&fDataClass, p, 2);
         memcpy (&dataInstance, p + 2, 2);
         if (swapit) {
            swap (&fDataClass);
            swap (&dataInstance);
         }
         fDataInstance = dataInstance;
         return 4;
      }
      // version 6
      else if (version < 8) {
         memcpy (&fDataClass, p, 2);
         memcpy (&fDataInstance, p + 2, 4);
         if (swapit) {
            swap (&fDataClass);
            swap (&fDataInstance);
         }
         return 6;
      }
      // version 8
      else {
         memcpy (&fDataClass, p, 2);
         memcpy (&fDataInstance, p + 2, 4);
         if (swapit) {
            swap (&fDataClass);
            swap (&fDataInstance);
         }
         return 6;
      }
    }

//______________________________________________________________________________
   long ptr_struct::write (int version, char* p, bool swapit) const
   {
      if (version < 6) {
         int_2u_t dataClass = fDataClass;
         int_2u_t dataInstance = fDataInstance;
         if (swapit) {
            swap (&dataClass);
            swap (&dataInstance);
         }
         memcpy (p, &dataClass, 2);
         memcpy (p + 2, &dataInstance, 2);
         return 4;
      }
      else if (version < 8) {
         int_2u_t dataClass = fDataClass;
         int_4u_t dataInstance = fDataInstance;
         if (swapit) {
            swap (&dataClass);
            swap (&dataInstance);
         }
         memcpy (p, &dataClass, 2);
         memcpy (p + 2, &dataInstance, 4);
         return 6;
      }
      else {
         int_2u_t dataClass = fDataClass;
         int_4u_t dataInstance = fDataInstance;
         if (swapit) {
            swap (&dataClass);
            swap (&dataInstance);
         }
         memcpy (p, &dataClass, 2);
         memcpy (p + 2, &dataInstance, 4);
         return 8;
      }
   }

//______________________________________________________________________________
   bool ptr_struct::set (int version, const char* name)
   {
      fDataClass = 0;
      fDataInstance = 0;
      int kSH = kSH_V4;
      if (version >= 6) kSH = kSH_V4;
      const SH_def_t*	cSH = cSH_V4;
      if (version >= 6) cSH = cSH_V6;
      for (int i = 0; i < kSH; i++) {
         if (strcmp (cSH[i].fName, name) == 0) {
            fDataClass = cSH[i].fClassNum;
            return true;
         }
      }
      return false;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Type ID						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

template <typename T>
   class typeOf {
   public:
      typedef T value_type;
      typedef double tempvalue_type;
      static const int value;
   };

template <typename T>
   const int typeOf<T>::value = -1;
template <>
   const int typeOf<int_1s_t>::value = typeID_int_1s_t;
template <>
   const int typeOf<int_2s_t>::value = typeID_int_2s_t;
template <>
   const int typeOf<real_8_t>::value = typeID_real_8_t;
template <>
   const int typeOf<real_4_t>::value = typeID_real_4_t;
template <>
   const int typeOf<int_4s_t>::value = typeID_int_4s_t;
template <>
   const int typeOf<int_8s_t>::value = typeID_int_8s_t;

template <>
   class typeOf<complex_8_t> {
   public:
      typedef complex_8_t value_type;
      typedef complex_16_t tempvalue_type;
      static const int value;
   };
   const int typeOf<complex_8_t>::value = typeID_complex_8_t;

template <>
   class typeOf<complex_16_t> {
   public:
      typedef complex_16_t value_type;
      typedef complex_16_t tempvalue_type;
      static const int value;
   };
   const int typeOf<complex_16_t>::value = typeID_complex_16_t;

template <>
   const int typeOf<char*>::value = typeID_string;
template <>
   const int typeOf<int_2u_t>::value = typeID_int_2u_t;
template <>
   const int typeOf<int_4u_t>::value = typeID_int_4u_t;
template <>
   const int typeOf<int_8u_t>::value = typeID_int_8u_t;
template <>
   const int typeOf<int_1u_t>::value = typeID_int_1u_t;


//______________________________________________________________________________
   inline int type_sizeof (int n)
   {
      switch (n) {
         case typeOf<int_1s_t>::value:
            return sizeof (int_1s_t);
         case typeOf<int_2s_t>::value:
            return sizeof (int_2s_t);
         case typeOf<int_4s_t>::value:
            return sizeof (int_4s_t);
         case typeOf<int_8s_t>::value:
            return sizeof (int_8s_t);
         case typeOf<int_1u_t>::value:
            return sizeof (int_1u_t);
         case typeOf<int_2u_t>::value:
            return sizeof (int_2u_t);
         case typeOf<int_4u_t>::value:
            return sizeof (int_4u_t);
         case typeOf<int_8u_t>::value:
            return sizeof (int_8u_t);
         case typeOf<real_4_t>::value:
            return sizeof (real_4_t);
         case typeOf<real_8_t>::value:
            return sizeof (real_8_t);
         case typeOf<complex_8_t>::value:
            return sizeof (complex_8_t);
         case typeOf<complex_16_t>::value:
            return sizeof (complex_16_t);
         default:
            return 0;
      }
   }

//______________________________________________________________________________
   inline bool checkalign (const char* p, int dtype)
   {
      switch (dtype) {
         case typeOf<int_1s_t>::value:
         case typeOf<int_1u_t>::value:
         case typeOf<char*>::value:
            return true;
         case typeOf<int_2s_t>::value:
         case typeOf<int_2u_t>::value:
            return (((ptrdiff_t)p & 0x01) == 0);
         case typeOf<int_4s_t>::value:
         case typeOf<int_4u_t>::value:
         case typeOf<real_4_t>::value:
            return (((ptrdiff_t)p & 0x03) == 0);
         case typeOf<int_8s_t>::value:
         case typeOf<int_8u_t>::value:
         case typeOf<real_8_t>::value:
         case typeOf<complex_8_t>::value:
            return (((ptrdiff_t)p & 0x07) == 0);
         case typeOf<complex_16_t>::value:
            return (((ptrdiff_t)p & 0x0F) == 0);
         default:
            return false;
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// FrVectFComp / FrVectFExpand				                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   int FrVectFComp(int_2s_t *out, 
                  unsigned long *compL,
                  real_4_t *data,
                  int nData,
                  int bSize)
   /*--------------------------------------------------------------------*/
   {real_4_t *outF, delta, deltaMax, deltaMin, resolution, scale, halfScale,
         factor, rebuild, nnew;
      real_4_t resol[] = {1.,1.,2.,4.,6.5,14.5,30.5,62.,126.,254.,510.,1022.,2046.,
         4094.,8190.,16382.,32766.,65532.};
      int i, step;
   
   /*---- store the bloc size -----*/
   
      outF = (float *) out;
      outF[0] = data[0];
   
      /*--------- first compute the step size  --------------*/
   
      deltaMax = 0.;
      deltaMin = 0.;
      for(i=0; i<nData-1; i++)  
      {delta = data[i+1] - data[i];
         if(delta < deltaMin) deltaMin = delta;
         if(delta > deltaMax) deltaMax = delta;}
      if(-deltaMin > deltaMax) deltaMax = -deltaMin;
      if(deltaMax == 0.) 
      {outF[1] = 0;
         *compL = 8;
         return(0);}
   
      resolution = resol[bSize];
      factor = resolution/deltaMax;
      scale  = 1./factor;
      halfScale = .5*scale;
      outF[1] = scale; 
   
      /*---------- then store the data --------------------*/
   
      rebuild   = data[0];
   
      for(i=1; i<nData; i++)
      {step = (int)(factor*(data[i] - data[i-1]));
         nnew = scale*step + rebuild;
         if(nnew > data[i])
         {step -= (int)(factor*( halfScale + nnew - data[i]));}
         else {step -= (int)(factor*(-halfScale + nnew - data[i]));}
         rebuild += scale*step;
         out[i+4] = step;}
   
      out[4]  = bSize;
   
      *compL = 8 + 2*nData;
   
      return(0);}

/*--------------------------------------------------FrVectFExpand-----*/
   void FrVectFExpand(float *out, 
		      const short *data, 
		      int nData) {
   /*--------------------------------------------------------------------*/
      const float* dataF = reinterpret_cast<const float*>(data);
      out[0]       = dataF[0];
      float  scale = dataF[1];
   
      if (scale == 0.) {
         for(int i=1; i<nData; i++) {out[i] = 0.;}
      }
      else {
         for(int i=1; i<nData; i++) {out[i] = out[i-1] + scale*data[4+i];}
      }
      return;
   }


   template<typename T>
   int getNBits(T val) {
      int hi = 8*sizeof(T);
      int lo = 0;
      while (hi > lo+1) {
	 int mid = (hi + lo)/2;
	 if (val < T(1 << mid)) hi = mid;
	 else                   lo = mid;
      }
      return hi;
   }

/*--------------------------------------------------------------FrVectZComp--*/
   int FrVectZComp(int_2u_t *out, unsigned long *compL,
		   int_2s_t *data, long nData, int bSize)
   /*--------------------------------------------------------------------*/
   {
      int_2u_t uData, limit;
      int_2s_t max, pOut;
      long iIn, iOut, maxOut;
      int_2s_t wMax[] = {0,0,1,3,7,0xf,0x1f,0x3f,0x7f,0xff,0x1ff,
			 0x3ff,0x7ff,0xfff,0x1fff,0x3fff,0x7fff};

      maxOut = *compL/2;

      /*---- store the bloc size -----*/

      out[0] = bSize;
      iOut = 0;
      pOut = 16;

      /*--------- store the data --------------*/

      iIn = 0;
      while(iIn<nData) { /*-------- tune the size of the last bloc -----*/
	    if(iIn + bSize > nData) {bSize = nData - iIn;}

	    /*---- get the maximum amplitude --------------*/
      
	    max = 0;
	    for(long i=0; i<bSize; i++) {
	       if(data[iIn+i] == -32768) return(-1);/*the positive value is outside
                                             the 16 bits integer range ------*/
	       if(data[iIn+i] > 0) {
		  max = max |  data[iIn + i];
	       }
	       else {
		  max = max | -data[iIn + i];
	       }
	    }

	    /*----- determine the number of bits needed -----*/
	    /*        (2*(max-1)  < 2**nBits)                */
	    int nBits = getNBits(max);
	    cout << "nBits=" << nBits << " for max " << max << endl;
     
	    /*---- encode the data size - we store nBits - 1 in 4 bits---*/

	    out[iOut] = out[iOut] | ((nBits-1) << pOut);
	    if(pOut > 12) {
	       iOut ++;
	       if(iOut >= maxOut) return(-1);
	       pOut = pOut - 16;
	       out[iOut] = (nBits-1) >> - pOut;
	    }
	    pOut  = pOut +4;

	    /*----------------------- encode the data itself ------*/
	    if(nBits > 1) {
	       limit = 16 - nBits;
	       for(long i = 0; i<bSize; i++) {
		  uData = (unsigned short) (data[iIn+i] + wMax[nBits]);

		  out[iOut] = out[iOut] | (uData << pOut);
		  if(pOut > limit) {
		     iOut ++;
		     if(iOut >= maxOut) return(-1);
		     pOut = pOut - 16;
		     out[iOut] = uData >> -pOut;
		  }
		  pOut  = pOut + nBits;
	       }  
	    } 
	    /*----------------------------- increase pointer -------*/
	    iIn = iIn + bSize;
      }

      *compL = 2*(iOut+1);
      return(0);
   }

/*-------------------------------------------------------------FrVectZCompI--*/
   int FrVectZCompI(int_4u_t *out, unsigned long *compL, int_4s_t *data,
		    unsigned long nData, int bSize) {
/*---------------------------------------------------------------------------*/
      int_4s_t uData;
      int_4u_t max, pOut;
      unsigned long iIn, iOut, maxOut;
      int wMax[] = {0,0,1,3,7,
		    0xf      ,0x1f      ,0x3f      ,0x7f,
		    0xff     ,0x1ff     ,0x3ff     ,0x7ff,
		    0xfff    ,0x1fff    ,0x3fff    ,0x7fff,
		    0xffff   ,0x1ffff   ,0x3ffff   ,0x7ffff,
		    0xfffff  ,0x1fffff  ,0x3fffff  ,0x7fffff,
		    0xffffff ,0x1ffffff ,0x3ffffff ,0x7ffffff,
		    0xfffffff,0x1fffffff,0x3fffffff,0x7fffffff};

      maxOut = *compL/4;

      /*----------------------------store the bloc size--*/
      out[0] = bSize;
      iOut = 0;
      pOut = 16;

      /*------------------------------- store the data --*/
      iIn = 0;
      while(iIn<nData) {
	 /*------------ tune the size of the last bloc -----*/

	 if ( (unsigned long)(iIn + bSize) > nData) {bSize = nData - iIn;}

                          /*-------- get the maximum amplitude --------------*/
	 max = 0;

	 for(int i=0; i<bSize; i++) {
	    if(unsigned(data[iIn+i])==0x80000000) // the positive value is 
	       return -1;                         // > 2^31
	    if(data[iIn+i] > 0) {
	       max = max |  data[iIn + i];
	    }
	    else {
	       max = max | -data[iIn + i];
	    }
	 }

	 /*--- determine the number of bits needed - (2*(max-1)  < 2**nBits) */
	 int nBits = getNBits(max);
	 cout << "nBits=" << nBits << " for max " << max << endl;
     
	 /*---- encode the data size ---- we store nBits - 1 in 5 bits---*/

	 if(pOut != 32) out[iOut] = out[iOut] | ((nBits-1) << pOut);
	 if(pOut > 27) {
	    iOut ++;
	    if(iOut >= maxOut) return(-1);
	    pOut = pOut - 32;
	    out[iOut] = (nBits-1) >> - pOut;
	 }
	 pOut  = pOut + 5;
                                    /*--------- encode the data itself ------*/
	 if(nBits > 1) {
	    int_4u_t limit = 32 - nBits;
	    for(int i = 0; i<bSize; i++) {
	       uData = (unsigned int) (data[iIn+i] + wMax[nBits]);

	       if(pOut != 32) out[iOut] = out[iOut] | (uData << pOut);
	       if(pOut > limit) {
		  iOut ++;
		  if(iOut >= maxOut) return(-1);
		  pOut = pOut - 32;
		  out[iOut] = uData >> -pOut;
	       }
	       pOut  = pOut + nBits;
	    }
	 }

	 /*------------------------- increase pointer -------*/
	 iIn = iIn + bSize;
      }

      *compL = 4*(iOut+1);
      return(0);
   }




/*------------------------------------------------------------FrVectZExpand--*/
   void FrVectZExpand(int_2s_t *out, int_2u_t *data, unsigned long nData) {
/*---------------------------------------------------------------------------*/
      unsigned short nBits, pIn, uData, *buf;
      unsigned short wMax[] = {0,0,1,3,7,0xf,0x1f,0x3f,0x7f,0xff,0x1ff,
			       0x3ff,0x7ff,0xfff,0x1fff,0x3fff,0x7fff};
      unsigned short mask[] = {0,1,3,7,0xf,0x1f,0x3f,0x7f,0xff,0x1ff,
			       0x3ff,0x7ff,0xfff,0x1fff,0x3fff,0x7fff,0xffff};
      unsigned int iBuf;
      int_8u_t iIn, iOut, bSize;

#if 0
                            /*------------------- check which endian --------*/
      unsigned int iAlpha = 0x12345678;
       unsigned short* alpha = (unsigned short *) &iAlpha;
#endif
                            /*------------------ retrieve the bloc size -----*/
      bSize = data[0];
      iIn = 1;
      pIn = 0;
                            /*------------- retrieve the data  --------------*/

      buf = (unsigned short *) &iBuf; 
      iOut = 0;
      do {
	 /*-------- extract nBits ---(we check if data is in 1 or 2 words) ------*/
        
	 if(pIn <= 12) {
	    uData = data[iIn] >> pIn;
	    pIn = pIn + 4;
	 }
	 else {
	    uData = (data[iIn] >> pIn) | (data[iIn+1] << (16-pIn) ) ;
	    iIn++;
	    pIn = pIn - 12;
	 }

	 nBits = 1 + (mask[4] & uData);
	 if(nBits == 1) nBits = 0;

                        /*----------------------------extract data ----------*/

	 for(unsigned int i=0; i<bSize; i++) {
	    if(iOut >= nData) break;

	    if(pIn + nBits <= 16) {
	       uData = data[iIn] >> pIn;
	       pIn = pIn + nBits;
	    }
	    else {
	       uData = (data[iIn] >> pIn) | (data[iIn+1] << (16-pIn) ) ;
	       iIn++;
	       pIn = pIn + nBits - 16;
	    }

	    out[iOut] = (mask[nBits] & uData) - wMax[nBits];
	    iOut++;
	 }
      } while(iOut<nData);

      return;
   }

/*-----------------------------------------------------------FrVectZExpandI--*/
   void FrVectZExpandI(int_4s_t *out, int_4u_t *data, unsigned long nData) {
/*---------------------------------------------------------------------------*/

      unsigned short *alpha;
      unsigned int nBits, pIn, uData, *buf;
      unsigned int wMax[] = {0,0,1,3,7,
			     0xf      ,0x1f      ,0x3f      ,0x7f,
			     0xff     ,0x1ff     ,0x3ff     ,0x7ff,
			     0xfff    ,0x1fff    ,0x3fff    ,0x7fff,
			     0xffff   ,0x1ffff   ,0x3ffff   ,0x7ffff,
			     0xfffff  ,0x1fffff  ,0x3fffff  ,0x7fffff,
			     0xffffff ,0x1ffffff ,0x3ffffff ,0x7ffffff,
			     0xfffffff,0x1fffffff,0x3fffffff,0x7fffffff};
      unsigned int mask[] = {0,1,3,      7,       0xf,
			     0x1f      ,0x3f      ,0x7f,      0xff,
			     0x1ff     ,0x3ff     ,0x7ff,     0xfff,
			     0x1fff    ,0x3fff    ,0x7fff,    0xffff,
			     0x1ffff   ,0x3ffff   ,0x7ffff,   0xfffff,
			     0x1fffff  ,0x3fffff  ,0x7fffff,  0xffffff,
			     0x1ffffff ,0x3ffffff ,0x7ffffff, 0xfffffff,
			     0x1fffffff,0x3fffffff,0x7fffffff,0xffffffff};
      unsigned int iBuf, iAlpha;
      int_8u_t  iIn, iOut, bSize;
   /*----------------- check which indian --------*/
      iAlpha = 0x12345678;
      alpha = (unsigned short *) &iAlpha;

                              /*---------------- retrieve the bloc size -----*/
      bSize = data[0] & 0xffff;
      iIn = 0;
      pIn = 16;
      /*------------------ retrieve the data---------*/

      buf = (unsigned int *) &iBuf; 
      iOut = 0;
      do {  /*----- extract nBits -(we check if data is in 1 or 2 words) ----*/
        
	 if(pIn <= 27) {
	    uData = data[iIn] >> pIn;
	    pIn = pIn + 5;
	 }
	 else {
	    uData = (data[iIn] >> pIn) & mask[32-pIn];
	    iIn++;
	    uData += data[iIn] << (32-pIn);
	    pIn = pIn - 27;
	 }

	 nBits = 1 + (0x1f & uData);
	 if(nBits == 1) nBits = 0;

	 /*----------------------extract data ----------*/
	 for(unsigned int i=0; i<bSize; i++) {
	    if(iOut >= nData) break;

	    if(pIn + nBits <= 32) {
	       uData = data[iIn] >> pIn;
	       pIn = pIn + nBits;
	    }
	    else {
	       uData = (data[iIn] >> pIn) & mask[32-pIn];
	       iIn++;
	       uData += data[iIn] << (32-pIn);
	       pIn = pIn + nBits - 32;
	    }

	    out[iOut] = (mask[nBits] & uData) - wMax[nBits];
	    iOut++;
	 }
      } while(iOut<nData);

      return;
   }


/*-----------------------------------------------------------FrVectZExpandL--*/
   void FrVectZExpandL(int_8s_t *out, int_8u_t *data, unsigned long nData) {
/*---------------------------------------------------------------------------*/

      int_8u_t nBits, pIn, uData, *buf, iBuf, iIn, iOut, bSize;
      static int_8u_t wMax[65],  mask[65], iFirst = 0;
          
      /*-------------------------------------------------initialize arrays---*/
      if(iFirst == 0) {
	 iFirst = 1;
	 mask[0] = 0;
	 mask[1] = 1;
	 wMax[0] = 0;
	 wMax[1] = 0;
	 for(int i=2; i< 65; i++) {
	    mask[i] = (mask[i-1]<< 1) + 1;
	    wMax[i] = (wMax[i-1]<< 1) + 1;
	 }
      }
 
      /*--------------------------------------- retrieve the bloc size -----*/
      bSize = data[0] & 0xffff;
      iIn = 0;
      pIn = 16;

      /*------------------------------------------------retrieve the data---*/
      buf = (int_8u_t*) &iBuf; 
      iOut = 0;
      do {  /*---- extract nBits -(we check if data is in 1 or 2 words) --- */
	 if(pIn <= (64-6)) {
	    uData = data[iIn] >> pIn;
	    pIn = pIn + 6;
	 }
	 else {
	    uData = (data[iIn] >> pIn) & mask[64-pIn];
	    iIn++;
	    uData += data[iIn] << (64-pIn);
	    pIn = pIn - (64-6);
	 }

	 nBits = 1 + (0x3f & uData);
	 if(nBits == 1) nBits = 0;
                               /*----------------------extract data ----------*/
	 for(unsigned int i=0; i<bSize; i++) {
	    if(iOut >= nData) break;

	    if(pIn + nBits <= 64) {
	       uData = data[iIn] >> pIn;
	       pIn = pIn + nBits;
	    }
	    else {
	       uData = (data[iIn] >> pIn) & mask[64-pIn];
	       iIn++;
	       uData += data[iIn] << (64-pIn);
	       pIn = pIn + nBits - 64;
	    }

	    out[iOut] = (mask[nBits] & uData) - wMax[nBits];
	    iOut++;
	 }
      } while(iOut<nData);

      return;
   }

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// (de)compression					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   /// Compression Types.
   enum Compression {
   NOCOMPR     = 0,
   GZIP        = 1,
   DIFF        = 2,
   GZIP_DIFF   = 3,
   DIFF_ZERO_SUPPRESS_SHORT = 5,
   DIFF_ZERO_SUPPRESS_SHORT_DIFF_GZIP_OTHER = 6,
   DIFF_ZERO_SUPPRESS_INT = 8,
   DIFF_ZERO_SUPPRESS_LONG = 10
   };

//______________________________________________________________________________
template <typename T>
   inline void diffData (T* dest, const T* src, int_8u_t num, 
			 bool swapit = false)
   {
      if (num > 0) {
         if (swapit) {
            for (int_8u_t i = num - 1; i > 0; --i) {
               dest[i] = src[i] - src[i-1];
               swap (dest + i);
            }
            dest[0] = src[0];
            swap (dest);
         }
         else {
            for (int_8u_t i = num - 1; i > 0; --i) {
               dest[i] = src[i] - src[i-1];
            }
            dest[0] = src[0];
         }
      }
   }

//______________________________________________________________________________
template <typename T>
   inline void undiffData (T* data, int_8u_t num)
   {
      for (int_8u_t i = 1; i < num; i++) {
         data [i] += data [i-1];
      }
   }

//______________________________________________________________________________
template <typename T>
   bool compress (const T* data, int_8u_t num, char* p, int_8u_t& len, 
                 int compr)
   {
      bool swapit = (littleendian() ^ (compr >= 256));

      len = 0;
      if (data == 0) {
         return false;
      }
      if ((compr % 256 == DIFF_ZERO_SUPPRESS_SHORT) &&
         !((typeOf<T>::value == typeID_int_2s_t) || 
          (typeOf<T>::value == typeID_int_2u_t))) {
         compr -= (DIFF_ZERO_SUPPRESS_SHORT - NOCOMPR);
      }
      if (compr % 256 == DIFF_ZERO_SUPPRESS_SHORT_DIFF_GZIP_OTHER) {
         if ((typeOf<T>::value == typeID_int_2s_t) || 
            (typeOf<T>::value == typeID_int_2u_t)) {
            compr -= (DIFF_ZERO_SUPPRESS_SHORT_DIFF_GZIP_OTHER -
                     DIFF_ZERO_SUPPRESS_SHORT);
         }
         else {
            compr -= (DIFF_ZERO_SUPPRESS_SHORT_DIFF_GZIP_OTHER -
                     GZIP_DIFF);
         }
      }
      // compress if necessary
      switch (compr % 256) {
         case NOCOMPR:
            {
               len = num * sizeof (T);
               if (swapit) {
                  T* y = new (nothrow) T [num];
                  if (y == 0) {
                     return false;
                  }
                  memcpy (y, data, len);
                  T* x = y;
                  for (int_8u_t i = 0; i < num; i++, x++) {
                     swap (x);
                  }
                  memcpy (p, y, len);
                  delete [] y;
               }
               else {
                  memcpy (p, data, len);
               }
               break;
            }
         case GZIP:
            {
               int_8u_t size = num * sizeof (T);
               T* y = (T*) data;
               if (swapit) {
                  T* y = new (nothrow) T [num];
                  if (y == 0) {
                     return false;
                  }
                  memcpy (y, data, size);
                  T* x = y;
                  for (int_8u_t i = 0; i < num; i++, x++) {
                     swap (x);
                  }
               }
               unsigned long bufferSize = 
                  (unsigned long) (size * 1.001 + 13);
            #if HAVE_LIBZ_COMPRESS2
               int ret = ::compress2((int_1u_t*)p, &bufferSize, 
                                    (int_1u_t*)y, size, kGZIPlevel);
            #else
               int ret = ::compress ((int_1u_t*)p, &bufferSize, 
                                   (int_1u_t*)y, size);
            #endif
               len = bufferSize;
               if (swapit) {
                  delete [] y;
               }
               if (ret != Z_OK) {
                  cerr << "compression error " << ret << endl;
                  return false;
               }
               break;
            }
         case DIFF:
            {
               int_8u_t size = num * sizeof (T);
               T* y = new (nothrow) T [num];
               if (y == 0) {
                  return false;
               }
               diffData (y, data, num, swapit);
               memcpy (p, y, size);
               delete [] y;
               len = size;
               break;
            }
         case GZIP_DIFF:
            {
               int_8u_t size = num * sizeof (T);
               T* y = new (nothrow) T [num];
               if (y == 0) {
                  return false;
               }
               diffData (y, data, num, swapit);
               unsigned long bufferSize = 
                  (unsigned long) (size * 1.001 + 13);
            #if HAVE_LIBZ_COMPRESS2
               int ret = ::compress2((int_1u_t*)p, &bufferSize, 
                                    (int_1u_t*)y, size, kGZIPlevel);
            #else
               int ret = ::compress ((int_1u_t*)p, &bufferSize, 
                                    (int_1u_t*)y, size);
            #endif
               delete [] y;
               if (ret != Z_OK) {
                  cerr << "compression error " << ret << endl;
                  return false;
               }
               len = bufferSize;
               break;
            }
         case DIFF_ZERO_SUPPRESS_SHORT:
            {
               int_8u_t size = num * sizeof (T);
               T* y = new (nothrow) T [num];
               if (y == 0) {
                  return false;
               }
               diffData (y, data, num);
               T* z = new (nothrow) T [num];
               if (z == 0) {
                  delete [] y;
                  return false;
               }
               unsigned long bufferSize = 
                  (unsigned long) (size * 1.001 + 13);;           
               int ret = FrVectZComp ((unsigned short*)z, &bufferSize, 
                                    (short*)y, num, size);
               delete [] y;
               if (ret != 0) {
                  delete [] z;
                  return false;
               }
               if (swapit) {
                  for (int_8u_t i = 0; i < (int_8u_t)bufferSize; i++) {
                     swap (z + i);
                  }
               }
               memcpy (p, z, bufferSize);
               len = bufferSize;
               delete [] z;
               break;
            }
         case DIFF_ZERO_SUPPRESS_INT:
            {
               int_8u_t size = num * sizeof (T);
               T* y = new (nothrow) T [num];
               if (y == 0) {
                  return false;
               }
               diffData (y, data, num);
               T* z = new (nothrow) T [num];
               if (z == 0) {
                  delete [] y;
                  return false;
               }
               unsigned long bufferSize = 
                  (unsigned long) (size * 1.001 + 13);;           
               int ret = FrVectZCompI ((int_4u_t*)z, &bufferSize, 
				       (int_4s_t*)y, num, size);
               delete [] y;
               if (ret != 0) {
                  delete [] z;
                  return false;
               }
               if (swapit) {
                  for (int_8u_t i = 0; i < (int_8u_t)bufferSize; i++) {
                     swap (z + i);
                  }
               }
               memcpy (p, z, bufferSize);
               len = bufferSize;
               delete [] z;
               break;
            }
         case DIFF_ZERO_SUPPRESS_LONG:
            {
	       // let it fall into the default....
	       //
               //int_8u_t size = num * sizeof (T);
               //T* y = new (nothrow) T [num];
               //if (y == 0) {
               //   return false;
               //}
               //diffData (y, data, num);
               //T* z = new (nothrow) T [num];
               //if (z == 0) {
               //   delete [] y;
               //   return false;
               //}
               //int_8u_t bufferSize = int_8u_t(size * 1.001 + 13);           
               //int ret = FrVectZCompL ((int_8u_t*)z, &bufferSize, 
	       //			       (int_8s_t*)y, num, size);
               //delete [] y;
               //if (ret != 0) {
               //   delete [] z;
               //   return false;
               //}
               //if (swapit) {
               //   for (int_8u_t i = 0; i < bufferSize; i++) {
               //      swap (z + i);
	       //   }
	       //}
               //memcpy (p, z, bufferSize);
               //len = bufferSize;
               //delete [] z;
               //break;
            }
         default:
            {
               len = 0;
               return false;
            }
      }
      return true;
   }

//______________________________________________________________________________
template <typename T>
   bool decompress (T* data, int_8u_t num, const char* p, 
                   int_8u_t len, int compr)
   {
      bool swapit = (littleendian() ^ (compr >= 256));
      int  cmode  = compr % 256;

      // check multi-compression methods
      if (cmode == DIFF_ZERO_SUPPRESS_SHORT && sizeof(T) != sizeof(short) ) {
         cmode = NOCOMPR;
      }
      else if (cmode == DIFF_ZERO_SUPPRESS_SHORT_DIFF_GZIP_OTHER) {
         if (sizeof(T) == sizeof(short)) {
            cmode = DIFF_ZERO_SUPPRESS_SHORT;
         }
         else {
            cmode = GZIP_DIFF;
         }
      }

      // if not compressed, copy and swap if necessary
      if (cmode == NOCOMPR) {
         if (len !=  int_8u_t(num * sizeof (T))) {
            return false;
         }
         memcpy (data, p, len);
         if (swapit) {
            for (int_8u_t i = 0; i < num; i++) {
               swap (data + i);
            }
         }
         return true;
      }
   
      // check for zero suppress
      else if (cmode == DIFF_ZERO_SUPPRESS_SHORT) {
         T* y = new (nothrow) T [num];
         if (y == 0) {
            return false;
         }
         memcpy (y, p, len);
         if (swapit) {
            for (int_8u_t i = 0; i < num; i++) {
               swap (y + i);
            }
            swapit = false;
         }
         FrVectZExpand ((int_2s_t*) data, (int_2u_t*) y, num);
         delete [] y;
      }
   
      // check for zero suppress
      else if (cmode == DIFF_ZERO_SUPPRESS_INT) {
         T* y = new (nothrow) T [num];
         if (y == 0) {
            return false;
         }
         memcpy (y, p, len);
         if (swapit) {
            for (int_8u_t i = 0; i < num; i++) {
               swap (y + i);
            }
            swapit = false;
         }
         FrVectZExpandI((int_4s_t*) data, (int_4u_t*) y, num);
         delete [] y;
      }
   
      // check for zero suppress
      else if (cmode == DIFF_ZERO_SUPPRESS_LONG) {
         T* y = new (nothrow) T [num];
         if (y == 0) {
            return false;
         }
         memcpy (y, p, len);
         if (swapit) {
            for (int_8u_t i = 0; i < num; i++) {
               swap (y + i);
            }
            swapit = false;
         }
         FrVectZExpandL((int_8s_t*) data, (int_8u_t*) y, num);
         delete [] y;
      }
      
      // check for gzip
      else if ((cmode == GZIP) || (cmode == GZIP_DIFF)) {
         unsigned long length = num * sizeof (T);
         int error = 
            ::uncompress ((int_1u_t*)data, &length, (int_1u_t*)p, len);
         if ((error != Z_OK) || (length != num * sizeof (T))) {
            return false;
         }
      }
      
      // check for diff
      else if (cmode == DIFF) {
         memcpy (data, p, len);
      }
      
      // error
      else {
         return false;
      }
   
      // swap it necessary
      if (swapit) {
         for (int_8u_t i = 0; i < num; i++) {
            swap (data + i);
         }
      }
   
      // check for integration
      if ((cmode == DIFF) || 
	  (cmode == GZIP_DIFF) ||
	  (cmode == DIFF_ZERO_SUPPRESS_SHORT)) {
         undiffData (data, num);
      }
      else if (cmode == DIFF_ZERO_SUPPRESS_INT) {
         undiffData (reinterpret_cast<int_4s_t*>(data), num);
      }
      else if (cmode == DIFF_ZERO_SUPPRESS_LONG) {
	 undiffData (reinterpret_cast<int_8s_t*>(data), num);
      }
      return true;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// data conversion					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

template <typename S, typename T>
   inline void convertdata (S* dest, const T* src, int_8u_t N)
   {
      if ((N <= 0) || (dest == 0) || (src == 0)) {
         return;
      }
      // memcpy if the same type
      if ((typeOf<S>::value != -1) && 
         (typeOf<S>::value == typeOf<T>::value)) {
         memcpy (dest, src, N * sizeof (T));
      }
      // element by element otherwise
      else {
         for (int_8u_t k = 0; k < N; k++) {
            dest[k] = S(src[k]);
         }
      }
   }

//______________________________________________________________________________
template <typename S, typename T>
   inline void convertdata (S* dest, const T* src, int_8u_t N, 
                     int dec, int ipol)
   {
      if ((N <= 0) || (dest == 0) || (src == 0)) {
         return;
      }
      // no decimation/interpolation
      if ((dec == 1) && (ipol == 1)) {
         for (int_8u_t k = 0; k < N; k++) {
            dest[k] = S(src[k]);
         }
      }
      // decimation (simple average)
      else if (dec > 1) {
         typename typeOf<S>::tempvalue_type x;
         for (int_8u_t k = 0; k < N; k++, dest++) {
            x = (typename typeOf<S>::tempvalue_type) (0);
            for (int l = 0; l < dec; l++, src++) {
               x += (typename typeOf<S>::tempvalue_type)(*src);
            }
            *dest = S(x / (real_8_t)dec);
         }
      }
      // interpolation (repeat value)
      else if (ipol > 1) {
         int_8u_t M = N / ipol;
         for (int_8u_t k = 0; k < M; k++, src++) {
            for (int l = 0; l < ipol; l++, dest++) {
               *dest = S(*src);
            }
         }
      }
   }

//______________________________________________________________________________
template <typename S, typename T>
   inline void convertdata_real_cmlx (S* dest, const T* src, int_8u_t N)
   {
      if ((N <= 0) || (dest == 0) || (src == 0)) {
         return;
      }
      for (int_8u_t k = 0; k < N; ++k) {
         dest[2*k] = static_cast <S> (src[k].real());
         dest[2*k+1] = static_cast <S> (src[k].imag());
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// strings						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   int readString (const char* p, bool swapit, char* s, int max)
   {
      int_2u_t	len;
      memcpy (&len, p, 2);
      if (swapit) {
         swap (&len);
      }
      if (len < max) {
         memcpy (s, p + 2, len);
      }
      else {
         memcpy (s, p + 2, max);
         s[max-1] = 0;
      }
      return len + 2;
   }

//______________________________________________________________________________
   int writeString (char* p, bool swapit, const char* s, int pad = 0)
   {
      int_2u_t len = strlen (s) + 1;
      int_2u_t l = len + pad;
      if (swapit) {
         swap (&l);
      }
      memcpy (p, &l, 2);
      memcpy (p + 2, s, len);
      if (pad > 0) memset (p + 2 + len, 0, pad);
      return len + pad + 2;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// fileheader_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   fileheader_t::fileheader_t()
   {
      memset (this, 0, sizeof (fileheader_t));
   }


//______________________________________________________________________________
   void fileheader_t::init (int version)
   {
      strcpy ((char*)fIGWD, "IGWD");
      fVersion = version;
      fMinorVersion = 0;
      fSize_int2 = sizeof (int_2u_t);
      fSize_int4 = sizeof (int_4u_t);
      fSize_int8 = sizeof (int_8u_t);
      fSize_real4 = sizeof (real_4_t);
      fSize_real8 = sizeof (real_8_t);
      fByteOrder2 = 0x1234;
      fByteOrder4 = 0x12345678;
      fByteOrder8 = 0x0123456789ABCDEFULL;
      fPi4 = acos(-1.0);
      fPi8 = acos(-1.0);
      if (version < 8) {
	 fAlpha[0] = 'A';
	 fAlpha[1] = 'Z';
      } else {
	 fAlpha[0] = 2;  // library number
	 fAlpha[1] = 0;  // checksum type
      }
   }

//______________________________________________________________________________
   long fileheader_t::read (const char* p)
   {
      memcpy (fIGWD, p, 14);
      memcpy (&fByteOrder4, p + 14, sizeof (int_4u_t));
      memcpy (&fByteOrder8, p + 18, sizeof (int_8u_t));
      memcpy (&fPi4, p + 26, sizeof (real_4_t));
      memcpy (&fPi8, p + 30, sizeof (real_8_t));
      memcpy (fAlpha, p + 38, sizeof (2));
      return 40;
   }


//______________________________________________________________________________
   long fileheader_t::write (char* p, bool swapit) const
   {
      fileheader_t* fh;
      if (swapit) {
         fh = new fileheader_t (*this);
         swap (&fh->fByteOrder2);
         swap (&fh->fByteOrder4);
         swap (&fh->fByteOrder8);
         swap (&fh->fPi4);
         swap (&fh->fPi8);
      }   
      else {
         fh = (fileheader_t*) this;
      }
      memcpy (p, fh->fIGWD, 14);
      memcpy (p + 14, &fh->fByteOrder4, sizeof (int_4u_t));
      memcpy (p + 18, &fh->fByteOrder8, sizeof (int_8u_t));
      memcpy (p + 26, &fh->fPi4, sizeof (real_4_t));
      memcpy (p + 30, &fh->fPi8, sizeof (real_8_t));
      memcpy (p + 38, fh->fAlpha, sizeof (2));
      if (swapit) {
         delete fh;
      }
      return 40;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// generic_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   generic_t::generic_t()
   {
      memset (this, 0, sizeof (generic_t));
   }

//______________________________________________________________________________
   long generic_t::read (int version, const char* p, bool swapit, bool skipSH)
   {
      const char* next = p;
      int len = 0;
      do {
         // version 4
         if (version < 6) {
            struct {
               int_4u_t len;
               int_2u_t clas;
               int_2u_t instance;
            } dummy;
         
            memcpy (&dummy, next, 8);
            if (swapit) {
               swap (&dummy.len);
               swap (&dummy.clas);
               swap (&dummy.instance);
            }
            fLen = dummy.len;
	    fCheck = 0;
            fClass = dummy.clas;
            fInstance = dummy.instance;
         }
         // version 6
         else if (version < 8) {
            memcpy (&fLen, next, 8);
	    fCheck = 0;
            memcpy (&fClass, next + 8, 2);
            memcpy (&fInstance, next + 10, 4);
            if (swapit) {
               swap (&fLen);
               swap (&fClass);
               swap (&fInstance);
            }
         }
         // version 8
         else {
	    const char* pp = next;
	    pp += swapin(pp, fLen, swapit);
	    fCheck = *pp++;
	    fClass = *pp++;
	    pp += swapin(pp, fInstance, swapit);
         }
         len += fLen;
         next += fLen;
      } while (skipSH && (fClass <= 2));
      return (len - fLen + (version < 6 ? 8 : 14));
   }

//______________________________________________________________________________
   long generic_t::write (int version, char* p, bool swapit) const
   {
      if (version < 6) {
         struct {
            int_4u_t len;
            int_2u_t clas;
            int_2u_t instance;
         } dummy = {int_4u_t(fLen), int_2u_t(fClass), int_2u_t(fInstance)};
         if (swapit) {
            swap (&dummy.len);
            swap (&dummy.clas);
            swap (&dummy.instance);
         }
         memcpy (p, &dummy.len, 8);
         return 8;
      }
      else if (version < 8) {
         if (swapit) {
            generic_t g = *this;
            swap (&g.fLen);
            swap (&g.fClass);
            swap (&g.fInstance);
            memcpy (p, &g.fLen, 8);
            memcpy (p + 8, &g.fClass, 2);
            memcpy (p + 10, &g.fInstance, 4);
         }
         else {
            memcpy (p, &fLen, 8);
            memcpy (p + 8, &fClass, 2);
            memcpy (p + 10, &fInstance, 4);
         }
         return 14;
      }
      else {
	 p += swapout(fLen, p, swapit);
	 *p++ = fCheck;
	 *p++ = fClass;
	 p += swapout(fInstance, p, swapit);
	 return 14;
      }
   }

//______________________________________________________________________________
   long 
   generic_t::fixlength (int version, char* p, int_8u_t len, bool swapit) const
   {
      const_cast<int_8u_t&>(fLen) = len;
      if (version < 6) {
         int_4u_t dummy = len;
         if (swapit) swap (&dummy);
         memcpy (p, &dummy, 4);
      }
      else if (version < 8) {
         if (swapit) swap (&len);
         memcpy (p, &len, 8);
      } 
      else {
	 int_4u_t chkSum(0);
	 const_cast<int_8u_t&>(fLen) += sizeof(chkSum);
	 swapout(fLen, p, swapit);

	 //-----------------------------  perform checksum calculation here.
	 // if (fCheck != 0) chkSum = calculate_checkSum(p, length, swapit);
	 swapout(chkSum, p + len, swapit);
      }
      return fLen;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// dict_element_t					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   dict_element_t::dict_element_t()
   {
      memset (this, 0, sizeof (dict_element_t));
   }

//______________________________________________________________________________
   long dict_element_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit);
      if (fClass != 2) {
         return 0;
      }
      pp += readString (pp, swapit, fName, maxName);
      pp += readString (pp, swapit, fType, maxName);
      pp += readString (pp, swapit, fComment, maxName);
      if (version >= 8) {
	 pp += swapin(pp, fChkSum, swapit);
      }
      return long(pp - p);
   }

//______________________________________________________________________________
   long dict_element_t::write (int version, char* p, bool swapit) const
   {
      char* pp = p + generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, fName);
      pp += writeString (pp, swapit, fType);
      pp += writeString (pp, swapit, fComment);
      return fixlength (version, p, long(pp - p), swapit);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// dict_header_t					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   dict_header_t::dict_header_t()
   {
      memset (this, 0, sizeof (dict_header_t));
   }

//______________________________________________________________________________
   dict_header_t::dict_header_t (const dict_header_t& dicth)
   {
      memset (this, 0, sizeof (dict_header_t));
      *this = dicth;
   }

//______________________________________________________________________________
   dict_header_t::~dict_header_t()
   {
      if (fElements) delete [] fElements;
   }

//______________________________________________________________________________
   dict_header_t& dict_header_t::operator= (const dict_header_t& dicth)
   {  
      if (this != &dicth) {
         *(generic_t*)this = dicth;
         save_strncpy (fName, dicth.fName, maxName);
         fClassNum = dicth.fClassNum;
         save_strncpy (fComment, dicth.fComment, maxName);
         fElementNum = dicth.fElementNum;
         if (fElements) delete [] fElements;
         fElements = new dict_element_t[fElementNum];
         for (int i = 0; i < fElementNum; ++i) {
            fElements[i] = dicth.fElements[i];
         }
      }
      return *this;
   }

//______________________________________________________________________________
   long dict_header_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit);
      if (fClass != 1) {
         return 0;
      }
      pp += readString (pp, swapit, fName, maxName);
      pp += swapin(pp, fClassNum, swapit);
      pp += readString (pp, swapit, fComment, maxName);
      if (version >= 8) pp += swapin(pp, fChkSum, swapit);

      fElementNum = 0;
      if (fElements) delete [] fElements;
      fElements = new (nothrow) dict_element_t [maxDictElements];
      int len;
      do {
         len = fElements[fElementNum].read (version, pp, swapit);
         if (len > 0) fElementNum++;
         pp += len;
      } while ((len > 0) && (fElementNum < maxDictElements));
      return (int)(pp - p);
   }

//______________________________________________________________________________
   long dict_header_t::write (int version, char* p, bool swapit) const
   {
      char* pp = p + generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, fName);
      pp += swapout(fClassNum, pp, swapit);
      pp += writeString (pp, swapit, fComment);
      pp = p + fixlength (version, p, long(pp - p), swapit);

      for (int i = 0; i < fElementNum; i++) {
         pp += fElements[i].write (version, pp, swapit);
      }
      return long(pp - p);
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// dict_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   void dict_t::standard (int version)
   {
      fDictNum = 0;
      int instanceSE = 0;
      int kSH;
      const SH_def_t* cSH;
      if (version < 6) {
	 kSH = kSH_V4;
	 cSH = cSH_V4;
      } else if (version < 8) {
	 kSH = kSH_V6;
	 cSH = cSH_V6;
      } else {
	 kSH = kSH_V8;
	 cSH = cSH_V8;
      }

      for (int i = 0; i < kSH; i++) {
         fDict[i].fLen = 0;
         fDict[i].fClass = 1;
         fDict[i].fInstance = i;
         save_strncpy (fDict[i].fName, cSH[i].fName, maxName);
         fDict[i].fClassNum = cSH[i].fClassNum;
         save_strncpy (fDict[i].fComment, cSH[i].fComment, maxName);
         if (fDict[i].fElements) delete [] fDict[i].fElements;
         fDict[i].fElements = 
            new (nothrow) dict_element_t [cSH[i].fElementNum];
         for (int j = 0; j < cSH[i].fElementNum; j++) {
            fDict[i].fElements[j].fLen = 0;
            fDict[i].fElements[j].fClass = 2;
            fDict[i].fElements[j].fInstance = instanceSE;
            if (version < 6) {
               save_strncpy (fDict[i].fElements[j].fName, cSE_V4[i][j].fName, 
                       maxName);
               save_strncpy (fDict[i].fElements[j].fType, cSE_V4[i][j].fType, 
                       maxName);
               save_strncpy (fDict[i].fElements[j].fComment, cSE_V4[i][j].fComment, 
                       maxName);
            }
            else {
               save_strncpy (fDict[i].fElements[j].fName, cSE_V6[i][j].fName, 
                       maxName);
               save_strncpy (fDict[i].fElements[j].fType, cSE_V6[i][j].fType, 
                       maxName);
               save_strncpy (fDict[i].fElements[j].fComment, cSE_V6[i][j].fComment, 
                       maxName);
            }
            instanceSE++;
         }
         fDict[i].fElementNum = cSH[i].fElementNum;
      }
      fDictNum = kSH;
   }

//______________________________________________________________________________
   long dict_t::write (int version, char* p, bool swap) const
   {
      char* pp = p;
      for (int i = 0; i < fDictNum; i++) {
         pp += fDict[i].write (version, pp, swap);
      }
      return (pp - p);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// frameheader_t					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   frameheader_t::frameheader_t()
   {
      memset (this, 0, sizeof (frameheader_t));
   }

//______________________________________________________________________________
   long frameheader_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      pp += readString (pp, swapit, fName, maxName);
      memcpy (&fRun, pp, 22);
      pp += 22;
      if (version == 4) {
         memcpy (&fLocalTime, pp, 4);
         pp += 4;
      }
      else {
         fLocalTime = 0;
      }
      memcpy (&fFrameLen, pp, 8);
      pp += 8;
      for (int i = 0; i < 14; ++i) {
         if ((version > 4) && (i == 7)) {
            fDir[i] = ptr_struct();
         }
         else {
            pp += fDir[i].read (version, pp, swapit);
         }
      }
      if (swapit) {
         swap (&fRun);
         swap (&fFrame);
         swap (&fDataQual);
         swap (&fGTimeS);
         swap (&fGTimeN);
         swap (&fULeapS);
         swap (&fLocalTime);
         swap (&fFrameLen);
      }
      pp += swapin(pp, fChkSum, swapit);
      return long(pp - p);
   }

//______________________________________________________________________________
   long frameheader_t::write (int version, char* p, bool swapit) const
   {
      frameheader_t* fr;
      if (swapit) {
         fr = new (nothrow) frameheader_t (*this);
         swap (&fr->fRun);
         swap (&fr->fFrame);
         swap (&fr->fDataQual);
         swap (&fr->fGTimeS);
         swap (&fr->fGTimeN);
         swap (&fr->fULeapS);
         swap (&fr->fLocalTime);
         swap (&fr->fFrameLen);
      }
      else {
         fr = (frameheader_t*) this;
      }
      char* pp = p + fr->generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, fr->fName);
      memcpy (pp, &fr->fRun, 22);
      pp += 22;
      if (version <= 4) {
         memcpy (pp, &fr->fLocalTime, 4);
         pp += 4;
      }
      memcpy (pp, &fr->fFrameLen, 8);
      pp += 8;
      for (int i = 0; i < 14; ++i) {
         if ((version > 4) && (i == 7)) {
            ;
         }
         else {
            pp += fDir[i].write (version, pp, swapit);
         }
      }
      if (swapit) {
         delete fr;
      }
      return fixlength (version, p, long(pp - p), swapit);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// detector_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   detector_t::detector_t()
   {
      memset (this, 0, sizeof (detector_t));
   }

//______________________________________________________________________________
   long detector_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      pp += readString (pp, swapit, fName, maxName);
      fPrefix[0] = 0;
      if (version <= 4) {
         memcpy (&fLongitudeD, pp, 7 * 4);
         pp += 7 * 4;
         for (int i = 0; i < 2; i++) {
            pp += fMore[i].read (version, pp, swapit);
         }
         if (swapit) {
            swap (&fLongitudeD);
            swap (&fLongitudeM);
            swap (&fLongitudeS);
            swap (&fLatitudeD);
            swap (&fLatitudeM);
            swap (&fLatitudeS);
            swap (&fElevation);
            swap (&fArmXAzimuth);
            swap (&fArmYAzimuth);
         }
         fArmXAltitude = 0;
         fArmYAltitude = 0;
         fArmXMidpoint = 0;
         fArmYMidpoint = 0;
         fLocalTime = 0;
         fDataQuality = 0;
         strcpy (fQaBitList, "");
         fMore[2] = ptr_struct();
      }
      else {
         if (version >= 6) {
            memcpy (fPrefix, pp, 2);
            pp += 2; 
            fPrefix[3] = 0;
         }
         real_8_t temp[2];
         memcpy (temp, pp, 2 * 8);
         pp += 2 * 8;
         memcpy (&fElevation, pp, 8 * 4);
         pp += 8 * 4;
         if (version >= 6) {
            fDataQuality = 0;
            strcpy (fQaBitList, "");
         }
         else {
            memcpy (&fDataQuality, pp, 4);
            pp += 4;
            pp += readString (pp, swapit, fQaBitList, maxName);
         }
         for (int i = 0; i < 3; i++) {
            pp += fMore[i].read (version, pp, swapit);
         }
         if (swapit) {
            for (int i = 0; i < 2; i++) {
               swap (temp + i);
            }
            swap (&fElevation);
            swap (&fArmXAzimuth);
            swap (&fArmYAzimuth);
            swap (&fArmXAltitude);
            swap (&fArmYAltitude);
            swap (&fArmXMidpoint);
            swap (&fArmYMidpoint);
            swap (&fLocalTime);
            swap (&fDataQuality);
         }
         set_longitude (temp[0]);
         set_latitude (temp[1]);
	 if (version >= 8) pp += swapin(pp, fChkSum, swapit);
      }
      return long(pp - p);
   }

//______________________________________________________________________________
   long detector_t::write (int version, char* p, bool swapit) const
   {
      detector_t* det;
      real_8_t longitude = get_longitude();
      real_8_t latitude = get_latitude();
      if (swapit) {
         det = new (nothrow) detector_t (*this);
         swap (&det->fLongitudeD);
         swap (&det->fLongitudeM);
         swap (&det->fLongitudeS);
         swap (&det->fLatitudeD);
         swap (&det->fLatitudeM);
         swap (&det->fLatitudeS);
         swap (&det->fElevation);
         swap (&det->fArmXAzimuth);
         swap (&det->fArmYAzimuth);
         swap (&det->fArmXAltitude);
         swap (&det->fArmYAltitude);
         swap (&det->fArmXMidpoint);
         swap (&det->fArmYMidpoint);
         swap (&det->fLocalTime);
         swap (&longitude);
         swap (&latitude);
      }
      else {
         det = (detector_t*) this;
      }
      char* pp = p + generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, fName);
      if (version < 6) {
         memcpy (pp, &det->fLongitudeD, 7 * 4);
         pp += 7 * 4;
         for (int i = 0; i < 2; i++) {
            pp += fMore[i].write (version, pp, swapit);
         }
      }
      else {
         memcpy (pp, &fPrefix, 2);
         pp += 2;
         memcpy (pp, &longitude, 8);
         pp += 8;
         memcpy (pp, &latitude, 8);
         pp += 8;
         memcpy (pp, &det->fElevation, 8*4);
         pp += 8*4;
         for (int i = 0; i < 3; i++) {
            pp += fMore[i].write (version, pp, swapit);
         }
      }
      if (swapit) {
         delete det;
      }
      return fixlength (version, p, long(pp - p), swapit);
   }

//______________________________________________________________________________
   real_8_t detector_t::get_longitude() const
   {
      int sign = fLongitudeD > 0 ? +1 : -1;
      return sign * pi / 180. *
         (fabs ((real_8_t)fLongitudeD) + (real_8_t)fLongitudeM / 60. +
         (real_8_t)fLongitudeS / 3600.);
   }

//______________________________________________________________________________
   void detector_t::set_longitude (real_8_t lon)
   {
      int sign = lon > 0 ? +1 : -1;
      lon = fabs (180. / pi * lon);
      fLongitudeD = (int)lon;
      fLongitudeM = (int)(60. * (lon - fLongitudeD));
      fLongitudeS = (3600. * lon - 3600 * fLongitudeD - 60 * fLongitudeM);
      fLongitudeD *= sign;
   }

//______________________________________________________________________________
   real_8_t detector_t::get_latitude() const
   {
      int sign = fLatitudeD > 0 ? +1 : -1;
      return  sign * pi / 180. *
         (fabs ((real_8_t)fLatitudeD) + (real_8_t)fLatitudeM / 60. +
         (real_8_t)fLatitudeS / 3600.);
   }

//______________________________________________________________________________
   void detector_t::set_latitude (real_8_t lat)
   {
      int sign = lat > 0 ? +1 : -1;
      lat = fabs (180. / pi * lat);
      fLatitudeD = (int)lat;
      fLatitudeM = (int)(60. * (lat - fLatitudeD));
      fLatitudeS = (3600. * lat - 3600 * fLatitudeD - 60 * fLatitudeM);
      fLatitudeD *= sign;
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// hist_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   hist_t::hist_t()
   {
      memset (this, 0, sizeof (hist_t));
   }

//______________________________________________________________________________
   long hist_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      pp += readString (pp, swapit, fName, maxName);
      memcpy (&fTime, pp, 4);
      pp += 4;
      pp += readString (pp, swapit, fComment, maxName);
      pp += fNext.read (version, pp, swapit);
      if (swapit) {
         swap (&fTime);
      }
      if (version >= 8) pp += swapin(pp, fChkSum, swapit);
      return long(pp - p);
   }

//______________________________________________________________________________
   long hist_t::write (int version, char* p, bool swapit) const
   {
      hist_t* hist;
      if (swapit) {
         hist = new (nothrow) hist_t (*this);
         swap (&hist->fTime);
      }
      else {
         hist = (hist_t*) this;
      }
      char* pp = p + generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, hist->fName);
      memcpy (pp, &hist->fTime, 4);
      pp += 4;
      pp += writeString (pp, swapit, hist->fComment);
      // memcpy (pp, &hist->fNext, 4);
      // pp += 4;
      pp += fNext.write (version, pp, swapit);
      if (swapit) {
         delete hist;
      }
      return fixlength (version, p, long(pp - p), swapit);
   }


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// rawdata_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   rawdata_t::rawdata_t()
   {
      memset (this, 0, sizeof (rawdata_t));
   }

//______________________________________________________________________________
   long rawdata_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      pp += readString (pp, swapit, fName, maxName);
      // memcpy (&fData, pp, 5 * 4);
      // pp += 5 * 4;
      // if (swapit) {
         // for (int i = 0; i < 5; i++) {
            // swap (&fData[i].fClass);
            // swap (&fData[i].fInstance);
         // }
      // }
      for (int i = 0; i < 5; i++) {
         pp += fData[i].read (version, pp, swapit);
      }
      return (int) (pp - p);
   }

//______________________________________________________________________________
   long rawdata_t::write (int version, char* p, bool swapit) const
   {
      // rawdata_t* raw;
      // if (swapit) {
         // raw = new (nothrow) rawdata_t (*this);
         // for (int i = 0; i < 2; i++) {
            // swap (&raw->fData[i].fClass);
            // swap (&raw->fData[i].fInstance);
         // }
      // }
      // else {
         // raw = (rawdata_t*) this;
      // }
      char* pp = p + generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, fName);
      // memcpy (pp, &raw->fData, 5 * 4);
      // pp += 5 * 4;
      // if (swapit) {
         // delete raw;
      // }
      for (int i = 0; i < 5; i++) {
         pp += fData[i].write (version, pp, swapit);
      }
      return fixlength (version, p, long(pp - p), swapit);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// adcdata_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   adcdata_t::adcdata_t () 
   {
      memset (this, 0, sizeof (adcdata_t));
   }

//______________________________________________________________________________
   long 
   adcdata_t::read (int version, datatype_t dtype, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      switch (dtype) {
         case kAdcData: 
            {
               pp += readString (pp, swapit, fName, maxName);
               pp += readString (pp, swapit, fComment, maxName);
               memcpy (&fChannelGroup, pp, 5 * 4);
               pp += 5 * 4;
               pp += readString (pp, swapit, fUnit, maxName);
               memcpy (&fSampleRate, pp, 8);
               pp += 8;
               if (version < 6) {
                  memcpy (&fTimeOffsetS, pp, 2 * 4);
                  pp += 2 * 4;
               }
               else {
                  real_8_t timeofs;
                  memcpy (&timeofs, pp, 8);
                  if (swapit) swap (&timeofs);
                  fTimeOffsetS = (int_4s_t)timeofs;
                  fTimeOffsetN = (int_4u_t)
                     (1E9*(fabs(timeofs)-abs(fTimeOffsetS)));
                  if (swapit) {
                     swap (&fTimeOffsetS);
                     swap (&fTimeOffsetN);
                  }
                  pp += 8;
               }
               memcpy (&fFShift, pp, 8);
               pp += 8;
               if (version > 4) {
                  memcpy (&fPhase, pp, 4);
                  pp += 4;
               }
               else {
                  fPhase = 0;
               }
               memcpy (&fDataValid, pp, 2);
               pp += 2;
               pp += fData[0].read (version, pp, swapit);
               pp += fData[1].read (version, pp, swapit);
               pp += fData[2].read (version, pp, swapit);
               fData[3] = ptr_struct();
               break;
            }
         case kProcData:
            {
               pp += readString (pp, swapit, fName, maxName);
               pp += readString (pp, swapit, fComment, maxName);
               memset (&fChannelGroup, 0, 5 * 4);
               fUnit[0] = 0;
               if (version < 6) {
                  memcpy (&fSampleRate, pp, 8);
                  pp += 8;
                  memcpy (&fTimeOffsetS, pp, 2 * 4);
                  pp += 2 * 4;
                  memcpy (&fFShift, pp, 8);
                  pp += 8;
                  if (version > 4) {
                     memcpy (&fPhase, pp, 4);
                     pp += 4;
                  }
                  else {
                     fPhase = 0;
                  }
               }
               else {
                  pp += 4; // skip type/subtype
                  real_8_t timeofs;
                  memcpy (&timeofs, pp, 8);
                  if (swapit) swap (&timeofs);
                  fTimeOffsetS = (int_4s_t)timeofs;
                  fTimeOffsetN = (int_4u_t)
                     (1E9*(fabs(timeofs)-abs(fTimeOffsetS)));
                  if (swapit) {
                     swap (&fTimeOffsetS);
                     swap (&fTimeOffsetN);
                  }
                  pp += 8;
                  pp += 8; // skip tRange
                  memcpy (&fFShift, pp, 8);
                  pp += 8;
                  memcpy (&fPhase, pp, 4);
                  pp += 4;
                  pp += 2*8; // skip fRange, BW
                  int_2u_t auxN;
                  memcpy (&auxN, pp, 2);
                  pp += 2;
                  if (swapit) swap (&auxN);
                  pp += auxN * 8; // skip aux parameters
                  char name[2];
                  for (int i = 0; i < auxN; ++i) {
                     pp += readString (pp, swapit, name, 2);
                  } // skip aux names
               }
               fDataValid = 0;
               pp += fData[0].read (version, pp, swapit);
               pp += fData[1].read (version, pp, swapit);
               if (version < 6) {
                  pp += fData[3].read (version, pp, swapit);
                  pp += fData[2].read (version, pp, swapit);
               }
               else {
                  pp += fData[3].read (version, pp, swapit);
                  ptr_struct dummy;
                  pp += dummy.read (version, pp, swapit);
                  pp += fData[2].read (version, pp, swapit);
               }
               break;
            }
         case kSimData:
            {
               pp += readString (pp, swapit, fName, maxName);
               pp += readString (pp, swapit, fComment, maxName);
               memset (&fChannelGroup, 0, 5 * 4 + 1);
               fUnit[0] = 0;

	       if (version < 7) {
		  real_4_t rate;
		  memcpy (&rate, pp, sizeof(rate));
		  if (swapit) {
		     swap (&rate);
		     fSampleRate = rate;
		     swap (&fSampleRate); //revert the swapping for later
		  }
		  else {
		     fSampleRate = rate;
		  }
		  pp += 4;
	       }
	       else {
		  memcpy (&fSampleRate, pp, sizeof(fSampleRate));
		  pp += sizeof(fSampleRate);
	       }

               if (version < 6) {
                  memset (&fTimeOffsetS, 0, 2 * 4);
               }
               else {
                  real_8_t timeofs;
                  memcpy (&timeofs, pp, 8);
                  if (swapit) swap (&timeofs);
                  fTimeOffsetS = (int_4s_t)timeofs;
                  fTimeOffsetN = (int_4u_t)
                     (1E9*(fabs(timeofs)-abs(fTimeOffsetS)));
                  if (swapit) {
                     swap (&fTimeOffsetS);
                     swap (&fTimeOffsetN);
                  }
                  pp += 8;
               }
               if (version < 6) {
                  fFShift = 0;
                  fPhase = 0;
               }
               else {
                  memcpy (&fFShift, pp, 8);
                  pp += 8;
                  memcpy (&fPhase, pp, 4);
                  pp += 4;
               }
               fDataValid = 0;
               // memcpy (fData, pp, 2 * 4);
               // pp += 2*4;
               // memcpy (fData+3, pp, 4);
               // pp += 4;
               // memcpy (fData+2, pp, 4);
               // pp += 4;
               pp += fData[0].read (version, pp, swapit);
               pp += fData[1].read (version, pp, swapit);
               pp += fData[3].read (version, pp, swapit);
               pp += fData[2].read (version, pp, swapit);
               break;
            }
         case kSerData:
            {
               pp += readString (pp, swapit, fName, maxName);
               memset (&fChannelGroup, 0, 5 * 4 + 1);
               fUnit[0] = 0;
               memcpy (&fTimeOffsetS, pp, 2 * 4);
               pp += 2 * 4;
	       if (version < 7) {
		  real_4_t rate;
		  memcpy (&rate, pp, sizeof(rate));
		  if (swapit) swap (&rate);
		  fSampleRate = rate;
		  if (swapit) swap (&fSampleRate); //revert swapping for later
		  pp += sizeof(rate);
	       }
	       else {
		  memcpy (&fSampleRate, pp, sizeof(fSampleRate));
		  pp += sizeof(fSampleRate);
	       }
               fFShift = 0;
               fPhase = 0;
               fDataValid = 0;
               pp += readString (pp, swapit, fComment, maxName);
               // memcpy (fData, pp, 4);
               // pp += 4;
               // memcpy (fData+3, pp, 4);
               // pp += 4;
               // memcpy (fData+2, pp, 4);
               // pp += 4;
               // memset (fData + 1, 0, 4);
               pp += fData[0].read (version, pp, swapit);
               pp += fData[3].read (version, pp, swapit);
               pp += fData[2].read (version, pp, swapit);
               fData[1] = ptr_struct();
               break;
            }
      }
      fDatatype = dtype;
      if (swapit) {
         swap (&fChannelGroup);
         swap (&fChannelNumber);
         swap (&fNBits);
         swap (&fBias);
         swap (&fSlope);
         swap (&fSampleRate);
         swap (&fTimeOffsetS);
         swap (&fTimeOffsetN);
         swap (&fFShift);
         swap (&fPhase);
         swap (&fDataValid);
         // for (int i = 0; i < 4; i++) {
            // swap (&fData[i].fClass);
            // swap (&fData[i].fInstance);
         // }
      }
      if (version >= 8) pp += swapin(pp, fChkSum, swapit);
      return long(pp - p);
   }

//______________________________________________________________________________
   long adcdata_t::write (int version, char* p, bool swapit) const
   {
      adcdata_t* adc;
      real_8_t timeofs = 
         fabs ((real_8_t)fTimeOffsetS) + (real_8_t)fTimeOffsetN/1E9;
      if (fTimeOffsetS < 0) timeofs *= -1;
      if (swapit) {
         adc = new (nothrow) adcdata_t (*this);
         swap (&adc->fChannelGroup);
         swap (&adc->fChannelNumber);
         swap (&adc->fNBits);
         swap (&adc->fBias);
         swap (&adc->fSlope);
         swap (&adc->fSampleRate);
         swap (&adc->fTimeOffsetS);
         swap (&adc->fTimeOffsetN);
         swap (&adc->fFShift);
         swap (&adc->fPhase);
         swap (&adc->fDataValid);
         swap (&timeofs);
         // for (int i = 0; i < 3; i++) {
            // swap (&adc->fData[i].fClass);
            // swap (&adc->fData[i].fInstance);
         // }   
      }
      else {
         adc = (adcdata_t*) this;
      }
      char* pp = p + adc->generic_t::write (version, p, swapit);
      pp += writeString (pp, swapit, adc->fName);
      pp += writeString (pp, swapit, adc->fComment);
      memcpy (pp, &adc->fChannelGroup, 5 * 4);
      pp += 5 * 4;
      pp += writeString (pp, swapit, adc->fUnit);
      memcpy (pp, &adc->fSampleRate, 8);
      pp += 8;
      if (version < 6) {
         memcpy (pp, &adc->fTimeOffsetS, 2 * 4);
         pp += 2 * 4;
      }
      else {
         memcpy (pp, &timeofs, 8);
         pp += 8;
      }
      memcpy (pp, &adc->fFShift, 8);
      pp += 8;
      if (version > 4) {
         memcpy (pp, &fPhase, 4);
         pp += 4;
      }
      memcpy (pp, &adc->fDataValid, 2);
      pp += 2;
      // memcpy (pp, adc->fData, 3 * 4);
      // pp += 3*4;
      pp += fData[0].write (version, pp, swapit);
      pp += fData[1].write (version, pp, swapit);
      pp += fData[2].write (version, pp, swapit);
      if (swapit) {
         delete adc;
      }
      return fixlength (version, p, (int)(pp - p), swapit);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// frvect_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   frvect_t::frvect_t () 
   : fData (0), fOwn (false)
   {
      memset (this, 0, sizeof (frvect_t));
   }

//______________________________________________________________________________
   frvect_t::frvect_t (const frvect_t& vect) 
   : fData (0), fOwn (false)
   {
      *this = vect;
   }

//______________________________________________________________________________
   frvect_t& frvect_t::operator= (const frvect_t& vect)
   {
      if (this != &vect) {
         deallocate();
         *(generic_t*)this = vect;
         save_strncpy (fName, vect.fName, maxName);
         fCompress = vect.fCompress;
         fType = vect.fType;
         fNData = vect.fNData;
         fNBytes = vect.fNBytes;
         fNCompBytes = vect.fNCompBytes;
         fNDim = vect.fNDim;
         save_strncpy (fUnitY, vect.fUnitY, maxName);
         for (int i = 0; i < 4; i++) {
            fNx[i] = vect.fNx[i];
            fDx[i] = vect.fDx[i];
            fStartX[i] = vect.fStartX[i];
            save_strncpy (fUnitX[i], vect.fUnitX[i], maxName);
         }
         fNext = vect.fNext;
         allocate (fType, fNData);
         if (fData && vect.fData) {
            memcpy (fData, vect.fData, fNBytes);
         }
      }
      return *this;
   }

//______________________________________________________________________________
   frvect_t::~frvect_t () 
   {
      deallocate ();
   }

#ifdef DEBUG
   static std::vector<int> memlist;
   int memalloc = 0;
#endif

//______________________________________________________________________________
   bool frvect_t::allocate (int_2u_t type, int size)
   {
      deallocate();
      fType = type;
      int len = type_sizeof (type) * size;
      fNData = size;
      fNBytes = len;
      if (len <= 0) {
         return true;
      }
      fData = new (nothrow) char [len];
      if (fData == 0) {
         return false;
      }
      else {
      #ifdef DEBUG
         //cerr << "allocate frvect_t   @ " << (int) fData << endl;
         memlist.push_back ((int) fData);
         memalloc++;
         if (memalloc % 1000 == 0) {
            cerr << "Number of allocations = " << memalloc <<
               "  Active allocations = " << memlist.size() << endl;
         }
      #endif
         memset (fData, 0, len);
         fOwn = true;
         return true;
      }
   }

//______________________________________________________________________________
   void frvect_t::deallocate ()
   {
      if (fOwn && fData) {
      #ifdef DEBUG
         // cerr << "deallocate frvect_t @ " << (int) fData << endl;
         for (vector<int>::iterator i = memlist.end();
             i != memlist.begin(); ) {
            --i;
            if (*i ==  (int) fData) {
               memlist.erase (i);
               break;
            }
         }
      #endif
         delete [] (char*)fData;
      }
      fData = 0;
      fOwn = false;
   }

//______________________________________________________________________________
   void frvect_t::clone (const frvect_t& vect)
   {
      if (this != &vect) {
         deallocate ();
         *(generic_t*)this = vect;
         save_strncpy (fName, vect.fName, maxName);
         fCompress = vect.fCompress;
         fType = vect.fType;
         fNData = vect.fNData;
         fNBytes = vect.fNBytes;
         fNCompBytes = vect.fNCompBytes;
         fNDim = vect.fNDim;
         save_strncpy (fUnitY, vect.fUnitY, maxName);
         for (int i = 0; i < 4; i++) {
            fNx[i] = vect.fNx[i];
            fDx[i] = vect.fDx[i];
            fStartX[i] = vect.fStartX[i];
            save_strncpy (fUnitX[i], vect.fUnitX[i], maxName);
         }
         fNext = vect.fNext;
      }
   }

//______________________________________________________________________________
   void frvect_t::image (const frvect_t& vect)
   {
      if (this != &vect) {
         deallocate ();
         *(generic_t*)this = vect;
         save_strncpy (fName, vect.fName, maxName);
         fCompress = vect.fCompress;
         fType = vect.fType;
         fNData = vect.fNData;
         fNBytes = vect.fNBytes;
         fNCompBytes = vect.fNCompBytes;
         fNDim = vect.fNDim;
         save_strncpy (fUnitY, vect.fUnitY, maxName);
         for (int i = 0; i < 4; i++) {
            fNx[i] = vect.fNx[i];
            fDx[i] = vect.fDx[i];
            fStartX[i] = vect.fStartX[i];
            save_strncpy (fUnitX[i], vect.fUnitX[i], maxName);
         }
         fNext = vect.fNext;
         fData = vect.fData;
         fOwn = false;
      }
   }

//______________________________________________________________________________
   long frvect_t::read (int version, const char* p, bool swapit, datacopy cpy)
   {
      const char* pp = p;
      pp = p + generic_t::read (version, pp, swapit, true);
      pp += readString (pp, swapit, fName, maxName);
      memcpy (&fCompress, pp, 4);
      pp += 4;
      if (swapit) {
         swap (&fCompress);
         swap (&fType);
      }
      if (version >= 6) {
         memcpy (&fNData, pp, 16);
         pp += 16;
         if (swapit) {
            swap (&fNData);
            swap (&fNBytes);
         }
      }
      else {
         int_4u_t dummy[2];
         memcpy (dummy, pp, 8);
         pp += 8;
         if (swapit) {
            swap (dummy + 0);
            swap (dummy + 1);
         }
         fNData = dummy[0];
         fNBytes = dummy[1];
      }
      //cerr << "fNBytes = " << fNBytes << endl;
      // get data
      int bytes = fNBytes;
      fNCompBytes = fNBytes;
      // no copy
      if (cpy == fv_nocopy) {
         deallocate ();
      }
      // copy
      else if ((cpy == fv_copy) || (fCompress % 256 != 0) || 
              swapit || !checkalign (pp, fType)) {
         // cerr << "fCompress = " << fCompress << endl;
         // cerr << "fType = " << fType << endl;
         // cerr << "fNData = " << fNData << endl;
         // cerr << "fNBytes = " << fNBytes << endl;
         allocate (fType, fNData);
         bool ok = true;
         switch (fType) {
            case typeOf<int_1s_t>::value:
               ok = decompress ((int_1s_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_2s_t>::value:
               ok = decompress ((int_2s_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_4s_t>::value:
               ok = decompress ((int_4s_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_8s_t>::value:
               ok = decompress ((int_8s_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_1u_t>::value:
               ok = decompress ((int_1u_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_2u_t>::value:
               ok = decompress ((int_2u_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_4u_t>::value:
               ok = decompress ((int_4u_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<int_8u_t>::value:
               ok = decompress ((int_8u_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<real_4_t>::value:
               ok = decompress ((real_4_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<real_8_t>::value:
               ok = decompress ((real_8_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<complex_8_t>::value:
               ok = decompress ((complex_8_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            case typeOf<complex_16_t>::value:
               ok = decompress ((complex_16_t*)fData, fNData, 
                               pp, bytes, fCompress);
               break;
            default:
               deallocate ();
               break;
         }
         if (!ok) cerr << "decompression error in " << fName << endl;
      }
      // avoid copy by using original data pointer
      else {
         deallocate ();
         if ((fType > 12) || (fType == 8)) {
         }
         else {
            fData = (void*) pp;
         }
      }
      pp += bytes;
      // now dimension stuff
      memcpy (&fNDim, pp, 4);
      pp += 4;
      if (swapit) {
         swap (&fNDim);
      }
      if (version >= 6) {
         for (unsigned int i = 0; i < fNDim; i++) {
            if (i < 4) memcpy (&fNx[i], pp, 8);
            pp += 8;
         }
      }
      else {
         int_4u_t nx;
         for (unsigned int i = 0; i < fNDim; i++) {
            if (i < 4) {
               memcpy (&nx, pp, 4);
               if (swapit) swap (&nx);
               fNx[i] = nx;
               if (swapit) swap (&fNx[i]);
            }
            pp += 4;
         }
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         if (i < 4) memcpy (&fDx[i], pp, 8);
         pp += 8;
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         if (i < 4) memcpy (&fStartX[i], pp, 8);
         pp += 8;
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         if (i < 4) {
            pp += readString (pp, swapit, fUnitX[i], maxName);
         }
         else {
            char dummy [maxName];
            pp += readString (pp, swapit, dummy, maxName);
         }
      }
      pp += readString (pp, swapit, fUnitY, maxName);
      // memcpy (&fNext, pp, 4);
      // pp += 4;
      pp += fNext.read (version, pp, swapit);
      if (fNDim > 4) fNDim = 4;
      if (swapit) {
         for (unsigned int i = 0; i < fNDim; i++) {
            swap (&fNx[i]);
            swap (&fDx[i]);
            swap (&fStartX[i]);
         }
      }
   
      return long(pp - p);
   }

//______________________________________________________________________________
   long frvect_t::write (int version, char* p, bool swapit) const
   {
      // ::TEMP:: cerr temp
      // if (fData) {
         // cerr << "WRITE1  Data = ";
         // for (int i = 0; i < 10; i++) {
            // cerr << ((float*)fData)[i] << " ";
         // }
         // cerr << endl;
      // }
   
      frvect_t* vec;
      if (swapit) {
         vec = new (nothrow) frvect_t;
         vec->clone (*this);
         swap (&vec->fCompress);
         swap (&vec->fType);
         swap (&vec->fNData);
      }
      else {
         vec = (frvect_t*) this;
      }
      // write header (make sure data vector is aligned to 64bit)
      char* pp = p + generic_t::write (version, p, swapit);
      int varlen = 12;
      if (version >= 6) varlen = 20;
      int pad = 8 - (long(pp + 2 + strlen (fName) + 1 + varlen) & 0X07);
      if (pad == 8) pad = 0;
      pp += writeString (pp, swapit, fName, pad);
      memcpy (pp, &vec->fCompress, 4);
      pp += 4;
      if (version >= 6) {
         memcpy (pp, &vec->fNData, 8);
         pp += 8;
      }
      else {
         int_4u_t ndata = fNData;
         if (swapit) swap (&ndata);
         memcpy (pp, &ndata, 4);
         pp += 4;
      }
      // write data before writing nbytes
      char* ppp = pp + (version >= 6 ? 8 : 4);
      int_8u_t nbytes = 0;
      switch (fType) {
         case typeOf<int_1s_t>::value:
            compress ((int_1s_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_2s_t>::value:
            compress ((int_2s_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_4s_t>::value:
            compress ((int_4s_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_8s_t>::value:
            compress ((int_8s_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_1u_t>::value:
            compress ((int_1u_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_2u_t>::value:
            compress ((int_2u_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_4u_t>::value:
            compress ((int_4u_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<int_8u_t>::value:
            compress ((int_8u_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<real_4_t>::value:
            compress ((real_4_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<real_8_t>::value:
            compress ((real_8_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<complex_8_t>::value:
            compress ((complex_8_t*)fData, fNData, 
                     ppp, nbytes, fCompress);
            break;
         case typeOf<complex_16_t>::value:
            compress ((complex_16_t*)fData, fNData, 
		      ppp, nbytes, fCompress);
            break;
         default:
            nbytes = 0;
            break;
      }
      //cerr << "nbytes = " << nbytes << endl;
      (int_8u_t&)fNCompBytes = nbytes;
      //cerr << "fNCompBytes = " << fNBytes << endl;
      // write nbytes now
      if (fNDim > 4) (int_4u_t&)fNDim = 4;
      if (swapit) {
         vec->fNBytes = nbytes;
         swap (&vec->fNBytes);
         swap (&vec->fNDim);
         for (unsigned int i = 0; i < fNDim; i++) {
            swap (&vec->fNx[i]);
            swap (&vec->fDx[i]);
            swap (&vec->fStartX[i]);
         }
         // swap (&vec->fNext.fClass);
         // swap (&vec->fNext.fInstance);
      }
      if (version >= 6) {
         memcpy (pp, &vec->fNBytes, 8);
      }
      else {
         int_4u_t nb = nbytes;
         if (swapit) swap (&nb);
         memcpy (pp, &nb, 4);
      }
      pp = ppp + nbytes;
      // now dimension stuff
      memcpy (pp, &vec->fNDim, 4);
      pp += 4;
      if (version >= 6) {
         for (unsigned int i = 0; i < fNDim; i++) {
            memcpy (pp, &vec->fNx[i], 8);
            pp += 8;
         }
      }
      else {
         for (unsigned int i = 0; i < fNDim; i++) {
            int_4u_t nx = fNx[i];
            if (swapit) swap (&nx);
            memcpy (pp, &nx, 4);
            pp += 4;
         }
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         memcpy (pp, &vec->fDx[i], 8);
         pp += 8;
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         memcpy (pp, &vec->fStartX[i], 8);
         pp += 8;
      }
      for (unsigned int i = 0; i < fNDim; i++) {
         pp += writeString (pp, swapit, fUnitX[i]);
      }
      pp += writeString (pp, swapit, fUnitY);
      // memcpy (pp, &vec->fNext, 4);
      // pp += 4;
      pp += fNext.write (version, pp, swapit);
      if (swapit) {
         delete vec;
      }
      return fixlength (version, p, long(pp - p), swapit);
   }

//______________________________________________________________________________
template <typename T>
   inline int frvect_get (T* dest, const void* src, int N, int max, 
                     int dtype)
   
   {
      if ((dest == 0) || (N <= 0) || (src == 0)) {
         return 0;
      }
      if (N > max) N = max;
      switch (dtype) {
         case typeOf<int_1s_t>::value:
            convertdata (dest, (int_1s_t*)src, N);
            break;
         case typeOf<int_2s_t>::value:
            convertdata (dest, (int_2s_t*)src, N);
            break;
         case typeOf<int_4s_t>::value:
            convertdata (dest, (int_4s_t*)src, N);
            break;
         case typeOf<int_8s_t>::value:
            convertdata (dest, (int_8s_t*)src, N);
            break;
         case typeOf<real_4_t>::value:
            convertdata (dest, (real_4_t*)src, N);
            break;
         case typeOf<real_8_t>::value:
            convertdata (dest, (real_8_t*)src, N);
            break;
         case typeOf<int_1u_t>::value:
            convertdata (dest, (int_1u_t*)src, N);
            break;
         case typeOf<int_2u_t>::value:
            convertdata (dest, (int_2u_t*)src, N);
            break;
         case typeOf<int_4u_t>::value:
            convertdata (dest, (int_4u_t*)src, N);
            break;
         case typeOf<int_8u_t>::value:
            convertdata (dest, (int_8u_t*)src, N);
            break;
         default:
            return 0;
      }
      return N;
   }

//______________________________________________________________________________
template <typename T>
   inline int frvect_get_real_cmlx (T* dest, const void* src, int N, int max, 
                     int dtype)
   
   {
      if ((dest == 0) || (N <= 0) || (src == 0)) {
         return 0;
      }
      if (2 * N > max) N = max / 2;
      switch (dtype) {
         case typeOf<complex_8_t>::value:
            convertdata_real_cmlx (dest, (complex_8_t*)src, N);
            break;
         case typeOf<complex_16_t>::value:
            convertdata_real_cmlx (dest, (complex_16_t*)src, N);
            break;
      }
      return 2 * N;
   }

//______________________________________________________________________________
template <typename T>
   inline int frvect_get_cmlx (T* dest, const void* src, int N, int max, 
                     int dtype)
   
   {
      if ((dest == 0) || (N <= 0) || (src == 0)) {
         return 0;
      }
      if (N > max) N = max;
      switch (dtype) {
         case typeOf<complex_8_t>::value:
            convertdata (dest, reinterpret_cast<const complex_8_t*>(src), N);
            break;
         case typeOf<complex_16_t>::value:
            convertdata (dest, reinterpret_cast<const complex_16_t*>(src), N);
            break;
      }
      return N;
   }

//______________________________________________________________________________
   int frvect_t::get (real_4_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_real_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
   int frvect_t::get (real_8_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_real_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
   int frvect_t::get (complex_8_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
   int frvect_t::get (complex_16_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
   int frvect_t::get (int_2s_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_real_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
   int frvect_t::get (int_4s_t* y, int N) const
   {
      if ((fType == typeOf<complex_8_t>::value) ||
         (fType == typeOf<complex_16_t>::value)) {
         return frvect_get_real_cmlx (y, fData, fNData, N, fType);
      }
      else {
         return frvect_get (y, fData, fNData, N, fType);
      }
   }

//______________________________________________________________________________
template <typename T>
   inline void frvect_datacpy_1 (void* pdest, const T* psrc, int dType, 
                     int num, int dec = 1, int ipol = 1)
   {
      switch (dType) {
         case typeOf<int_1s_t>::value:
            convertdata ((int_1s_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_2s_t>::value:
            convertdata ((int_2s_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_4s_t>::value:
            convertdata ((int_4s_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_8s_t>::value:
            convertdata ((int_8s_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_1u_t>::value:
            convertdata ((int_1u_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_2u_t>::value:
            convertdata ((int_2u_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_4u_t>::value:
            convertdata ((int_4u_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<int_8u_t>::value:
            convertdata ((int_8u_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<real_4_t>::value:
            convertdata ((real_4_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<real_8_t>::value:
            convertdata ((real_8_t*) pdest, psrc, num, dec, ipol);
            return;
         case typeOf<complex_8_t>::value:
	    convertdata (reinterpret_cast<complex_8_t*>(pdest), psrc, 
			 num, dec, ipol);
            return;
         case typeOf<complex_16_t>::value:
	    convertdata (reinterpret_cast<complex_16_t*>(pdest), psrc, 
			 num, dec, ipol);
            return;
         default:
            return;
      }
   }

//______________________________________________________________________________
   inline void frvect_datacpy (void* pdest, const void* psrc, int dType, 
                     int sType, int num, int dec = 1, int ipol = 1)
   {
      switch (sType) {
         case typeOf<int_1s_t>::value:
            frvect_datacpy_1 (pdest, (const int_1s_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_2s_t>::value:
            frvect_datacpy_1 (pdest, (const int_2s_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_4s_t>::value:
            frvect_datacpy_1 (pdest, (const int_4s_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_8s_t>::value:
            frvect_datacpy_1 (pdest, (const int_8s_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_1u_t>::value:
            frvect_datacpy_1 (pdest, (const int_1u_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_2u_t>::value:
            frvect_datacpy_1 (pdest, (const int_2u_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_4u_t>::value:
            frvect_datacpy_1 (pdest, (const int_4u_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<int_8u_t>::value:
            frvect_datacpy_1 (pdest, (const int_8u_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<real_4_t>::value:
            frvect_datacpy_1 (pdest, (const real_4_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<real_8_t>::value:
            frvect_datacpy_1 (pdest, (const real_8_t*) psrc, dType, 
                             num, dec, ipol);
            return;
         case typeOf<complex_8_t>::value:
            switch (dType) {
               case typeOf<complex_8_t>::value:
		  convertdata (reinterpret_cast<complex_8_t*>(pdest), 
			       reinterpret_cast<const complex_8_t*>(psrc), 
			       num, dec, ipol);
                  return;
               case typeOf<complex_16_t>::value:
		  convertdata (reinterpret_cast<complex_16_t*>(pdest), 
			       reinterpret_cast<const complex_8_t*>(psrc), 
			       num, dec, ipol);
                  return;
            }
            return;
         case typeOf<complex_16_t>::value:
            switch (dType) {
               case typeOf<complex_8_t>::value:
		  convertdata (reinterpret_cast<complex_8_t*>(pdest), 
			       reinterpret_cast<const complex_16_t*>(psrc), 
			       num, dec, ipol);
                  return;
               case typeOf<complex_16_t>::value:
                  convertdata (reinterpret_cast<complex_16_t*>(pdest), 
			       reinterpret_cast<const complex_16_t*>(psrc), 
			       num, dec, ipol);
                  return;
            }
            return;
         default:
            return;
      }
   }

//______________________________________________________________________________
   bool frvect_t::fill (int_4u_t pos, int_4u_t num, const frvect_t& src, 
                     int_4u_t ofs, real_8_t mul)
   {
      // ::TEMP:: cerr temp
      // if (src.fData) {
         // cerr << "FILL1   Data = ";
         // for (int i = 0; i < 10; i++) {
            // cerr << ((float*)(src.fData))[i] << " ";
         // }
         // cerr << endl;
      // }
   
      // determine decimation rate / interpolation ratio
      int dec = 1;
      int ipol = 1;
      int_4u_t srcnum = num;
      if (mul < 1) {
         dec = (int) (1 / mul + 0.5);
         srcnum = num * dec;
      }
      else {
         ipol = (int) (mul + 0.5);
         srcnum = num / ipol;
      }
      int size = type_sizeof (fType);
      // cerr << "dec = " << dec << " ipol = " << ipol << endl;
      // cerr << "fData = " << fData << " pos = " << pos << " num = " << num <<
         // " fNData = " << fNData << " size = " << size << 
         // " fBytes = " << fNBytes << endl;
      // cerr << "src.fData = " << src.fData << " ofs = " << ofs << 
         // " srcnum = " << srcnum << " src.fNData = " << src.fNData << 
         // " src.fBytes = " << src.fNBytes << endl;
      // checks first
      if ((fData == 0) || (pos + num > fNData) || (size <= 0) ||
         (src.fData == 0) || (ofs + srcnum > src.fNData)) {
         return false;
      }
      // caculate start addresses
      void* pdest = (char*)fData + size * pos;
      const void* psrc =  (const char*)src.fData + size * ofs;
      // do memcpy if data rates and types are equal
      if ((dec == 1) && (ipol == 1) && (fType == src.fType)) {
         memcpy (pdest, psrc, num * size);
      }
      // else copy element by element
      else {
         frvect_datacpy (pdest, psrc, fType, src.fType, num, dec, ipol);
      }
   
      return true;
   }

//______________________________________________________________________________
   bool frvect_t::fill (int_4u_t pos, int_4u_t num, const char* values,
                     bool swapit)
   {
      if ((fData == 0) || (pos >= fNData)) {
         return false;
      }
      int len = num;
      if (pos + len > fNData) {
         len = fNData - pos;
      }
      int size = type_sizeof (fType);
      void* pdest = (char*)fData + size * pos;
      memcpy (pdest, values, len * size);
      // check swap
      if (swapit) {
         if (size == 2) {
            int_2s_t* p = (int_2s_t*)pdest;
            for (int i = 0; i < len; i++) {
               swap (p + i);
            }
         }
         else if (size == 4) {
            int_4s_t* p = (int_4s_t*)pdest;
            for (int i = 0; i < len; i++) {
               swap (p + i);
            }
         }
         else if (size == 8) {
            int_8s_t* p = (int_8s_t*)pdest;
            for (int i = 0; i < len; i++) {
               swap (p + i);
            }
         }
         else if (size == 16) {
            complex_16_t* p = (complex_16_t*)pdest;
            for (int i = 0; i < len; i++) {
               swap (p + i);
            }
         }
      }
      return true;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// data_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   data_t::data_t ()
   {
   }

//______________________________________________________________________________
   data_t::~data_t ()
   {
      deallocate();
   }

//______________________________________________________________________________
   bool data_t::allocate (int_2u_t type, int size)
   {
      return fVect.allocate (type, size);
   }

//______________________________________________________________________________
   void data_t::deallocate ()
   {
      fVect.deallocate ();
   }

//______________________________________________________________________________
   void data_t::clone (const data_t& templ)
   {
      fADC = templ.fADC;
      fVect.clone (templ.fVect);
   }

//______________________________________________________________________________
   bool data_t::fill (int_4u_t pos, int_4u_t num, const data_t& src, 
                     int_4u_t ofs, real_8_t mul)
   {
      return fVect.fill (pos, num, src.fVect, ofs, mul);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// endof_frame_t					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   endof_frame_t::endof_frame_t()
   {
      memset (this, 0, sizeof (endof_frame_t));
   }

//______________________________________________________________________________
   long endof_frame_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      memcpy (&fRun, pp, 8);
      pp += 8;
      if (swapit) {
         swap (&fRun);
         swap (&fFrame);
      }
      if (version == 4) {
         fChkType = 0;
         fChkSum = 0;
	 fGTimeS = 0;
	 fGTimeN = 0;
      }
      else if (version < 8) {
         memcpy (&fChkType, pp, 2 * 4);
         pp += 2 * 4;
	 fGTimeS = 0;
	 fGTimeN = 0;
	 if (swapit) {
	    swap (&fChkType);
	    swap (&fChkSum);
	 }
      }
      else {
         fChkType = 0;
	 pp += swapin(pp, fGTimeS, swapit);
	 pp += swapin(pp, fGTimeN, swapit);
	 pp += swapin(pp, fChkSum, swapit);
      }
      return long(pp - p);
   }

//______________________________________________________________________________
   long endof_frame_t::write (int version, char* p, bool swapit) const
   {
      char* pp = p + generic_t::write (version, p, swapit);
      pp += swapout(fRun, pp, swapit);
      pp += swapout(fFrame, pp, swapit);
      if (version > 4) {
	 if (version < 8) {
	    pp += swapout(fChkType, pp, swapit);
	 }
	 else {
	    pp += swapout(fGTimeS, pp, swapit);
	    pp += swapout(fGTimeN, pp, swapit);
	 }
      }
      return fixlength (version, p, long(pp - p), swapit);
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// endof_file_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   endof_file_t::endof_file_t()
   {
      memset (this, 0, sizeof (endof_file_t));
   }

//______________________________________________________________________________
   long endof_file_t::read (int version, const char* framestart, 
                     int_8u_t framelen, bool swapit)
   {
      const char* p = framestart + framelen - size (version);
      return read (version, p, swapit);
   }

//______________________________________________________________________________
   long endof_file_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      if (version < 6) {
         int_4u_t nbytes;
         int_4u_t seektoc;
         memcpy (&fNFrames, pp, 4); 
         pp += 4;
         memcpy (&nbytes, pp, 4);
         pp += 4;
         memcpy (&fChkType, pp, 8); 
         pp += 8;
         memcpy (&seektoc, pp, 4);
         pp += 4;
         if (swapit) {
            swap (&fNFrames);
            swap (&nbytes);
            swap (&fChkType);
            swap (&fChkSum);
            swap (&seektoc);
         }
         fNBytes = nbytes;
         fSeekTOC = seektoc;
      } 
      else if (version < 8) {
         memcpy (&fNFrames, pp, 4); 
         pp += 4;
         memcpy (&fNBytes, pp, 24);
         pp += 24;
         if (swapit) {
            swap (&fNFrames);
            swap (&fNBytes);
            swap (&fChkType);
            swap (&fChkSum);
            swap (&fSeekTOC);
         }
      }
      else {
	 pp += swapin(pp, fNFrames,      swapit);
	 pp += swapin(pp, fNBytes,       swapit);
	 pp += swapin(pp, fSeekTOC,      swapit);
	 pp += swapin(pp, fHeaderChkSum, swapit);
	 pp += swapin(pp, fChkSum,       swapit);
	 pp += swapin(pp, fileChkSum,    swapit);
      }
      return (int)(pp - p);
   }

//______________________________________________________________________________
   long endof_file_t::write (int version, char* p, bool swapit) const
   {
      char* pp = p + generic_t::write (version, p, swapit);
      if (version < 6) {
         int_4u_t dummy[5];
         dummy[0] = fNFrames;
         dummy[1] = fNBytes;
         dummy[2] = fChkType;
         dummy[3] = fChkSum;
         dummy[4] = fSeekTOC;
         if (swapit) {
            for (int i = 0; i < 5; ++i) swap (dummy + i);
         }
         memcpy (pp, dummy, 5*4);
         pp += 5*4;
      }
      else if (version < 8) {
         endof_file_t* eof;
         if (swapit) {
            eof = new (nothrow) endof_file_t (*this);
            swap (&eof->fNFrames);
            swap (&eof->fNBytes);
            swap (&eof->fChkType);
            swap (&eof->fChkSum);
            swap (&eof->fSeekTOC);
         }
         else {
            eof = (endof_file_t*) this;
         }
         memcpy (pp, &eof->fNFrames, 4);
         pp += 4;
         memcpy (pp, &eof->fNBytes, 24);
         pp += 24;
         if (swapit) {
            delete eof;
         }
	 fixlength (version, p, long(pp - p), swapit);
      }
      else {
	 pp += swapout(fNFrames, pp, swapit);
	 pp += swapout(fNBytes,  pp, swapit);
	 pp += swapout(fSeekTOC, pp, swapit);
	 pp += swapout(fHeaderChkSum, pp, swapit);
	 pp = p + fixlength (version, p, long(pp - p), swapit);
	 pp += swapout(fileChkSum, pp, swapit);
      }
      return long(pp - p);
   }

//______________________________________________________________________________
   int endof_file_t::size (int version)
   {
      if (version < 6) {
	 return 28;
      }
      else if (version < 8) {
	 return 42;
      }
      else {
	 return 32 + 14;
      }
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// toc_frame_t / toc_SH_t / toc_detector_t / toc_stat_t / 		//
// toc_data_t / toc_Event_t						//
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   toc_frame_t::toc_frame_t ()
   {
      memset (this, 0, sizeof (toc_frame_t));
   }

//______________________________________________________________________________
   toc_SH_t::toc_SH_t ()
   {
      memset (this, 0, sizeof (toc_SH_t));
   }

//______________________________________________________________________________
   toc_detector_t::toc_detector_t ()
   {
      memset (this, 0, sizeof (toc_detector_t));
   }

//______________________________________________________________________________
   toc_stat_t::toc_stat_t ()
   {
      memset (this, 0, sizeof (toc_stat_t));
   }

//______________________________________________________________________________
   toc_data_t::toc_data_t ()
   {
      memset (this, 0, sizeof (toc_data_t));
   }

//______________________________________________________________________________
   bool toc_data_t::operator== (const toc_data_t& d2) const
   {
      return (strncasecmp (fName, d2.fName, maxName) == 0);
   }

//______________________________________________________________________________
   bool toc_data_t::operator< (const toc_data_t& d2) const
   {
      return (strncasecmp (fName, d2.fName, maxName) < 0);
   }

//______________________________________________________________________________
   toc_Event_t::toc_Event_t ()
   {
      memset (this, 0, sizeof (toc_Event_t));
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// toc_t						                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   toc_t::toc_t()
   {
      memset (this, 0, sizeof (toc_t));
   }

//______________________________________________________________________________
   toc_t::toc_t (const toc_t& toc)
   {
      memset (this, 0, sizeof (toc_t));
      *this = toc;
   }

//______________________________________________________________________________
   toc_t::~toc_t()
   {
      if (fFrames) delete [] fFrames;
      if (fSH) delete [] fSH;      
      if (fDetInfo) delete [] fDetInfo;
      if (fStat) delete [] fStat;      
      for (int j = 0; j < 5; j++) {
         allocate (0, j);
         // if (fDataPosBuf[j]) delete [] fDataPosBuf[j];
         // if (fData[j]) delete [] fData[j];
      }
   }

//______________________________________________________________________________
   toc_t& toc_t::operator= (const toc_t& toc)
   {
      if (this != &toc) {
         // free mmeory if necessary
         if (fFrames) delete [] fFrames;
         if (fSH) delete [] fSH;
         if (fDetInfo) delete [] fDetInfo;
         if (fStat) delete [] fStat;
         for (int j = 0; j < 5; j++) {
            allocate (0, j);
            // if (fDataPosBuf[j]) delete [] fDataPosBuf[j];
            // if (fData[j]) delete [] fData[j];
         }
         // copy basics
         memcpy (this, &toc, sizeof (toc_t));
         // copy frames
         fFrames = new (nothrow) toc_frame_t [fNFrame];
         memcpy (fFrames, toc.fFrames, fNFrame * sizeof (toc_frame_t));
         // copy frame structures
         fSH = new (nothrow) toc_SH_t [fNSH];
         memcpy (fSH, toc.fSH, fNSH * sizeof (toc_SH_t));
         // copy detector structures
         fDetInfo = new (nothrow) toc_detector_t [fNDetector];
         memcpy (fDetInfo, toc.fDetInfo, fNDetector * sizeof (toc_detector_t));
         // copy static structures
         fStat = new (nothrow) toc_stat_t [fNStatType];
         memcpy (fStat, toc.fStat, fNStatType * sizeof (toc_stat_t));
         // copy data block entries
         for (int j = 0; j < 5; j++) {
            fData[j] = 0;
            fDataPosBuf[j] = 0;
            allocate (fNData[j] + 1, j);
            // fData[j] = new (nothrow) toc_data_t [fNData[j]];
            // fDataPosBuf[j] = new (nothrow) int_8u_t [fNFrame * fNData[j]];
            memcpy (fData[j], toc.fData[j], fNData[j] * sizeof (toc_data_t));
            for (unsigned int i = 0; i < fNData[j]; i++) {
               fData[j][i].fPosition = fDataPosBuf[j] + i * fNFrame;
               memcpy (fData[j][i].fPosition, toc.fData[j][i].fPosition,
                      fNFrame * sizeof (int_8u_t));
            }
         }
      }
      return *this;
   }

//______________________________________________________________________________
   bool toc_t::allocate (int newsize, int i)
   {
      if ((i < 0) || (i > 4)) {
         return false;
      }
      // save old
      toc_data_t* oldData = fData[i];
      int_8u_t* oldDataPosBuf = fDataPosBuf[i];
      // deallocate?
      if (newsize <= 0) {
         fData[i] = 0;
         fDataPosBuf[i] = 0;
      }
      // new array
      else {
         // create new arrays
         fData[i] = (toc_data_t*) 
            new (nothrow) char [newsize * sizeof (toc_data_t)];
         fDataPosBuf[i] = new (nothrow) int_8u_t [newsize * fNFrame];
         // failed?
         if (!fData[i] || !fDataPosBuf[i]) {
            if (fData[i]) delete [] (char*)fData[i];
            if (fDataPosBuf[i]) delete [] fDataPosBuf[i];
            fData[i] = oldData;
            fDataPosBuf[i] = oldDataPosBuf;
            return false;
         }
         // copy old elements over, init new ones
         int ncpy = ((int)fNData[i] < newsize) ? fNData[i] : newsize;
         if (oldData) {
            memcpy (fData[i], oldData, ncpy * sizeof (toc_data_t));
            if (ncpy < newsize) {
               memset (fData[i] + ncpy, 0, (newsize-ncpy)*sizeof (toc_data_t));
            }
         }
         else {
            memset (fData[i], 0, newsize*sizeof (toc_data_t));
         }
         if (oldDataPosBuf) {
            memcpy (fDataPosBuf[i], oldDataPosBuf, 
                   ncpy * fNFrame * sizeof (int_8u_t));
            if (ncpy < newsize) {
               memset (fDataPosBuf[i] + ncpy*fNFrame, 0, 
                      (newsize-ncpy)*fNFrame*sizeof (int_8u_t));
            }
         }
         else {
            memset (fDataPosBuf[i], 0, newsize*fNFrame*sizeof (int_8u_t));
         }
         // set position pointers
         if (oldData && oldDataPosBuf) {
            for (int j = 0; j < ncpy; j++) {
               fData[i][j].fPosition = fDataPosBuf[i] +
                  (int)(oldData[j].fPosition - oldDataPosBuf);
            }
            for (int j = ncpy; j < newsize; ++j) {
               fData[i][j].fPosition = fDataPosBuf[i] + j * fNFrame;
            }
         }
         else {
            for (int j = 0; j < newsize; ++j) {
               fData[i][j].fPosition = fDataPosBuf[i] + j * fNFrame;
            }
         }
      }
      // cleanup
      fChannelMax[i] = newsize;
      if (oldData) delete [] (char*)oldData;
      if (oldDataPosBuf) delete [] oldDataPosBuf;
      return true;
   }

//______________________________________________________________________________
   bool toc_t::init (int version)
   {
      // set structure header
      ptr_struct ptr;
      ptr.set (version, kFrTOC);
      fClass = ptr.fDataClass;
      fInstance = ptr.fDataInstance;
      // de-allocate arrays
      if (fFrames) delete [] fFrames;
      fFrames = 0;
      if (fSH) delete [] fSH;  
      fSH = 0;  
      fNSH = 0;  
      if (fDetInfo) delete [] fDetInfo;
      fDetInfo = 0;
      fNDetector = 0;
      if (fStat) delete [] fStat;
      fStat = 0;      
      fNStatType = 0;
      for (int j = 0; j < 5; j++) {
         allocate (0, j);
         fNData[j] = 0;
         // if (fDataPosBuf[j]) delete [] fDataPosBuf[j];
         // fDataPosBuf[j] = 0;
         // if (fData[j]) delete [] fData[j];
         // fData[j] = 0;
         // fNData[j] = 0;
         // fChannelMax[j] = 0;
      }
      fNEvt[0] = fNEvt[1] = 0;
      if (fNFrame < 1) {
         return false;
      }
      // allocate new structures
      fFrames = new (nothrow) toc_frame_t [fNFrame];
      int kSH = kSH_V4;
      if (version >= 6) kSH = kSH_V4;
      const SH_def_t*	cSH = cSH_V4;
      if (version >= 6) cSH = cSH_V6;
      fSH = new (nothrow) toc_SH_t [kSH];
      if ((fFrames == 0) || (fSH == 0)) {
         return false;
      }
      for (int i = 0; i < 5; ++i) {
         if (!allocate (kDefaultTocSize, i)) {
            return false;
         }
      }
      // fData[0] = new (nothrow) toc_data_t [maxChannels];
      // fDataPosBuf[0] = new (nothrow) int_8u_t [maxChannels * fNFrame];
      // if ((fFrames == 0) || (fSH == 0) || 
         // (fData[0] == 0) || (fDataPosBuf[0] == 0)) {
         // return false;
      // }
      // for (int i = 0; i < maxChannels; i++) {
         // fData[0][i].fPosition = fDataPosBuf[0] + i * fNFrame;
      // }
   
      // fill in defaults
      fNSH = kSH;
      for (int i = 0; i < kSH; i++) {
         fSH[i].fSHid = cSH[i].fClassNum;
         save_strncpy (fSH[i].fSHName, cSH[i].fName, maxName);
      }
      // fChannelMax = maxChannels;
      return true;
   }

//______________________________________________________________________________
   long  toc_t::read (int version, const char* p, bool swapit)
   {
      const char* pp = p + generic_t::read (version, p, swapit, true);
      if (version < 8 ) {
	 memcpy (&fULeapS, pp, 2);
	 pp += 2;
	 if (version == 4) {
	    memcpy (&fLocalTime, pp, 4);
	    pp += 4;
	 }
	 else {
	    fLocalTime = 0;
	 }
	 memcpy (&fNFrame, pp, 4);
	 pp += 4;
	 if (swapit) {
	    swap (&fULeapS);
	    swap (&fLocalTime);
	    swap (&fNFrame);
	 }
	 // read list of frames
	 if (fFrames) delete [] fFrames;
	 fFrames = new (nothrow) toc_frame_t [fNFrame];
	 int frlen = 16 * 4;
	 char* buf = new (nothrow) char[fNFrame * frlen];
	 if (version > 4) {
	    memcpy (buf, pp, fNFrame * 4);
	    pp += fNFrame * 4;
	    for (unsigned int i = 0; i < fNFrame; i++) {
	       fFrames[i].fDataQual = *(int_4u_t*)(buf + i*4);
	    }
	 }
	 memcpy (buf, pp, fNFrame * frlen);
	 for (unsigned int i = 0; i < fNFrame; i++) {
	    if (version == 4) fFrames[i].fDataQual = 0;
	    fFrames[i].fGTimeS = *(int_4u_t*)(buf + (0*fNFrame +i)*4);
	    fFrames[i].fGTimeN = *(int_4u_t*)(buf + (1*fNFrame +i)*4);
	    fFrames[i].fDt = *(real_8_t*)(buf + (1*fNFrame +i)*8);
	    fFrames[i].fRun = *(int_4s_t*)(buf + (4*fNFrame +i)*4);
	    fFrames[i].fFrame = *(int_4u_t*)(buf + (5*fNFrame +i)*4);
#ifdef __FILEOFS_4U
	    // ::WRONG:: incorrect frame format
	    fFrames[i].fPositionH = *(int_4u_t*)(buf + (6*fNFrame +i)*4);
	    fFrames[i].fNFirstADC = *(int_4u_t*)(buf + (7*fNFrame +i)*4);
	    fFrames[i].fNFirstSer = *(int_4u_t*)(buf + (8*fNFrame +i)*4);
	    fFrames[i].fNFirstTable = *(int_4u_t*)(buf + (9*fNFrame +i)*4);
	    fFrames[i].fNFirstMsg = *(int_4u_t*)(buf + (10*fNFrame +i)*4);
	    frlen -= 5 * 4;
#else
	    fFrames[i].fPositionH = *(int_8u_t*)(buf + (3*fNFrame +i)*8);
	    fFrames[i].fNFirstADC = *(int_8u_t*)(buf + (4*fNFrame +i)*8);
	    fFrames[i].fNFirstSer = *(int_8u_t*)(buf + (5*fNFrame +i)*8);
	    fFrames[i].fNFirstTable = *(int_8u_t*)(buf + (6*fNFrame +i)*8);
	    fFrames[i].fNFirstMsg = *(int_8u_t*)(buf + (7*fNFrame +i)*8);
#endif
	    if (swapit) {
	       swap (&fFrames[i].fDataQual);
	       swap (&fFrames[i].fGTimeS);
	       swap (&fFrames[i].fGTimeN);
	       swap (&fFrames[i].fDt);
	       swap (&fFrames[i].fRun);
	       swap (&fFrames[i].fFrame);
	       swap (&fFrames[i].fPositionH);
	       swap (&fFrames[i].fNFirstADC);
	       swap (&fFrames[i].fNFirstSer);
	       swap (&fFrames[i].fNFirstTable);
	       swap (&fFrames[i].fNFirstMsg);
	    }
	 }
	 pp += fNFrame * frlen;
	 delete [] buf;
	 
	 // read list of frame structure headers
	 memcpy (&fNSH, pp, 4);
	 pp += 4;
	 if (swapit) {
	    swap (&fNSH);
	 }
	 if (fSH) delete [] fSH;
	 fSH = new (nothrow) toc_SH_t [fNSH];
	 for (unsigned int i = 0; i < fNSH; i++) {
	    memcpy (&fSH[i].fSHid, pp, sizeof (int_2u_t));
	    pp += 2;
	 }
	 for (unsigned int i = 0; i < fNSH; i++) {
	    pp += readString (pp, swapit, fSH[i].fSHName, maxName);
	 }
	 if (swapit) {
	    for (unsigned int i = 0; i < fNSH; i++) {
	       swap (&fSH[i].fSHid);
	    }
	 }
	 
	 // read list of detector information indices
	 if (version > 4) {
	    memcpy (&fNDetector, pp, sizeof (int_4u_t));
	    pp += 4;
	    if (swapit) {
	       swap (&fNDetector);
	    }
	    if (fDetInfo) delete [] fDetInfo;
	    fDetInfo = new (nothrow) toc_detector_t [fNDetector];
	    for (unsigned int i = 0; i < fNDetector; i++) {
	       pp += readString (pp, swapit, fDetInfo[i].fDetector, maxName);
	    }
	    for (unsigned int i = 0; i < fNDetector; i++) {
	       memcpy (&fDetInfo[i].fPos, pp, 8);
	       pp += 8;
	       if (swapit) {
		  swap (&fDetInfo[i].fPos);
	       }
	    }
	 }
	 else {
	    fNDetector = 0;
	    if (fDetInfo) delete [] fDetInfo;
	    fDetInfo = 0;
	 }
	 
	 // read static struct
	 memcpy (&fNStatType, pp, sizeof (int_4u_t));
	 pp += 4;
	 if (swapit) {
	    swap (&fNStatType);
	 }
	 if (fNStatType == (int_4u_t)(-1)) fNStatType = 0;
	 if (fStat) delete [] fStat;
	 fStat = new (nothrow) toc_stat_t [fNStatType];
	 for (unsigned int i = 0; i < fNStatType; i++) {
	    pp += readString (pp, swapit, fStat[i].fName, maxName);
	    pp += readString (pp, swapit, fStat[i].fDetector, maxName);
	    memcpy (&fStat[i].fStatInstance, pp, 4);
	    pp += 4;
	    if (swapit) {
	       swap (&fStat[i].fStatInstance);
	    }
	    // skip info
	    pp += fStat[i].fStatInstance * 20;
	 }
	 
	 // read list of data blocks: ADC, Proc, Sim, Ser, Summary
	 for (int j = 0; j < 5; j++) {
	    memcpy (&fNData[j], pp, sizeof (int_4u_t));
	    pp += 4;
	    if (swapit) {
	       swap (fNData + j);
	    }
	    if (fNData[j] == (int_4u_t)(-1)) fNData[j] = 0;
	    allocate (0, j); // do it in two steps to avoid wrong copy!
	    allocate (fNData[j] + 1, j);
	    for (unsigned int i = 0; i < fNData[j]; i++) {
	       pp += readString (pp, swapit, fData[j][i].fName, maxName);
	    }
	    if (j == 0) {
	       for (unsigned int i = 0; i < fNData[j]; i++) {
		  memcpy (&fData[j][i].fChannelID, pp, sizeof (int_4u_t));
		  pp += 4;
	       }
	       for (unsigned int i = 0; i < fNData[j]; i++) {
		  memcpy (&fData[j][i].fGroupID, pp, sizeof (int_4u_t));
		  pp += 4;
	       }
	       if (swapit) {
		  for (unsigned int i = 0; i < fNData[j]; i++) {
		     swap (&fData[j][i].fChannelID);
		     swap (&fData[j][i].fGroupID);
		  }
	       }
	    }
#ifdef __FILEOFS_4U
	    // ::WRONG:: incorrect frame format
	    for (unsigned int i = 0; i < fNFrame * fNData[j]; i++) {
            int_4u_t temp;
            memcpy (&temp, ((int_4u_t*)pp) + i, sizeof (int_4u_t));
            fDataPosBuf[j][i] = temp;
	    }
	    pp += fNFrame * fNData[j] * sizeof (int_4u_t);
#else
	    memcpy (fDataPosBuf[j], pp, 
		    fNFrame * fNData[j] * sizeof (int_8u_t));
	    pp += fNFrame * fNData[j] * sizeof (int_8u_t);
#endif
	    if (swapit) {
	       for (unsigned int i = 0; i < fNFrame * fNData[j]; i++) {
		  swap (fDataPosBuf[j] + i);
	       }
	    }
	    // make sure entries are sorted by name
	    // toc_data_t* begin = fData[j];
	    // toc_data_t* end = fData[j] + fNData[j];
	    // sort (begin, end);
	 }
      }

      //--------------------------------  Version >=8
      else {
	 pp += swapin(pp, fULeapS, swapit);
	 fLocalTime = 0;
	 pp += swapin(pp, fNFrame, swapit);

	 // read list of frames
	 if (fFrames) delete [] fFrames;
	 fFrames = new (nothrow) toc_frame_t [fNFrame];

	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fDataQual,    swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fGTimeS,      swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fGTimeN,      swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fDt,          swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fRun,         swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fFrame,       swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fPositionH,   swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fNFirstADC,   swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fNFirstSer,   swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fNFirstTable, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapin(pp, fFrames[i].fNFirstMsg,   swapit);
	 
	 // read list of frame structure headers
	 pp += swapin(pp, fNSH, swapit);
	 if (fSH) delete [] fSH;
	 fSH = new (nothrow) toc_SH_t [fNSH];
	 for (unsigned int i=0; i<fNSH; ++i) 
	    pp += swapin(pp, fSH[i].fSHid,   swapit);
	 for (unsigned int i = 0; i < fNSH; i++) {
	    pp += readString (pp, swapit, fSH[i].fSHName, maxName);
	 }

	 // read list of detector information indices
	 pp += swapin(pp, fNDetector, swapit);
	 if (fDetInfo) delete [] fDetInfo;
	 fDetInfo = new (nothrow) toc_detector_t [fNDetector];
	 for (unsigned int i = 0; i < fNDetector; i++) {
	    pp += readString (pp, swapit, fDetInfo[i].fDetector, maxName);
	 }
	 for (unsigned int i = 0; i < fNDetector; i++) {
	    pp += swapin(pp, fDetInfo[i].fPos, swapit);
	 }

	 // read static struct(s)
	 pp += swapin(pp, fNStatType, swapit);
	 if (fStat) delete [] fStat;
	 if (fNStatType == (int_4u_t)(-1)) {
	    fNStatType = 0;
	    fStat = 0;
	    pp += swapin(pp, fNTotalStat, swapit);
	 }
	 else {
	    fStat = new (nothrow) toc_stat_t [fNStatType];
	    for (unsigned int i = 0; i < fNStatType; i++) 
	       pp += readString (pp, swapit, fStat[i].fName, maxName);
	    for (unsigned int i = 0; i < fNStatType; i++) 
	       pp += readString (pp, swapit, fStat[i].fDetector, maxName);
	    for (unsigned int i = 0; i < fNStatType; i++) 
	       pp += swapin(pp, fStat[i].fStatInstance, swapit);
	    pp += swapin(pp, fNTotalStat, swapit);
	    // skip info
	    pp += fNTotalStat * 20;
	 }
	 
	 // read list of data blocks: ADC, Proc, Sim, Ser, Summary
	 for (int j = 0; j < 5; j++) {
	    pp += swapin(pp, fNData[j], swapit);
	    if (fNData[j] == (int_4u_t)(-1)) {
	       fNData[j] = 0;
	       allocate (0, j); // do it in two steps to avoid wrong copy!
	    }
	    else {
	       allocate (0, j); // do it in two steps to avoid wrong copy!
	       allocate (fNData[j] + 1, j);
	       for (unsigned int i = 0; i < fNData[j]; i++) 
		  pp += readString (pp, swapit, fData[j][i].fName, maxName);
	       if (j == 0) {
		  for (unsigned int i = 0; i < fNData[j]; i++) 
		     pp += swapin(pp, fData[j][i].fChannelID, swapit);
		  for (unsigned int i = 0; i < fNData[j]; i++) 
		     pp += swapin(p, fData[j][i].fGroupID, swapit);
	       }

	       for (unsigned int i = 0; i < fNFrame * fNData[j]; i++) 
		  pp += swapin(pp, fDataPosBuf[j][i], swapit);
	    }
	    // make sure entries are sorted by name
	    // toc_data_t* begin = fData[j];
	    // toc_data_t* end = fData[j] + fNData[j];
	    // sort (begin, end);
	 }
      }
      //-------------------------------- Ignore trigger entries
      for (int j=0; j<2; ++j) {
	 fNEvt[j] = 0;
	 fEvtTotal[j] = 0;
	 fEvt[j] = 0;
      }
      return fLen;
   }

//______________________________________________________________________________
   bool toc_t::scan (const char* p, int len, bool swapit)
   {
      // skip frame file header
      fileheader_t fh;
      const char* pp = p + fh.read (p);
      int version = fh.fVersion;
   
      // reset SH class numbers
      for (unsigned int i = 0; i < fNSH; i++) {
         fSH[i].fSHid = 0;
      }
   
      // loop over frame structures
      int frame = -1;
      bool firstADC = true;
      bool firstSer = true;
      generic_t el;
      while (pp < p + len) {
         // read element head
         el.read (version, pp, swapit, false);
      
         // SH structure
         if (el.fClass == 1) {
            // read it
            dict_header_t sh;
            sh.read (version, pp, swapit);
            // check if in list
            for (unsigned int i = 0; i < fNSH; i++) {
               if (strncmp (sh.fName, fSH[i].fSHName, maxName) == 0) {
                  fSH[i].fSHid = sh.fClassNum;
                  break;
               }
            }
         }
         
         // SE structure (skip)
         else if (el.fClass == 2) {
         }
         
         // if others look up which 
         else {
            string name;
            for (unsigned int i = 0; i < fNSH; i++) {
               if (el.fClass == fSH[i].fSHid) {
                  name = fSH[i].fSHName;
                  break;
               }
            }
            datatype_t dtype = (datatype_t)-1;
            if (name == "FrAdcData") {
               dtype = kAdcData; 
            }
            else if (name == "FrProcData") {
               dtype = kProcData; 
            }
            else if (name == "FrSimData") {
               dtype = kSimData; 
            }
            else if (name == "FrSerData") {
               dtype = kSerData; 
            }
            // adc data
            if ((frame >= 0) && (dtype != (datatype_t)-1)) {
               if (firstADC && (dtype == kAdcData)) {
                  fFrames[frame].fNFirstADC = (int_8u_t) (pp - p);
                  firstADC = false;
               }
               if (firstSer && (dtype == kSerData)) {
                  fFrames[frame].fNFirstSer = (int_8u_t) (pp - p);
                  firstSer = false;
               }
               adcdata_t adc;
               if (adc.read (version, dtype, pp, swapit) <= 0) {
                  continue;
               }
               toc_data_t* chn;
               if (frame == 0) {
                  chn = add (adc.fName, dtype);
                  if (chn) {
                     chn->fChannelID = adc.fChannelNumber;
                     chn->fGroupID = adc.fChannelGroup;
                  }
               }
               else {
                  chn = find (adc.fName);
               }
               if (chn) {
                  chn->fPosition[frame] = (int) (pp - p);
               }
            }
            
            // frame header
            else if (name == "FrameH") {
               if (frame + 1 >= (int)fNFrame) {
                  return false;
               }
               frame++;
               frameheader_t fh;
               if (fh.read (version, pp, swapit) <= 0) {
                  return false;
               }
               fFrames[frame].fDataQual = fh.fDataQual;
               fFrames[frame].fGTimeS = fh.fGTimeS;
               fFrames[frame].fGTimeN = fh.fGTimeN;
               fFrames[frame].fDt = fh.fFrameLen;
               fFrames[frame].fRun = fh.fRun;
               fFrames[frame].fFrame = fh.fFrame;
               fFrames[frame].fPositionH = (int) (pp - p);
               fFrames[frame].fNFirstADC = 0;
               fFrames[frame].fNFirstSer = 0;
               fFrames[frame].fNFirstTable = 0;
               fFrames[frame].fNFirstMsg = 0;
               firstADC = true;
            }
            // detector structure
            else if (name == "Detector") {
               detector_t ndet;
               if (ndet.read (version, pp, swapit) <= 0) {
                  return false;
               }
               toc_detector_t* det = addDetector (ndet.fName);
               if (det != 0) det->fPos = (int_8u_t) (pp - p);
            }
            // vector data (skip)
            // detector structure (skip)
            // history structure (skip)
            // raw data structure (skip)
            // end of frame (skip)
            // end of file (skip)
            // TOC (skip)
         }
         if (el.fLen <= 0) {
            return false;
         }
         pp += el.fLen;
      }
   
      fNFrame = frame + 1;
      return true;
   }

//______________________________________________________________________________
   long toc_t::write (int version, char* p, bool swapit) const
   {
      // write toc intro
      char* pp = p + generic_t::write (version, p, swapit);

      if (version < 8) {
	 // need a copy! this guarantees that the addresses stored in
	 // fDataPosBuf are in order
	 toc_t* toc = new (nothrow) toc_t (*this);

	 // swap if necessary
	 if (swapit) {
	    swap (&toc->fULeapS);
	    swap (&toc->fLocalTime);
	    swap (&toc->fNFrame);
	    for (unsigned int i = 0; i < fNFrame; i++) {
	       swap (&toc->fFrames[i].fDataQual);
	       swap (&toc->fFrames[i].fGTimeS);
	       swap (&toc->fFrames[i].fGTimeN);
	       swap (&toc->fFrames[i].fDt);
	       swap (&toc->fFrames[i].fRun);
	       swap (&toc->fFrames[i].fFrame);
	       swap (&toc->fFrames[i].fPositionH);
	       swap (&toc->fFrames[i].fNFirstADC);
	       swap (&toc->fFrames[i].fNFirstSer);
	       swap (&toc->fFrames[i].fNFirstTable);
	       swap (&toc->fFrames[i].fNFirstMsg);
	    }
	    swap (&toc->fNSH);
	    for (unsigned int i = 0; i < fNSH; i++) {
	       swap (&toc->fSH[i].fSHid);
	    }
	    swap (&toc->fNDetector);
	    for (unsigned int i = 0; i < fNDetector; i++) {
	       swap (&toc->fDetInfo[i].fPos);
	    }
	    swap (&toc->fNStatType);
	    for (int j = 0; j < 5; j++) {
	       swap (toc->fNData + j);
	       if (j == 0) {
		  for (unsigned int i = 0; i < fNData[j]; i++) {
		     swap (&toc->fData[j][i].fChannelID);
		     swap (&toc->fData[j][i].fGroupID);
		  }
	       }
	       for (unsigned int i = 0; i < fNData[j]; i++) {
		  for (unsigned int k = 0; k < fNFrame; k++) {
		     swap (toc->fData[j][i].fPosition + k);
		  }
	       }
	    }
	 }
   
	 // write toc intro
	 memcpy (pp, &toc->fULeapS, 2);
	 pp += 2;
	 if (version == 4) {
	    memcpy (pp, &toc->fLocalTime, 4);
	    pp += 4;
	 }
	 memcpy (pp, &toc->fNFrame, 4);
	 pp += 4;
   
	 // write list of frames
	 int frlen = 16 * 4;
	 char* buf = new (nothrow) char[fNFrame * frlen];
	 if (version >= 6) {
	    for (unsigned int i = 0; i < fNFrame; i++) {
	       *(int_4u_t*)(buf + i*4) = toc->fFrames[i].fDataQual;
	    }
	    memcpy (pp, buf, fNFrame * 4);
	    pp += fNFrame * 4;
	 }
	 for (unsigned int i = 0; i < fNFrame; i++) {
	    *(int_4u_t*)(buf + (0*fNFrame +i)*4) = toc->fFrames[i].fGTimeS;
	    *(int_4u_t*)(buf + (1*fNFrame +i)*4) = toc->fFrames[i].fGTimeN;
	    *(real_8_t*)(buf + (1*fNFrame +i)*8) = toc->fFrames[i].fDt;
	    *(int_4s_t*)(buf + (4*fNFrame +i)*4) = toc->fFrames[i].fRun;
	    *(int_4u_t*)(buf + (5*fNFrame +i)*4) = toc->fFrames[i].fFrame;
	    *(int_8u_t*)(buf + (3*fNFrame +i)*8) = toc->fFrames[i].fPositionH;
	    *(int_8u_t*)(buf + (4*fNFrame +i)*8) = toc->fFrames[i].fNFirstADC;
	    *(int_8u_t*)(buf + (5*fNFrame +i)*8) = toc->fFrames[i].fNFirstSer;
	    *(int_8u_t*)(buf + (6*fNFrame +i)*8) = toc->fFrames[i].fNFirstTable;
	    *(int_8u_t*)(buf + (7*fNFrame +i)*8) = toc->fFrames[i].fNFirstMsg;
	 }
	 memcpy (pp, buf, fNFrame * frlen);
	 pp += fNFrame * frlen;
	 delete [] buf;
   
	 // write list of frame structure headers
	 memcpy (pp, &toc->fNSH, sizeof (int_4u_t));
	 pp += 4;
	 for (unsigned int i = 0; i < fNSH; i++) {
	    memcpy (pp, &toc->fSH[i].fSHid, sizeof (int_2u_t));
	    pp += 2;
	 }
	 for (unsigned int i = 0; i < fNSH; i++) {
	    pp += writeString (pp, swapit, fSH[i].fSHName);
	 }

	 // write detector struct
	 if (version >= 6) {
	    memcpy (pp, &toc->fNDetector, sizeof (int_4u_t));
	    pp += 4;
	    for (unsigned int i = 0; i < fNDetector; ++i) {
	       pp += writeString (pp, swapit, fDetInfo[i].fDetector);
	       memcpy (pp, &toc->fDetInfo[i].fPos, sizeof (int_8u_t));
	       pp += 8;
	    }
	 }

	 // write static struct
	 toc->fNStatType = 0;
	 memcpy (pp, &toc->fNStatType, sizeof (int_4u_t));
	 pp += 4;
  
	 // write list of data blocks: ADC, Proc, Sim, Ser, Summary
	 for (int j = 0; j < 5; j++) {
	    memcpy (pp, &toc->fNData[j], sizeof (int_4u_t));
	    pp += 4;
	    for (unsigned int i = 0; i < fNData[j]; i++) {
	       pp += writeString (pp, swapit, fData[j][i].fName);
	    }
	    if (j == 0) {
	       for (unsigned int i = 0; i < fNData[j]; i++) {
		  memcpy (pp, &toc->fData[j][i].fChannelID, sizeof (int_4u_t));
		  pp += 4;
	       }
	       for (unsigned int i = 0; i < fNData[j]; i++) {
		  memcpy (pp, &toc->fData[j][i].fGroupID, sizeof (int_4u_t));
		  pp += 4;
	       }
	    }
	    memcpy (pp, toc->fDataPosBuf[j],
		    fNFrame * fNData[j] * sizeof (int_8u_t));
	    pp += fNFrame * fNData[j] * sizeof (int_8u_t);
	 }
   
	 // write list of events
	 for (int i = 0; i < 2; i++) {
	    int_4u_t nTrig = 0;
	    memcpy (pp, &nTrig, sizeof (int_4u_t));
	    pp += 4;
	 }
	 delete toc;
      }

      //--------------------------------  Version > 8
      else {
	 pp += swapout(fULeapS, pp, swapit);
	 pp += swapout(fNFrame, pp, swapit);

	 // read list of frames
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fDataQual,    pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fGTimeS,      pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fGTimeN,      pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fDt,          pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fRun,         pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fFrame,       pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fPositionH,   pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fNFirstADC,   pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fNFirstSer,   pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fNFirstTable, pp, swapit);
	 for (unsigned int i=0; i<fNFrame; ++i) 
	    pp += swapout(fFrames[i].fNFirstMsg,   pp, swapit);
	 
	 // read list of frame structure headers
	 pp += swapout(fNSH, pp, swapit);
	 for (unsigned int i=0; i<fNSH; ++i) 
	    pp += swapout(fSH[i].fSHid,   pp, swapit);
	 for (unsigned int i = 0; i < fNSH; i++) {
	    pp += writeString (pp, swapit, fSH[i].fSHName);
	 }

	 // read list of detector information indices
	 pp += swapout(fNDetector, pp, swapit);
	 for (unsigned int i = 0; i < fNDetector; i++) {
	    pp += writeString (pp, swapit, fDetInfo[i].fDetector);
	 }
	 for (unsigned int i = 0; i < fNDetector; i++) {
	    pp += swapout(fDetInfo[i].fPos, pp, swapit);
	 }

	 // read static struct(s)
	 if (fNStatType == 0) {
	    pp += swapout(int_4u_t(-1), pp, swapit);
	    pp += swapout(fNTotalStat, pp, swapit);
	 }
	 else {
	    pp += swapout(fNStatType, pp, swapit);	    
	    for (unsigned int i = 0; i < fNStatType; i++)
	       pp += writeString (pp, swapit, fStat[i].fName);
	    for (unsigned int i = 0; i < fNStatType; i++)
	       pp += writeString (pp, swapit, fStat[i].fDetector);
	    for (unsigned int i = 0; i < fNStatType; i++)
	       pp += swapout(fStat[i].fStatInstance, pp, swapit);
	    pp += swapout(fNTotalStat, pp, swapit);
	    // skip info
	    pp += fNTotalStat * 20;
	 }
	 
	 // read list of data blocks: ADC, Proc, Sim, Ser, Summary
	 for (int j = 0; j < 5; j++) {
	    pp += swapout(fNData[j], pp, swapit);
	    for (unsigned int i = 0; i < fNData[j]; i++) 
	       pp += writeString (pp, swapit, fData[j][i].fName);
	    if (j == 0) {
	       for (unsigned int i = 0; i < fNData[j]; i++) 
		  pp += swapout(fData[j][i].fChannelID, pp, swapit);
	       for (unsigned int i = 0; i < fNData[j]; i++) 
		  pp += swapout(fData[j][i].fGroupID, pp, swapit);
	    }

	    for (unsigned int i = 0; i < fNFrame * fNData[j]; i++) 
	       pp += swapout(fDataPosBuf[j][i], pp, swapit);
	 }
   
	 // write list of events (real, simulated)
	 for (int j = 0; j < 2; ++j) {
	    pp += swapout(fNEvt[j],  pp, swapit);
	    pp += swapout(fEvtTotal[j], pp, swapit);
	 }
      }
      return fixlength (version, p, long(pp - p), swapit);
   }

//______________________________________________________________________________
   int toc_t::size (int version) const 
   {
      if (version < 6) {
         // toc intro
         int len = 8 + 10;
         // list of frames
         len += fNFrame * 64;
         // list of frame structure headers
         len += 4;
         for (unsigned int i = 0; i < fNSH; i++) {
            len += 2 + 2 + strlen (fSH[i].fSHName) + 1;
         }
         // static struct
         len += 4;
         // list of data blocks: ADC, Proc, Sim, Ser, Summary
         for (int j = 0; j < 5; j++) {
            len += 4;
            for (unsigned int i = 0; i < fNData[j]; i++) {
               len += 2 + strlen (fData[j][i].fName) + 1;
               if (j == 0) len += 8;
            }
            len += fNFrame * fNData[j] * sizeof (int_8u_t);
         }
         // list of events
         len += 2 * 4;
         return len;
      }
      else {
         // toc intro
         int len = 14 + 6;
         // list of frames
         len += fNFrame * 68;
         // list of frame structure headers
         len += 4;
         for (unsigned int i = 0; i < fNSH; i++) {
            len += 2 + 2 + strlen (fSH[i].fSHName) + 1;
         }
         // detector strcuture
         len += 4;
         for (unsigned int i = 0; i < fNDetector; i++) {
            len += 8 + 2 + strlen (fDetInfo[i].fDetector) + 1;
         }
         // static struct
         len += 4;
         // list of data blocks: ADC, Proc, Sim, Ser, Summary
         for (int j = 0; j < 5; j++) {
            len += 4;
            for (unsigned int i = 0; i < fNData[j]; i++) {
               len += 2 + strlen (fData[j][i].fName) + 1;
               if (j == 0) len += 8;
            }
            len += fNFrame * fNData[j] * sizeof (int_8u_t);
         }
         // list of events
         len += 2 * 4;
         return len;
      }
   }

//______________________________________________________________________________
   toc_data_t* toc_t::find (const char* chnname) const
   {
      toc_data_t key;
      save_strncpy (key.fName, chnname, maxName);
      for (int i = 0; i < 5; ++i) {
         if (fNData[i] > 0) {
            toc_data_t* begin = fData[i];
            toc_data_t* end = fData[i] + fNData[i];
            toc_data_t* f = lower_bound (begin, end, key);
            if ((f != end) && (*f == key)) {
               return f;
            }
         }
      }
      return 0;
   }

//______________________________________________________________________________
   toc_data_t* toc_t::find (const char* chnname, int i) const
   {
      if ((i < 0) || (i > 4)) {
         return 0;
      }
      toc_data_t key;
      save_strncpy (key.fName, chnname, maxName);
      toc_data_t* begin = fData[i];
      toc_data_t* end = fData[i] + fNData[i];
      toc_data_t* f = lower_bound (begin, end, key);
      if ((f != end) && (*f == key)) {
         return f;
      }
      else {
         return 0;
      }
   }

//______________________________________________________________________________
   toc_data_t* toc_t::add (const char* chnname, int i)
   {
      if ((i < 0) || (i > 4)) {
         return 0;
      }
      // make sure we have enough room
      if ((int)fNData[i] + 1 >= fChannelMax[i]) {
         if (!allocate (2 * fChannelMax[i], i)) {
            return 0;
         }
      }
      toc_data_t key;
      save_strncpy (key.fName, chnname, maxName);
      toc_data_t* begin = fData[i];
      toc_data_t* end = fData[i] + fNData[i];
      toc_data_t* f = lower_bound (begin, end, key);
      // end of chain
      if (f == end) {
         ++fNData[i];
         save_strncpy (f->fName, chnname, maxName);
         return f;
      }
      // already in there
      else if (*f == key) {
         return f;
      }
      // insert
      else {
         toc_data_t temp = *end;
         int len = (char*)end - (char*)f;
         memmove (f + 1, f, len);
         *f = temp;
         save_strncpy (f->fName, chnname, maxName);
         fNData[i]++;
         return f;
      }
      return 0;
   }

//______________________________________________________________________________
   toc_detector_t* toc_t::findDetector (const char* name) const
   {
      unsigned int ins = 0;
      for (; ins < fNDetector; ++ins) {
         int res = strcmp (name, fDetInfo[ins].fDetector);
         if (res == 0) {
            return fDetInfo + ins; // exact match
         }
         else if (res > 0) {
            break;
         }
      }
      return 0;
   }

//______________________________________________________________________________
   toc_detector_t* toc_t::addDetector (const char* name)
   {
      unsigned int ins = 0;
      for (; ins < fNDetector; ++ins) {
         int res = strncmp (name, fDetInfo[ins].fDetector, maxName-1);
         if (res == 0) {
            return 0; // exact match
         }
         else if (res < 0) {
            break;
         }
      }
      toc_detector_t* det = new toc_detector_t[fNDetector+1];
      for (unsigned int i = 0; i < ins; ++i) {
         det[i] = fDetInfo[i];
      }
      for (unsigned int i = ins; i < fNDetector; ++i) {
         det[i+1] = fDetInfo[i];
      }
      save_strncpy (det[ins].fDetector, name, maxName);
      delete [] fDetInfo;
      fDetInfo = det;
      ++fNDetector;
      return fDetInfo + ins;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// output operators					                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

   std::ostream& dumpframe (std::ostream& os, const char* p, int len)
   {
      // frame file header
      fileheader_t fh;
      const char* pp = p + fh.read (p);
      dump (os, fh);
      bool swapit = (fh.fByteOrder2 != 0x1234);
      int version = fh.fVersion;
   
      // SH class number array
      const int kSHMax = 100;
      int_4u_t		fNSH = 0;
      toc_SH_t		fSH[kSHMax];
   
      // loop over frame structures
      generic_t el;
      while (pp < p + len) {
         // read element head
         el.read (version, pp, swapit, false);
      
         // SH structure
         if (el.fClass == 1) {
            // read it
            dict_header_t sh;
            sh.read (version, pp, swapit);
            dump (os << endl, sh, version, false);
            // check if in list
            bool found = false;
            for (unsigned int i = 0; i < fNSH; i++) {
               if (strncmp (sh.fName, fSH[i].fSHName, maxName) == 0) {
                  found = true;
                  break;
               }
            }
            // add to list
            if (!found && ((int)fNSH < kSHMax)) {
               save_strncpy (fSH[fNSH].fSHName, sh.fName, maxName);
               fSH[fNSH].fSHid = sh.fClassNum;
               fNSH++;
            }
         }
         
         // SE structure (skip)
         else if (el.fClass == 2) {
            // read it
            dict_element_t se;
            se.read (version, pp, swapit);
            dump (os << endl, se, version);
         }
         
         // if others look up which 
         else {
            string name;
            for (unsigned int i = 0; i < fNSH; i++) {
               if (el.fClass == fSH[i].fSHid) {
                  name = fSH[i].fSHName;
                  break;
               }
            }
            datatype_t dtype = (datatype_t)-1;
            if (name == "FrAdcData") {
               dtype = kAdcData; 
            }
            else if (name == "FrProcData") {
               dtype = kProcData; 
            }
            else if (name == "FrSimData") {
               dtype = kSimData; 
            }
            else if (name == "FrSerData") {
               dtype = kSerData; 
            }
         
            // adc, proc, sim, ser data
            if (dtype != (datatype_t)-1) {
               // read it
               adcdata_t adc;
               if (adc.read (version, dtype, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, adc, version);
            }
            // vector data
            else if (name == "FrVect") {
               // read it
               frvect_t vec;
               if (vec.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, vec, version);
            }
            // frame header
            else if (name == "FrameH") {
               frameheader_t fh;
               if (fh.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, fh, version);
            }
            // detector structure
            else if (name == "FrDetector") {
               detector_t det;
               if (det.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, det, version);
            }
            // history structure
            else if (name == "FrHistory") {
               hist_t h;
               if (h.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, h, version);
            }
            // raw data structure
            else if (name == "FrRawData") {
               rawdata_t raw;
               if (raw.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, raw, version);
            }
            // end of frame
            else if (name == "FrEndOfFrame") {
               endof_frame_t eof;
               if (eof.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, eof, version);
            }
            // end of file
            else if (name == "FrEndOfFile") {
               endof_file_t eof;
               if (eof.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, eof, version);
            }
            // TOC
            else if (name == "FrTOC") {
               toc_t toc;
               if (toc.read (version, pp, swapit) <= 0) {
                  return os;
               }
               dump (os << endl, toc, version);
            }
            // unknown structure
            else {
               ios::fmtflags oldFlags = os.flags();
               os << endl << name << ".length ";
               for (int i = name.size(); i < 14; i++) os << " ";
               os << " = " << el.fLen;
               os << endl << name << ".address";
               for (int i = name.size(); i < 14; i++) os << " ";
               os << " = (" << el.fClass << "," <<
                  el.fInstance << ")";
               os.setf (oldFlags);
            }
         }
         pp += el.fLen;
         if (el.fLen <= 0) {
            cerr << "Illegal structure length " << el.fLen << endl;
            return os;
         }
      }
   
      return os;
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const fileheader_t& h)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "FileHeader.magic       = " << h.fIGWD << endl;
      os << "FileHeader.version     = " << (int)h.fVersion << endl;
      os << "FileHeader.mvers       = " << (int)h.fMinorVersion << endl;
      os << "FileHeader.size_int2   = " << (int)h.fSize_int2 << endl;
      os << "FileHeader.size_int4   = " << (int)h.fSize_int4 << endl;
      os << "FileHeader.size_int8   = " << (int)h.fSize_int8 << endl;
      os << "FileHeader.size_real4  = " << (int)h.fSize_real4 << endl;
      os << "FileHeader.size_real8  = " << (int)h.fSize_real8 << endl;
      os << "FileHeader.byteord2    = " << hex << showbase << 
         h.fByteOrder2 << endl;
      os << "FileHeader.byteord4    = " << hex << showbase <<
         h.fByteOrder4 << endl;
      os << "FileHeader.byteord8    = " << hex << showbase <<
         h.fByteOrder8 << dec << endl;
      os << "FileHeader.Pi4         = " << setprecision (7) << 
         h.fPi4 << endl;
      os << "FileHeader.Pi8         = " << setprecision (15) << 
         h.fPi8 << endl;
      os << "FileHeader.alpha       = " << h.fAlpha[0] <<  
         h.fAlpha[1];
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const fileheader_t& h)
   {
      return dump (os, h);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const dict_element_t& el,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "FrSE.length            = " << el.fLen << endl;
      os << "FrSE.address           = (" << el.fClass << "," <<
         el.fInstance << ")" << endl;
      os << "FrSE.name              = " << el.fName << endl;
      os << "FrSE.type              = " << el.fType << endl;
      os << "FrSE.comment           = " << el.fComment;
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const dict_element_t& el)
   {
      return dump (os, el);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const dict_header_t& h,
                     int version, bool elements)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "FrSH.length            = " << h.fLen << endl;
      os << "FrSH.address           = (" << h.fClass << "," <<
         h.fInstance << ")" << endl;
      os << "FrSH.name              = " << h.fName << endl;
      os << "FrSH.class             = " << h.fClassNum << endl;
      os << "FrSH.comment           = " << h.fComment << endl;
      os << "FrSH.number            = " << h.fElementNum;
      if (elements) {
         for (int i = 0; i < h.fElementNum; i++) {
            os << endl << h.fElements[i];
         }
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const dict_header_t& h)
   {
      return dump (os, h);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const dict_t& d,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "FrSH.number            = " << d.fDictNum << endl;
      for (int i = 0; i < d.fDictNum; i++) {
         os << d.fDict[i];
         if (i < d.fDictNum - 1) os << endl;
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const dict_t& d)
   {
      return dump (os, d);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const frameheader_t& h,
                     int version)
   {
      const char* const ptr4[] = {"type  ", "user  ", "detSim", 
         "detPrc", "hist  ", "raw   ", "proc  ", "strain", "sim   ", 
         "trig  ", "evnt  ", "sum   ", "aux   ", "table ", 0};
      const char* const ptr5[] = {"type  ", "user  ", "detSim", 
         "detPrc", "hist  ", "raw   ", "proc  ", "sim   ", 
         "trig  ", "evnt  ", "sum   ", "aux   ", "table ", 0};
      ios::fmtflags oldFlags = os.flags();
      os << "FrameHeader.length     = " << h.fLen << endl;
      os << "FrameHeader.address    = (" << h.fClass << "," <<
         h.fInstance << ")" << endl;
      os << "FrameHeader.project    = " << h.fName << endl;
      os << "FrameHeader.run        = " << h.fRun << endl;
      os << "FrameHeader.frame      = " << h.fFrame << endl;
      os << "FrameHeader.dataQuality= " << h.fDataQual << endl;
      os << "FrameHeader.GPS sec    = " << h.fGTimeS << endl;
      os << "FrameHeader.GPS nsec   = " << h.fGTimeN << endl;
      os << "FrameHeader.leap sec   = " << h.fULeapS << endl;
      if (version == 4) {
         os << "FrameHeader.local time = " << h.fLocalTime << endl;
      }
      os << "FrameHeader.dt         = " << h.fFrameLen << endl;
      const char* const* ptr = (version == 4) ? ptr4 : ptr5;
      for (int i = 0; (i < 14) && ptr[i]; i++) {
         os << "FrameHeader." << ptr[i] << "     = (" << 
            h.fDir[i].fDataClass << "," << 
            h.fDir[i].fDataInstance << ")";
         if (ptr[i+1]) os << endl;
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const frameheader_t& h)
   {
      return dump (os, h);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const detector_t& det,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "Detector.length        = " << det.fLen << endl;
      os << "Detector.address       = (" << det.fClass << "," <<
         det.fInstance << ")" << endl;
      os << "Detector.name          = " << det.fName << endl;
      os << "Detector.longitude(o)  = " << det.fLongitudeD << endl;
      os << "Detector.longitude(')  = " << det.fLongitudeM << endl;
      os << "Detector.longitude(\")  = " << det.fLongitudeS << endl;
      os << "Detector.latitude(o)   = " << det.fLatitudeD << endl;
      os << "Detector.latitude(')   = " << det.fLatitudeM << endl;
      os << "Detector.latitude(\")   = " << det.fLatitudeS << endl;
      os << "Detector.elevation     = " << det.fElevation << endl;
      os << "Detector.azimuth X arm = " << det.fArmXAzimuth << endl;
      os << "Detector.azimuth Y arm = " << det.fArmYAzimuth << endl;
      if (version > 4) {
         os << "Detector.altitude X arm= " << det.fArmXAltitude << endl;
         os << "Detector.altitude Y arm= " << det.fArmYAltitude << endl;
         os << "Detector.midpoint X arm= " << det.fArmXMidpoint << endl;
         os << "Detector.midpoint Y arm= " << det.fArmYMidpoint << endl;
         os << "Detector.localtime     = " << det.fLocalTime << endl;
         os << "Detector.dataquality   = " << det.fDataQuality << endl;
         os << "Detector.qabits        = " << det.fQaBitList << endl;
      }
      os << "Detector.more          = (" << det.fMore[0].fDataClass << 
         "," << det.fMore[0].fDataInstance << ")" << endl;
      os << "Detector.moreTable     = (" << det.fMore[1].fDataClass << 
         "," << det.fMore[1].fDataInstance << ")";
      if (version > 4) {
         os << endl << "Detector.next          = (" << 
            det.fMore[2].fDataClass << "," << 
            det.fMore[2].fDataInstance << ")";
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const detector_t& det)
   {
      return dump (os, det);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const hist_t& hist,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "History.length         = " << hist.fLen << endl;
      os << "History.address        = (" << hist.fClass << "," <<
         hist.fInstance << ")" << endl;
      os << "History.name           = " << hist.fName << endl;
      os << "History.time           = " << hist.fTime << endl;
      os << "History.comment        = " << hist.fComment << endl;
      os << "History.next           = (" << hist.fNext.fDataClass << 
         "," << hist.fNext.fDataInstance << ")";
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const hist_t& hist)
   {
      return dump (os, hist);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const rawdata_t& raw,
                     int version)
   {
      const char* const ptr[] = {"serial", "ADC   ", "table ", 
         "logmsg", "more  "};
      ios::fmtflags oldFlags = os.flags();
      os << "RawData.length         = " << raw.fLen << endl;
      os << "RawData.address        = (" << raw.fClass << "," <<
         raw.fInstance << ")" << endl;
      os << "RawData.name           = " << raw.fName << endl;
      for (int i = 0; i < 5; i++) {
         os << "RawData." << ptr[i] << "         = (" << 
            raw.fData[i].fDataClass << "," << 
            raw.fData[i].fDataInstance << ")";
         if (i < 4) os << endl;
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const rawdata_t& raw)
   {
      return dump (os, raw);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const adcdata_t& adc,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      string n;
      switch (adc.fDatatype) {
         case kAdcData:
            n = "ADC";
            break;
         case kProcData:
            n = "Prc";
            break;
         case kSimData:
            n = "Sim";
            break;
         case kSerData:
            n = "Ser";
            break;
         default:
            n = "XXX";
            break;
      }
      os << n << ".length             = " << adc.fLen << endl;
      os << n << ".address            = (" << adc.fClass << "," <<
         adc.fInstance << ")" << endl;
      os << n << ".name               = " << adc.fName << endl;
      os << n << ".comment            = " << adc.fComment << endl;
      if (adc.fDatatype == kAdcData) {
         os << n << ".channelGroup       = " << adc.fChannelGroup << endl;
         os << n << ".channelNumber      = " << adc.fChannelNumber << endl;
         os << n << ".nBits              = " << adc.fNBits << endl;
         os << n << ".bias               = " << adc.fBias << endl;
         os << n << ".slope              = " << adc.fSlope << endl;
         os << n << ".unit               = " << adc.fUnit << endl;
      }
      os << n << ".sampleRate         = " << adc.fSampleRate << endl;
      os << n << ".timeOffsetS        = " << adc.fTimeOffsetS << endl;
      os << n << ".timNOffsetN        = " << adc.fTimeOffsetN << endl;
      if (adc.fDatatype == kAdcData) {
         os << n << ".fShift             = " << adc.fFShift << endl;
         os << n << ".fPhase             = " << adc.fPhase << endl;
         os << n << ".dataValid          = " << adc.fDataValid << endl;
      }
      os << n << ".data               = (" << 
         adc.fData[0].fDataClass << "," <<
         adc.fData[0].fDataInstance << ")" << endl;
      os << n << ".user               = (" << 
         adc.fData[1].fDataClass << "," <<
         adc.fData[1].fDataInstance << ")" << endl;
      os << n << ".next               = (" << 
         adc.fData[2].fDataClass << "," <<
         adc.fData[2].fDataInstance << ")";
      if (adc.fDatatype != kAdcData) {
         os << endl << n << ".table              = (" 
            << adc.fData[3].fDataClass << "," <<
            adc.fData[3].fDataInstance << ")";
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const adcdata_t& adc)
   {
      return dump (os, adc);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const frvect_t& vec,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "Vec.length             = " << vec.fLen << endl;
      os << "Vec.address            = (" << vec.fClass << "," <<
         vec.fInstance << ")" << endl;
      os << "Vec.name               = " << vec.fName << endl;
      os << "Vec.compress           = " << vec.fCompress << endl;
      os << "Vec.type               = " << vec.fType << endl;
      os << "Vec.nData              = " << vec.fNData << endl;
      os << "Vec.nBytes             = " << vec.fNCompBytes << endl;
      os << "Vec.nDim               = " << vec.fNDim << endl;
      for (unsigned int i = 0; (i < vec.fNDim) && (i < 4); i++) {
         os << "Vec.nx[" << i << "]              = " << vec.fNx[i] << endl;
         os << "Vec.dx[" << i << "]              = " << vec.fDx[i] << endl;
         os << "Vec.x0[" << i << "]              = " << vec.fStartX[i] << endl;
         os << "Vec.unitX[" << i << "]           = " << vec.fUnitX[i] << endl;
      }
      os << "Vec.unitY              = " << vec.fUnitY << endl;
      os << "Vec.next               = (" << vec.fNext.fDataClass << "," <<
         vec.fNext.fDataInstance << ")";
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const frvect_t& vec)
   {
      return dump (os, vec);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const endof_frame_t& eof,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "Endframe.length        = " << eof.fLen << endl;
      os << "Endframe.address       = (" << eof.fClass << "," <<
         eof.fInstance << ")" << endl;
      os << "Endframe.run           = " << eof.fRun << endl;
      os << "Endframe.frame         = " << eof.fFrame;
      if (version > 4) {
         os << endl << "Endframe.chkType       = " << eof.fChkType;
         os << endl << "Endframe.chkSum        = " << eof.fChkSum;
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const endof_frame_t& eof)
   {
      return dump (os, eof);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const endof_file_t& eof,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "Eof.length             = " << eof.fLen << endl;
      os << "Eof.address            = (" << eof.fClass << "," <<
         eof.fInstance << ")" << endl;
      os << "Eof.nFrames            = " << eof.fNFrames << endl;
      os << "Eof.nBytes             = " << eof.fNBytes << endl;
      os << "Eof.chkFlag            = " << eof.fChkType << endl;
      os << "Eof.chkSum             = " << eof.fChkSum << endl;
      os << "Eof.seekTOC            = " << eof.fSeekTOC;
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const endof_file_t& eof)
   {
      return dump (os, eof);
   }

//______________________________________________________________________________
   std::ostream& dump (std::ostream& os, const toc_t& toc,
                     int version)
   {
      ios::fmtflags oldFlags = os.flags();
      os << "TOC.length             = " << toc.fLen << endl;
      os << "TOC.address            = (" << toc.fClass << "," <<
         toc.fInstance << ")" << endl;
      os << "TOC.ULeapS             = " << toc.fULeapS << endl;
      os << "TOC.localTime          = " << toc.fLocalTime << endl;
      os << "TOC.nFrame             = " << toc.fNFrame << endl;
      for (unsigned int i = 0; i < toc.fNFrame; i++) {
         os << "TOC.dataQual[" << i << "]        = " << 
            toc.fFrames[i].fDataQual << endl;
         os << "TOC.GTimeS[" << i << "]          = " << 
            toc.fFrames[i].fGTimeS << endl;
         os << "TOC.GTimeN[" << i << "]          = " << 
            toc.fFrames[i].fGTimeN << endl;
         os << "TOC.dt[" << i << "]              = " << 
            toc.fFrames[i].fDt << endl;
         os << "TOC.run[" << i << "]             = " << 
            toc.fFrames[i].fRun << endl;
         os << "TOC.frame[" << i << "]           = " << 
            toc.fFrames[i].fFrame << endl;
         os << "TOC.positionH[" << i << "]       = " << 
            toc.fFrames[i].fPositionH << endl;
         os << "TOC.nFirstADC[" << i << "]       = " << 
            toc.fFrames[i].fNFirstADC << endl;
         os << "TOC.nFirstSer[" << i << "]       = " << 
            toc.fFrames[i].fNFirstSer << endl;
         os << "TOC.nFirstTable[" << i << "]     = " << 
            toc.fFrames[i].fNFirstTable << endl;
         os << "TOC.nFirstMsg[" << i << "]       = " << 
            toc.fFrames[i].fNFirstMsg << endl;
      }
      os << "TOC.nSH                = " << toc.fNSH << endl;
      for (unsigned int i = 0; i < toc.fNSH; i++) {
         os << "TOC.SHid[" << i << "]            = " << 
            toc.fSH[i].fSHid << endl;
         os << "TOC.SHName[" << i << "]          = " << 
            toc.fSH[i].fSHName << endl;
      }
      if (version > 4) {
         os << "TOC.nDetector          = " << toc.fNDetector << endl;
         for (unsigned int i = 0; i < toc.fNDetector; i++) {
            os << "TOC.name[" << i << "]            = " << 
               toc.fDetInfo[i].fDetector << endl;
            os << "TOC.position[" << i << "]        = " << 
               toc.fDetInfo[i].fPos << endl;
         }
      }
      os << "TOC.nStatType          = " << toc.fNStatType;
      const char* const dtype[] = {"ADC", "Prc", "Sim", "Ser", "Sum"};
      for (int j = 0; j < 5; j++) {
         os << endl << "TOC.nData[" << dtype[j] << "]         = " << 
            toc.fNData[j];
         for (unsigned int i = 0; i < toc.fNData[j]; i++) {
            os << endl << "TOC." << dtype[j] << "[" << setw (4) << i << 
               "].name     = " << toc.fData[j][i].fName;
            os << endl << "TOC." << dtype[j]<< "[" << setw (4) << i << 
               "].pos      = ";
            for (unsigned int k = 0; k < toc.fNFrame; k++) {
               os << toc.fData[j][i].fPosition[k] << " ";
            }
         }
      }
      os.setf (oldFlags);
      return os;
   }

//______________________________________________________________________________
   std::ostream& operator<< (std::ostream& os, const toc_t& toc)
   {
      return dump (os, toc);
   }


}
