/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DAQC_API_HH
#define DAQC_API_HH
//
//    This is a C++ interface for frame data.
//
#include <sys/time.h>
#include <time.h>
#include <vector>
#include <string>
#include <map>
#include "gmutex.hh"

/**  nds1/nds2 client interface in C++. This interface is independent (for 
  *  now, at least) of the nds2-client package and uses a single base-class 
  *  to access both NDS versions.
  *  \brief C++ client for nds2
  *  \author J. Zweizig (john.zweizig@ligo.org)
  */
namespace sends {

/** @page Network Data Access API (C++)
    A typical online NDS client would do the following:
    \verbatim
    DAQC_api* nds(0);

    //---------  Construct a concrete DAQC_api socket with
    nds = new NDS1Socket;

    //---------  or with
    nds = new NDS2Socket;

    //---------  Open a socket to the specified server port.
    const char* servername = "nds-server:port";
    nds.open(servername);
    if (!nds.TestOpen()) fail("Open failed!");

    //---------  Specify the channel to be read.
    const char* chan = "channel name";
    if (!nds.AddChannel(chan)) fail("Add channel failed!");
    if (nds.RequestOnlineData()) fail("Data Request failed");

    //---------  Specify the channel to be read.
    float* Samples = new float[data_len];
    while (1) {
        int nData = nds.GetData((char*) Samples, data_len);
	if (nData <= 0) break;
        ...  Process data ...
    }
    \endverbatim
    @memo Access channel data through the network data server
    @author John Zweizig
    @version 0.1; Last modified March 5, 2008
    @ingroup IO_daqs
************************************************************************/

//--------------------------------------  Default protocol ports
#ifndef DAQD_PORT
#define DAQD_PORT 8088
#endif
#define NDS2_PORT 31200

/*@{*/

/**  Define channel types. The channel types are used to distinguish 
  *  the requested source of the data. 
  *  \remarks
  *  This expands on and replaces the channel group code in the NDS1 API
  *  which seems to have a very few values (0=normal fast channel, 1000=dmt 
  *  trend channel, 1001=obsolete dmt channel).
  *  \brief %Channel type code enumerator.
  */
enum chantype {
    cUnknown,   //<  Unknown or unspecified default type.
    cOnline,    //<  Online data channel
    cRaw,       //<  Archived raw data channel
    cRDS,       //<  Processed/RDS data channel
    cSTrend,    //<  Second trend data
    cMTrend,    //<  Minute trend data
    cTestPoint  //<  Test point channel

};

/**  Data type enumerator.
  *  \remarks numbering must be contiguous 
  */
enum datatype {
    _undefined = 0,      //<  Undefined data type
    _16bit_integer = 1,  //<  short (16-bit integer) data
    _32bit_integer = 2,  //<  int (32-bit integer) data
    _64bit_integer = 3,  //<  long (64-bit integer) data
    _32bit_float = 4,    //<  float (32-bit floating point) data
    _64bit_double = 5,   //<  double (64-bit floating point) data
    _32bit_complex = 6,  //<  complex<float> (32-bit float complex) data
    _32bit_uint = 7      //<  unsigned int data
};

/**  The channel database contains a description of all the available 
  *  channels including their names, sampling rates and group numbers.
  *  @brief %Channel data-base entry.
  *  @author John Zweizig
  *  @version 1.0; last modified March 5, 2008
  *  @ingroup IO_daqs
  */
struct DAQDChannel {
    ///                  The channel name
    std::string mName;
    ///                  The channel type
    chantype mChanType;
    ///                  The channel sample rate
    double mRate;
    ///                  Data type
    datatype mDatatype;
    ///                  %Channel byte offset in record
    int mBOffset;
    ///                  Data length or error code
    int mStatus;
    ///                  Front-end gain
    float mGain;
    ///                  Unit conversion slope
    float mSlope;
    ///                  Unit conversion offset
    float mOffset;
    ///                  Unit name.
    std::string mUnit;

    DAQDChannel(void) 
	: mChanType(cUnknown), mRate(0.0), mDatatype(_undefined), mBOffset(0), 
	  mStatus(0), mGain(1.0), mSlope(1.0), mOffset(0.0)
    {}

    /// Number of data words
    long nwords(double dt) const {
	return long(dt*mRate + 0.5);
    }

    ///  Convert type name string to channel type
    static chantype cvt_str_chantype(const std::string& str);

    ///  Convert channel type to name string
    static const char* cvt_chantype_str(chantype typ);

    ///  Convert type name string to channel type
    static datatype cvt_str_datatype(const std::string& str);

    ///  Convert channel data type to name string
    static const char* cvt_datatype_str(datatype typ);

    ///  Convert channel data type to name string
    static int datatype_size(datatype typ);
};

/**  The DAQD header record is sent before each block of data.
  *  @brief DAQD header record.
  *  @ingroup IO_daqs
  */
struct DAQDRecHdr {
    ///            Data block length (in bytes) excluding this word.
    int Blen;
    ///            Data length in seconds.
    int Secs;
    ///            GPS time (in seconds) of the start of this data.
    int GPS;
    ///            Time offset of the data from the GPS second (in ns).
    int NSec;
    ///            Data block sequence number (first reply to a request is 0).
    int SeqNum;
};

/**  DAQC_api provides a client interface to the Network Data Server.
  *  The server provides data in the CDS proprietary format or as standard
  *  frames. The interface may also be used to list channel names, or to
  *  specify the channels to be read in.
  *  @brief The DAQD socket class.
  *  @author John Zweizig and Daniel Sigg
  *  @version 1.2; last modified March 5, 2008
  *  @ingroup IO_daqs
  */
class DAQC_api {
public:
    /// list of channels: map between channel name and channel info
    typedef std::vector<DAQDChannel> chan_list;

    /// channel list iterator
    typedef chan_list::iterator channel_iter;

    /// channel list iterator
    typedef chan_list::const_iterator const_channel_iter;

    ///  Wait time type definition.
    typedef double wait_time;

    /// Counter data type definition.
    typedef unsigned long count_type;

public:
    /**  Construct an unopened socket
      *  \brief Default constructor.
      */
    DAQC_api(void);

    /**  Disconnect and close the socket. Release any allocated storage.
      *  \brief Destroy an interface. 
      */
    virtual ~DAQC_api(void);

    /**  Open an existing socket and connect it to a server.
      *  The argument, ipaddr, specifies the IP address of the node on which 
      *  the network data server is running. It may be specified either as 
      *  a symbolic name or as four numeric fields separated by dots.
      *  \brief Connect to a server.
      *  \param ipaddr Server IP address string.
      *  \param ipport Server IP port number.
      *  \param buflen Default receive buffer length.
      *  \return zero if successful, a positive non-zero error code if one was 
      *  returned by DAQD or -1.
      */
    virtual int open (const std::string& ipaddr, int ipport, 
		      long buflen = 1048576) = 0;

    /**  Disconnect and close a socket.
      *  \brief Close the socket.
      */
    virtual void close() = 0;

    /**  Flushes any pending input data from the socket.
      *  \brief Flush input data.
      */
    virtual void flush() = 0;

    /**  Test whether the socket is open.
      *  \brief Test open.
      *  \return true if socket is open and connected
      */
    virtual bool isOpen(void) const;

    /**  Test whether the server is processing a request from this client.
      *  \brief Test if transaction in progress.
      *  \return true if request was sent
      */
    virtual bool isOn(void) const;

    /**  Add the channel specified by a DAQDChannel structure to the request 
      *  list.
      *  \brief Add a channel to the request list.
      *  \param chns   %Channel description.
      *  \return 1 if successful.
      */
    virtual int AddChannel(const DAQDChannel& chns);

    /**  Add the specified channel to the request list. All channels may be 
      *  added by specifying "all" instead of a channel name.
      *  \brief Add a channel to the request list.
      *  \param chan   %Channel name.
      *  \param ty     %Channel type code.
      *  \param rate   Requested sample rate.
      *  \return 1 if if successful.
      */
    virtual int AddChannel(const std::string& chan, chantype ty, double rate);

    /**  The names, sample rates, etc. of all channels known by the server are 
      *  appended into the channel vector. Available() returns the number of 
      *  entries found or -1 if an error occurred.
      *  \brief List all known channels.
      *  \param typ     %Channel type to be requested.
      *  \param gps     GPS tiem for the request.
      *  \param list    %Channel list vector to receive selected channel 
      *                 meta-data.
      *  \param timeout Maximum wait time.
      *  \return Number of channels or -1 if an error occurred.
      */
    virtual int addAvailable(chantype typ, long gps, chan_list& list, 
			     wait_time timeout=-1) = 0;

    /**  The names, sample rates, etc. of all channels known by the server are 
      *  copied into the channel vector. Available() returns the number of 
      *  entries found or -1 if an error occurred.
      *  \brief List all known channels.
      *  \param typ     %Channel type to be requested.
      *  \param gps     GPS time for the request.
      *  \param list    %Channel list vector to receive selected channel 
      *                 meta-data.
      *  \param timeout Maximum wait time.
      *  \return Number of channels or -1 if an error occurred.
      */
    virtual int Available(chantype typ, long gps, chan_list& list, 
			  wait_time timeout=-1);

    /**  Request list start position iterator value.
      *  \brief Request list start.
      *  \return Iterator pointing to the start position of the request list.
      */
    const_channel_iter chan_begin(void) const;

    /**  Request list end position iterator value.
      *  \brief Request list end.
      *  \return Iterator pointing to the end position of the request list.
      */
    const_channel_iter chan_end(void) const;

    /**  Find a channel request list entry correspondeing to the specified 
      *  channel name. If the channel is not found, FundChannel will return
      *  chan_end()
      *  \brief Find a requested channel entry.
      *  \param chan Requested channel name string.
      *  \return Iterator pointing to the channel entry.
      */
    const_channel_iter FindChannel(const std::string& chan) const;

    /**  Find a channel request list entry correspondeing to the specified 
      *  channel name. If the channel is not found, FundChannel will return
      *  chan_end()
      *  \brief Find a requested channel entry.
      *  \param chan Requested channel name string.
      *  \return Iterator pointing to the channel entry.
      */
    channel_iter FindChannel(const std::string& chan);

    /**  Copy data for a specified channel to a user buffer.
      *  @brief Get channel data
      *  @param chan %Channel name
      *  @param data User provided data buffer
      *  @param maxlen Length of user buffer in bytes.
      *  @return Number of float words copied or negative error code.
      */
    virtual int GetChannelData(const std::string& chan, float* data,
			       long maxlen) const;
   
    /**  Receive block of data in the CDS proprietary format.
      *  A single block of data (including the header) is received and stored
      *  in an internal buffer. 
      *  @brief Get a data block from the server.
      *  @param timeout Maximum wait time.
      *  @return data length, or failure.
      */
     virtual int GetData(wait_time timeout = -1);

    /**  Receive a block of data. Transfer the header and data into a
      *  specified pre-allocated buffer.
      *  \brief Receive a data buffer.
      *  \param buf     Pointer to a receive the data header and buffer.
      *  \param timeout Maximum wait time.
      *  \return Number of bytes read or negative.
      */
    virtual int GetData(char** buf, wait_time timeout = -1);

    /**  Get the number of requested channels.
      *  \brief Number of requested channels.
      *  \return Number of requested channels.
      */
    virtual unsigned long nRequest(void) const;

    /**  The network data server is requested to start a net-writer task.
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading CDS data.
      *  \param stride Number of seconds of data to to transfer (1/16th sec 
      *         -> inf).
      *  \param timeout Maximum time to wait for a ewsponse (-1 = infinite)
      *  \return Zero if successful, NDS response or -1.
      */
    virtual int RequestOnlineData (double stride, wait_time timeout) = 0;
   
    /**  The network data server is requested to start a net-writer task
      *  for past data. Start time and duration are given in GPS seconds. 
      *  Only channels explicitly specified by AddChannel() will be written.
      *  \brief Start reading archived data.
      *  \param start GPS time of start of data to be read.
      *  \param duration Number of seconds of data to read.
      *  \param timeout Maximum time to wait for a response (-1 = infinite)
      *  \return Zero if successful, NDS response or -1.
      */
    virtual int RequestData (unsigned long start, unsigned long duration, 
			     wait_time timeout = -1) = 0;

    /**  Remove the specified channel from the request list.
      *  \brief Remove channel from the request list
      *  \param chan %Channel name.
      */
    virtual void RmChannel(const std::string& chan);
   
    /**  Set the abort flag variable. If an abort "button" is specified, 
      *  recv/send will periodically check its state and if it has become 
      *  true, will abort the transaction (\e i.e. return immediately with 
      *  an error condition).
      *  \note The abort variable must be set to false before calling a 
      *  method that receives data or sends a request.
      *  \brief Set abort button variable.
      *  \param abort Pointer to a boolean abort variable.
      */
    virtual void setAbort (bool* abort);

    /**  Set a number debug level. This is equivalent to the boolean entry
      *  described below.
      *  \brief  Set debug print level.
      *  \param debug a non-zero \a debug value requests debugging output.
      */
    virtual void setDebug(int debug=1);

    /**  Setting debug mode to true causes the following to be printed to cout: 
      *  - All request text
      *  - The status code and reply text for each request
      *  - The header of each data block received
      *  - The length of each data/text block received and its buffer size.
      *
      *  \brief  Set debug mode.
      *  \param debug True to set debug mode
      */
    virtual void setDebug(bool debug=true);
   
    /**  StopWriter effectively countermands the RequestXXX() functions.
      *  \brief Stop a data writer.
      *  \return Server response code or -1 if no writer is active.
      */
    virtual int StopWriter() = 0;

    /**  The network data server is requested to return start time and 
      *  duration of the data stored on disk (NDS1 only).
      *  \brief Request known time intervals.
      *  \param type     %Channel type code to query
      *  \param start    Start time of available data [returned]
      *  \param duration Available data duration [returned].
      *  \param timeout  Maximum ewat time for reply.
      *  \return Zero on success.
      */
    virtual int Times(chantype type, unsigned long& start, 
		      unsigned long& duration, wait_time timeout = -1);

    /**  The version and revision numbers of the server software are returned 
      *  in a single float as (Version + 0.01*Revision).
      *  \brief Get the server version ID.
      *  \return Server version plus 0.01 times the spftware revision.
      */
     virtual float Version(void) const;

    /**  Execution is blocked until data are available to the socket. This
      *  can be used to wait for data after a request has been made. The
      *  calling function can then e.g. allocate a buffer before calling 
      *  GetData(), GetName() or GetFrame(). If poll is true the function
      *  returns immediately with the return value indicating whether data 
      *  are present.
      *  \brief Wait for data.
      *  \param poll If true WaitforData returns immediately. The return 
      *              codes indicates whether data are available.
      *  \return 0: no data available, &gt;0: data are available, &lt; error.
      */
    virtual int  WaitforData (bool poll = false) = 0;

protected:
    /**  Swap the data words in the in the receive buffer header.
      *  \brief Swap the receive buffer header.
      */
    void SwapHeader(void);

    /**  Swap all the data in the receive buffer.
      *  \brief Swap received data.
      *  \return Zero on success.
      */
    int SwapData(void);

    /**  Convert the hexidecimal number in \a text to an integer. 
      *  \brief convert a hex number to integer.
      *  \param text Hexidemimal number.
      *  \param N    Number of hex digits to convert.
      *  \return Converted hex number or -1.
      */
    int CVHex(const char* text, int N);
   
    /**  Receive a data record consisting of a header record followed by 
      *  a data block. The header is checked for an out-of-band message 
      *  type (e.g. reconfigure block) and the return code is set accordingly.
      *  A memory buffer of correct length is allocated automatically.
      *  \brief Receive a data header and data block.
      *  \param timeout Maximum wait time
      *  \return Data length, -1 on error, -2 on reconfigure block.
      */
    virtual int RecvData(wait_time timeout=-1) = 0;
   
    /**  Receive a float data word, swap the byte ordering if necessary
      *  and store thedata in the argument location.
      *  \brief Receive a float data word.
      *  \param data    Float into which received data is to be placed.
      *  \param timeout Maximum time to wait for data.
      *  \return Number of bytes received or -1.
      */
    virtual int RecvFloat(float& data, wait_time timeout=-1);

    /**  Receive an integer data word, swap the byte ordering if necessary
      *  and store the data in the argument location.
      *  \brief Receive an integer data word.
      *  \param data    Integer into which received data is to be placed.
      *  \param timeout Maximum time to wait for data.
      *  \return Number of bytes received or -1.
      */
    virtual int RecvInt(int& data, wait_time timeout=-1);

    /**  Read up to \a len bytes are read from the socket into \c *buf. If 
      *  \a readall is true, RecvRec will perform as many reads as is 
      *  necessary to fill \a buf.
      *  \brief Receive data from the socket.
      *  \param buf Pointer to buffer to oreceive data.
      *  \param len Maximum number of bytes to revceive.
      *  \param readall Read specified (maximum) number of bytes from one or
      *                 more packets.
      *  \param maxwait Maximum wait time, or -1 for indefinite wait.
      *  \return Number of bytes read, 0 on End-of-file or negative on error.
      */
    virtual int RecvRec(char *buf, long len, bool readall=false, 
			wait_time maxwait=-1) = 0;
    
    /**  Receive a reconfigure structure of channel meta-data. The data
      *  are unpacked from the receive buffer into the requested channel 
      *  list.
      *  \brief Receive a reconfigure block.
      *  \param len     Length of reconfigure block (excluding header)
      *  \param maxwait Maximum wait time.
      *  \return On error -1 or -2 on success.
      */
    virtual int RecvReconfig(count_type len, wait_time maxwait=-1) = 0;
   
    
    /**  Receive a string consisting of a 4-byte length followed by a
      *   text field.
      *  \brief Receive a string.
      *  \param str     String into which input data will be stored.
      *  \param maxwait Maximum wait time.
      *  \return length of string or -1.
      */
    virtual int RecvStr(std::string& str, wait_time maxwait=-1);

public:
    /**  Receive buffer class holds the receive buffer and its header.
      *  \brief receive buffer.
      */
    class recvBuf {
    public:
	///  Data type used for size values
	typedef unsigned long size_type;
    public:
	/**  Construct a receive buffer of a specified length.
	  *  \brief receive buffer constructor.
	  *  \param length Buffer length in bytes.
	  */
	recvBuf(size_type length=0);

	/**  Destroy a receive buffer.
	  *  \brief Receive buffer destructor.
	  */
	~recvBuf(void);

	/**  Get the currently allocated buffer capacity.
	  *  \brief Current buffer capacity.
	  *  \return Buffer capacity in bytes.
	  */
	size_type capacity(void) const;

	/**  Release any allocated storage and est the current capacity to 
	  *  zero.
	  *  \brief Release the buffer.
	  */
	void clear(void);

	/**  Return a pointer to the data portion of the buffer.
	  *  \brief Get data pointer.
	  *  \return Character pointer to the start of the buffered data.
	  */
	char* ref_data(void);

	/**  Return a constant pointer to the data portion of the buffer.
	  *  \brief Get data pointer.
	  *  \return Constant pointer to the start of the buffered data.
	  */
	const char* ref_data() const;

	/**  Return a reference to the header structure corresponding to 
	  *  the current data buffer. 
	  *  \brief Get header reference.
	  *  \return Reference to buffer header.
	  */
	DAQDRecHdr& ref_header();

	/**  Return a constant reference to the header structure 
	  *  corresponding to the current data buffer. 
	  *  \brief Get header reference.
	  *  \return Constant reference to buffer header.
	  */
	const DAQDRecHdr& ref_header() const;

	/**  Reserve the requested number of data bytes fo an input buffer.
	  *  If the current capacity exceeds the requested length, no
	  *  further action will be taken.
	  *  \brief Reserve space for a data buffer.
	  *  \param length number of bytes to reserve.
	  */
	void reserve(size_type length);
    private:
	size_type  mLength;
	DAQDRecHdr mHeader;
	char*      mData;
    };

    ///  Receive buffer
    recvBuf mRecvBuf;

protected:
    ///  mutex to protect object.   
    mutable thread::recursivemutex	mux;
   
    ///  Set to true if socket is open and connected.
    bool  mOpened;

    ///  Debug mode specified.
    int mDebug;

    /**  This enumerator specifies the type of data requested through the 
      *  socket. It is set when a data writer is started and tested by the 
      *  data read methods when they are called.
      *  \brief Writer type enumeration.
      */
    enum writer_type {
	NoWriter,      ///<  No write currently running
	NameWriter,    ///<  A name writer is being run by the server
	DataWriter,    ///<  A data writer is being run by the server
	FrameWriter,   ///<  A frame writer is being run by the server
	FastWriter,    ///<  Server is running a fast online data writer
	NDS2Writer     ///<  NDS2 server is in progrss.
    } mWriterType;
   
    ///  Offline flag sent when the writer is started.
    int   mOffline;

    ///  Server version number.
    int   mVersion;

    ///  Server revision number.
    int   mRevision;

    ///  Abort flag pointer
    bool* mAbort;

    ///  %Channel request list.
    chan_list mRequest_List;
};

//======================================  Inline methods
inline DAQC_api::const_channel_iter 
DAQC_api::chan_begin(void) const {
    return mRequest_List.begin();
}

inline DAQC_api::const_channel_iter 
DAQC_api::chan_end(void) const {
    return mRequest_List.end();
}

inline bool 
DAQC_api::isOpen(void) const {
    return mOpened;
}

inline float 
DAQC_api::Version(void) const {
    return mVersion + 0.01*mRevision;
}

inline bool 
DAQC_api::isOn() const {
    return mWriterType != NoWriter;
}

inline unsigned long 
DAQC_api::nRequest(void) const {
    return mRequest_List.size();
}

inline DAQC_api::recvBuf::size_type 
DAQC_api::recvBuf::capacity(void) const {
    return mLength;
}

inline char* 
DAQC_api::recvBuf::ref_data(void) {
    return mData;
}

inline const char* 
DAQC_api::recvBuf::ref_data() const {
    return mData;
}

inline DAQDRecHdr& 
DAQC_api::recvBuf::ref_header() {
    return mHeader;
}

inline const DAQDRecHdr&
DAQC_api::recvBuf:: ref_header() const {
    return mHeader;
}

} // namespace sends

/*@}*/

#endif  //  DAQSOCKET_HH

