/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_METASTAB_HH
#define GENERATOR_METASTAB_HH

#include "generator/DataSource.hh"

namespace generator {

    /**
     *  %Metastable is a one-shot data generator. The generator parameters
     *  are:
     *  <table>
     *  <tr><td>\c t0 </td>
     *      <td> 1.0 </td>
     *      <td> Time duration of off data </td>
     *  </tr>
     *  <tr><td>\c t1  </td>
     *      <td>  1.0 </td>
     *      <td> Time durartion of on data </td>
     *  </tr>
     *  <tr><td>\c Off </td>
     *      <td> 0  </td>
     *      <td> Off data value </td>
     *  </tr>
     *  <tr><td>\c On </td>
     *      <td> 1.0 </td>
     *      <td> On data value </td>
     *  </tr>
     *  </table>
     *
     *  Note that by default the Metastable source retriggers automatically
     *  at the end of each event. This will cause any random parameters to
     *  be regenerated and a new event to follow immediately. 
     *
     *  \brief %Metastable waveform generator.
     *  \version $Id: Metastable.hh 7277 2015-01-12 07:21:33Z john.zweizig@LIGO.ORG $
     *  \author  John Zweizig (john.zweizig@ligo.org)
     */
    class Metastable : public DataSource {
    public:
	/**
	 *  Construct a %Metastable instance with the default parameter
	 *  values, \e i.e. \c t0=1, \c t1=1, \c Off=0 and \c On=1.
	 *  \brief Construct a default %Metastable instance.
	 */
	Metastable(void);

	/**  Destroy a %Metastable data source.
	 *  \brief Destructor
	 */
	~Metastable(void);

	/**
	 *  Make an exact copy of a %Metastable instance.
	 *  \brief Clone and instance/
	 *  \return Pointer tothe new instance.
	 */
	Metastable* clone(void) const;

	/**  Return a constant string with the data source type name
	 *   ("Metastable").
	 *  \brief Data type string.
	 *  \return Constant pointer to the character string "Metastable".
	 */
	const char* getDataType(void) const;

	/**  Returns the duration of the next off-on-off cycle.
	 *  \brief Cycle duration.
	 *  \return Duration of off-on-off cycle.
	 */
	Interval getDuration(void) const;

	/**  Returns the starting offset which is defined to be 0 seconds.
	 *  \return Starting offset.
	 */
	Interval getStartOffset(void) const;

    private:

	/**  Generate the signal data for a %Metastable data source.
	 *  \brief Generate data.
	 *  \param t0   Start time.
	 *  \param dT   Sample time.
	 *  \param N    Number of samples to generate.
	 *  \param data Ouput data array.
	 */
	void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);

    private:
	Time     mT0;   ///< Current event start time 
	Interval mDt0;  ///< Current event Off duration
	Interval mDt1;  ///< Current event On duration
	double   mVoff; ///< Current event Off value
	double   mVon;  ///< Current event On value
    };

    //====================================  Inline methods
    inline const char*
    Metastable::getDataType(void) const {
	return "Metastable";
    }

    inline Interval
    Metastable::getStartOffset(void) const {
	return Interval(0.0);
    }

} // namespace generator

#endif  // GENERATOR_METASTAB_HH
