#ifndef GENERATOR_DAMPEDSINE_HH
#define GENERATOR_DAMPEDSINE_HH

#include "DataSource.hh"

namespace generator {

  /**  %DampedSine is a damped sinusoid generator. The generated waveform is:
    *
    *  \f[ x(t) = A \sin(\omega (t - t_0) + \phi_0) e^{- \omega (t-t_0)/Q } \f]
    *
    *  where the parameters are derived from the generator parameters
    *  as follows: \f$\omega = 2 \pi F\f$ and \f$ \phi_0 = Phi\f$. 
    *
    *  The generator parameters and their defaults are as follows:
    *
    *  <table>
    *  <tr><td>Parameter </td>
    *      <td> Default </td>
    *      <td> Meaning </td>
    *  </tr>
    *  <tr><td>\c A </td>
    *      <td> - </td>
    *      <td> Amplitude of envelope at \c t=to </td>
    *  </tr>
    *  <tr><td>\c F </td>
    *      <td> - </td>
    *      <td> Sinusoid frequency (in Hz) </td>
    *  </tr>
    *  <tr><td>\c Phi </td>
    *      <td> - </td>
    *      <td> Phase at \c t=t0 </td>
    *  </tr>
    *  <tr><td>\c Q </td>
    *      <td> -  </td>
    *      <td> Quality factor </td>
    *  </tr>
    *  <tr><td>\c Width </td>
    *      <td> 6 </td>
    *      <td> Width specifies the duration of the waveform to be generated
    *           generated and is specified in units of \f$ Q / \omega\f$. At 
    *           the default Width value of  6.0, the gaussian envelope has 
    *           decreased to \f$ 1.25 \times 10^{-4}\f$ of its initial
    *           amplitude. </td>
    *  </tr>
    *  </table> 
    *
    *  @memo Damped sinusoid data source.
    *  @version 1.1; Last modified February 23, 2004
    *  @author John Zweizig
    */
  class DampedSine : public DataSource {
  public:

      /**  Construct a defaul damped sin source.
        *  @memo Default constructor
        */
      DampedSine(void);

      /**  Construct a damped sine data source with specified parameters.
        *  @memo %Parameter constructor
        *  @param Amp  is the amplitude (A).
	*  @param Freq is the sinusoid frequency (F).
	*  @param Q    is the quality factor (Q).
	*  @param Phi  is the initial phase (Phi).
	*  @param wid  is the generation width.
	*/
      DampedSine(double Amp, double Freq, double Q, double Phi=0.0, 
		 double wid=6.0);

      /**  Destroy a damped sine data source
        *  @memo Destructor
        */
      ~DampedSine(void);

      /**  Construct an identical data source.
        *  @memo Clone the dameped sine data source
	*  @return pointer to the newly created object.XS 
        */
      DampedSine* clone(void) const;

      /**  This function returns a literal string "DampedSine".
        *  @memo Get the data source type name.
	*  @return Source data type nams string.
        */
      const char* getDataType(void) const;

      /**  Returns the duration of generated data. This is calculated on an
        *  event-by-event basis as \f$Width \times Q / (2 \pi F) \f$.
        *  @memo Get the duration of data to be generated.
        *  @return Duration of generated data.
        */
      Interval getDuration(void) const;

      /**  Returns the start of generated data relative to the critical 
        *  time specified. For %DampedSine this always returns zero.
        *  @memo Get the start time offset of data to be generated.
        *  @return Start time offset relative to \c t=to.
        */
      Interval getStartOffset(void) const;

  private:
      /**  Get digitized data series for a specified time epoch.
        *  @memo  Get generated data series.
	*  @param t0   Start tie of epoch.
	*  @param dT   Digitization sample time
	*  @param N    Number of samples to generate/return
	*  @param data Pre-allocate data array.
	*/
      void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);
  };

  //====================================  Inline functions.
  inline const char*
  DampedSine::getDataType(void) const {
      return "DampedSine";
  }

  inline Interval 
  DampedSine::getStartOffset(void) const {
      return Interval(0.0);
  }

} // namespace generator

#endif  // GENERATOR_DAMPEDSINE_HH
