#include "LscPdBackout.hh"
#include "DynConfig.hh"
#include "PConfig.h"
#include <math.h>
#include <iostream>
#include <sstream>

using namespace std;

//======================================  Constructor
LscPdBackout::LscPdBackout(const char* ifo, const char* port, int pd) 
  : mIfo(ifo), mPort(port), mPD(pd)
{
}

//======================================  Set calibration constants.
void
LscPdBackout::setCalibrate(DynConfig& cal) {
    if (mFiltDB.empty()) {
        string fmFile = mIfo + "LSC.txt";
        setFilterDB(fmFile);
    }
    ostringstream lscPfx;
    lscPfx << getBaseName();
    if (mPD > 0) lscPfx << mPD;
    cal.addVbl(lscPfx.str()+"_Q_GAIN",   mQGain,   1.0);
    cal.addVbl(lscPfx.str()+"_I_GAIN",   mIGain,   1.0);
    cal.addVbl(lscPfx.str()+"_Phase",    mPhase,   0.0);
    cal.addVbl(lscPfx.str()+"_Q_OFFSET", mQOffset, 0.0);
    cal.addVbl(lscPfx.str()+"_I_OFFSET", mIOffset, 0.0);
    cal.addVbl("LscEpicsPi"            , mLscPi,   3.1415);
}

//=======================================  Inline function to add TSeries
inline TSeries 
operator*(const TSeries& ts, double rhs) {
    TSeries rc(ts);
    rc *= rhs;
    return rc;
}

//=======================================  Get data from Filter database
void
LscPdBackout::setFilterDB(const string& fmFile) {

    //----------------------------------  Get the filter database
    mFiltDB.init(fmFile.c_str());
    if (mFiltDB.empty()) return;

    //----------------------------------  Figure out the port name
    ostringstream pdname;
    pdname << mPort;
    if (mPD > 0) pdname << mPD;

    //----------------------------------  Invert the Q-Phase whitening filter
    string filtName = pdname.str()+"_Q";
    if (mFiltDB.exists(filtName)) {
        FilterModule PdQF = mFiltDB[filtName];
	PdQF.setMask(1);
	mPdQFm = PdQF.getInverse();
	mPdQFm.dumpSosData(cout);
    } else {
        cout << "Couldn't find filter " << filtName << " for PD " 
	     << getBaseName() << endl;
    }

    //----------------------------------  Invert the I-Phase whitening filter
    filtName = pdname.str()+"_I";
    if (mFiltDB.exists(filtName)) {
        FilterModule PdIF = mFiltDB[filtName];
	PdIF.setMask(1);
	mPdIFm = PdIF.getInverse();
    } else {
        cout << "Couldn't find filter " << filtName << " for PD " 
	     << getBaseName() << endl;
    }
}

//=======================================  Back out Q-Phase PD data
TSeries 
LscPdBackout::getPdQ(const TSeries& PortQ, const TSeries& PortI) {
    if (mQGain == 0.0) {
        return TSeries(PortQ.getStartTime(), PortQ.getTStep());
    }
#ifdef USE_SINCOS
    double cosphi, sinphi;
    sincos(mPhase * mLscPi / 180.0, &sinphi, &cosphi);
#else
    double cosphi = cos(mPhase * mLscPi / 180.0);
    double sinphi = sin(mPhase * mLscPi / 180.0);
#endif
    TSeries tsPdQ;
    if (cosphi != 0.0) tsPdQ  = PortQ * cosphi;
    if (sinphi != 0.0) tsPdQ += PortI * sinphi;
    TSeries filtPdQ(mPdQFm(tsPdQ));
    filtPdQ *= 1./mQGain;
    filtPdQ -= mQOffset;
    return filtPdQ;
}

//=======================================  Back out I-Phase PD data
TSeries 
LscPdBackout::getPdI(const TSeries& PortQ, const TSeries& PortI) {
    if (mIGain == 0.0) {
        return TSeries(PortI.getStartTime(), PortI.getTStep());
    }
#ifdef USE_SINCOS
    double cosphi, sinphi;
    sincos(mPhase * mLscPi / 180.0, &sinphi, &cosphi);
#else
    double cosphi = cos(mPhase * mLscPi / 180.0);
    double sinphi = sin(mPhase * mLscPi / 180.0);
#endif
    TSeries tsPdI;
    if (cosphi != 0.0) tsPdI  = PortI * cosphi;
    if (sinphi != 0.0) tsPdI -= PortQ * sinphi;
    TSeries filtPdI(mPdIFm(tsPdI));
    filtPdI *= 1./mIGain;
    filtPdI -= mIOffset;
    return filtPdI;
}
