/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FILTERMODULE_HH
#define FILTERMODULE_HH

#include "IIRFilter.hh"
#include <string>
#include <vector>

/**  Single maskable stage of a filter module.
  */
class FilterStage {
public:
    /** Default constructor.
      */
    FilterStage(void);

    /**  Data Constructor
      *  \param name  Filter stage name
      *  \param id    Stage number
      *  \param flags filter flags
      *  \param f     IIR filter.
      */
    FilterStage(const char* name, int id, int flags, const IIRFilter& f);

    /**  Get a constant reference to the filter stage.
      *  \return Filter stage constant reference.
      */
    const IIRFilter& getFilter(void) const;

    /**  Get a reference to the filter stage.
      *  \return Filt stage reference.
      */
    IIRFilter& getFilter(void);

    /**  Get an inverse of the filter stage.
      *  \return Inverse of the filter.
      */
    IIRFilter getInverse(void) const;

    /**  Get apointer to the filter stage name.
      *  \return Constant pointer to the filter name string.
     */
    const char* getName(void) const;

private:
    std::string mName;
    int mFlags;
    int mID;
    IIRFilter mFilter;
};

//======================================  Inline methods.
inline const IIRFilter&
FilterStage::getFilter(void) const {
    return mFilter;
}

inline IIRFilter&
FilterStage::getFilter(void) {
    return mFilter;
}

inline const char*
FilterStage::getName(void) const {
    return mName.c_str();
}

/**  Filter module composed of an arbirary number of switchable filters
  *  This filter module is modeled after the LSC filter module, but has 
  *  an arbitray number of arbitrary length filters.
  *  @memo Filter Module
  *  @author J. Zweizig
  *  @version 1.0; Last modified May 28, 2003
  */
class FilterModule {
public:
  /**  Constructor
    */
  FilterModule(void);

  /** Destructor
    */
  ~FilterModule(void);

  /**  Get reference to a filter.
    *  \return Reference to the currently defined IIR filter.
    */
  const IIRFilter& getFilter(void) const;

  /**  Get the inverse of an IIR filter.
    *  \return IIR filter with the poles and zeroes swapped.
    */
  IIRFilter getInverse(void) const;

  /**  Get the stage mask
    *  \return Currently active bit mask
    */
  int getMask(void) const;

  /**  Filter a time series with the selected stages.
    *  \param ts %Time series to be filtered.
    *  \return Filtered time series.
    */
  TSeries filter(const TSeries& ts);

  /**  Add or replace a stage of the filter module.
    *  @memo Add a stage.
    *  @param N Stage number to be added/replaced.
    *  @param s Stage to be added.
    */
  void addStage(unsigned int N, const FilterStage& s);

  /**  Set the stage mask and calculate the resulting filter.
    *  \param mask Filter stage bit-mask.
    */
  void setMask(int mask);

private:
  int mMask;
  IIRFilter mFilter;
  typedef std::vector<FilterStage> StageList;
  typedef StageList::const_iterator stage_iter;
  StageList mStages;
};

//======================================  Inline methods.
inline const IIRFilter& 
FilterModule::getFilter(void) const {
    return mFilter;
}

inline int 
FilterModule::getMask(void) const {
    return mMask;
}

#endif // FILTERMODULE_HH
