/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GRIDWARE_HH
#define GRIDWARE_HH

#ifndef __CINT__
#include "MultiDacc.hh"
#include "SigFlag.hh"
#include "TrigRaw.hh"
#endif //  !defined(__CINT__)

/**  <h2> Introduction </h2>
  *  The data processing base class defines the basic routines necessary 
  *  for a data processing application. The %GridWare class is designed to
  *  provide a similar environment whether running in the foreground 
  *  (\e i.e. under root) or in the background. The functions
  *  provided by the environment are:
  *  <ul>
  *    <li> Parse common command line arguments.
  *    <li> Read frames.
  *    <li> Copy requested data into time series (TSeries) containers.
  *    <li> Handling attention interrupts.
  *    <li> Clean up on termination.
  *  </ul>
  *
  *  <h2> Writing a %GridWare Monitor </h2>
  *  The strategy for writing a monitor program is to create a class 
  *  (MyClass) based on %GridWare including a constructor, destructor and
  *  ProcessData method. An attention handler may also be provided if
  *  desired. The operations to be performed by each of these methods
  *  are:
  *
  *  <b> Constructor (MyClass(int argc, const char* argv[])) </b>
  *  <ul>
  *    <li> %Process user specific arguments
  *    <li> Define parameters (see TrigPar class).
  *    <li> Configure the process.
  *    <li> Request data channels with MultiDacc::addChannel.
  *  </ul>
  *
  *  <b> Destructor (~MyClass(void)) </b>
  *  <ul>
  *    <li> Save modified parameters
  *    <li> Print results and statistics.
  *  </ul>
  *
  *  <b> Epoch processing (void ProcessData(void)) </b>
  *  <ul>
  *    <li> %Process data from one epoch.
  *    <li> Generate trigger is necessary.
  *    <li> Keep statistics.
  *  </ul>
  *
  *  <b> Interrupt handler (void Attention(void)) </b>
  *  <ul>
  *    <li> Monitor dependent interrupt processing.
  *  </ul>
  *
  *  The \c EXECDAT(myclass) macro is used to generate the main
  *  function. Definition and implementation templates are available 
  *  from DatTemplate.hh and DatTemplate.cc.
  *
  *  The \c EXEROOT(myclass) macro is used to generate a main function
  *  with root graphics. Note that the "TROOT.h" and "TApplication.h"
  *  header files must be included in your monitor implementation file 
  *  if \c EXEROOT is used.
  *
  *  <h2> Running the Monitor </h2>
  *  The base class constructor recognizes the following command line 
  *  arguments: 
  *  <table>
  *  <tr><td>-debug \<lvl\>     </td>
  *      <td>Set debug level to \<lvl\> </td></tr>
  *  <tr><td>-infile \<file\>   </td>
  *      <td><td>Read frames from \<file\> </td></tr>
  *  <tr><td>-inlist \<list\>   </td>
  *      <td><td>Read frames from files listed in \<list\> </td></tr>
  *  <tr><td>-maxframes \<nFr\> </td>
  *      <td><td>Stop after reading \<nFr\> frames </td></tr>
  *  <tr><td>-partition \<part\></td>
  *      <td><td>Read from shared memory partition \<part\>. </td></tr>
  *  </table>
  *  If no input option is specified, the frames are read from the partition
  *  specified by the \c LIGOSMPART environment variable or, if this is not 
  *  defined, from "LIGO_Online".
  *
  *  When running under \c ROOT, the base class definition and implementation 
  *  files must be loaded before the monitor can be run. If these files 
  *  aren't provided automatically by the start-up script, they may be 
  *  loaded with:
  *  \verbatim
        gROOT->LoadMacro("GridWare.hh");
        gROOT->LoadMacro("GridWare.cc"); \endverbatim
  *  The monitor is run with the \c .x command or by loading it and calling 
  *  it as a function. See the note entitled "Online monitoring of LIGO Data
  *  Using ROOT" for more details.
  *  \brief Data processing monitor base class.
  *  @author John G. Zweizig
  *  @version 1.4; Last Modified March 4, 2003
  */
class GridWare {
  public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  \brief Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
    GridWare(int argc, const char *argv[]);

    /**  Close the input stream and go away.
      *  \brief Class destructor.
      */
    virtual ~GridWare();

    /**  #MainLoop# reads in data and invokes #ProcessData# for each data 
      *  epoch successfully read in. #MainLoop# exits when a SIGINT or 
      *  SIGTERM signal is caught, or when #finish# is called by a subsidiary 
      *  function (usually by #ProcessData#). This function is provided by 
      *  the base class.
      *  \brief Loop over data epochs.
      */
    void MainLoop(void);

    /**  #ProcessData# performs the monitor specific processing data from
      *  a single epoch. It must be defined as part of the derived monitor 
      *  class implementation.
      *  \brief Process one epoch (Supplied by user).
      */
    virtual void ProcessData(void)=0;

    /**  #Attention# performs the monitor specific attention interrupt 
      *  processing. Its implementation is optional. Execution of 
      *  the attention handler is deferred until the processing of the 
      *  current epoch has completed. This means that there may be a 
      *  significant delay between the delivery of the SIGUSR1 signal and
      *  the invocation of the handler. The ROOT version uses the ROOT 
      *  event handler as a source of When used with ROOT, the default action is to
      *  terminate the monitor. When used in the background, it defaults
      *  to no action. The execution of the attention handler is 
      *  synchronized to the 
      *  \brief Attention interrupt handler (Supplied by user).
      */
    virtual void Attention(void);

    /**  finish stops processing by setting a flag that breaks the main 
      *  loop. Note that finish returns to the calling function (generally 
      *  the user's ProcessData) which must exit normally for the process 
      *  to terminate.
      *  \brief Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  Debug returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  \brief Get the debug level.
      *  \return Debug printout level.
      */
    int Debug(void) const {return mDebug;}

    /**  Return a constant reference to the data accessor.
      *  \brief Get the data accessor.
      *  \return Reference to the multi-data accessor.
      */
    const MultiDacc& getDacc(void) const {return mIn;}

    /** Get number of streams
      * \return Number of streams
      */
    unsigned int getNStream(void) {return mIn.getNDacc();}

    /** Get individual Dacc
      * \param i Dacc instance number
      * \return pointer to the specified data accessor
      */
    Dacc* getDacc(unsigned int i) {return mIn.getDacc(i);}

    /**  Test for a %GridWare command line argument. All %GridWare command line 
      *  arguments take a value in the following argument position.
      *  \brief Test if string is a %GridWare command line argument.
      *  \param arg Argument string to test.
      *  \return true if the specified string is a %GridWare argument.
      */
    bool isGridWareArg(const char* arg) const;

    /**  Align the time stride to an integer multiple of seconds.
      *  \brief set stride alignment.
      *  @param mult boundary alignment.
      *  @param off  boundary offset
      */
    void setStrideAlignment(unsigned long mult, double off);

    /**  Test whether the specified signal has been caught.
      *  \brief Test for a specified signal.
      *  \param sig Signal number
      *  \return true if the specified signal was caught.
      */
    bool testAttnSignal(int sig) const;

  //------------------------------------  Private methods
  private:
    /**  Test if a termination interrupt occurred.
      *  \brief Test for termination.
      */
    bool testTerm(void) const;

    /**  Test if an attention interrupt occurred.
      *  \brief Test for attention.
      */
    bool testAttn(void);

  protected:
    /**  Return a reference to the data accessor.
      *  \brief Get the data accessor.
      *  \return Reference to the monitor multi-data accessor.
      */
    MultiDacc& getDacc(void) {return mIn;}

    ///  Frame Input object
    MultiDacc  mIn;

#ifndef __CINT__
    /**  A raw trigger data identifier is encoded for each data epoch read.
      *  This field is omitted in the ROOT version.
      *  \brief Raw trigger data identifier for current epoch.
      */
    trig::TrigRaw  mRawID;
#endif  // __CINT__

  private:
    ///  Internal run flag
    bool     mActive;

#ifndef __CINT__
    ///  Termination signal handling object
    SigFlag mTerm;

    ///  Attention signal handling object
    SigFlag mAttn;
#endif // !def(__CINT__)
  
    ///  Debug Level
    int  mDebug;

    /**  Frame read error.
      *  mEOF is set when a frame read error occurs.
      */
    bool mEOF;

    /**  Attention signals.
      *  mAttnMask is set to a or of the bits corresponding to the Signals 
      *  that have been caught.
      */
    long mAttnMask;

    long mMaxFrames;

    long mAlignMult;
    double mAlignOff;

  /** The file is used to store the GPS time of current processed data
   */
    string mStatusFile;
    void Status( Time );
  
};

inline bool 
GridWare::testAttnSignal(int sig) const {
    return ((1 << sig) & mAttnMask) != 0;
}

//
//   The EXECDAT macro generates the main function to execute a monitor
//   based on GridWare. The argument is whatever name you have chosen
//   for your class.
//
#define EXECDAT(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

//
//   The EXEROOT macro generates the main function to execute a monitor
//   based on GridWare. This macro differs from EXECDAT in that it also 
//   sets up the root object structure which allows the derived class to 
//   use any root classes. The argument is whatever name you have chosen
//   for your class.
//
#define EXEROOT(myclass) \
TROOT root("GridWare", "Graphical background monitor"); \
int main(int argc, const char* argv[]) { \
    int bc(argc); const char** bv=new const char*[bc]; \
    for (int i=0 ; i<bc ; i++) bv[i]=argv[i]; \
    TApplication theApp("App", &bc, const_cast<char**>(bv)); \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

#endif     //  GRIDWARE_HH
