/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//  Class PSD:  Computes power spectrum (with windowing and detrending) 
//  from TSeries data.
//    version:  1.8 (Sept 24 2004)
//    authors:  Kevin C. Schlaufman (kcs149@psu.edu)
//              Patrick J. Sutton (psutton@ligo.caltech.edu)
//
///////////////////////////////////////////////////////////////////////////

#include "FSeries.hh"
#include "FSpectrum.hh"
#include "TSeries.hh"
#include "FDFilter.hh"
#include "PSD.hh"
#include "fSeries/DFT.hh"
#include "fSeries/PSD.hh"
#include <iostream>

using namespace std;

PSD::PSD(const window_api *win_choice, int intervals)
{
        n = intervals;
        if (win_choice) win = win_choice->clone();
	else            win = 0;
}

PSD::PSD(const char *square, int intervals)
{
        n = intervals;
        win = 0;
}

PSD::PSD(int intervals)
{
	n = intervals;
	win = new Hamming;
}

PSD::PSD(void)
{
	n = 1;
	win = new Hamming;
}

PSD::~PSD(void)
{
    if (win) delete win;
}

void 
PSD::generate(FSpectrum& total, const TSeries* ts)
{
    // Divide TSeries interval into n sub-intervals of equal length
    Interval seg_length = ts->getInterval();
    seg_length /= double(n);
    Time t_zero = ts->getStartTime();

    // Test to determine if seg_length is an integer.  If not, display
    // possible error message
    if ( double(seg_length) != int(double(seg_length)) ) {
	cout << "The window function may not work properly if "
	     << "the window is applied to time segments not an "
	     << "integer number of seconds long." << endl << endl;
    }

    // Set window length
    if(win) win->setWindow((int)((double)seg_length));

    // Extract data starting with t_zero and lasting for seg_length
    TSeries sub_series(ts->extract(t_zero, seg_length));

    // Detrend, window sub_series
    sub_series -= sub_series.getAverage();
    if(win) sub_series = win->apply(sub_series);

    // Add to running total for power spectrum
    total = FSpectrum(FSeries(sub_series));

    // Increment t_zero
    t_zero += seg_length;

    // Loop to divide TSeries into n sub-series
    for(int i = 1; i < n; i++) {
	// Extract data starting with t_zero and lasting for length seg_length
	sub_series = ts->extract(t_zero, seg_length);

	// Detrend, window sub_series
	sub_series -= sub_series.getAverage();
	if (win) sub_series = win->apply(sub_series);

	// Add to running total for power spectrum
	total += FSpectrum(FSeries(sub_series));

	// Increment t_zero
	t_zero += seg_length;
    }

    // Normalize to mean power per data point
    if (n > 1) total *= 1.0 / double(n);
}

//======================================  Filtered PSD
void 
PSD::generate(FSpectrum& total, const TSeries* ts, FDFilter& fd) {
    // Divide TSeries interval into n sub-intervals of equal length
    Interval seg_length = ts->getInterval();
    seg_length /= double(n);
    Time t_zero = ts->getStartTime();

    // Test to determine if seg_length is an integer.  If not, display
    // possible error message
    if ( double(seg_length) != int(double(seg_length)) ) {
	cout << "The window function may not work properly if "
	     << "the window is applied to time segments not an "
	     << "integer number of seconds long." << endl << endl;
    }

    // Set window length
    if(win) win->setWindow((int)((double)seg_length));

    // Extract data starting with t_zero and lasting for seg_length
    TSeries sub_series(ts->extract(t_zero, seg_length));

    // Detrend, window sub_series
    sub_series -= sub_series.getAverage();
    if(win) sub_series = win->apply(sub_series);

    // Add to running total for power spectrum
    total = FSpectrum(fd.Apply(FSeries(sub_series)));

    // Increment t_zero
    t_zero += seg_length;

    // Loop to divide TSeries into n sub-series
    for(int i=1; i < n; i++) {
	// Extract data starting with t_zero and lasting for length seg_length
	sub_series = ts->extract(t_zero, seg_length);

	// Detrend, window sub_series
	sub_series -= sub_series.getAverage();
	if (win) sub_series = win->apply(sub_series);

	// Add to running total for power spectrum
	total += FSpectrum(fd.Apply(FSeries(sub_series)));

	// Increment t_zero
	t_zero += seg_length;
    }

    // Normalize to mean power per data point
    if (n > 1) total *= 1.0 / double(n);
}

void 
PSD::generate(containers::PSD& total, const TSeries* ts) {
    // Divide TSeries interval into n sub-intervals of equal length
    Interval seg_length = ts->getInterval();
    seg_length /= double(n);
    Time t_zero = ts->getStartTime();

    // Test to determine if seg_length is an integer.  If not, display
    // possible error message
    if ( double(seg_length) != int(double(seg_length)) ) {
	cout << "The window function may not work properly if "
	     << "the window is applied to time segments not an "
	     << "integer number of seconds long." << endl << endl;
    }

    // Set window length
    if(win) win->setWindow((int)((double)seg_length));

    // Extract data starting with t_zero and lasting for seg_length
    TSeries sub_series(ts->extract(t_zero, seg_length));

    // Detrend, window sub_series
    sub_series -= sub_series.getAverage();
    if(win) sub_series = win->apply(sub_series);

    // Add to running total for power spectrum
    total = containers::PSD(sub_series);

    // Increment t_zero
    t_zero += seg_length;

    // Loop to divide TSeries into n sub-series
    for(int i=1; i < n; i++) {
	// Extract data starting with t_zero and lasting for length seg_length
	sub_series = ts->extract(t_zero, seg_length);

	// Detrend, window sub_series
	sub_series -= sub_series.getAverage();
	if (win) sub_series = win->apply(sub_series);

	// Add to running total for power spectrum
	total += containers::PSD(sub_series);

	// Increment t_zero
	t_zero += seg_length;
    }

    // Normalize to mean power per data point
    if (n > 1) total *= 1.0 / double(n);
}
