/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef IFORESPONSE_HH
#define IFORESPONSE_HH

#include "FDFilterBase.hh"
#include "FSeries.hh"
#include "TSeries.hh"
#include <string>

class FSpectrum;

/**  Convert a Frequency series from strain units to the ADC response.
 *  @author John Zweizig
 *  @version 1.1; March 30, 2005
 */
class IfoResponse : public FDFilterBase {
public:
    using FDFilterBase::Apply;

    /**  Construct a null IFO response filter.
     */
    IfoResponse(void);

    /**  Construct and initialize an IFO response filter.
      *  \brief Constructor
      *  \param title Calibration title string.
      *  \param file  Clibration file path.  
      */
    IfoResponse(const std::string& title, const std::string& file);

    /**  Ifo response filter destructor.
      *  \brief Destructor.
      */
    virtual ~IfoResponse(void) {}

    /**  Clone an ifo response filter.
      *  \brief Clone the filter
      *  \return Pointer to the cloned filter.
      */
    IfoResponse* clone(void) const;

    /**  Read in a filter response transfer functions and parameters from 
      *  a specified file. At present, only xml files are accepted, but in 
      *  the future, read will determine whether the calibration file is an 
      *  xml or frame file.
      *  @memo Initialize an IFO response filter from a specified file.
      *  \brief Read a calibration file.
      *  \param title Calibration title string.
      *  \param file Calibration file path string.
      */
    void read(const std::string& title, const std::string& file);

    /**  Read calibration information from an xml file.
      *  \brief Read a calibration xml file.
      *  \param title Calibration title string.
      *  \param istr Calibration xml file name.
     */
    void readXml(const std::string& title, const std::string& istr);

    /**  Read calibration information from a framefile.
      *  \brief Read a calibration frame file.
      *  \param title Calibration title string.
      *  \param istr Calibration frame file name.
      */
    void readFrame(const std::string& title, const std::string& istr);

    /**  Set the frequency range to be filtered.
      *  \brief Set the frequency range
      *  @param fMin Minimum frequency to be filtered.
      *  @param fMax Maximum frequency to be filtered.
      *  @param dF   Frequency increment.
      */
    void setFreq(double fMin, double fMax, double dF);

    /**  Calculate the time-dependent response transfer function for the 
      *  specified time.
      *  \brief Set the response transfer function
      *  \param t Time for which the response is to be set.
      */
    void setResponse(const Time& t) const;

    /**  Filter a frequency series in strain units to produce an output 
      *  frequency series colored by the IFO response function and in
      *  ADC units. If the frequency limits have not been set, the input
      *  frequency series range is used.
      *  @memo Filter a frequency series.
      *  @param fIn Input frequency series in strain units.
      *  @param fOut Output frequency series in ADC units.
      */
    virtual void Apply(const FSeries& fIn, FSeries& fOut);

    /**  Filter a spectrum instance.
      *  \brief Filter a spectrum instance.
      *  \param fIn Spectrum to be filtered.
      *  \param fOut Filter spectrum (output).
      */
    virtual void Apply(const FSpectrum& fIn, FSpectrum& fOut);

    /**  Filter a DFT
      *  \brief Filter a DFT instance.
      *  \param fIn DFT to be filtered.
      *  \param fOut Filtered DFT (output).
       */
    virtual void Apply(const containers::DFT& fIn, containers::DFT& fOut);

    /**  Filter a PSD instance
      *  \brief Filter a PSD instance.
      *  \param fIn PSD to be filtered.
      *  \param fOut Filtered PSD (output).
       */
    virtual void Apply(const containers::PSD& fIn, containers::PSD& fOut);

private:
    double  mFmin;
    double  mFmax;
    double  mDf;
    FSeries mOpenLoopGain;
    FSeries mSensing;
    TSeries mAlpha;
    TSeries mAlphaBet;

    //------------------------------------  Mutable FD status 
    mutable FSeries mResponse;
    mutable float   mLastAlpha;
    mutable float   mLastAlphaBet;
};

#endif  // IFORESPONSE_HH
