/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef STD_LCL_ARRAY_HH
#define STD_LCL_ARRAY_HH
#include <cstring>

/**  \brief Local array template.
  *
  *  The lcl_array class can be used for local data storage.
  *  It auto-destructs when the program leaves the scope the object was 
  *  defined in. This is especially useful if exceptions are used to return
  *  control from a point other than the end of the function.
  *  \author John G Zweizig
  *  \version 1.1; December 4, 2007
  */
template<class T>
class lcl_array {
public:
    /** Array size type.
      */
    typedef unsigned long size_type;
private:
    /** Local storage data array.
     */
    T* _data;
public:
    /** \brief Constructor
      *
      * Allocate a temporary array of length n.
      * @param _n number of element to allocate.
      */
    lcl_array(size_type _n) {_data = new T[_n];}

    /** \brief Initializing constructor
      *
      *  Allocate a temporary array of length n and initialize it to values 
      *  at the specified location.
      *  @param _n number of element to allocate.
      */
    lcl_array(size_type _n, const T* ival);

    /** \brief Destructor
      *
      * Delete the local array.
      */
    ~lcl_array(void) {delete [] _data;}

    /** \brief Get the array pointer
      *
      * Return a writable array pointer.
      * \return Writable pointer to the first element of the temporary array.
      */
    operator T* (void) {return _data;}

    /** \brief Get the array pointer
      *
      * Return a constant array pointer.
      * \return Constant pointer to the first element of the temporary array.
      */
    operator const T* (void) const {return _data;}

    /**  \brief Get the array pointer
      *
      *  Get a writeable pointer.
      *  \return Writable pointer to the first element of the temporary array.
      */
    T* get(void) {return _data;}

    /** \brief Get the array pointer
      *
      * Return a constant array pointer.
      * \return Constant pointer to the first element of the temporary array.
      */
    const T* get(void) const {return _data;}

private:
    /**  Dummy copy constructor to inhibit copying.
      *  @param x dummy argument
      */
    lcl_array(const lcl_array& x);

    /**  Dummy assignment operator to inhibit copying.
      *  @param x dummy argument
      */
    lcl_array& operator=(const lcl_array& x);
};

template <class T>
lcl_array<T>::lcl_array(size_type _n, const T* ival) {
    _data = new T[_n];
    if (ival) memcpy(_data, ival, _n*sizeof(T));
}

#endif   // !defined(STD_LCL_ARRAY_HH)
