//////////////////////////////////////////////////////////////////////////
//                                                                      //
// pipe_exec							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
#include "PConfig.h"
#include "pipe_exec.hh"
#include "fdstream.hh"
#define __BIDIRECTIONAL_PIPE
#include "Time.hh"
#include "Interval.hh"

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#include <cstdlib>
#include <cstring>


namespace std {


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// prog_exec							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   prog_exec::prog_exec (const char* cmd)
   : fPID (-1), fError (false), fArgs (0)
   {
      // parse before fork to avoid allocating memory in child process
      if (!parse (cmd)) {
         fError = true;
         return;
      }
      const char* Path = path();
      char* const* Args = args();
      bool ret = false;
   
      // fork
      fPID = fork();
      // error
      if (fPID == -1) {
         fError = true;
         return;
      }
      // child
      else if (fPID == 0) {
         ret = (execvp (Path, Args) != -1);
         exit (!ret); // never reached
      }
   }

//______________________________________________________________________________
   prog_exec::prog_exec ()
   : fPID (-1), fError (false), fArgs (0)
   {
   }

//______________________________________________________________________________
   prog_exec::~prog_exec () 
   {
      if (fPID && (fPID != -1)) {
         //cerr << "send kill to " << fPID << endl;
         kill (SIGKILL);
      }
      parse (0);
   }

//______________________________________________________________________________
   int prog_exec::wait (int* ret, bool poll)
   {
      if (fPID && (fPID != -1)) {
         int stat_loc;
	 int options = poll ? WNOHANG : 0;
         int pid = waitpid (fPID, &stat_loc, options);
         if (pid == fPID) {
            fPID = 0;
            if (ret) *ret = stat_loc;
         }
         return pid;
      }
      else {
         return -1;
      }
   }

//______________________________________________________________________________
   int prog_exec::wait (double timeout, int* stat_loc)
   {
      // invalid pid
      if ((fPID == 0) || (fPID == -1)) {
         return -1;
      }
      // poll
      if (timeout == 0) {
         return wait (stat_loc, true);
      }
      // block
      else if (timeout < 0) {
         return wait (stat_loc, false);
      }
      // true timeout
      int ret;
      Time start = Now();
      Time stop = start + Interval (timeout);
      while ((ret = wait (stat_loc, true)) == 0) {
         if (Now() > stop) {
            return 0;
         }
         timespec tick = {1,0};
         nanosleep (&tick, 0);
      }
      return ret;
   }

//______________________________________________________________________________
   bool prog_exec::kill (int sig)
   {
      if (fPID && (fPID != -1)) {
         return ::kill (fPID, sig) == 0;
      }
      else {
         return false;
      }
   }

//______________________________________________________________________________
   bool prog_exec::parse (const char* cmd)
   {
      // parse command
      setcmd (cmd);
      // free memory
      if (fArgs) {
         for (char** p = fArgs; *p; ++p) {
            delete [] *p; *p = 0;
         }
         delete [] fArgs;
         fArgs = 0;
      }
      if (!cmd) {
         return false;
      }
      // process command line arguments
      int maxarg = 100;
      fArgs = new char* [maxarg];
      fArgs[0] = 0;
      int argn = 0;
      fFilename = "";
      char* buf = new char [strlen (cmd) + 10];
      if (!buf) {
         return false;
      }
      strcpy (buf, cmd);
      char* last = 0;
      char* tok = buf;
      do {
         while (isspace (*tok)) ++tok;
	 if (*tok == '\'') {
	    last = strchr (++tok, '\'');
	 }
	 else if (*tok == '"') {
	    last = strchr (++tok, '"');
	 }
	 else {
	    for (last = tok; *last && !isspace (*last); ++last) ;
	    if (!*last) last = 0;
	 }
	 if (last) *last = 0;
	 char* p = tok;
         if (fFilename.empty()) {
            fFilename = tok;
            if (strrchr (tok, '/')) p = strrchr (tok, '/') + 1;
         }
         fArgs[argn] = new char [strlen (p) + 1];
         if (!fArgs[argn]) {
            delete [] buf;
            return false;
         }
         strcpy (fArgs[argn], p);
         fArgs[++argn] = 0;
	 //cout << "Argument[" << argn-1 << "] = " << p << endl;
	 if (last) tok = last + 1;
      } while (last);
      delete [] buf;
      return true;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// pipe_exec							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   pipe_exec::pipe_exec (const char* cmd, const char* mode)
   : iostream((streambuf*)0), fSBuf (0)
   {
      // set mode
      if (!mode) {
         setmode (mode);
      }
      else {
         string m;
         if ((strchr (mode, 'r') != 0) ||
            (strchr (mode, 'R') != 0)) m += "r";
         if ((strchr (mode, 'w') != 0) ||
            (strchr (mode, 'W') != 0)) m += "w";
         setmode (m.c_str());
      }
      // parse before fork to avoid allocating memory in child process
      if (!parse (cmd)) {
         setstate (ios::badbit);
         return;
      }
      initcmd();
   }

//______________________________________________________________________________
   void pipe_exec::initcmd()
   {
      // set path & args;
      const char* Path = path();
      char* const* Args = args();
      bool ret = false;
   
      // create pipes
      if (pipe (fIn) == -1) {
         setstate (ios::badbit);
         closepipe();
         return;
      }
   #ifdef __BIDIRECTIONAL_PIPE
      fOut[0] = fIn[1];
      fOut[1] = fIn[0];
   #else
      if (pipe (fOut) == -1) {
         setstate (ios::badbit);
         closepipe();
         return;
      }
   #endif
      // fork
      setpid (fork());
      // error
      if (pid() == -1) {
         setstate (ios::badbit);
         closepipe();
         return;
      }
      // child
      else if (pid() == 0) {
         // dup stdin and stdout
         //printf ("MODE IS %s\n", fMode.c_str());
         if (fMode == "r") {
            // readonly
            if ((dup2 (fOut[1], 1) == -1) ||
               (dup2 (fOut[1], 2) == -1)) {
               exit (1);
            }
         }
         else if (fMode == "w") {
            // write only
            int nul = ::open ("/dev/null", 2);
            if ((dup2 (fIn[0], 0) == -1) ||
               (dup2 (nul, 1) == -1) ||
               (dup2 (nul, 2) == -1)) {
               exit (1);
            }
            // if ((dup2 (fIn[0], 0) == -1)) {
               // exit (1);
            // }
         }
         else {
            // read and write
            if ((dup2 (fIn[0], 0) == -1) ||
               (dup2 (fOut[1], 1) == -1) ||
               (dup2 (fOut[1], 2) == -1)) {
               exit (1);
            }
         }
         // start child process
         ret = (execvp (Path, Args) != -1);
         exit (!ret); // never reached
      }
      // parent
      else {
         // create stream buffer and connect it to pipe
      #ifndef __GNU_STDC_OLD
         fSBuf = new fd_streambuf (fOut[0], fIn[1]);
         init (fSBuf);
         clear();
      #else
         #ifndef __BIDIRECTIONAL_PIPE
         #error "Error: Need bidirectional pipes"
         #endif
         fSBuf = new fd_streambuf (fOut[0]);
         init (fSBuf);
         clear();
         // attach (fOut[0]);
      #endif
      }
   }

//______________________________________________________________________________
   pipe_exec::~pipe_exec () 
   {
      if (fSBuf) {
         init (0);
         delete fSBuf;
      }
      closepipe();
   }

//______________________________________________________________________________
   void pipe_exec::close()
   {
      closepipe();
   }

//______________________________________________________________________________
   int pipe_exec::wait (int* ret, bool poll)
   {
      int pid = prog_exec::wait (ret, poll);
      if (pid > 0) {
         closepipe();
      }
      return pid;
   }

//______________________________________________________________________________
   int pipe_exec::wait (double timeout, int* stat_loc)
   {
      int pid = prog_exec::wait (timeout, stat_loc);
      if (pid > 0) {
         closepipe();
      }
      return pid;
   }

//______________________________________________________________________________
   void pipe_exec::closepipe()
   {
      if (fIn[0] != -1) ::close (fIn[0]);
      if (fIn[1] != -1) ::close (fIn[1]);
   #ifndef __BIDIRECTIONAL_PIPE
      if (fOut[0] != -1) ::close (fOut[0]);
      if (fOut[1] != -1) ::close (fOut[1]);
   #endif
      fIn[0] = -1;
      fIn[1] = -1;
      fOut[0] = -1;
      fOut[1] = -1;
   }

}

// 
   // using namespace std;
// 
// extern "C"
   // int main (int argn, char** argv) 
   // {
      // pipe_exec pe ("/home/sigg/diag/addons/fantom/src/p2open -e s/e/E/g");
      // if (!pe) {
         // cerr << "pipe_exec failed" << endl;
         // return 1;
      // }
      // cerr << "pipe exec done " << (int) pe.rdstate() << endl;
   // 
      // string line;
      // string answer;
   //    //cin >> line;
      // cerr << "Send a test pattern ... and ..." << endl;
      // pe << "Send a test pattern" << endl;
      // cerr << "pipe send done " << (int) pe.rdstate() << endl;
      // getline (pe, answer);
      // cerr << "ANSWER = " << answer << endl;
      // while (pe) {
         // getline (cin, line);
         // pe << line << endl;
         // if (!pe) {
            // break;
         // }
         // getline (pe, answer);
         // if (!pe) {
            // break;
         // }
         // cerr << "ANSWER = " << answer << endl;
         // sleep (1);
         // pe.wait (0, true);
      // }
      // return 0;
   // }
