/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PConfig.h"
#include "sockutil.h"
#include "lsmp_con.hh"
#include "lsmp_prod.hh"
#include "SigFlag.hh"
#include "SysError.hh"
#include "Time.hh"
#include "Interval.hh"
#include "checksum_crc32.hh"
#include <string>
#include <stdexcept>
#include <inttypes.h>
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstdio>
#include <cstring>
#include <cerrno>
#include <fcntl.h>
#include <unistd.h>
#include <sys/socket.h>

struct link_header {
    uint32_t block_id;
    uint32_t length;
    uint32_t gps_secs;
    uint32_t data_time;
};

/**  The FrameLink program connects two shared memory partitions together 
  *  with a tcp link. Two instances of the framelink program are used to
  *  connect the processes. The program may run in one of four modes:
  *  <ol>
  *    <li> push: establish a connection to a receive process and send 
  *         it frames
  *    <li> pull: establish a connection to a send process and receive
  *         frames.
  *    <li> send: listen for a connection. Send frames as they arrive in 
  *         the specified shared memory partition (partner of pull mode).
  *    <li> receive: listen for a connection. Copy received data to the 
  *         specified shared memory partition (partner of push mode).
  *  </ol>
  */
class FrameLink {
public:
    enum pgm_mode {
	kUnknown,
	kPush,
	kPull,
	kReceive,
	kSend
    };
    typedef unsigned long gps_type;

    enum sock_status {
	kOK,
	kRetry,
	kReconnect
    };

    /**  Maximum time to maintain the client side of a connection if 
      *  no data are received.
      */
    static Interval disconnect_timeout;
public:
    FrameLink(int argc, const char* argv[]);
    ~FrameLink(void);

    /**  Test whether this process is a TCP client, \e i.e. whether 
      *  the program mode is \c kSend or \c kReceive.
      *  \brief Test whether this is a client.
      *  \return True if this process is a TCP client.
      */
     bool client(void) const;

    /**  Close the server socket
     */
    void close_server(void);

    /**  Close the peer socket.
     */
    void close_sock(void);

    /**  Release the shared memory partition.
      *  \brief close sharedmemory
      */
    void close_shm(void);

    /**  Connect the socket. For client instances (push or pull mode) a connect
      *  request is issued. For server instances a listen is issued on the 
      *  first connect request followed by an accept request. If the connect()
      *  or accept() request fails or times out, connect_sock returns false.
      *  \brief Connect the socket.
      *  \return True if the conection was made.
      */
    bool connect_sock(void);

    /**  Attach the shared memory partition.
     */
    bool connect_shm(void);

    /**  Flush the specified number of bytes from the input socket.
     */
    int flush_packet(int len, bool wait);

    /** Main looop over frames.
     */
    void process(void);

    /**  Receive the specified number of bytes into the buffer pointed to by
      *  \a buf. Wait for all data if \a wait is set. The receive call will 
      *  be reissued if an EINTR is set and the mTerm signal flag has not 
      *  been set.
      *  \brief Read data from the open socket.
      *  \param buf  Pointer to the buffer to receive the data.
      *  \param len  Length of data to be received.
      *  \param wait Wait for data to be read if asserted.
      *  \return Length of data received, or 0 on eof or -1 on error.
      */
    int read(void* buf, int len, bool wait) const;

    /**  Read in a header structure. Indicate an error if the header 
      *  structure is the wrong length.
      *  \brief Read header structure
      *  \param hdr Reference to the output header structure.
      *  \return Error code or zero.
      */
    sock_status read_header(link_header& hdr);

    /**  Test whether this process is a TCP data receiver, \e i.e. whether 
      *  the program mode is \c kReceive or \c kPull.
      *  \brief test whether this is a receiver.
      *  \return True if this process is a TCP receiver.
      */ 
    bool receiver(void) const;

    /**  Test whether this process is a TCP data sender, \e i.e. whether 
      *  the program mode is \c kSend or \c kPush.
      *  \brief test whether this is a send.
      *  \return True if this process is a TCP sender.
      */ 
    bool sender(void) const;

    /**  Test whether this process is a TCP server, \e i.e. whether 
      *  the program mode is \c kPush or \c kPull.
      *  \brief test whether this is a server.
      *  \return True if this process is a TCP server.
      */ 
    bool server(void) const;

    /**  Test whether the verbose stream is requested and working.
      *  \brief Check verbose output stream.
      *  \param reopen (re-)Open verbose stream if it is not open.
      */
    bool v_check(bool reopen);

    void start_stats(void);
    void end_stats(const Time& gps);
    void print_stats(void) const;
private:
    //----------------------------------  Command line parameters
    pgm_mode    mPgmMode;
    std::string mPeerName;
    std::string mPartition;
    std::string mVerbose;
    int         mDebug;
    Interval    mStatisticsPeriod;
    uint32_t    mFrameDt;

    //----------------------------------  Status info.
    SigFlag     mTerm;       /// Termination signal flag
    sockaddr_in mServerAddr; /// Server address (bind or connect)
    int         mServer;     /// Server port 
    int         mSocket;
    bool        mBound;
    bool        mConnected;
    uint32_t    mBlockID;
    LSMP_CON*   mInPart;
    LSMP_PROD*  mOutPart;

    //----------------------------------  Timing statistics
    gps_type    mLastGps;
    long        mBlocks;
    long        mChkSum;
    Time        mLastStats;
    Interval    mSumStartLatency;
    Interval    mSumEndLatency;
    long        mSumFrames;
    long        mFramesFlushed;
    Time        mLastRecv;

    //----------------------------------  Verbose output stream
    std::ofstream mVerbStream;
    int         mVerbCount;
};

inline bool
FrameLink::client(void) const {
    return (mPgmMode == kPush || mPgmMode == kPull);
}

inline bool
FrameLink::receiver(void) const {
    return (mPgmMode == kReceive || mPgmMode == kPull);
}

inline bool
FrameLink::sender(void) const {
    return (mPgmMode == kSend || mPgmMode == kPush);
}

inline bool
FrameLink::server(void) const {
    return (mPgmMode == kSend || mPgmMode == kReceive);
}

using namespace std;
Interval FrameLink::disconnect_timeout(600.0);
static const double Timeout = 60;

//======================================  logstamp - Timestamp a log file line
std::ostream&
logstamp(std::ostream& out) {
    return out << TimeString(Now(), "%Y.%02m.%02d-%02H:%02N:%02S ");
}

//======================================  COntrol constants
static const int max_verbline = 100;

//======================================  Main program
int
main(int argc, const char* argv[]) {
    FrameLink pgm(argc, argv);
    try {
	pgm.process();
    } catch (SysError& e) {
	logstamp(cerr) << "framelink caught exception: " << e.what() << endl;
    } catch (exception& e) {
	logstamp(cerr) << "framelink caught exception: " << e.what() << endl;
    }
    pgm.close_shm();
    pgm.close_sock();
    pgm.close_server();
    pgm.print_stats();
}

//======================================  Initialization
FrameLink::FrameLink(int argc, const char* argv[]) 
    : mPgmMode(kUnknown), mDebug(0), mStatisticsPeriod(0.0), mFrameDt(0),
      mServer(-1), mSocket(-1), mBound(false), mConnected(false), 
      mBlockID(0), mInPart(0), mOutPart(0), 
      mLastGps(0), mBlocks(0), mChkSum(0), mLastStats(0),
      mVerbCount(0)
{
    //----------------------------------  Scan through arguments 
    bool syntax = false;
    bool daemon = false;
    const  char* logfile=0;
    int iarg = 1;
    for (iarg=1; iarg < argc; iarg++) {
	if (*argv[iarg] != '-') break;
	string argi = argv[iarg];
	if (argi == "--daemon") {
	    daemon = true;
	} 
	else if (argi == "--debug") {
	    mDebug = strtol(argv[++iarg], 0, 0);
	} 
	else if (argi == "--delta") {
	    mFrameDt = strtol(argv[++iarg], 0, 0);
	} 
	else if (argi == "--help") {
	    syntax = true;
	    break;
	}
	else if (argi == "--log") {
	    logfile = argv[++iarg];
	}
	else if (argi == "--stats") {
	    mStatisticsPeriod = strtol(argv[++iarg], 0, 0);
	} 
	else if (argi == "--verbose") {
	    mVerbose = argv[++iarg];
	} 
	else {
	    cerr << "Argument not recognized: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Print syntax message and fail.
    if (argc-iarg < 3 || syntax) {
	cerr << "Framelink copies frame file images to or from a  Ligo shared"
	     << endl;
	cerr << "memory partition over a tcp link." << endl;
	cerr << "framelink command syntax: " << endl << endl;
	cerr << "  framelink [options] <mode> <from> <to>" 
	     << endl << endl;
	cerr << "where: " << endl;
	cerr << "<mode> is the program mode: push, pull, send or receive" 
	     << endl;
	cerr << "<from> is the data source partition or socket" << endl;
	cerr << "<to>   is the data destination partition or socket" << endl;
	cerr << endl << "Options are: " << endl;
	cerr << "--daemon     Run process in the background" << endl;
	cerr << "--debug <n>  Set debug print level to <n>" << endl;
	cerr << "--delta <s>  Set frame size to <s> seconds" << endl;
	cerr << "--help       Print this message" << endl;
	cerr << "--log   <l>  Write output and errors to file <l>" << endl;
	cerr << "--stats <p>  Set statistics period to <p> seconds" << endl;
	cerr << "--verbose <f> Verbose logging file name" << endl;
	mTerm.setSig0();
	return;
    }

    //----------------------------------  Get the program mode.
    string mode = argv[iarg];
    if (mode == "push") {
	mPgmMode   = kPush;
	mPartition = argv[iarg+1];
	mPeerName  = argv[iarg+2];
	if (!mFrameDt) mFrameDt = 1;
    } else if (mode == "pull") {
	mPgmMode   = kPull;
	mPeerName  = argv[iarg+1];
	mPartition = argv[iarg+2];
    } else if (mode == "send") {
	mPgmMode   = kSend;
	mPartition = argv[iarg+1];
	mPeerName  = argv[iarg+2];
	if (!mFrameDt) mFrameDt = 1;
    } else if (mode == "receive") {
	mPgmMode   = kReceive;
	mPeerName  = argv[iarg+1];
	mPartition = argv[iarg+2];
    } else {
	cerr << "Invalid program mode: " << mode << endl;
	mTerm.setSig0();
	return;
    }

    //----------------------------------  Get the socket address
    mServerAddr.sin_family = AF_INET;
    mServerAddr.sin_port   = ntohs(23232);
    string::size_type i = mPeerName.find(":");
    if (i != string::npos) {
	mServerAddr.sin_port = ntohs(strtol(mPeerName.c_str()+i+1, 0, 0));
	mPeerName.erase(i);
    }
    if (mDebug) {
	logstamp(cout) << "Look up peer: " << mPeerName << ":" 
		       << mServerAddr.sin_port << endl;
    }
    int rc = nslookup(mPeerName.c_str(), &mServerAddr.sin_addr);
    if (rc < 0) {
	
	cerr << "Can't find address for server " << mPeerName << endl;
	mTerm.setSig0();
	return;
    }

    //-----------------------------------  Fork a daemon
    if (daemon) {
	switch (fork()) {
	case -1:
	    exit(1);
	case 0:
	    break;
	default:
	    exit(0);
	}
    }

    //-----------------------------------  Optionally switch to a log file
    if (logfile) {
	int flags = O_WRONLY | O_CREAT;
	int fd = open(logfile, flags, 0664);
	close(1); // stdout
	if (dup(fd) != 1) {
	    cerr << "stdout not diverted!" << endl;
	}
	close(2); // stderr
	if (dup(fd) != 2) {
	    cout << "stderr not diverted!" << endl;
	}
	close(fd);
    }

    //----------------------------------  Identify myself (after log is open)
    logstamp(cout) << "Starting framelink: $Id: framelink.cc 7471 2015-09-16 20:00:56Z john.zweizig@LIGO.ORG $" << endl;
    logstamp(cout) << "arguments are:";
    for (int i=1; i<argc; i++) cout << " " << argv[i];
    cout << endl;

    //----------------------------------  Open the verbose log stream
    if (v_check(true)) {
	logstamp(cout) <<  "Verbose logging enabled to: " << mVerbose << endl;
    }

    //----------------------------------  Catch termination signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGINT);
}


//======================================  Destructor
FrameLink::~FrameLink(void) {
}

//======================================  Connect to socket
bool
FrameLink::connect_sock(void) {
    int rc = 0;

    //----------------------------------  Open a socket
    if (mSocket < 0) {
	mSocket = socket(PF_INET, SOCK_STREAM, 0);
	if (mSocket < 0) {
	    throw SysError("Socket request failed");
	}
	mBound = false;
    }

    //-----------------------------  A client binds a socket to a local address
    //                               and connects to the remote server.
    if (client()) {

	//--------------------------  Bind socket if not already bound.
	if (!mBound) {
	    sockaddr_in socknam;
	    socknam.sin_family      = AF_INET;
	    socknam.sin_port        = 0;
	    socknam.sin_addr.s_addr = INADDR_ANY;
	    int len                 = sizeof(socknam);
	    rc = ::bind(mSocket, (struct sockaddr *)&socknam, len);
	    if (rc < 0) {
		throw SysError("framelink: client port bind failed");
	    }
	    mBound = true;
	}

	//-------------------------------  Connect to server.
	if (!mConnected) {
	    logstamp(cout) << "Connecting to server: " << mPeerName << endl;
	    rc = connectWithTimeout(mSocket, (sockaddr*)&mServerAddr, 
				    sizeof(mServerAddr), Timeout);
	    if (rc == 0) {
		mConnected = true;
		mLastRecv = Now();
		if (v_check(true)) {
		    mVerbStream << "# Connected to: " << mPeerName << endl;
		}
	    }

	    //--------------------------  Handle errors
	    else {
		int sverrno = errno;
		logstamp(cout) << "connect failed: " << strerror(errno) << endl;
		switch(sverrno) {
		case ETIMEDOUT:
		    logstamp(cout) << "framelink: waiting for server " 
				   << mPeerName << endl;
		    break;
		case ECONNREFUSED:
		case ENOTCONN:
		case ENETUNREACH:
#ifdef EHOSTUNREACH
		case EHOSTUNREACH:
#endif
#ifdef ECONNABORTED
		case ECONNABORTED:
#endif
		    close(mSocket);
		    mSocket = -1;
		    sleep(30);
		    break;
		case EINTR:
		    break;
		default:
		    throw SysError(string("Connection failed to: ")+mPeerName);
		}
	    }
	    mBlockID = 0;
	}
    } 

    //----------------------------------  Server instances
    else if (server()) {
	//------------------------------  bind server sockets to specified port
	if (!mBound) {
	    logstamp(cout) << "Binding socket to server address: " 
			   << mPeerName << endl;
	    mServer = mSocket;
	    mSocket = -1;
	    int reuse = 1;
	    rc = setsockopt(mServer, SOL_SOCKET, SO_REUSEADDR, 
			    &reuse, sizeof(int));
	    if (rc != 0) {
		throw SysError("Unable to set reuse address");
	    }

	    rc = ::bind(mServer, (sockaddr*)&mServerAddr, sizeof(mServerAddr));
	    if (rc < 0) throw SysError("Server port bind failed");

	    logstamp(cout) << "Listening to server socket" << endl;
	    if (listen(mServer, 1)) {
		throw SysError("Listen failed");
	    }
	    mBound = true;
	}

	//------------------------------  Accept a connection
	if (!mConnected) {
	    logstamp(cout) << "Waiting for client connection" << endl;
	    mSocket = accept(mServer, 0, 0);
	    if (mSocket < 0) {
		perror("framelink: Accept error");
	    } else {
		logstamp(cout) << "Client connected" << endl;
		mConnected = true;
		mLastRecv = Now();
	    }
	    mBlockID = 0;
	}
    }
    return mConnected;
}

//======================================  Connect to shared memory
bool
FrameLink::connect_shm(void) {
    bool rc = false;
    //----------------------------------  Open the partition
    switch (mPgmMode) {

    case kPush:
    case kSend:
	if (!mInPart) {
	    logstamp(cout) << "framelink: Attaching consumer to partition " 
			   << mPartition << endl;
	    mInPart  = new LSMP_CON(mPartition.c_str());
	    if (!mInPart || !mInPart->attached()) {
		throw SysError("Failed to attach partition");
	    }
	}
	rc = (mInPart != NULL && mInPart->attached());
	break;

    case kPull:
    case kReceive:
	if (!mOutPart) {
	    logstamp(cout) << "framelink: Attaching producer to partition " 
			   << mPartition << endl;
	    mOutPart = new LSMP_PROD(mPartition.c_str());
	    if (!mOutPart || !mOutPart->attached()) {
		throw SysError("Failed to attach partition");
	    }
	    if (mOutPart->lock(true)) {
		int lockerr = errno;
		cerr << "framelink: attempt to lock partition: " << mPartition
		     << " failed with error: " << strerror(lockerr) << endl;
	    }
	}
	rc = (mOutPart != NULL && mOutPart->attached());
	break;

    default:
	rc = false;
	break;
    }
    return rc;
}

//======================================  Close shared memory connection
void
FrameLink::close_server(void) {
    if (mServer >= 0) {
	close(mServer);
	mServer = -1;
	mBound = false;
    }
}

//======================================  Close shared memory connection
void
FrameLink::close_shm(void) {
    if (mInPart)  {
	delete mInPart;
	mInPart = 0;
    }
    if (mOutPart) {
	delete mOutPart;
	mOutPart = 0;
    }
}

//======================================  Close socket connection
void
FrameLink::close_sock(void) {
    shutdown(mSocket, SHUT_RDWR);
    close(mSocket);
    mSocket = -1;
    if (client()) {
	mBound = false;
    }
    mConnected = false;
}

//=======================================  flush the specified length data
int
FrameLink::flush_packet(int len, bool wait) {
    if (len <= 0) return -1;
    int rc = recv(mSocket, 0, len, (MSG_TRUNC | MSG_WAITALL));
    mFramesFlushed++;
    return rc;
}

//======================================  Copy frames
void
FrameLink::process(void) {
    while (!mTerm) {
	int rc = 0;
	int saverrno = 0;

	//------------------------------  Wait for a connection if necessary
	if (!mConnected && !connect_sock()) continue;

	//------------------------------  Test shm connection.
	if (!connect_shm()) {
	    string("Unable to attach partition ")+mPartition;
	    throw runtime_error(string("Unable to attach partition: ")
				+ mPartition);
	}

	//------------------------------  Receiver processing...
	if (receiver()) {
	    link_header h;
	    sock_status hdrOK = read_header(h);
	    if (hdrOK == kRetry) {
		continue;
	    } else if  (hdrOK == kReconnect) {
		close_shm();
		close_sock();
		continue;
	    }

	    //--------------------------  Get a buffer from the output partition
	    uint32_t bufl = mOutPart->getBufferLength();
	    if (h.length > bufl) {
		logstamp(cerr) << "Frame length (" << h.length 
			       << ") > partition buffer length ("
			       << bufl << ")." << endl;
		flush_packet(h.length, true);
		if (v_check(false)) {
		    mVerbStream << " shm buffer too small" << endl;
		    mVerbStream.flush();
		}
		continue;
	    }

	    char* buf = mOutPart->get_buffer();
	    if (!buf) {
		if (v_check(false)) {
		    mVerbStream << " no buffer available" << endl;
		    mVerbStream.flush();
		}
		throw runtime_error("No buffer available");
	    }

	    //--------------------------  Read frame into shared memory
	    int rLen = read(buf, h.length, true);
	    if (rLen <= 0) {
		int save_errno = errno;
		mOutPart->return_buffer();
		errno = save_errno;
		if (v_check(false)) {
		    mVerbStream << " error receiving frame" << endl;
		    mVerbStream.flush();
		}
		throw SysError("Error receiving frame");
	    }

	    //--------------------------  Make a crc check
	    if (uint32_t(rLen) != h.length) {
		mOutPart->return_buffer();
		if (v_check(false)) {
		    mVerbStream << " Length error, read " << rLen 
				<< "/" << h.length << endl;
		    mVerbStream.flush();
		}
		logstamp(cerr) << "Length error (" << rLen << "/" << h.length 
			       << ") in frame at gps: " << h.gps_secs << endl;
		close_shm();
		close_sock();
		continue;
	    }

	    //--------------------------  Make a crc check
	    if (buf[39] == 1) { // crc only valid if FrHeader[39] == crc32
		uint32_t crc32 = 0;
		memcpy(&crc32, buf+h.length-4, sizeof(crc32));
		checksum_crc32 chk;
		chk.add((const unsigned char*)buf, h.length-4);
		if (crc32 != chk.result()) {
		    mChkSum++;
		    mOutPart->return_buffer();
		    if (v_check(false)) {
			mVerbStream << " checksum error." << endl;
			mVerbStream.flush();
		    }
		    logstamp(cerr) << "Checksum error in frame at gps: " 
				   << h.gps_secs << endl;
		    continue;
		}
	    }

	    //--------------------------  Warn if no crc found
	    else if (!(mBlocks % 1000)) {
		logstamp(cerr) << "Warning: Frame received without crc" << endl;
	    }

	    //--------------------------  Log receipt
	    if (v_check(false)) {
		mVerbStream << " OK." << endl;
		mVerbStream.flush();
	    }

	    //--------------------------  Count the buffer and release
	    if (v_check(true)) {
		Time now = Now();
		mVerbStream << now.getS() << " put " << h.gps_secs 
			    << " " << h.length;
	    }
	    mBlocks++;
	    mOutPart->SetID(h.gps_secs);
	    mOutPart->release(h.length);
	    if (v_check(false)) {
		mVerbStream << " OK." << endl;
		mVerbStream.flush();
	    }	    
	    end_stats(Time(h.gps_secs));
	}

	//------------------------------  Sender processing...
	else {
	    //--------------------------  Get a full frame buffer.
	    link_header h;
	    const char* buf = 0;
	    bool loop = true;
	    while (loop) {
		buf = mInPart->get_buffer();
		loop = (buf == NULL) && (errno == EINTR) && !mTerm;
		if (loop) {
		    logstamp(cout) << "get_buffer interrupted." << endl;
		}
	    }
	    if (!buf) {
		throw runtime_error("Unable to get input buffer");
	    }
	    start_stats();

	    //--------------------------  Get buffer params, check sequential
	    int blen = mInPart->getLength();
	    gps_type gps = mInPart->getEvtID();
	    Time now(Now());
	    if (v_check(true)) {
		mVerbStream << now.getS() << " get " << gps << " " << blen;
	    }
	    if (gps <= mLastGps) {
		logstamp(cerr) 
		    << "GPS of new frame (" << gps << ") before previous (" 
		    << mLastGps << ")" << endl;
		if (Now().getS() + 600 < mLastGps ) {
		    logstamp(cerr) 
			<< "Last frame was more than 600s in the future. "
			<< "Sequence error ignored!" << endl;
		    mInPart->free_buffer();
		    if (v_check(false)) {
			mVerbStream << " Sequence error" << endl;
			mVerbStream.flush();
		    }
		    continue;
		}
	    }
	    if (v_check(false)) {
		mVerbStream << " OK." << endl;
		mVerbStream.flush();
	    }

	    //--------------------------  Build a header, send it to the peer
	    h.block_id  = htonl(mBlockID++);
	    h.length    = htonl(blen);
	    h.gps_secs  = htonl(gps);
	    h.data_time = htonl(mFrameDt);
	    if (v_check(true)) {
		now = Now();
		mVerbStream << now.getS() << " put " << gps << " " << blen;
	    }
	    int send_flag = 0;
#ifdef P__DARWIN
	    int set = 1;
	    setsockopt(mSocket, SOL_SOCKET, SO_NOSIGPIPE, &set, sizeof(int));
#else
	    send_flag = MSG_NOSIGNAL;
#endif
	    rc = send(mSocket, &h, sizeof(h), send_flag);
	    saverrno = errno;
	    if (rc < 0) {
		if (v_check(false)) {
		    mVerbStream << " header send error: " << strerror(saverrno) 
				<< endl;
		    mVerbStream.flush();
		}
		mInPart->free_buffer();
		perror("framelink: Error sending header");
		switch (saverrno) {
		case EINTR:
		    logstamp(cerr) << "Buffer flushed after interrupt" << endl;
		    break;
		case ENOTCONN:
		case ECONNRESET:
		case EPIPE:
		    logstamp(cerr) << "Connection closed by peer" << endl;
		    close_shm();
		    close_sock();
		    break;
		default:
		    errno = saverrno;
		    throw SysError("Error sending header");
		}
		continue;
	    } else if (rc != int(sizeof(h))) {
		mInPart->free_buffer();
		if (v_check(false)) {
		    mVerbStream << " header length error." << endl;
		    mVerbStream.flush();
		}
		throw runtime_error("Header length error");
	    }

	    if (mDebug) {
		logstamp(cout) 
		    << "sent header block, len, gps, time = {" 
		    << ntohl(h.block_id) << ", " << blen << ", " << gps 
		    << ", " << ntohl(h.data_time) << "}" << endl;
	    }

	    //--------------------------  Send the frame data in one swell foop
	    rc = send(mSocket, buf, blen, send_flag);
	    saverrno = errno;
	    mInPart->free_buffer();
	    if (rc < 0) {
		cerr << "framelink: Error sending frame" << strerror(saverrno)
		     << endl;
		if (v_check(false)) {
		    mVerbStream << " data send error: " << strerror(saverrno) 
				<< endl;
		    mVerbStream.flush();
		}
		switch (errno) {
		case EINTR:
		    cerr << "Buffer flushed after interrupt" << endl;
		    break;
		case ENOTCONN:
		case ECONNRESET:
		case EPIPE:
		    logstamp(cerr) << "Connection closed by peer" << endl;
		    close_shm();
		    close_sock();
		    break;
		default:
		    throw SysError("Error sending frame");
		}
		continue;
	    } else if (rc != int(blen)) {
		if (v_check(false)) {
		    mVerbStream << " error sending frame." << endl;
		    mVerbStream.flush();
		}
		logstamp(cerr) 
		    << "framelink: Error sending frame, length sent = " 
		    << rc << " of " << blen << " bytes." << endl;
		close_shm();
		close_sock();
		continue;
	    }
	    mLastGps = gps;
	    end_stats(Time(gps));
	    if (v_check(false)) {
		mVerbStream << " OK." << endl;
		mVerbStream.flush();
	    }
	}   //----------------------------  if (!receiver())

    }   //--------------------------------  while(loop)

    //------------------------------------  Print a normal termination message.
    SigFlag::mask_t mask = mTerm.getSigFlags();
    int sig = 0;
    while ((mask >>= 1) != 0) sig++;
    logstamp(cout) << "frame link terminated by signal: " << sig << endl;
}

//=======================================  Read the specified length data
int
FrameLink::read(void* buf, int len, bool wait) const {
    int flags = 0;
    if (wait) flags = MSG_WAITALL;
    int nLeft = len;
    char* pBuf = reinterpret_cast<char*>(buf);
    int nRead = 0;
    while (nLeft > 0) {
	int rc = recv(mSocket, pBuf, nLeft, flags);
	if (!rc || (rc < 0 && errno != EINTR) || bool(mTerm)) {
	    nRead = rc;
	    break;
	}
	if (rc > 0) {
	    nRead += rc;
	    nLeft -= rc;
	    pBuf   += rc;
	}
    }
    return nRead;
}

//======================================  Read the specified length data
FrameLink::sock_status 
FrameLink::read_header(link_header& h) {

    //----------------------------------  Wait for a frame
    int rc = socketWait(mSocket, Timeout, wm_read);
    if (!rc) {
	logstamp(cout) << "framelink: Waiting for frame from " 
			       << mPeerName << endl;
	//continue;
    }
    else if (rc < 0) {
	switch (errno) {
	case EINTR:
	    break;
	default:
	    throw SysError("framelink: socketWait failed: ");
	}
	return kRetry;
    }

    //----------------------------------  Set up statistics
    start_stats();

    //----------------------------------  Read in the header
    Time now(Now());
    rc = read(&h, sizeof(h), false);
    if (rc < 0) {
	switch (errno) {
	case EINTR:
	case EAGAIN:
#if EWOULDBLOCK != EAGAIN
	case EWOULDBLOCK:
#endif
	    if(now < mLastRecv + disconnect_timeout) return kRetry;
	case ENOTCONN:
	case ECONNRESET:
	    logstamp(cout) << "Connection closed by sender." << endl;
	    return kReconnect;
	default:
	    throw SysError("Error reading header");
	}
    }

    //---------------------------------- End-of file on the read socket.
    else if (!rc) {
	logstamp(cout) << "end-of-file on read" << endl;
	return kReconnect;
    } 

    //---------------------------------- Length error on read
    else if (rc != int(sizeof(h))) {
	logstamp(cout) << "length error on read" << endl;
	return kReconnect;
    }

    //----------------------------------  Reorder the header bytes
    mLastRecv = now;
    h.block_id  = ntohl(h.block_id);
    h.length    = ntohl(h.length);
    h.gps_secs  = ntohl(h.gps_secs);
    h.data_time = ntohl(h.data_time);

    if (mDebug) {
	logstamp(cout) << "received header block, len, gps, time = {" 
		       << h.block_id << ", " << h.length << ", " 
		       << h.gps_secs << ", "<< h.data_time << "}" 
		       << endl;
    }

    //----------------------------------  Announce that we have a new frame
    if (v_check(true)) {
	mVerbStream << now.getS() << " get " << h.gps_secs << " " << h.length;
    }

    //----------------------------------  Check the block ID and length.
    if (!mBlockID) {
	if (h.block_id) {
	    logstamp(cout) << "First block id is not zero! block_id = " 
			   << h.block_id << endl;
	    mBlockID = h.block_id;
	}
	if (!mFrameDt) {
	    logstamp(cout) << "Data length inferred from first block. dt = " 
			   << h.data_time << endl;
	    mFrameDt = h.data_time;
	}
    }

    //----------------------------------  Check the block ID
    else if (mBlockID != h.block_id) {
	logstamp(cout) << "Unexpected block ID, block_id = " 
		       << h.block_id << endl;
	if (v_check(false)) {
	    mVerbStream << " Invalid block id." << endl;
	    mVerbStream.flush();
	}
	return kReconnect;
    }

    //----------------------------------  Check the frame length
    else if (mFrameDt != h.data_time) {
	logstamp(cout) << "Unexpected data length, data_time = " 
			   << h.data_time << endl;
	if (v_check(false)) {
	    mVerbStream << " Invalid data time." << endl;
	    mVerbStream.flush();
	}
	return kReconnect;
    }
    mBlockID++;
    return kOK;
}

//=======================================  Open verbose stream if requested
bool
FrameLink::v_check(bool reopen) {
    if (mVerbose.empty()) return false; // not requested
    bool open = mVerbStream.is_open();
    if (mVerbStream.fail() || (reopen && mVerbCount > max_verbline)) {
	if (open) mVerbStream.close();
	mVerbStream.clear();
	open = false;
    }
    if (!open && reopen) {
	mVerbStream.open(mVerbose.c_str(), 
			 std::ofstream::out | std::ofstream::app);
	open = mVerbStream.is_open();
	if (!open) {
	    cout << "Error: Unable to open output file: " << mVerbose << endl;
	}
	mVerbCount = 0;
    }
    if (open && reopen) mVerbCount++;
    return open;
}

//=======================================  Record start time statistics
void
FrameLink::start_stats(void) {
    if (mStatisticsPeriod != Interval(0)) {
	Time now = Now();
	if (now < mLastStats + mStatisticsPeriod) {
	    mSumStartLatency += now - mLastStats;
	} else {
	    if (mLastStats != Time(0)) print_stats();
	    mLastStats = now;
	    mSumStartLatency = 0;
	    mSumEndLatency   = 0;
	    mSumFrames       = 0;
	    mFramesFlushed   = 0;
	}
    }
}

//=======================================  Record start time statistics
void
FrameLink::end_stats(const Time& gps) {
    if (mStatisticsPeriod != Interval(0)) {
	Time now = Now();
	mSumStartLatency -= gps - mLastStats;
	mSumEndLatency   += now - gps;
	mSumFrames++;
    }
}

//=======================================  Print latency statistics
void
FrameLink::print_stats(void) const {
    if (mStatisticsPeriod != Interval(0) && mLastStats != Time(0)) {
	double nFrame = mSumFrames;
	if (!nFrame) nFrame = 1;
        cout << "Latency statistics starting at GPS " << mLastStats.getS()
	     << " start latency " << mSumStartLatency/nFrame
	     << " end latency "   << mSumEndLatency/nFrame << endl;
    }
}
