/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DMT base class methods.
//

//-------------------------------------  Header files.
#include <time.h>
#include <iostream>
#include <string>
#include <cstdlib>
#include <cstring>
#include <stdio.h>
#include <fstream>
#include <inttypes.h>
#ifndef NOTGDS
#include "iSMbuf.hh"
#include "SigFlag.hh"
#endif
//
//    Base class for the Data Monitor processing routines!
//

using namespace std;

#define MaxID 256

struct IGWhdr {
    long  len;   // note that the signed int is a cheat.
    short chksm;
    short id;
    int   inst;
};


/**  Frame processor base class for data monitors.
  */
class FrTest {
  public:
    typedef int64_t int8_t;
    typedef int32_t int4_t;
    typedef int16_t int2_t;

  public:
    FrTest(int argc, const char *argv[]);
    virtual ~FrTest();
    void ProcessFrame();
    void MainLoop(void);
    void finish(void) {mActive = false;}
    int Debug(void) const {return mDebug;}
    int  getNFrames(void) const {return mNFrames;}
    void Dump(const IGWhdr& h);

    char   getChar(void);
    int    getInt(void);
    long   getLong(void);
    short  getShort(void);
    long   getString(char* p, long max);
    long   skipStr(long max);
    void ignore(long off);

    bool procSH(const IGWhdr& h);
    bool procAdc(const IGWhdr& h);

  private:
    bool       mActive;
#ifndef NOTGDS
    SigFlag    mTerm;
#else
    bool       mTerm;
#endif
    streambuf* mBuffer;
    istream*   mStream;
    int        mDebug;
    int        mNFrames;
    int        maxFrames;
    bool       mSwap;
    bool       mDumpID[MaxID];
    bool       mDefID[MaxID];
    int        mAdcDataID;
    int        mEOFrID;
    int        mEOFID;
    char       mVersion[2];
    int        mHdrSz;
    bool       mDaqErr;
};

inline void
FrTest::ignore(long off) {
    mStream->seekg(off, ios::cur);
}

//-------------------------------------  FrTest main function.
int main(int argc, const char *argv[]) {
    FrTest MyObject(argc,argv); 
    MyObject.MainLoop(); 
    return 0;}

//-------------------------------------  FrTest constructor.
FrTest::FrTest(int argc, const char *argv[]) 
  : mActive(false), mBuffer(0), mStream(0), mDebug(0), mNFrames(0),
    maxFrames(0), mAdcDataID(-1), mEOFrID(-1), mEOFID(-1), mDaqErr(false)
{
#ifndef NOTGDS
    const char*     partition = getenv("LIGOSMPART");
    if (!partition) partition = "LIGO_Online";
#endif
    const char*     file      = (const char*) 0;
    for (int i=0 ; i<MaxID ; i++) mDumpID[i] = false;

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc && argv ; i++) {
        string argi = argv[i];
        if (argi == "-infile" || argi == "-i") {
	    if (++i >= argc) continue;
	    file = argv[i];
#ifndef NOTGDS
	} else if (argi == "-partition" || argi == "-p") {
	    if (++i >= argc) continue;
	    partition = argv[i];
#endif
	} else if (argi == "-daqerr") {
	    mDaqErr = true;
	} else if (argi == "-debug") {
	    if (++i == argc) mDebug = 1;
	    else             mDebug = strtol(argv[i], 0, 0);
	} else if (argi == "-dumpid") {
	    if (++i >= argc) continue;
	    mDumpID[strtol(argv[i], 0, 0)] = true;
	} else if (argi == "-maxframes") {
	    if (++i == argc) maxFrames = 1;
	    else             maxFrames = strtol(argv[i], 0, 0);
	} else {
	    cerr << "Invalid argument: " << argv[i] << endl;
	    cerr << "Syntax:" << endl;
	    cerr << "FrTest [{-i | -infile} <file>] [-debug <debug-level>]" 
		 << " [-dumpid <struct-id>]" << endl;
	    cerr << "       [-maxframes <max-fr>] [-daqerr]";

#ifndef NOTGDS
	    cerr << " [{-p | -partition} <part>]";
#endif
	    cerr << endl;
	    return;
	}
    }

#ifndef NOTGDS
    //---------------------------------  Handle signals
    mTerm.add(SIGTERM);
    mTerm.add(SIGHUP);
    mTerm.add(SIGINT);
#else
    mTerm = false;
#endif

    //---------------------------------  Open a frame stream (FrameCPP)
    if (file) {
        filebuf* fbuf = new filebuf;
	if (fbuf->open((char*)file, ios::in)) {
	    cout << "FrTest: Opened file " << file 
		 << " for frame input." << endl;
	} else {
	    cerr << "FrTest: Unable to open file " << file 
		 << " for frame input." << endl;
	    return;
	}
	mBuffer = fbuf;
    } else {
#ifdef NOTGDS
        cerr << "FrTest: No file specified with -infile " << endl;
#else
        iSMbuf* sbuf = new iSMbuf;
	if (sbuf->open((char*)partition, ios::in)) {
	    cout << "FrTest: Opened partition " << partition
		 << " for frame input." << endl;
	} else {
	    cerr << "FrTest: Unable to access partiton " << partition
		 << " for frame input." << endl;
	    return;
	}
	mBuffer = sbuf;
#endif
    }
    mStream = new istream(mBuffer);
    mActive = true;
}

//-------------------------------------  FrTest object destructor.
FrTest::~FrTest(void) 
{
    //---------------------------------  Close the Input file/partition.
    if (mStream) delete mStream;
    if (mBuffer) delete mBuffer;

    //---------------------------------  Print statistics
    if (mDebug != 999) {
        cout << "Frame reading terminated after " << mNFrames 
	     << " frames." << endl;
    }
}

//-------------------------------------  Main processing loop function.
void
FrTest::MainLoop(void) 
{
    //---------------------------------  Loop until terminated.
    while(mActive && !mTerm) {
        //-----------------------------  Process a frame.
	ProcessFrame();
	if (maxFrames && mNFrames>=maxFrames) mActive = false;
    }
}

void
FrTest::ProcessFrame() {
    char   igwd[5];
    IGWhdr h;

    //---------------------------------  Check the IGWD specifier
    if (mStream->read(igwd, sizeof(igwd)).gcount() < int(sizeof(igwd))) {
	finish();
	return;
    }

    if (strcmp("IGWD", igwd)) {
        cout << "Error: Data isn't IGWD format" << endl;
	finish();
	return;
    }

    if (!mNFrames) {
        mStream->read(mVersion, sizeof(mVersion));
	cout << "Frame version is " << int(mVersion[0]) << "." 
	     << int(mVersion[1]) << endl;
	ignore(5);
    } else {
	ignore(7);
    }

    //---------------------------------  Test for swapped bytes
    unsigned short SwapCode;
    mStream->read((char*)&SwapCode, sizeof(SwapCode));
    mSwap = (SwapCode != 0x1234);
    ignore(26);

    //---------------------------------  Set up header size
    if (mVersion[0] >= 6) {
        mHdrSz = sizeof(int8_t) + sizeof(int2_t) + sizeof(int4_t);
    } else {
        mHdrSz = sizeof(int4_t) + sizeof(int2_t) + sizeof(int2_t);
    }

    //---------------------------------  Initialize the instance counts.
    int strinst[MaxID];
    for (int i=0 ; i<MaxID ; i++) {
        strinst[i] = 0;
	mDefID[i]  = false;
    }
    mDefID[1] = true;
    mDefID[2] = true;

    //---------------------------------  Loop over structures
    while (!mTerm) {

        //-----------------------------  Read the structure header
        if (mVersion[0] >= 8) {
	    h.len   = getLong();
	    h.chksm = getChar();
	    h.id    = getChar();
	    h.inst  = getInt();
	} else if (mVersion[0] >= 6) {
	    h.len   = getLong();
	    h.chksm = 0;
	    h.id    = getShort();
	    h.inst  = getInt();
	} else {
	    h.len   = getInt();
	    h.chksm = 0;
	    h.id    = getShort();
	    h.inst  = getShort();
	}

        //-----------------------------  Check for valid structure length
	if (h.len < mHdrSz) {
	    cout << "Error: Illegal structure length " << h.len << endl;
	    finish();
	    return;
	}

        //-----------------------------  Check for valid structure ID
	if (h.id <= 0 || h.id >= MaxID || !mDefID[h.id]) {
	    cout << "Error: Illegal or undefined structure ID " << h.id 
		 << endl;
	    finish();
	    return;
	}

        //-----------------------------  Check for valid instance number
	if (h.inst != strinst[h.id]++) {
	    if (h.inst == 1 && strinst[h.id] == 1) {
	        cout << "Warning: First instance of structure " << h.id
		     << " is 1 instead of zero." << endl;
	        strinst[h.id]++;
	    } else {
	        cout << "Error: Illegal structure instance " << h.inst 
		     << " of ID " << h.id << ", expected " 
		     << strinst[h.id]-1 << endl;
	    }
	}

	if (Debug() > 1) cout << "Length, id, instance =" << h.len << " " 
			      << h.id << " " << h.inst << endl;
	if (h.id == 1) {
            if (procSH(h)) return;
	} else if (mDaqErr && h.id == mAdcDataID) {
	    procAdc(h);
	} else if (mDumpID[h.id]) {
	    Dump(h);
	} else {
	    ignore(h.len - mHdrSz);
	}

	if (h.id == mEOFrID) {
	    mNFrames++;
	    if (mVersion[0] >= 4) {
	        for (int i=1 ; i<MaxID ; i++) strinst[i] = 0;
	    }
	}
	if (h.id == mEOFID) break;
    }
    if (mDebug) cout << "End of Frame " << getNFrames() << endl;
}

inline void 
swap(unsigned short& x) {
    unsigned short t = (x>>8) & 255; 
    x = ((x&255)<<8) + t;
}

#define LSTRUCTNAME 32
bool 
FrTest::procSH(const IGWhdr& h) {
    char  name[LSTRUCTNAME];
    char* comment;
    unsigned short lname=getShort();
    if ((lname < 2) || (lname > h.len-mHdrSz-6)) {
        cout << "Error: Inconsistent name length in SH, l=" << lname << endl;
	ignore(h.len-mHdrSz-2);
	return true;
    }
    mStream->read(name, lname);

    short ID=getShort();
    short lcomment = getShort();
    if (mVersion[0] < 8) {
	if (h.len != mHdrSz + 6 + lname + lcomment) {
	    cout << "Error: Inconsistent comment length in SH, l=" << lname 
		 << endl;
	    ignore(h.len-mHdrSz-6-lname);
	    return true;
	}
    } else {
	if (h.len != mHdrSz + 10 + lname + lcomment) {
	    cout << "Error: Inconsistent comment length in SH, l=" << lname 
		 << endl;
	    ignore(h.len-mHdrSz-10-lname);
	    return true;
	}
    }

    if (mDumpID[h.id]) {
        printf ("Structure type SH, Instance %i, Length %li\n",
		  h.inst, h.len);
	comment = new char[lcomment];
	mStream->read(comment, lcomment);
	printf ("  Name:%-20s  ID: %4i Comment: %s\n", name, ID, comment);
	delete[] comment;
    } else {
        ignore(lcomment);
    }
    if (mVersion[0] >= 8) {
	getInt(); // get the checksum
    }
    if (!ID || ID > MaxID) {
        cout << "Error: Illegal structure ID defined" << endl;
	return true;
    }
    mDefID[ID] = true;
    if (!strcmp(name, "FrEndOfFile"))  mEOFID = ID;
    if (!strcmp(name, "FrEndOfFrame")) mEOFrID = ID;
    if (!strcmp(name, "FrAdcData"))    mAdcDataID = ID;
    return false;
}

bool 
FrTest::procAdc(const IGWhdr& h) {
    char name[64];
    long len = h.len - mHdrSz;
    long l = getString(name, sizeof(name));
    if (l < 0) {
        ignore(len+l);
	return true;
    }
    len -= l;
    l = skipStr(len);
    if (l < 0) {
        ignore(len+l);
	return true;
    }
    len -= l;
    ignore(20);
    len -= 20;
    l = skipStr(len);
    if (l < 0) {
        ignore(len+l);
	return true;
    }
    len -= l;
    ignore(28);
    len -= 28;
    short dValid = getShort();
    len -= 2;
    ignore(len);
    if (dValid) {
        cout << "Daq error (" << dValid << ") in channel: "  << name << endl;
    }
    return false;
}

char
FrTest::getChar(void)  {
    char a;
    mStream->read(&a, sizeof(a));
    if (mStream->eof()) return 0;
    return a;
}

int
FrTest::getInt(void)  {
    int4_t a;
    mStream->read((char*)&a, sizeof(a));
    if (mStream->eof()) return 0;
    if (mSwap) {
        int4_t b(a);
	char* from = (char*) (&b + 1);
	char* to   = (char*) &a;
	for (unsigned int i=0 ; i<sizeof(a) ; i++) *to++ = *--from;
    }
    return a;
}

long
FrTest::getLong(void) {
    int8_t a;
    mStream->read((char*)&a, sizeof(a));
    if (mStream->eof()) return 0;
    if (mSwap) {
        int8_t b(a);
	char* from = (char*) (&b + 1);
	char* to   = (char*) &a;
	for (unsigned int i=0 ; i<sizeof(a) ; i++) *to++ = *--from;
    }
    return a;
}

short
FrTest::getShort(void) {
    int2_t a;
    mStream->read((char*)&a, sizeof(a));
    if (mStream->eof()) return 0;
    if (mSwap) {
        int2_t b(a);
	char* from = (char*) (&b + 1);
	char* to   = (char*) &a;
	for (unsigned int i=0 ; i<sizeof(a) ; i++) *to++ = *--from;
    }
    return a;
}

long 
FrTest::getString(char* content, long max) {
    short l = getShort();
    if (l>max) {
        cout << "Invalid string length" << endl;
        return -2;
    }
    mStream->read(content, l);
    return l+2;
}

long 
FrTest::skipStr(long max) {
    short l = getShort();
    if (l>max) {
        cout << "Invalid string length" << endl;
        return -2;
    }
    ignore(l);
    return l+2;
}

#define TEXTLEN 16
void 
FrTest::Dump(const IGWhdr& h) {
    char string[TEXTLEN+1];

    //-----------------------------  Read in a frame
    int nbyt = h.len - sizeof(h);
    int nwd  = (nbyt + sizeof(int) - 1)/sizeof(int);
    int* buffer =  new int[nwd];
    if (!buffer) return;
    buffer[nwd-1] = 0;
    mStream->read((char*)buffer, nbyt);

    //-----------------------------  Dump the frame to stdout.
    printf ("Structure type %i, instance %i, Length %li\n", 
	   h.id, h.inst, h.len);
    for (int i=0 ; i<nwd ; i+=4) {
        for (int j=0 ; j<TEXTLEN ; j++) {
	    char tchar = *((char*) (buffer + i) +j);
	    if      (tchar >= 'a' && tchar <= 'z') string[j] = tchar;
	    else if (tchar >= 'A' && tchar <= 'Z') string[j] = tchar;
	    else if (tchar >= '0' && tchar <= '9') string[j] = tchar;
	    else if (tchar == ' ' || tchar == '_') string[j] = tchar;
	    else if (tchar == ':' || tchar == '-') string[j] = tchar;
	    else                                   string[j] = '.';
	}
	string [TEXTLEN] = 0;
        printf ("%8lx  %08x %08x %08x %08x |%s|\n", i*sizeof(int),
	       buffer[i], buffer[i+1], buffer[i+2], buffer[i+3], string);
    }
    delete[] buffer;
}
