/* -*-  mode: c++; c-basic-offset: 3; -*- */
#ifndef SEGLIST_HH
#define SEGLIST_HH

#include "LockSegment.hh"
#include <stdexcept>
#include <vector>

//======================================  Segment list class
/**  LockSegList is a list of LockSegment objects ordered by Start Time.
  *
  *  @memo Segment list class.
  *  @author J.Zweizig
  *  @version 1.3; last modified 7/30/2010
  */
class LockSegList {
public:
   /** Data size type.
     */
   typedef unsigned long size_type;

public:

   /**  Construct an empty segment list.
     *  @memo Default segment list constructor.
     */
   LockSegList(void) {}

   /**  Construct and Segment list and optionally fill it from a file.
     *  @memo Segment list constructor.
     *  @param lid  List ID string
     *  @param file File from which the list is to be initialized.
     */
   explicit LockSegList(const char* lid, const char* file=0);

   /**  Destroy a segment list and free all storage.
     *  @memo Segment list destructor.
     */
   virtual ~LockSegList(void) {}

   /**  Check that segments are inserted in order... print errors to cerr.
     *  @memo Check list ordering.
     *  @return True if an error was discovered.
     */
   bool check(void) const;

   /**  Erase all the segments in the list.
     *  @memo Clear the segment list.
     */
   void clear(void);

   /**  Coalesce overlapping segments. The list must be sorted before it 
     *  coalesced.
     *  @memo Coalesce segment in the list.
     */
   void coalesce(void);

   /**  Look for the segment containing the specified time.
     *  @memo Fins segment containing a specified time
     *  @param t Time to search for.
     *  @return Index of segment into the list.
     */
   size_type find(const Time& t) const throw(std::domain_error);

   /**  Find the first segment that ends after the specified time. The index
     *  of the found segment is returned. If there are no segments after the 
     *  specified time, an index equal to the size of segment list is returned.
     *  @memo Find segment after a specified time.
     *  @param t Earliest time for segment.
     *  @return Index of segment into the list.
     */ 
   size_type findafter(const Time& t) const;

   /**  Return the list title.
     *  \return List ID string pointer.
     */
   const char* getListID(void) const;

   /**  Test whether the specified time is within one of the segments in the
     *  list.
     *  @memo Test if time is in segment.
     *  @param t Time to be tested
     *  @return True if the time is in a segment.
     */
   bool inSegment(const Time& t) const;

   /**  Test whether any time between the specified start and end time is
     *  within one or more of the list segments.
     *  @memo Test if time range is in a segment.
     *  @param t Range start time
     *  @param t2 Range end time.
     *  @return True if range overlaps a segment.
     */
   bool inSegment(const Time& t, const Time& t2) const;

   /**  Insert a copy of the specified segment into the list. The segment
     *  is inserted to maintain an increasing start-time order.
     *  @memo Insert a segment into the list
     *  @param seg Reference to the segment to be inserted.
     */
   void insert(const LockSegment& seg);

   /**  Invert the segment list. The gaps between successive segments are 
     *  converted to segments and the original segments are removed.
     */
   void invert(void);

   /**  @memo Calculate the "live time" of the segments list.
     *  Return the sum of all segment durations between the start and the 
     *  end time.
     */
   Interval live(Time start=Time(0), Time end=Time(SEGMENT_TINF)) const;

   /**  Insert a segment into the list. If the segment overlaps existing
     *  one or more segments, the segments are combined.
     *  @memo Merge a segment into the list.
     *  @param seg Segment to be merged into the list.
     */
   void merge(const LockSegment& seg);

   /**  Pad the front and back of each segment in the list by pFront and 
     *  pBack. The pFront and pBack intervals are defined such that a positive 
     *  interval increases the segment duration and a negative interval 
     *  decreases the resulting segment duration. i.e. the resulting segment 
     *  is \f$[start - pStart, end + pBack]\f$. Segments are coalesced after 
     *  the padding is added.
     *  @param pFront Length of padding to add to the start of the segment.
     *  @param pBack  Length of padding to add to the back of the segment.
     */
   void pad(Interval pFront=0, Interval pBack=0);

   /**  Fill segment list from the contents of the specified file.
     *  @memo Read segments
     *  @param file File containing segment data.
     */
   void read(const std::string& file);

   /**  Set the debug printout level.
     *  @memo Set debug print level.
     *  @param lvl Debug print level.
     */
   void setDebug(int lvl);

   /**  Return the number of segments in the list.
     *  @memo Return segment list size.
     *  @return Number of segments in list.
     */
   size_type size(void) const;

   /**  Sort the list in order of increasing start times. No coalescing 
     *  of overlapping segments is performed.
     *  @memo Sort the list.
     */
   void sort(void);

   /**  Add a segment to the end of the list (unsorted insertion).
     *  @memo Add a segment to the end of the list.
     *  @param seg Segment to be added to the list.
     */
   void stuff(const LockSegment& seg);

   /**  Add a list of segments segment to the end of the list (unsorted 
     *	insertion).
     *  @memo Add a segment to the end of the list.
     *  @param segl List of segments to be added to the list.
     */
   void stuff(const LockSegList& segl);

   /**  Return a constant reference to the specified segment.
     *  @memo Get reference to ith segment.
     *  @param seg Index of segment to be referenced.
     *  @return Constant reference to specified segment.
     */
   const LockSegment& operator[](size_type seg) const;

   /**  Return a constant reference to the segment containing the specified
     *  time.
     *  @memo Get reference to segment.
     *  @param t Time of segment to be referenced.
     *  @return Constant reference to specified segment.
     */
   const LockSegment& operator[](const Time& t) const throw(std::domain_error);

   /**  Replace the current list with the union of the list and the argument.
     *  \brief OR two segment lists
     *  \param t List to be ORed into the current list
     *  \return Reference to modified list.
     */
   LockSegList& operator|=(const LockSegList& t);

   /**  Replace the current list with the intersection of the list and the 
     *  argument.
     *  \brief AND two segment lists
     *  \param t List to be ANDed into the current list
     *  \return Reference to modified list.
     */
   LockSegList& operator&=(const LockSegList& t);

   /**  Replace the current list with the differences between the current 
     *  list and the argument list argument.
     *  \brief XOR two segment lists
     *  \param t List to be XORed into the current list
     *  \return Reference to modified list.
     */
   LockSegList& operator^=(const LockSegList& t);

   /**  Test whether any segments in the list overlap the specified time span.
     *  @memo Test for segment overlapping a range
     *  @param t Start of range
     *  @param t2 End of range
     *  @return True if one or more segments in the list overlaps the range.
     */
   bool overlap(const Time& t, const Time& t2) const;

   /**  Write a segment specifier to the list. The list specifier consists 
     *	of the List ID followed by a hyphen and the segment identifier.
     *  @memo Write segment ID
     *  @param out Output stream
     *  @param inx Segment index
     *  @return Output stream reference.
     */
   std::ostream& putID(std::ostream& out, size_type inx) const;

private:
   LockSegment& ref(size_type seg);
   const LockSegment& ref(size_type seg) const;
   void checkhdr(void) const;
   void stretch(size_type N=1);

public:
   /** Locked segment list storage type.
     */
   typedef std::vector<LockSegment> SegList_type;

   /**  Constant locked segment list iterator.
     */
   typedef SegList_type::const_iterator seg_iter;

private:
   std::string  mListID;
   SegList_type mList;
   int          mDebug;
};

//======================================  Segment list inlined methods
inline const char*
LockSegList::getListID(void) const {
   return mListID.c_str();
}

inline const LockSegment&
LockSegList::operator[](size_type seg) const {
   return mList[seg];
}

inline const LockSegment&
LockSegList::operator[](const Time& t) const throw(std::domain_error) {
   return mList[find(t)];
}

inline LockSegment&
LockSegList::ref(size_type seg) {
   return mList[seg];
}

inline const LockSegment&
LockSegList::ref(size_type seg) const {
   return mList[seg];
}

inline LockSegList::size_type
LockSegList::size(void) const {
   return mList.size();
}

#endif  // !defined(SEGLIST_HH)
