/* -*- mode: c++; c-basic-offset: 4; -*- */
// #define SEGLIST_DEBUG_PRINT 1

#include "SegList.hh"
#include "ParseLine.hh"
#include <iostream>
#include <algorithm>

using namespace std;

const unsigned long alloc_quantum(4096);

inline void
LockSegList::stretch(size_type N) {
    size_type need = mList.size() + N;
    size_type cap  = mList.capacity();
    if (need > cap) {
	if (need < cap + alloc_quantum) {
	    mList.reserve(cap + alloc_quantum);
	} 
	else {
	    mList.reserve(cap + N);
	}
    }
}

//======================================  test wheter to merge
inline bool
testMerge(const LockSegment& l1, const LockSegment& l2) {
    return l1.touch(l2) && (l1.getID() == l2.getID());
}

//======================================  Segment list constructor
LockSegList::LockSegList(const char* lid, const char* file) 
  : mListID(lid)
{
    stretch();
    if (file) read(file);
}

//======================================  clear segment list
void
LockSegList::clear(void) {
    mList.clear();
}

//======================================  check segment list
void
LockSegList::checkhdr(void) const {
    cerr << "Errors found in segment list: " << mListID << endl;
}

//======================================  check segment list
bool
LockSegList::check(void) const {
    size_type N=size();
    bool error=false;
    for (size_type i=0; i<N; ++i) {
	if (ref(i).getDuration() <= Interval(0)) {
	    if (!error)checkhdr();
	    error = true;
	    cerr << "Segment: " << i << " duration: " << ref(i).getDuration()
		 << " not positive definite." << endl;
	} else if (i && ref(i).getStartTime() <= ref(i-1).getStartTime()) {
	    if (!error)checkhdr();
	    error = true;
	    cerr << "Segment: " << i << " [" << ref(i) 
		 << "] does not start after segment: " << i-1 
		 << " [" << ref(i-1) << "]." << endl;	    
	}
    }
    return error;
}

//======================================  Coalesce segments in list
void
LockSegList::coalesce(void) {
    size_type N=size();
    size_type i0=0;
    while (i0<N && ref(i0).getDuration() <= Interval(0.0)) i0++;
    if (i0 == N) return;
    if (i0) ref(0) = ref(i0);

    size_type j=0;
    for (size_type i=i0+1; i<N; ++i) {
        if (ref(i).getDuration() <= Interval(0.0)) continue;
        if (testMerge(ref(j), ref(i))) {
	    ref(j) |= ref(i);
	} else {
	    j++;
	    if (j != i) ref(j) = ref(i);
	}
    }
    j++;
    if (j < N) mList.erase(mList.begin()+j, mList.begin()+N);
    if (check()) cerr << "Previous error found in: coalesce(" << mListID 
		      << ")." << endl;
}

//======================================  Find segment for specified time
LockSegList::size_type
LockSegList::find(const Time& t) const throw(domain_error) {
    for (size_type inx=findafter(t); inx; ) {
	if (mList[--inx].inSegment(t)) return inx;
    }
    throw domain_error("Time not in segment");
}

//======================================  Find first segment starting after t
LockSegList::size_type
LockSegList::findafter(const Time& t) const {
    size_type low=0, high=size();
    while (low != high) {
        size_type inx = (low+high) >> 1;
	if (t <= mList[inx].getStartTime()) high = inx;
	else if (inx > low)               low  = inx;
	else                              ++low;
    }
    return high;
}

//======================================  Test if time is in segment
bool
LockSegList::inSegment(const Time& t) const {
    for (size_type inx=findafter(t); inx; ) {
	if (mList[--inx].inSegment(t)) return true;
    }
    return false;
}

//======================================  Test if time range is in segment
bool
LockSegList::inSegment(const Time& t, const Time& t2) const {
    for (size_type inx=findafter(t); inx; ) {
	if (mList[--inx].inSegment(t, t2)) return true;
    }
    return false;
}

//======================================  Insert a segment into the list
void
LockSegList::insert(const LockSegment& l) {
    stretch();
    if (mList.empty() || mList.back() < l) {
	mList.push_back(l);
    } else {
	size_type inx = findafter(l.getStartTime());
	mList.insert(mList.begin()+inx, l);
    }
}

//======================================  Insert a segment into the list
void
LockSegList::invert(void) {
    coalesce();
    size_type N = size();
    size_type j = 0;
    Time last(0);
    for (size_type i=0; i<N; i++) {
        Time start = mList[i].getStartTime();
	Time end = mList[i].getEndTime();
	if (start > last) mList[j++] = LockSegment(0, last, start, 0);
	last = end;
    }

    //----------------------------------  Fill in the last entry
    const Time tInf(SEGMENT_TINF);
    if (j==N) {
        mList.push_back(LockSegment(0, last, tInf, 0));
    } else {
        mList[j++] = LockSegment(0, last, tInf, 0);
	if (j<N) mList.erase(mList.begin()+j, mList.end());
    }
}

//======================================  clear segment list
Interval
LockSegList::live(Time start, Time end) const {
    size_type N=size();
    Interval r(0);
    if (end <= start) return r;
    for (size_type i=0; i<N; ++i) {
	Time tEnd = mList[i].getEndTime();
	if (tEnd <= start) continue;
	Time tStart = mList[i].getStartTime();
	if (tStart >= end) break;
	if (tStart < start) tStart = start;
	if (tEnd > end) tEnd = end;
        r += tEnd - tStart;
    }
    return r;
}

//======================================  Insert a segment into the list
void
LockSegList::merge(const LockSegment& l) {
    size_type inx = findafter(l.getStartTime());
    size_type N   = size();
    if (inx < N && testMerge(l, mList[inx])) {
	mList[inx] |= l;
	if (inx && testMerge(mList[inx-1], mList[inx])) {
	    mList[inx-1] |= mList[inx];
	    mList.erase(mList.begin()+inx);
	}
    } else if (inx && testMerge(l, mList[inx-1])) {
	mList[inx-1] |= l;
    } else {
	stretch();
	if (inx == N) mList.push_back(l);
	else          mList.insert(mList.begin()+inx, l);
    }
}


//======================================  pad each segment in the list.
void
LockSegList::pad(Interval pFront, Interval pBack) {
    size_type N=size();
    for (size_type i=0; i<N; ++i) {
	mList[i].pad(pFront, pBack);
    }
    coalesce();
}

//======================================  write a segment ID
ostream&
LockSegList::putID(ostream& out, size_type inx) const {
    out << mListID << "-" << mList[inx].getID();
    return out;
}

//======================================  Read segments from a file.
void
LockSegList::read(const std::string& file) {
    ParseLine pl(file.c_str());
    if (!pl.isOpen()) {
        cerr << "Unable to open file: " << file << endl;
	return;
    }

    enum {
      unknown,
      segwizard,
      startend} filetype(unknown);

    while (pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;

	//------------------------------  Identify file type.
	if (filetype == unknown) {
	    if (nArg == 1 && string(pl[0]) == "seg_cit") {
	        filetype = segwizard;
		continue;
	    } else if (nArg == 2 && pl.getInt(1) > pl.getInt(0)) {
	        filetype = startend;
	    } else if (nArg >= 4 &&
		       pl.getInt(3) == pl.getInt(2)-pl.getInt(1)) {
	        filetype = segwizard;
	    } else {
	        throw runtime_error("Unrecognized segment file format");
	    }
	}

	//------------------------------  Get segment parameters.
	LockSegment::id_type id = 0;
	unsigned long start, duration;
	LockSegment::flag_type flags = 0;
	switch (filetype) {
	case startend:
	    start = pl.getInt(0);
	    duration = pl.getInt(1) - start;
	    break;
	case segwizard:
	    id = pl.getInt(0);
	    start = pl.getInt(1);
	    duration = pl.getInt(3);
	    if (nArg >= 5) flags = pl.getInt(4);
	    break;
	default:
	    throw runtime_error("Unrecognized segment file format");
	}

	//-------------------------------  Insert the segments
	if (start < 700000000 || duration < 0 || duration > 700000000) {
	    cerr << "Invalid segment id/start/duration=" << id << " " 
		 << start << " " << duration << endl;
	}
	insert(LockSegment(id, Time(start), Interval(duration), flags));
    }
}

//========================================================================
//
//                  Segment  Binary  Operators
//

//======================================  Calculate union of segment lists.
LockSegList&
LockSegList::operator|=(const LockSegList& l) {
    size_type N1=size();
    size_type N2=l.size();

    //----------------------------------  Count the number of output segments
    size_type N=0;
    Time tEnd(0);
    for (size_type i=0, j=0; i<N1 || j<N2;) {
	if (i>=N1) {
	    if (l[j].getStartTime() > tEnd) N++;
	    tEnd = l[j++].getEndTime();
	} else if (j>=N2) {
	    if (ref(i).getStartTime() > tEnd) N++;
	    tEnd = ref(i++).getStartTime();
	} else if (ref(i) < l[j]) {
	    Time iEnd = ref(i).getEndTime();
	    if (ref(i).getStartTime() > tEnd) N++;
	    if (iEnd > tEnd) tEnd = iEnd;
	    i++;
	} else {
	    Time jEnd = l[j].getEndTime();
	    if (l[j].getStartTime() > tEnd) N++;
	    if (jEnd > tEnd) tEnd = jEnd;
	    j++;
	}
    }

    //----------------------------------  Allocate a temporary list
    SegList_type t;
    t.reserve(N);
    if (N1 && (!N2 || ref(0) < l[0]) ) {
	t.push_back(ref(0));
    } else if (N2) {
	t.push_back(l[0]);
    }

    //----------------------------------  Build up the list
    for (size_type i=0, j=0; i<N1 || j<N2;) {
	try {
	    if (i>=N1) {
		if (t.back().touch(l[j])) t.back() |= l[j++];
		else                      t.push_back(l[j++]);
	    } else if (j>=N2) {
		if (t.back().touch(ref(i))) t.back() |= ref(i++);
		else                        t.push_back(ref(i++));
	    } else if (ref(i) < l[j]) {
		if (t.back().touch(ref(i))) t.back() |= ref(i++);
		else                        t.push_back(ref(i++));
	    } else {
		if (t.back().touch(l[j])) t.back() |= l[j++];
		else                      t.push_back(l[j++]);
	    }
	} catch (exception& e) {
	    cerr << "Exception: " << e.what() 
		 << " caught in SegList::operator|=" << endl;
	    throw;
	}
    }
    mList = t;
    if (check()) cerr << "Previous error found in: " << mListID 
		      << " |= " << l.mListID << endl;
    return *this;
}

//======================================  Calculate intersection of two lists
LockSegList&
LockSegList::operator&=(const LockSegList& l) {

    //---------------------------------- count the number of entries
    size_type N2=l.size();
    size_type N1=size();
    size_type N=0;
    for (size_type i=0, j=0; j<N2 && i<N1; ) {
	if (ref(i).overlap(l[j])) N++;
	if (ref(i).getEndTime() < l[j].getEndTime()) i++;
	else                                         j++;
    }

    //---------------------------------  Allocate a temporary result list.
    SegList_type t;
    t.reserve(N);

    //---------------------------------  Loop over pairs of segments.
    for (size_type i=0, j=0; j<N2 && i<N1; ) {
	if (ref(i).overlap(l[j])) {
	    t.push_back(ref(i));
	    t.back() &= l[j];
	}
	if (ref(i).getEndTime() < l[j].getEndTime()) {
	    i++;
	} else {
	    j++;
	}
    }
    mList = t;
    if (check()) cerr << "Previous error found in: " << mListID 
		      << " &= " << l.mListID << endl;
    return *this;
}

//======================================  Calculate differences two lists
LockSegList&
LockSegList::operator^=(const LockSegList& l) {
    LockSegList temp(*this);
    temp &= l;
    temp.invert();
    *this |= l;
    *this &= temp;
    if (check()) cerr << "Previous error found in: " << mListID 
		      << " ^= " << l.mListID << endl;
    return *this;
}

//======================================  Test if [t1:t2] overlaps a segment
bool 
LockSegList::overlap(const Time& t, const Time& t2) const {
    size_type N = size();
    size_type inx = findafter(t);
    if (inx == N) return false;
    if (mList[inx].overlap(t, t2)) return true;
    inx++;
    if (inx < N && mList[inx].overlap(t, t2)) return true;
    return false;
}

//======================================  Set the debug level.
void
LockSegList::setDebug(int lvl) {
    mDebug = lvl;
}

//======================================  Sort the list by start-time
void
LockSegList::sort(void) {
    std::sort(mList.begin(), mList.end());
}

//======================================  Stuff a segment at the then
void
LockSegList::stuff(const LockSegment& seg) {
    stretch(1);
    mList.push_back(seg);
}

//======================================  Stuff unsorted segments at the end
void
LockSegList::stuff(const LockSegList& segl) {
    stretch(segl.size());
    mList.insert(mList.end(), segl.mList.begin(), segl.mList.end());
}
