/* -*- mode: c++; c-basic-offset: 3; -*- */
//////////////////////////////////////////////////////////////////////////
//  									//
//  ColumnCache								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include "events/ColumnCache.hh"
#include "events/Event.hh"
#include "events/Value.hh"
#include "events/LayoutInfo.hh"
#include "events/Factory.hh"


namespace events {

//______________________________________________________________________________
   ColumnCache& ColumnCache::operator= (const ColumnCache& cc)
   {
      if (this != &cc) {
         mValid = cc.mValid;
         mName = cc.mName;
         if (mColCache) delete mColCache;
         mColCache = cc.mColCache ? cc.mColCache->Copy() : 0;
         mColIsFixed = cc.mColIsFixed;
         mFixColOfs = cc.mFixColOfs;
         mFixColType = cc.mFixColType;
         mCache = cc.mCache;
         mCacheVers = cc.mCacheVers;
      }
      return *this;
   }

//______________________________________________________________________________
   bool ColumnCache::Get (const Event& event, Value& val) const
   {
      // must be valid!
      if (!mValid || !event.GetData()) {
         return false;
      }
      // Lookup data address and type
      ColumnType::data_ptr data;
      ColumnType::Enum type;
      if (!CacheLookup (const_cast<Event&>(event), data, type)) {
         return false; // column not part of this event
      }
      // if no additional column cache, this is it
      if (!mColCache) {
         return val.Read (type, data); // default value, if data == 0
      }
      // If not, this must be an event
      if (type != ColumnType::kEvent) {
         return false;
      }
      // Use the default event, if data is zero
      if (data) {
         return mColCache->Get (*(const Event*)data, val);
      }
      else {
         return mColCache->Get (Event::Default(), val);
      }
   }

//______________________________________________________________________________
   bool ColumnCache::Set (Event& event, const Value& val)
   {
      // must be valid!
      if (!mValid || !event.GetData()) {
         return false;
      }
      // Lookup data address and type
      ColumnType::data_ptr data;
      ColumnType::Enum type;
      if (!CacheLookup (event, data, type)) {
         return false; // column not part of this event
      }
      // Check if column exists, if not, expand event columns
      if (!data) {
         if (!event.Update()) {
            return false;
         }
         CacheLookup (event, data, type); // this does not fail!
      }
      // if no additional column cache, this is it
      if (!mColCache) {
         return val.Write (type, data);
      }
      // If not, this must be an event
      if (type != ColumnType::kEvent) {
         return false;
      }
      return mColCache->Set (*(Event*)data, val);
   }

//______________________________________________________________________________
   Event* ColumnCache::GetEvent (Event& event)
   {
      // must be valid!
      if (!mValid || !event.GetData()) {
         return 0;
      }
      // Lookup data address and type
      ColumnType::data_ptr data;
      ColumnType::Enum type;
      if (!CacheLookup (event, data, type)) {
         return 0; // column not part of this event
      }
      // Check if column exists; expand event columns, if column 
      // is of event type
      if ((data == 0) && (type == ColumnType::kEvent)) {
         if (!event.Update()) {
            return 0;
         }
         CacheLookup (event, data, type); // this does not fail!
      }
      // if no additional column cache, this is it
      if (!mColCache) {
         return (type == ColumnType::kEvent) ? (Event*)data: &event;
      }
      // If not, this must be an event
      if (type != ColumnType::kEvent) {
         return 0;
      }
      return mColCache->GetEvent (*(Event*)data);
   }

//______________________________________________________________________________
   const Event* ColumnCache::GetEvent (const Event& event) const
   {
      // must be valid!
      if (!mValid || !event.GetData()) {
         return 0;
      }
      // Lookup data address and type
      ColumnType::data_ptr data;
      ColumnType::Enum type;
      if (!CacheLookup (const_cast<Event&>(event), data, type)) {
         return 0; // column not part of this event
      }
      // if no additional column cache, this is it
      if (!mColCache) {
         return (type == ColumnType::kEvent) ?
            (data ? (const Event*)data : &Event::Default()): &event;
      }
      // If not, this must be an event
      if (type != ColumnType::kEvent) {
         return 0;
      }
      return mColCache->GetEvent (*(Event*)data);
   }

//______________________________________________________________________________
   void ColumnCache::SetName (const char* name)
   {
      // Initialize values
      mValid = true;
      mName = name;
      if (mColCache) delete mColCache;
      mColCache = 0;
      mColIsFixed = false;
      mFixColOfs = 0;
      mFixColType = ColumnType::kInvalid;
      CacheClear();
   
      std::string nameFirst,nameRem;
      LayoutInfo::Parse (mName.c_str(),nameFirst,nameRem);
      if ( nameFirst.empty()) {
         mValid = false;
         return;
      }
      else mName = nameFirst; // column name can not be empty
   
      if (!nameRem.empty()) {
         mColCache = new ColumnCache (nameRem.c_str());
      }
   
      // eliminate spaces
      std::string::size_type pos;
      while ((pos = mName.find_first_of (" \t\f\n\r\v")) != 
      std::string::npos) {
         mName.erase (pos, 1);
      }
   
      if (mColCache) {
         mValid = mColCache->IsValid();
      }
      if (mValid) CacheInit();
   }

//______________________________________________________________________________
   std::string ColumnCache::GetName() const
   {
      std::string name = mName;
      if (mColCache) {
         name += "." + mColCache->GetName();
      }
      return name;
   }

//______________________________________________________________________________
   void ColumnCache::Reset() const
   {
      CacheClear();
      if (mColCache) mColCache->Reset();
   }

//______________________________________________________________________________
   void ColumnCache::CacheClear() const
   {
      mCache.clear();
      // get new version number
      mCacheVers = Factory::Get().GetLayoutAddColVers();
   }

//______________________________________________________________________________
   void ColumnCache::CacheInit()
   {
      // check for fixed column
      const ColumnInfo* info = 
         Factory::Get().GetFixedColumn (mName.c_str());
      // setup a single cache line
      if (info) {
         mColIsFixed = true;
         mFixColOfs = info->GetOffset();
         mFixColType = info->GetType();
      }
   }

//______________________________________________________________________________
   bool ColumnCache::CacheLookup (Event& event, ColumnType::data_ptr& data, 
   ColumnType::Enum& type) const
   {
      // fixed column? do it directly!
      if (mColIsFixed) {
         data = ColumnType::Address (event.GetData(), mFixColOfs);;
         type = mFixColType;
         return true; 
      }
      // Make sure no new columns have been added to a layout
      if (mCacheVers != Factory::Get().GetLayoutAddColVers()) {
         CacheClear();
      }
      Type eventtype;
      if (!event.GetLayout().GetType (eventtype)) {
         return false;
      }
      //CacheLine& cline = mCache[eventtype];
      return mCache[eventtype].Lookup (data, type, event, mName);
   }

//______________________________________________________________________________
   bool ColumnCache::CacheLine::Lookup (ColumnType::data_ptr& data, 
   ColumnType::Enum& type, const Event& event, 
   const std::string& name) 
   {
      switch (mStatus) {
         // Event does not contain this column
         case kNotFound:
            {
               return false;
            }
         // Event does contain this column
         case kValid:
            {
               // check if event data contains the column!
               // return address, if yes, zero otherwise
               data = mColIndex < 
                  ((ColumnType::Common_t*)event.GetData())->mColumns ?
                  ColumnType::Address (event.GetData(), mOffset) : 0;
               type = mDataType;
               return true; 
            }
         // Cache line wasn't initialized yet: do it first
         case kNeedInit:
            {
               mStatus = kNotFound;
               const ColumnInfo* info = 
                  event.GetLayout().GetColumn (name.c_str());
               if (info) {
                  mOffset = info->GetOffset();
                  mColIndex = info->GetColumn();
                  mDataType = info->GetType();
                  mStatus = kValid;
               }
               else {
                  mStatus = kNotFound;
               }
               // now do the lookup again
               return Lookup (data, type, event, name);
            }
      }
      return false;
   }

//______________________________________________________________________________
   bool ColumnCache::CacheLine::operator== (const CacheLine& cl) const
   {
      return (mStatus == cl.mStatus) &&
         (mColIndex == cl.mColIndex) && 
         (mOffset == cl.mOffset) &&
         (mDataType == cl.mDataType);
   }

//______________________________________________________________________________
   bool ColumnCache::CacheLine::operator< (const CacheLine& cl) const
   {
      return (mStatus < cl.mStatus) || 
         (mStatus == cl.mStatus && (mColIndex < cl.mColIndex ||
         (mColIndex == cl.mColIndex && (mOffset < cl.mOffset ||
         (mOffset == cl.mOffset && mDataType < cl.mDataType)))));
   }

}
