#ifndef _LIGO_EVENTALGORITHM_H
#define _LIGO_EVENTALGORITHM_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Algorithm						*/
/*                                                         		*/
/* Module Description: Defines event algorithms				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Algorithm.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <iosfwd>
#include <functional>
#include <algorithm>
#include "events/Event.hh"
#include "events/Iterator.hh"
#include "events/Function.hh"
#include "events/FunctionPtr.hh"
#include "events/Condition.hh"
#include "events/ConditionPtr.hh"
#include "events/List.hh"
#include "events/Chain.hh"
#include "events/Set.hh"


namespace std {

// An efficient way to swap a layout
template<> 
   inline void swap<> (events::Layout& l1, events::Layout& l2) {
      l1.Swap (l2); }
// An efficient way to swap an event
template<> 
   inline void swap<> (events::Event& e1, events::Event& e2) {
      e1.Swap (e2); }

// An efficient way to swap an event list
template<>
   inline void swap<> (events::List& e1, events::List& e2) {
      e1.Swap (e2); }
// An efficient way to swap an event list pointer
template<> 
   inline void swap<> (events::ListPtr& e1, events::ListPtr& e2) {
      e1.Swap (e2); }

// An efficient way to swap an event chain
template<> 
   inline void swap<> (events::Chain& e1, events::Chain& e2) {
      e1.Swap (e2); }
// An efficient way to swap an event chain pointer
template<> 
   inline void swap<> (events::ChainPtr& e1, events::ChainPtr& e2) {
      e1.Swap (e2); }

// An efficient way to swap an event set
template<> 
   inline void swap<> (events::Set& e1, events::Set& e2) {
      e1.Swap (e2); }

#ifdef __CINT__
   inline void swap (events::Layout& l1, events::Layout& l2) {
      l1.Swap (l2); }
   inline void swap (events::Event& e1, events::Event& e2) {
      e1.Swap (e2); }
   inline void swap (events::List& e1, events::List& e2) {
      e1.Swap (e2); }
   inline void swap (events::Chain& e1, events::Chain& e2) {
      e1.Swap (e2); }
   inline void swap (events::Set& e1, events::Set& e2) {
      e1.Swap (e2); }
#endif
}


   class Histogram1;
   class Histogram2;
   class TSeries;


namespace events {


   class Column;

/** @name Modify events
    Algorithms to modify events and their column values.
    @memo Modify events
 ************************************************************************/
//@{

/** Set column to a new value. Loops through the list of events
    and calculates the expression for each event. The specified 
    column is the set to then new value. The column name can be
    either a string or a column class.

    Example:
    \begin{verbatim}
    // Calculate the significance from the amplitude and sigma
    Set s1 ("input.xml");
    SetColumn (s1.Begin(), s1.End(), 
              Column ("Significance"),
              Column ("Amplitude") / Column ("Sigma"));
    \end{verbatim}

    @memo Set column to a new value
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param column Name of column/Column object
    @param expression New column value
    @return void
 ************************************************************************/
   void SetColumn (const Iterator& sourceBegin, 
                  const Iterator& sourceEnd,
                  const Column& column,
                  const Function& expression);
   inline void SetColumn (const Iterator& sourceBegin, 
                     const Iterator& sourceEnd,
                     const Column& column,
                     const FunctionPtr& expression) {
      SetColumn (sourceBegin, sourceEnd, column, *expression); }

/** Set column to a new value. Loops through the list of events
    and calculates the expression for each event which fullfills
    the condition. The specified column is then set to the new value.
    The column name can be either a string or a column class.
   
    Example:
    \begin{verbatim}
    // Calculate the significance from the amplitude and sigma
    // for all burts events
    Set s1 ("input.xml");
    SetColumn (s1.Begin(), s1.End(), 
              Column ("Significance"),
              Column ("Amplitude") / Column ("Sigma"),
              Filter ("burst::*"));
    \end{verbatim}
   
    @memo Set column to a new value of selected events
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param column Name of column/Column object
    @param expression New column value
    @param cond Event condition
    @param window Time window for analysis
    @return void
   ************************************************************************/
   void SetColumn (const Iterator& sourceBegin, 
                  const Iterator& sourceEnd,
                  const Column& column,
                  const Function& expression,
                  const Condition& cond,
                  const TimeWindow& window = TimeWindow (1.));
   void SetColumn (const Iterator& sourceBegin,
                     const Iterator& sourceEnd,
                     const Column& column,
                     const FunctionPtr& expression,
                     const ConditionPtr& cond,
                     const TimeWindow& window = TimeWindow (1.));
   inline void SetColumn (const Iterator& sourceBegin, 
                     const Iterator& sourceEnd,
                     const Column& column,
                     const FunctionPtr& expression,
                     const ConditionPtr& cond,
                     const TimeWindow& window) {           
      SetColumn (sourceBegin, sourceEnd, column, 
                *expression, *cond, window); }
//@}

/** @name Events analysis
    Algorithms to analyze a series of events.
    @memo Events analysis
 ************************************************************************/
//@{

/** Selects the events from the specified iterator range and
    copies them to the destination when they fullfill the 
    event condition. The destination iterator in most cases is
    a back inserter.

    Example:
    \begin{verbatim}
    // Select all events in s1 which have an amplitiude
    // greater than 5 and add them to s2.
    Set s1 ("input.xml");
    Set s2;
    Select (s1.Begin(), s1.End(),
            back_inserter (s2), Column("Amplitude") > 5);
    \end{verbatim}

    @memo Select events
    @param sourceBegin Start iterator to pick events from
    @param sourceEnd End iterator to pick events from
    @param dest Output iterator to store events
    @param cond Event condition
    @param window Time window for analysis
    @return End iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Select (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd, 
                     DestIterator dest, 
                     const Condition& cond,
                     const TimeWindow& window = TimeWindow (1.));
template <class DestIterator>
   inline DestIterator Select (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd, 
                     DestIterator dest, 
                     const ConditionPtr& cond,
                     const TimeWindow& window = TimeWindow (1.)) {
      return Select (sourceBegin, sourceEnd, dest, *cond, window); }

/** Move events which fulfill the condition to the end of the list.
    Remove doesn't really remove events but rather moves them 
    to the end of the list. The function returns an iterator to the 
    first of the "removed" events. The caller MUST delete these 
    events after the call. 

    For example:
    \begin{verbatim}
    // Remove all non-burst events from the event set.
    Set s1 ("input.xml");
    Iterator rem = Remove (s1.Begin(), s1.End(),
                         !Filter ("burst:*"));
    s1.Erase (rem, s1.End());
    \end{verbatim}
    is a correct use of Remove.

    @memo Remove events
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param cond Event condition
    @param window Time window for analysis
    @return End iterator of stored events
 ************************************************************************/
   Iterator Remove (const Iterator& sourceBegin, 
                   const Iterator& sourceEnd, 
                   const Condition& cond,
                   const TimeWindow& window = TimeWindow (1.));
   Iterator Remove (const Iterator& sourceBegin,
                     const Iterator& sourceEnd,
                     const ConditionPtr& cond,
                     const TimeWindow& window = TimeWindow (1.));
   inline Iterator Remove (const Iterator& sourceBegin, 
                     const Iterator& sourceEnd, 
                     const ConditionPtr& cond,
                     const TimeWindow& window) {
      return Remove (sourceBegin, sourceEnd, *cond, window); }


/** Selects the events which fullfill the event coincidence 
    condition. This algorithm expects two window iterators.
    This algorithm is typically not used directly.

    @memo Coincidence analysis (window iterators)
    @param sourceBegin Start window iterator
    @param sourceEnd End window iterator
    @param dest Output iterator to store events
    @param cond Event condition
    @return Next iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstWindowIterator& beg, 
                     const ConstWindowIterator& end,
                     DestIterator dest, 
                     const Condition& cond);
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstWindowIterator& beg, 
                     const ConstWindowIterator& end,
                     DestIterator dest, 
                     const ConditionPtr& cond) {
      return Coincidence (beg, end, dest, *cond); }

/** Selects the events which fullfill the event coincidence 
    condition. The coincidence order is selected by the
    window iterator. This algorithm can therefore be used
    for triple or multiple conicidence analysis as well.

    Example:
    \begin{verbatim}
    // Select all coincidence events in s1 using a coincidence 
    // window of 100ms and add them to s2
    Set s1 ("input.xml");
    Set s2;
    Coincidence (s1.Begin(2), s1.End(2),
                 back_inserter (s2), 0.1);
    \end{verbatim}

    @memo Coincidence analysis (single event set, all order)
    @param sourceBegin Start iterator to pick events from
    @param sourceEnd End iterator to pick events from
    @param order Conicidence order
    @param dest Output iterator to store events
    @param window Time window for coincidence
    @param cond Event condition
    @return Next iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     int order,
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond = Condition::True());
template <class DestIterator>
   inline DestIterator Coincidence (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     int order,
                     DestIterator dest, 
                     const TimeWindow& window,
                     const ConditionPtr& cond) {
      return Coincidence (sourceBegin, sourceEnd, order, 
                         dest, window, *cond); }

/** Selects the events from two independent event sets which 
    fullfill a coincidence condition. The coincidence 
    order is two.

    Example:
    \begin{verbatim}
    // Select all events in s1 which are in coincidence with
    // an event from s2 using a coincidence window of 100ms,
    // and add them to s3
    Set s1 ("input1.xml");
    Set s2 ("input2.xml");
    Set s3;
    Coincidence (s1.Begin(), s1.End(),
                 s2.Begin(), s2.End(),
                 back_inserter (s3), 0.1);
    \end{verbatim}

    @memo Double coincidence analysis (two event sets)
    @param source1Begin Start iterator of first event set
    @param source1End End iterator of first event set
    @param source2Begin Start iterator of second event set
    @param source2End End iterator of second event set
    @param dest Output iterator to store events
    @param window Conicidence window in sec
    @param cond Event condition on coincident events
    @return Next iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond = Condition::True());
template <class DestIterator>
   inline DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const ConditionPtr& cond) {
      return Coincidence (source1Begin, source1End, source2Begin, 
                         source2End, dest, window, *cond); }

/** Selects the events from three independent event sets which 
    fullfill a triple coincidence condition. The coincidence 
    order is three.

    Example:
    \begin{verbatim}
    // Select all tripe coincidence events between sets s1, s2
    // and s3 using a coincidence window of 100ms, and add them 
    // to s4. Additionally, we ask that each event has an amplitude
    // greater than 10.
    Set s1 ("input1.xml");
    Set s2 ("input2.xml");
    Set s3 ("input3.xml");
    Set s4;
    Coincidence (s1.Begin(), s1.End(),
                 s2.Begin(), s2.End(),
                 s3.Begin(), s3.End(),
                 back_inserter (s4), 0.1, 
                 Column ("Amplitude[0]") > 10 &&
                 Column ("Amplitude[1]") > 10 &&
                 Column ("Amplitude[2]") > 10);
    \end{verbatim}

    @memo Triple coincidence analysis (three event sets)
    @param source1Begin Start iterator of first event set
    @param source1End End iterator of first event set
    @param source2Begin Start iterator of second event set
    @param source2End End iterator of second event set
    @param source3Begin Start iterator of third event set
    @param source3End End iterator of third event set
    @param dest Output iterator to store events
    @param window Conicidence window in sec
    @param cond Event condition on coincident events
    @return Next iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     const ConstIterator& source3Begin, 
                     const ConstIterator& source3End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond = Condition::True());
template <class DestIterator>
   inline DestIterator Coincidence (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     const ConstIterator& source2Begin, 
                     const ConstIterator& source2End, 
                     const ConstIterator& source3Begin, 
                     const ConstIterator& source3End, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const ConditionPtr& cond) {
      return Coincidence (source1Begin, source1End, source2Begin, 
                         source2End, source3Begin, source3End, 
                         dest, window, *cond); }

/** Selects the events from multiple independent event sets which 
    fullfill a coincidence condition. The coincidence 
    order is the size of the specified input list.

    Example:
    \begin{verbatim}
    // Select all events in s1 which are in coincidence with an event
    // from s2 through s5 using a coincidence window of 100ms,
    // and add them to s6
    Set s1 ("input1.xml");
    Set s2 ("input2.xml");
    Set s3 ("input3.xml");
    Set s4 ("input4.xml");
    Set s5 ("input5.xml");
    Set s6;
    WindowIterator::InputStateList state;
    state.push_back (WindowIterator::InputState (s1.Begin(), s1.End()));
    state.push_back (WindowIterator::InputState (s2.Begin(), s2.End()));
    state.push_back (WindowIterator::InputState (s3.Begin(), s3.End()));
    state.push_back (WindowIterator::InputState (s4.Begin(), s4.End()));
    state.push_back (WindowIterator::InputState (s5.Begin(), s5.End()));
    Coincidence (state, back_inserter (s6), 0.1);
    \end{verbatim}

    @memo Double coincidence analysis (multiple event sets)
    @param state List of event set ranges
    @param dest Output iterator to store events
    @param window Conicidence window in sec
    @param cond Event condition on coincident events
    @return Next iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Coincidence (
                     const BasicWindowIterator::InputStateList& state, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const Condition& cond = Condition::True());
template <class DestIterator>
   inline DestIterator Coincidence (
                     const BasicWindowIterator::InputStateList& state, 
                     DestIterator dest, 
                     const TimeWindow& window,
                     const ConditionPtr& cond) {
      return Coincidence (state, dest, window, *cond); }

/** Selects the events which fullfill the event coincidence 
    condition. The coincidence order is selected by the
    window iterator. 

    Example:
    \begin{verbatim}
    // Find all clusters which consist of at least 3 events
    // within a time window of 1s, and add them to s2.
    Set s1 ("input.xml");
    Set s2;
    SelectClusters (s1.Begin(), s1.End(),
                    back_inserter (s2), 3, 1.0);
    \end{verbatim}

    @memo Cluster analysis
    @param sourceBegin Start iterator to pick events from
    @param sourceEnd End iterator to pick events from
    @param dest Output iterator to store events
    @param threshold Minimum number of events to be considered a cluster
    @param window Conicidence window in sec
    @param cond Event condition
    @return End iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator SelectClusters (
                     const ConstIterator& source1Begin, 
                     const ConstIterator& source1End, 
                     DestIterator dest, 
                     int threshold,
                     const TimeWindow& window = TimeWindow (1.), 
                     const Condition& cond = Condition::True());
template <class DestIterator>
   inline DestIterator SelectClusters (
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd, 
                     DestIterator dest, 
                     int threshold,
                     const TimeWindow& window, 
                     const ConditionPtr& cond) {
      return SelectClusters (sourceBegin, sourceEnd, dest, threshold,
                           window, *cond); }


/** VetoGate treats sets as lists of intervals (defined by the columns
    "Time" and "Duration") and splits both imput lists into 2. The first
    (postfix p) contains events that do not overlap with any events from
    the other input list; the second (postfix f) contains events that 
    overlap with at least event from the other input list.
    
    Both input sets have to be time ordered.
    Note that the second input set also has to be disjoint, i.e. its 
    events must not overlap in time.


    @memo Algorithm looking for time coincidence between 2 sets 
          (template)
    @param beg1 start of the 1st input list
    @param end1 end of the 1st input list
    @param beg2 start of the 2nd input list
    @param end2 end of the 2nd input list
    @param out1p list 1 events that passed the gate (were not in 
                 coincidence with any list 2 events)
    @param out1f list 1 events that failed to pass the gate (were in 
                 coincidence with at least one list 2 event)
    @param out2p list 2 events that passed the gate (were not in 
                 coincidence with any list 1 events)
    @param out2f list 2 events that failed to pass the gate (were in 
                 coincidence with at least one list 1 event)
    @return True if all Duration columns were readable
 ************************************************************************/
template <class DestIterator1, class DestIterator2, 
class DestIterator3, class DestIterator4>
   bool VetoGate(const ConstIterator& beg1, const ConstIterator& end1,
                const ConstIterator& beg2, const ConstIterator& end2,
                DestIterator1 out1p, DestIterator2 out1f,
                DestIterator3 out2p, DestIterator4 out2f,
                Interval offset = 0.0);


/** VetoGate treats sets as lists of intervals (defined by the columns
    "Time" and "Duration") and splits both imput lists into 2. The first
    (postfix p) contains events that do not overlap with any events from
    the other input list; the second (postfix f) contains events that 
    overlap with at least event from the other input list.
    
    Both input sets have to be time ordered.
    Note that the second input set also has to be disjoint, i.e. its 
    events must not overlap in time.


    @memo Algorithm looking for time coincidence between 2 sets 
    @param in1 1st input Set
    @param in2 2st input Set (has to be disjoint)
    @param out1p list 1 events that passed the gate (were not in 
                 coincidence with any list 2 events)
    @param out1f list 1 events that failed to pass the gate (were in 
                 coincidence with at least one list 2 event)
    @param out2p list 2 events that passed the gate (were not in 
                 coincidence with any list 1 events)
    @param out2f list 2 events that failed to pass the gate (were in 
                 coincidence with at least one list 1 event)
    @return True if all Duration columns were readable
 ************************************************************************/
   bool VetoGate (Set& in1, Set& in2,
                 Set& out1p, Set& out1f, Set& out2p, Set& out2f,
                 Interval offset = 0.0);			   
//@}


/** @name 
    Check if set is disjoint (template)
    @memo Is Disjoint template
 ************************************************************************/
//@{

/** IsDisjoint checks if a set is disjoint, i.e. none of the triggers 
    overlap.

    @memo Check if set is disjoint (template)
    @param beg start of the set
    @param end end of the set    
    @return True if the set is disjount.
 ************************************************************************/
   bool IsDisjoint (const ConstIterator& beg, const ConstIterator& end);
//@}

/** @name 
    Check if set is disjoint
    @memo Is Disjoint
 ************************************************************************/
//@{

/** IsDisjoint checks if a set is disjoint, i.e. none of the triggers overlap.

    @memo Check if set is disjoint
    @param s Set
    @return True if s is disjount.
 ************************************************************************/
   bool IsDisjoint (events::Set& s);
//@}



/** @name Event ordering
    Algorithms for sorting events.
    @memo Event ordering
 ************************************************************************/
//@{

/** Check the time order of an event range.

    Example:
    \begin{verbatim}
    // Sort a set if it is not ordered.
    Set s1;
    // ... append some unordered events to s1
    if (!CheckOrder (s1.Begin(), s2.End()) {
       s1.Sort();
    }
    \end{verbatim}

    @memo Check events for proper time order
    @param begin First iterator to check
    @param end One beyond the last iterator to check
    @return True if ordered in ascending time order
 ************************************************************************/
   bool CheckOrder (const ConstIterator& begin, const ConstIterator& end);

// /** Sorts the events using the specified event function.
//     The function must return a real value or must be convertable
//     to a real value.
// 
//     Example:
//     \begin{verbatim}
//     // Sort all events of s with descending amplitude
//     Set s1 ("input.xml");
//     Sort (s1.Begin(), s1.End(), Column("Amplitude"), false);
//     \end{verbatim}
// 
//     @memo Sort events in place
//     @param sourceBegin Start iterator of event list
//     @param sourceEnd End iterator of event list
//     @param func Event function (Sort order)
//     @param ascending Sort order (true default)
//  ************************************************************************/
//    void Sort (Iterator sourceBegin, 
//    Iterator sourceEnd,
//    const Function& func, 
//    bool ascending = true);
//    inline void Sort (Iterator sourceBegin, 
//    Iterator sourceEnd,
//    const FunctionPtr& func, 
//    bool ascending = true) {
//       Sort (sourceBegin, sourceEnd, *func, ascending); }

/** Sorts the events using the specified event function.
    The function must return a real value or must be convertable
    to a real value.

    Example:
    \begin{verbatim}
    // Pick the 20 events with largest amplitude and put
    // them in descending order into s2.
    Set s1 ("input.xml");
    Set s2;
    Sort (s1.Begin(), s1.End(), back_inserter (s2),
          Column("Amplitude"), false, 20);
    \end{verbatim}

    @memo Sort events into a new set
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param dest Output iterator to store sorted events
    @param func Event function (Sort order)
    @param ascending Sort order (true default)
    @param n Maximum number of sorted events (0 = all)
    @return End iterator of stored events
 ************************************************************************/
template <class DestIterator>
   DestIterator Sort (const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     DestIterator dest, 
                     const Function& func, 
                     bool ascending = true, 
                     int n = 0);
template <class DestIterator>
   inline DestIterator Sort (const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     DestIterator dest, 
                     const FunctionPtr& func, 
                     bool ascending = true, 
                     int n = 0);
 
template <class DestIterator>
   inline DestIterator Sort (const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     DestIterator dest,
                     const FunctionPtr& func,
                     bool ascending,
                     int n) {
      return Sort (sourceBegin, sourceEnd, dest, *func, ascending, n); }


//@}
/** @name Histogramming and plotting
    Routines to histogram and plot events.
    @memo Histogramming and plotting
 ************************************************************************/
//@{

/** Make a one-dimensional histogram. The specified event function 
    is used to add events to the histogram which fullfill the
    event condition.

    Example:
    \begin{verbatim}
    // Histogram the amplitude of events from set s1 which are
    // within the time interval 677000000 to 678000000.
    Set s1 ("input.xml");
    Histogram1 h1 (20, 0, 100);
    MakeHistogram (h1, s1.Begin(), s1.End(), Column("Amplitude"),
                   Time(677000000) <= Column("Time") < Time(678000000));
    \end{verbatim}

    @memo 1D histogram generation
    @param hist 1D histogram
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param func Event function used for histogram axis
    @param cond Condition to select events
    @param window Time window for analysis
    @return Number of events added to histogram
 ************************************************************************/
   int MakeHistogram (Histogram1& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& func,
                     const Condition& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));

   int MakeHistogram (Histogram1& hist,
                     const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& func,
                     const ConditionPtr& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));
 
   inline int MakeHistogram (Histogram1& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& func,
                     const ConditionPtr& cond,
                     const TimeWindow& window) {
      return MakeHistogram (hist, sourceBegin, sourceEnd, 
                           *func, *cond, window); }

/** Make a two-dimensional histogram. The specified event functions 
    are used to add events to the histogram which fullfill the
    event condition.

    Example:
    \begin{verbatim}
    // Histogram the amplitude of events from set s1 which are
    // within the time interval 677000000 to 678000000 against
    // their time.
    Set s1 ("input.xml");
    Histogram2 h2 (100, 0, 100, 100, 0, 1000000);
    MakeHistogram (h2, s1.Begin(), s1.End(), 
                   Column("Time") - Time(677000000),
                   Column("Amplitude"),
                   Time(677000000) <= Column("Time") < Time(678000000));
    \end{verbatim}

    @memo 2D histogram generation
    @param hist 2D histogram
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param f1 Event function used for first histogram axis
    @param f2 Event function used for second histogram axis
    @param cond Condition to select events
    @param window Time window for analysis
    @return Number of events added to histogram
 ************************************************************************/
   int MakeHistogram (Histogram2& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& f1, 
                     const Function& f2, 
                     const Condition& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));

   int MakeHistogram (Histogram2& hist,
                     const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& f1,
                     const FunctionPtr& f2,
                     const ConditionPtr& cond,
                     const TimeWindow& window);
                                                                                
   int MakeHistogram (Histogram2& hist,
                     const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& f1,
                     const FunctionPtr& f2,
                     const ConditionPtr& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));
 
   inline int MakeHistogram (Histogram2& hist, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& f1, 
                     const FunctionPtr& f2, 
                     const ConditionPtr& cond,
                     const TimeWindow& window) {
      return MakeHistogram (hist, sourceBegin, sourceEnd, 
                           *f1, *f2, *cond, window); }

/** Make a time series. The number of events is plotted against time.
    Example:
    \begin{verbatim}
    // Make a time series of the event rate from 677000000 to 677003600.
    Set s1 ("input.xml");
    TSeries ts (Time (677000000, 0), Interval (1.0), 3600);
    MakeTimeSeries (ts, s1.Begin(), s1.End());
    \end{verbatim}

    @memo Time series generation
    @param ts Time series
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param integrated If true plots the integrated event rate
    @param cond Condition to select events
    @return Number of plotted events
 ************************************************************************/
   int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     bool integrated = false,
                     const Condition& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));
   int MakeTimeSeries (TSeries& ts,
                     const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     bool integrated,
                     const ConditionPtr& cond,
                     const TimeWindow& window = TimeWindow (1.));
   inline int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     bool integrated,
                     const ConditionPtr& cond,
                     const TimeWindow& window) {
      return MakeTimeSeries (ts, sourceBegin, sourceEnd, integrated,
                           *cond, window); }

/** Make a time series. The specified event function is plotted 
    against time.

    Example:
    \begin{verbatim}
    // Make a time series of the event amplitudes within the time 
    // interval 677000000 to 678000000.
    Set s1 ("input.xml");
    TSeries ts (Time (677000000), Interval (100), 10000);
    MakeTimeSeries (ts, s1.Begin(), s1.End(), 
                    Column("Amplitude"),
                    Time(677000000) <= Column("Time") < Time(678000000));
    \end{verbatim}

    @memo Time series generation
    @param ts Time series
    @param sourceBegin Start iterator of event list
    @param sourceEnd End iterator of event list
    @param func Event function used for Y axis
    @param cond Condition to select events
    @param window Time window for function/condition
    @return Number of plotted events
 ************************************************************************/
   int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const Function& func,
                     const Condition& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));
   int MakeTimeSeries (TSeries& ts,
                     const ConstIterator& sourceBegin,
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& func,
                     const ConditionPtr& cond = Condition::True(),
                     const TimeWindow& window = TimeWindow (1.));

   inline int MakeTimeSeries (TSeries& ts, 
                     const ConstIterator& sourceBegin, 
                     const ConstIterator& sourceEnd,
                     const FunctionPtr& func,
                     const ConditionPtr& cond,
                     const TimeWindow& window) {
      return MakeTimeSeries (ts, sourceBegin, sourceEnd, 
                           *func, *cond, window); }

//@}
/** @name Input and Output
    Routines to write and read events to and from streams, respectively.
    @memo Input and Output
 ************************************************************************/
//@{

/** Read events from a stream. The events are read from the stream
    and inserted starting at the position indicated by the
    specified itereator. A maximum of max events are read. If
    max is negative, events are read until a end-of-file marker
    is encounterd or the fail bit is set.

    Example:
    \begin{verbatim}
    // Read events from a file and add them to an event list.
    List l1;
    ifstream inp ("input.xml");
    Read (inp, back_inserter (l1));
    \end{verbatim}

    @memo Read events from a stream
    @param inp Input stream
    @param insert Position where to insert events (output iterator)
    @param max Maximum number of events to read
    @return Iterator to next insert position
 ************************************************************************/
template <class DestIterator>
   DestIterator Read (std::istream& inp, 
                     DestIterator insert, int max = -1);

/** Writes events to a stream. A maximum of max events are written. If
    max is negative, all events in the specified range are written.

    Example:
    \begin{verbatim}
    // Write events to a series of files; 1000 at a time
    Set s1 ("input.xml");
    Iterator next = s1.begin();
    for (int i = 0; next != s1.end(); ++i) {
       char filename[128];
       sprintf (filename, "%s%i.xml", "output", i);
       ofstream out (filename);
       if (!out) break;
       next = Write (out, next, s1.end(), 1000);
    }
    \end{verbatim}

    @memo Write events to a stream
    @param out Output stream
    @param begin Start iterator of event list
    @param end End iterator of event list
    @param max Maximum number of events to write
    @return Iterator to next event to be written
 ************************************************************************/
template <class SrcIterator> 
   SrcIterator Write (std::ostream& out, 
                     SrcIterator begin, SrcIterator end, 
                     int max = -1);

}
namespace std {

/** Output operator for an event list. All events form the list
    are written.
    @memo Write all events of a list to a stream
    @param out Output stream
    @param list Event list
    @return Output stream
 ************************************************************************/
   ostream& operator<< (ostream& out, const events::List& list);

/** Input operator for an event list. The events will be added to
    the list.
    @memo Read events from a stream to a list
    @param inp Input stream
    @param list Event list
    @return Input stream
 ************************************************************************/
   istream& operator>> (istream& inp, events::List& list);

/** Output operator for an event chain.  All events form the chain
    are written.
    @memo Write all events of a chain to a stream
    @param out Output stream
    @param chain Event chain
    @return Output stream
 ************************************************************************/
   ostream& operator<< (ostream& out, const events::Chain& chain);

/** Input operator for an event chain. This will add a new list
    to the chain.
    @memo Read events from a stream to a chain
    @param inp Input stream
    @param chain Event chain
    @return Input stream
 ************************************************************************/
   istream& operator>> (istream& inp, events::Chain& chain);

/** Output operator for an event set. All events form the set
    are written.
    @memo Write all events of a set to a stream
    @param out Output stream
    @param set Event set
    @return Output stream
 ************************************************************************/
   ostream& operator<< (ostream& out, const events::Set& set);

/** Input operator for an event set. This will add a new chain to
    the set.
    @memo Read events from a stream to a set
    @param inp Input stream
    @param set Event set
    @return Input stream
 ************************************************************************/
   istream& operator>> (istream& inp, events::Set& set);

//@}

}

// Template defintion
#ifdef __GNUG__
#define _NEED_EVENTALGORITHM_TEMPL
#include "Algorithm.cc"
#endif
#endif // _LIGO_EVENTALGORITHM_H
