/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Time.hh"
#include <string>

/**  The %frame_name class constructs a frame file path in the canonical 
  *  format, \em i.e. \c \<directory>/\<prefix>-\<gps>-\<dt>.\<ext>. In 
  *  constructing the frame path %frame_name substitutes any format sequences 
  *  (as described in the table below) in the directory name and supplies 
  *  tools to create any missing directories as appripriate.
  *
  *  The substitution sequences are as follows:
  *  <table>
  *  <tr>
  *    <th>Sequence</th><th>Meaning</th>
  *  </tr>
  *  <tr>
  *    <td>%[n]d</td>
  *    <td>Frame duration with maximum field length of \e n digits</td>
  *  </tr>
  *  <tr>
  *    <td>%[n]g</td>
  *    <td>\e n most significant digits of GPS time</td>
  *  </tr>
  *  <tr>
  *    <td>%[n]r</td>
  *    <td>GPS time with right-most \e n digits suppressed.</td>
  *  </tr>
  *  </table>
  *  \brief Frame file path generation.
  *  \author John Zweizig
  *  \version $Id$
  */
class frame_name {
public:
    /**  Construct an empty frame name instance. By default the extension is 
      *  ".gwf" the frame length is 1s and the maximum depth is 1.
      *  \brief Construct an empty frame name.
      */
    frame_name(void);

    /**  Construct a %frame_name instance and initialized all the name fields.
      *  \brief Construct a frame name.
      *  \param dir_pat Directory patter string.
      *  \param pfx_pat Frame type (prefix) pattern string.
      *  \param dt      Default frame duration.
      *  \param ext     Frame  file extension string.
      */
    frame_name(const std::string& dir_pat, const std::string& pfx_pat, int dt=1,
	       const std::string& ext=std::string("gwf"));

    /**  Get the directory name for the specified time. The directory name is
      *  constructed from the directory pattern string and the time argument.
      *  \brief Build a directory name.
      *  \return Directory name string.
      */
    std::string dir_name(const Time& t) const;

    /**  Fetch the directory pattern string.
      *  \brief Directory pattern string.
      *  \return Directory pattern string reference.
      */
    const std::string& directory(void) const;

    /**  Fetch the frame file extension string.
      *  \brief  Frame file extension string.
      *  \return  Frame file extension string reference.
      */
    const std::string& extension(void) const;

    /**  Complete file path for the specified time and frame length. If the 
      *  time is zero, the GPS and Dt fields are suppressed. If the extension
      *  string is empty the \c '.<ext>' field is suppressed.
      *  \brief Generate frame file path.
      *  \param t Gps time of frame start.
      *  \param dt Frame length in seconds
      *  \return Frame file path string.
      */
    std::string file_path(const Time& t, int dt=0) const;

    /**  Make the specified directory. Missing parent directories are will be 
      *  constructed to the specified depth. For example, if the immediate 
      *  parent  of the specified directory is missing and the depth is at 
      *  least 2, the missing parent will be created. All created directories
      *  will be assigned the file access mode specified by \a mode.
      *  \brief Construct a directory.
      *  \param dir   Directory path.
      *  \param depth Maximum number of directories to be created.
      *  \param mode  Access mode of the directory(ies) created.
      *  \return True if the specified directory exists on return.
      */
    bool make_dir(const std::string& dir, int depth=1, int mode=0775);

    /**  Move the specified file name (usually the generated temp name) to
      *  the assigned file path.
      *  \brief Rename from temporary file name.
      *  \param temp temporary foile path.
      *  \param t0   file path start time.
      *  \param dt   file path duration.
      *  \return True if the move was successful.
      */
    bool move_to_path(const std::string& file, const Time& t0, int dt=0);

    /**  Return the frame file prefix string.
      *  \brief Frame file prefix string.
      *  \return Constant reference to the frame file prefix string.
      */
    const std::string& prefix(void) const;

    /**  Set the directory path pattern The directory name(s) are constructed 
      *  by replacing escape sequences with values derived from the frame start
      *  time.
      *  \brief Set directory pattern.
      *  \param dir_pat Directory pattern
      */
    void set_directory(const std::string& dir_pat);

    /**  Set the maximum number of new directories to create.
      *  \brief directory creation limit.
      *  \param deep Maxumium directory count.
      */
    void set_depth(int deep);

    /**  Set the default frame file length.
      *  \brief Set default frame length.
      *  \param dt defaul frame file length in seconds.
      */
    void set_dt(int dt);

    /**  Set the extension string. The default value is "gwf" and should not be
      *  changed for frame files.
      *  \brief Set extension.
      *  \param ext Extension string.
      */ 
    void set_extension(const std::string& ext);

    /**  Set the frame file name prefix. The prefix includes both the 
      *  observatory identifier(s) and the frame type code.
      *  \brief Set frame file prefix.
      *  \param pfx Prefix string.
      */ 
    void set_prefix(const std::string& pfx);

    /**  Generate a temporary path for a frame file with the the specified 
      *  time and frame length. This path is a 
      *  \brief Generate temporary file path.
      *  \param t Gps time of frame start.
      *  \param dt Frame length in seconds
      *  \return Frame file path string.
      */
    std::string temp_path(const Time& t, int dt=0) const;

private:
    std::string _dir;
    std::string _prefx;
    std::string _ext;
    int         _dt;
    int         _deep;
    std::string _lastDir;
};

//======================================  Inline accessors
inline const std::string&
frame_name::directory(void) const {
    return _dir;
}

inline const std::string&
frame_name::extension(void) const {
    return _ext;
}

inline const std::string&
frame_name::prefix(void) const {
    return _prefx;
}
