/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DACCIN_HH
#define DACCIN_HH

#include "Time.hh"
#include "Interval.hh"
#include "FileList.hh"
#include "FSeries.hh"
#include <string>
#include <iosfwd>

#define DACC_ONLDEV "/online/"

//======================================  Lost of frameCPP headers.
#include "FramePointerTypes.hh"
#ifndef __CINT__
  #include "framecpp/IFrameStream.hh"
  #include "framecpp/FrRawData.hh"
  #include "framecpp/FrStatData.hh"
  #include "framecpp/FrameH.hh"
  typedef FrameCPP::FrRawData::const_firstAdc_iterator AdcDataIterator;
  typedef FrameCPP::FrameH::const_procData_iterator    ProcDataIterator;
  typedef FrameCPP::FrameH::const_simData_iterator     SimDataIterator;
#endif

namespace FrameCPP {
    namespace Common {
	class FrameBufferInterface;
    }
}

class DVector;
class FrStatDataRef;

/**  %DaccIn methods provide an input interface to frame files.
  *  @memo DMT frame input class.
  *  @author  John G. Zweizig
  *  @version 1.3; Last modified August 8, 2008
  */
class DaccIn {
public:
    //--------------------------------------  Constructors and destructors.
    /**  Default (Null) constructor.
      *  \brief Default constructor.
      */
    DaccIn(void);

#ifndef __CINT__
    /**  Construct using an already allocated frame reader.
      *  \brief Stram constructor
      *  \param reader Opened FrameCPP frame stream.
      */
    explicit DaccIn(FrameCPP::IFrameStream* reader);
#endif

    /**  Destructor.
      *  \brief Destructor.
      */
    virtual ~DaccIn(void);

    //---------------------------------------  Status manipulation
    /**  This parameter controls the fraction of the data read into 
      *  the shared memory to be copied to the data stream and what
      *  part of the shared memory resources can be held allocated the 
      *  stream. 'nbuf' == 0 indicates data is taken as requested.
      *  'nbuf' == -1 indicates that all data written to the shared 
      *  memory must be copied to the data stream. 'nbuf'==N indicates
      *  that up to N buffers (frames) of data should be held until the 
      *  stream is ready for them.
      *  @memo Set the number of online buffers to reserve.
      *  @param N Number of buffers to reserve, or -1 for all.
      */
    void setBuffer(int N);

    /**  If the debug flag is set, explanatory messages will be printed 
      *  when an error is encountered.
      *  \brief Set the debug flag.
      *  \param debug New debug flag setting.
      */
    void setDebug(int debug=0);

    /**  Set the table-of-contents read mode. If the TOC flag is set, the
      *  table of contents will be read when each frame file is opened.
      *  @memo Set the readmode.
      *  \param toc New TOC read setting.
      */
    void setTOCMode(bool toc=true);

    //--------------------------------------  File list manipulation
    /**  One or more file names are added to the file list. Wilcard
      *  characters ('*'s) can be used to add several files simultaneously.
      *  The file names are sorted to keep them in an increaseing sequence.
      *  @memo Add a file to the file list.
      *  @param File Name of the file to be added to the input list.
      */
    void addFile(const std::string& File);

    /**  One or more frame file descriptors are read from a text file and 
      *  added to the list of frames to be processed. Each line of the 
      *  text file must contain a single file specifier like the those 
      *  accepted by addFile().
      *  @memo Add a file to the file list.
      *  @param File Name of the file to be added to the input list.
      */
    void addFileList(const std::string& File);

    /**  The current file is closed and its name is removed from the 
      *  file list. Any open shared memory partition is released.
      *  @memo Close the current file.
      */
    virtual void close(void);

    /**  The first file in the file list is opened. If a file is already 
      *  open, it will be closed, and the next file will be opened. If the 
      *  file name is of the form "/online/<partname>" the input stream is 
      *  connected to the the shared memory partition specified by \<partname\>.
      *  @memo Open a file.
      *  @return Zero if open was sucessful.
      */
    virtual int open(void);

    //----------------------------------  Data processing.
    /**  The current data frame is released. In Online data mode, the 
      *  shared memory partition buffer containing the frame is also 
      *  released. No new frame is read.
      *  @memo Release the current data frame.
      */
    void endFrame(void);

    /**  The current data frame is deleted and a new frame is read. 
      *  If the input file is not open or if an error occurs on reading
      *  the current file, the file is close and a new file is opened.
      *  @memo Read in a new data frame.
      *  @return Zero if valid frame read in, otherwise -1.
      */
    int nextFrame(void);

    //---------------------------------  Accessors
    /**  Returns the current frame time interval;
      *  @memo Get the Current time.
      *  \return Length of the current frame.
      */
    Interval getDt(void) const;

    /**  Returns the level of debug information that was requested.
      *  @memo Get the debug flag.
      *  @return Debug print level.
      */
    int getDebug(void) const {return mDebug;}

    /**  Returns a pointer to a string containing the last input file 
      *  path opened. If an online frame is used, the path is connstructed
      *  from the partition name and the gps time.
      *  \brief Get the input file name.
      *  \return Current frame file path.
      */
    const char* getFile(void) const;

    /**  Get the time and comment of the specified history record.
      *  @memo Get history data.
      *  @param name History record name.
      *  @param t    History record time (returned);
      *  @param com  History record comment.
      *  @return true if history found.
      */
    bool getHistory(const std::string& name, Time& t, std::string& com) const;

#ifndef __CINT__
    /**  Returns a pointer to the current frame header.
      *  \brief  Get a pointer to the current frame header.
      *  \return Pointer to the in-memory frame header.
      */
    frameh_pointer getFrame(void) const {return mFrame;}
#endif

    /**  The current frame path is returned. For online data (\e i.e. data
      *  from a shared memory partition), an identifier of the form 
      *  \c \<ifo-id>-\<partition>-\<GPS-time>-\<dt>.gwf is generated.
      *  @memo Copy a Frame ID into the specified string.
      *  @return string containing the current frame identifier.
      */
    std::string getFrameID(void) const;

    /**  Fetch all specified static data.
      *  @memo Get a static data.
      *  @param name FrStat name.
      *  @param det  Detector name.
      *  @param startGps  Start time of static validity.
      *  @param endGps End time of static validity.
      *  @return Number of static structures read.
      */
    int getStaticData(const std::string& name=".*", 
		      const std::string& det=".*", 
		      const Time& startGps=Time(0), 
		      const Time& endGps=Time(0));
  
    /**  Get a frequency series (FSeries) from a previously read static 
      *  structure. Note that getStaticData must have been used previously 
      *  to insure that the necessary static data reside in memory.
      *  @memo Get a static fseries.
      *  @param name FrStat name.
      *  @param gps  Time for of static validity.
      *  @return FSeries.
      */
    FSeries getStaticFSeries(const std::string& name, const Time& gps);
  
    /**  Get a time series (TSeries) from a previously read static structure.
      *  Note that getStaticData must have been used previously to insure that
      *  the necessary static data reside in memory.
      *  @memo Get a static TSeries.
      *  @param name FrStat name.
      *  @param gps  Time for of static validity.
      *  @return TSeries.
      */
    TSeries getStaticTSeries(const std::string& name, const Time& gps);

    /**  Returns the start time of the current (or previous) frame. If 
      *  the Dacc has no currently valid frame, getTime returns the start 
      *  time of the last frame read. If no frames have been read, getTime 
      *  returns Time(0). 
      *  @memo Get start time of the last frame read.
      *  @return start time of current or previous frame.
      */
    Time getTime(void) const {return mT0;}

    /**  Get the number of frames read.
      *  \brief Number of frames in read.
      *  \return Number of frames.
      */
    long getTotalFrames(void) const {return mTotalFrames;}

    /**  Test whether a frame header has been read in.
      *  \brief Test for frame.
      *  \return True if frame header is available.
      */
    bool haveFrame(void) const {return ((mFrame) ? true : false );}

    /**  Test whether the data are being streamed (\e e.g. from an online 
      *  shared memory partition).
      *  @memo Test if online data.
      *  @return True if %DaccIn input source provides online streaming data.
      */
    virtual bool isOnline(void) const {return (mSource==s_online);}

    /**  Test whether a frame file has been opened successfully.
      *  \brief Test if frame file is open.
      *  \return True if frame file ois open.
      */
    bool isOpen(void) const {return mReader;}

    /**  Write a list of input files to the specified STL output stream.
      *  \brief Write frame file list.
      *  \param out STL output stream to receive list
      *  \return Output stream reference.
      */
    std::ostream& listFiles(std::ostream& out) const;

    /**  Return a reference to the list of remaining frame files for this 
      *  %DaccIn instance. Files are processesed in the listed order and are
      *  removed after the frame file is opened.
      *  \brief Reference frame file list.
      *  \return Reference to the frame file list for this %DaccIn instance.
      */
    FileList& refList(void);

    /**  Return a constant reference to the list of remaining frame files for
      *  this %DaccIn instance. Files are processesed in the listed order and 
      *  are removed after the frame file is opened.
      *  \brief Reference frame file list.
      *  \return Constant reference to the frame file list for this %DaccIn 
      *  instance.
      */
    const FileList& refList(void) const;

    /**  Wait for data or a signal. waitData returns when a data buffer is
      *  found in the online partition or when a signal is caught. If the
      *  data are available, the return value is true. If \a test_only is 
      *  true waitData() returns immediately.
      *  @memo Wait for data or a signal.
      *  @param test_only Return immediately if true.
      *  @return true if data are availble.
      */
    bool waitData(bool test_only=false) const;

#ifndef __CINT__

    /**  Find an Adc data structure with the given channel name and
      *  return a pointer to the object. If the object isn't found,
      *  a null pointer is returned.
      *  \brief Find an FrAdcData structure.
      *  \param name Channel name
      *  \return Constant pointer to the found FrAdcData structure or NULL.
      */
    fradcdata_pointer findAdc(const std::string& name) const;

    /**  Find an Adc data structure with the given channel name and
      *  return a pointer to the object. An internal pointer is set to the
      *  position of the requested structure. findAdcOrder returns zero if
      *  the requested channel was not found or 1 (-1) if the structure was 
      *  found after (before) the previous requested structure.
      *  \brief Find an FrAdcData structure and return its relative position.
      *  \param name Channel name
      *  \param p    Pointer to requested structure.
      *  \return Relative position of the structure or 0 if not found.
      */
    int findAdcOrdered(const std::string& name, fradcdata_pointer& p) const;

    /**  Find a Processed Data structure with the given channel name and
      *  return a pointer to the object. If the object isn't found,
      *  a null pointer is returned.
      *  \brief Find an FrProcData structure.
      *  \param name Channel name
      *  \return Constant pointer to the found FrProcData structure or NULL.
      */
    frprocdata_pointer findProc(const std::string& name) const;

    /**  Find an FrProc data structure with the given channel name and
      *  return a pointer to the object. An internal pointer is set to the
      *  position of the requested structure. findProcOrder returns zero if
      *  the requested channel was not found or 1 (-1) if the structure was 
      *  found after (before) the previous requested structure.
      *  \brief Find an FrAdcData structure and return its relative position.
      *  \param name Channel name
      *  \param p    Pointer to requested structure.
      *  \return Relative position of the structure or 0 if not found.
      */
    int findProcOrdered(const std::string& name, frprocdata_pointer& p) const;

    /**  Find a Simulated Data structure with the given channel name and
      *  return a pointer to the object. If the object isn't found,
      *  a null pointer is returned.
      *  \brief Find an FrSimData structure.
      *  \param name Channel name
      *  \return Constant pointer to the found FrSimData structure or NULL.
      */
    frsimdata_pointer findSim(const std::string& name) const;

    /**  Find an FrSimData structure with the given channel name and
      *  return a pointer to the object. An internal pointer is set to the
      *  position of the requested structure. findSimOrder returns zero if
      *  the requested channel was not found or 1 (-1) if the structure was 
      *  found after (before) the previous requested structure.
      *  \brief Find an FrSimData structure and return its relative position.
      *  \param name Channel name
      *  \param p    Pointer to requested structure.
      *  \return Relative position of the structure or 0 if not found.
      */
    int findSimOrdered(const std::string& name, frsimdata_pointer& p) const;

    /**  Find a Static Data structure with the given channel name and
      *  valid at the specified time and version. If the versionis negative,
      *  the latest version number is returned. A valid FrStatDataRef is 
      *  returned if found. If the object isn't found, the FrStatDataRef is  
      *  a null reference (i.e. FrStatDataRef::null() returns true).
      *  \brief Find FrStat structure.
      *  \param name Name of static data to be found
      *  \param gps  Time of validity
      *  \param vsn  Version number
      *  \return Reference to an FrStat structure.
      */
  FrStatDataRef findStat(const std::string& name, const Time& gps, int vsn=-1);
#endif

  private:
    int closeFile(void);
    int closeReader(void);
    int deleteFrame(void);
    int flushFile(void);
    int openFile(const std::string& Source);
    int openReader(void);
    int readFrame(void);

#ifndef __CINT__
    fradcdata_pointer  readAdc(const std::string& name) const;
    frprocdata_pointer readProc(const std::string& name) const;
    frsimdata_pointer  readSim(const std::string& name) const;
#endif

  private:
    FileList mFile;

    /**  Debug mode flag.
      */
    int mDebug;

    /**  Time of current (Next) Frame.
      *  @memo Time of current (Next) Frame.
      */
    Time mT0;

    /** Frame count in this file
      */
    unsigned int mFramesInFile;

    /** Frame number
      */
    unsigned int mFrameNo;

    /** Frame number
      */
    enum {s_noTOC, s_TOC}  mTOCMode;

    /**  Current data source.
      *  @memo Current data source.
      */
    enum {s_none, s_online, s_file} mSource;

#if LDAS_VERSION_NUMBER < 109360
    /**  Pointer to input stream buffer object.
      *  @memo Pointer to input stream buffer object.
      */
    std::streambuf* mSbuf;

    /**  Pointer to input stream buffer object.
      *  @memo Pointer to input stream buffer object.
      */
    std::istream* mStream;
#else
    /**  Pointer to input frame buffer object.
      *  @memo Pointer to input frame buffer object.
      */
    FrameCPP::Common::FrameBufferInterface* mFrameBuf;
#endif

#ifndef __CINT__
    /**  Pointer to current frame reader.
      *  @memo Pointer to current frame reader.
      */
    FrameCPP::IFrameStream* mReader;

    /**  Pointer to current frame. If no frame hase been read, mFrame is
      *  set to NULL.
      *  @memo Pointer to current frame.
      */
    frameh_pointer mFrame;

    mutable AdcDataIterator  mAdcIter;
    mutable ProcDataIterator mProcIter;
    mutable SimDataIterator  mSimIter;

    FrameCPP::Common::FrStatData::Query mStatData;
#endif

    /**  File name of the last frame successfully opened.
      */
    std::string mLastFrame;

    long mTotalFrames;
};

//======================================  Inline methods
#ifndef __CINT__
inline void
DaccIn::addFile(const std::string& name) {
    return mFile.addFile(name);
}

inline void
DaccIn::addFileList(const std::string& name) {
    return mFile.addList(name);
}

inline const char*
DaccIn::getFile(void) const {
    return mLastFrame.c_str();
}

inline std::ostream& 
DaccIn::listFiles(std::ostream& out) const {
    return mFile.print(out);
}

inline FileList& 
DaccIn::refList(void) {
    return mFile;
}

inline const FileList& 
DaccIn::refList(void) const {
    return mFile;
}
#endif // !defined(__CINT__)

#endif // DACCIN_HH
