/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    Implementation of Data accessor channel class.
//
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif

#include "Channel.hh"
#include "DVecType.hh"
#include "framecpp/FrameH.hh"
#include "framecpp/FrRawData.hh"
#include "framecpp/FrVect.hh"
#include "FrVectRef.hh"
#include "ldas/ldasconfig.hh"
#if LDAS_VERSION_NUMBER < 200000
#include "general/gpstime.hh"
using General::GPSTime;
#else
#include "framecpp/GPSTime.hh"
using FrameCPP::GPSTime;
#endif

#include "TSeries.hh"
#include "FSeries.hh"
#include <iostream>

using FrameCPP::FrAdcData;
using FrameCPP::FrProcData;
using FrameCPP::FrRawData;
using FrameCPP::FrSimData;
using FrameCPP::FrVect;

using namespace std;

//======================================  Translate from FrameCPP::GPSTime
inline Time
GetGPSTime(const GPSTime& T) {
    return Time(T.GetSeconds(), T.GetNanoseconds());
}

//======================================  Default constructor
Channel::Channel(void) 
  : mDebug(false), mSample(0.0), mLast(0), mNSample(0), mAccVal(0.0),
    mUserPtr(0), mFUserPtr(0), mType(kUnknown), mChanFlags(0)
{
    myPtr.tsp = 0;
}

//======================================  Define a channel
Channel::Channel(const std::string& name, TSeries** ctlv, uint_type decim, 
		 ChanType t) 
  : mDebug(false), mSample(0.0), mLast(0), mNSample(0), mAccVal(0.0),
    mFUserPtr(0), mType(t), mChanFlags(0)
{
    myPtr.tsp = 0;

    //----------------------------------  Copy the name string.
    mName = name;

    //----------------------------------  Set the decimation factor;
    mDecim8 = decim;
    if (decim == 0) mDecim8 = 1;

    //----------------------------------  Set up the user pointer.
    mUserPtr = ctlv;
    if (!mUserPtr) mUserPtr = &myPtr.tsp;
}

//======================================  Define a channel
Channel::Channel(const std::string& name, FSeries** ctlv) 
  : mName(name), mDecim8(0), mDebug(false), mSample(0.0), mLast(0), 
    mNSample(0), mAccVal(0.0), mUserPtr(0), mType(kFSeries), mChanFlags(0)
{
    myPtr.fsp = 0;

    //----------------------------------  Set up the user pointer.
    mFUserPtr = ctlv;
    if (!mFUserPtr) mFUserPtr = &myPtr.fsp;
}

//======================================  Define a channel
Channel::Channel(const Channel& d) 
  : mUserPtr(0), mFUserPtr(0)
{
    myPtr.tsp = 0;
    *this = d;
}

//======================================  Define a channel
Channel&
Channel::operator=(const Channel& d) 
{
    mDebug   = d.mDebug;
    mSample  = d.mSample;
    mLast    = d.mLast;
    mNSample = d.mNSample;
    mAccVal  = d.mAccVal;
    mType    = d.mType;
    mName    = d.mName;
    mDecim8  = d.mDecim8;
    deleteSeries();
    if (d.mUserPtr != &d.myPtr.tsp) {
        mUserPtr = d.mUserPtr;
    } else {
	if (d.myPtr.tsp) myPtr.tsp = new TSeries(*d.myPtr.tsp);
        mUserPtr = &myPtr.tsp;
    }
    if (d.mFUserPtr != &d.myPtr.fsp) {
        mFUserPtr = d.mFUserPtr;
    } else {
        if (d.myPtr.fsp) myPtr.fsp = new FSeries(*d.myPtr.fsp);
        mFUserPtr = &myPtr.fsp;
    }
    mChanFlags = d.mChanFlags;
    return *this;
}

//======================================  Destroy a channel entry.
Channel::~Channel() {
    deleteSeries();
}

//======================================  Delete locally manageed series
void 
Channel::deleteSeries(void) {
    if (mUserPtr == &myPtr.tsp && myPtr.tsp) {
        delete myPtr.tsp;
    } else if (mFUserPtr == &myPtr.fsp && myPtr.fsp) {
        delete myPtr.fsp;
    }
    myPtr.tsp = 0;
}

//======================================  Allocate the time or frequency series.
void 
Channel::allocate(Interval Stride) {

    //----------------------------------  Allocate frequency-series storage
    if (mType == kFSeries ) {
	FSeries* uPtr = *mFUserPtr;
	if (uPtr) {
	    uPtr->clear();

	//------------------------------  DVec type unknown for direct copies
	} else {
	    uPtr = new FSeries();
	    uPtr->setName(mName.c_str());
	    *mFUserPtr = uPtr;
	}
    } 

    //----------------------------------  Allocate time-series storage
    else {

	//------------------------------  Calculate the number of words
        size_type nw = 0;
	if (double(mSample) > 0 && double(Stride) > 0) {
	    nw = size_type(Stride/mSample + mDecim8 - 0.5) / mDecim8;
	}

	//------------------------------  Clear and Resize TSeries if it exists
	TSeries* uPtr = *mUserPtr;
	if (uPtr) {
	    uPtr->Clear();
	    reserve(Stride);
	}

	//------------------------------  DVec type unknown 
	else {
	    uPtr = new TSeries(mLast, mSample);
	    uPtr->setName(mName.c_str());
	}
	*mUserPtr = uPtr;
    }

    mLast    = Time(0);
    mAccVal  = 0.0;
}

//======================================  Reserve space in the target series.
void 
Channel::reserve(Interval dT) {
    if (mType != kFSeries && double(mSample) > 0 && double(dT) > 0) {
        size_type nw = size_type(dT/mSample + mDecim8 - 0.5) / mDecim8;
	if (*mUserPtr) {
	    DVector* dv = (*mUserPtr)->refDVect();
	    if (dv) dv->reserve(nw);
	}
    }
}

//======================================  Set a flag state
void 
Channel::setFlag(ChanFlags flag, bool state) {
    uint_type bit = 1 << int(flag);
    if (state) mChanFlags |=  bit;
    else       mChanFlags &= ~bit;
}

//======================================  Fill frame data into the series.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  TSeries is not allocated.
//	 -7  Unsupported data type
//       -9  Invalid data (NaN, Inf or Unnormalized).
//
int
Channel::FillSeries(frameh_pointer frame, Interval off, Interval dT)
{
    Time t0(GetGPSTime(frame->GetGTime()));

    frrawdata_pointer  raw  = frame->GetRawData();
    fradcdata_pointer  adc;
    frprocdata_pointer proc;
    frsimdata_pointer  sim;

    //----------------------------------  Get the adc data.
    switch (mType) {
    case kRaw:
	if (!raw) {
	    if (mDebug) cout << "No raw data in frame." << endl;
	} else {
	    adc = *(raw->RefFirstAdc().find(mName, raw->RefFirstAdc().begin()));
	    if (adc) return FillSeries(adc, t0, off, dT);
	    if (mDebug) cout << "ADC " << mName << " not in frame." << endl;
	}
	break;

    //----------------------------------  Get the processed data.
    case kProcessed:
    case kFSeries:
	proc = *(frame->RefProcData().find(mName, frame->RefProcData().begin()));
	if (proc) return FillSeries(proc, t0, off, dT);
	if (mDebug) cout << "ProcData " << mName << " not in frame." << endl;
	break;

    //----------------------------------  Get the simulated data.
    case kSimulated:
	sim = *(frame->RefSimData().find(mName, frame->RefSimData().begin()));
	if (sim) return FillSeries(sim, t0, off, dT);
	if (mDebug) cout << "SimData " << mName << " not in frame." << endl;
	break;

    //----------------------------------  Look everywhere
    case kUnknown:
	if (raw) {
	    adc = *(raw->RefFirstAdc().find(mName, raw->RefFirstAdc().begin()));
	    if (adc) return FillSeries(adc, t0, off, dT);
	}
	proc = *(frame->RefProcData().find(mName, frame->RefProcData().begin()));
	if (proc) return FillSeries(proc, t0, off, dT);
	if (mDebug) cout << "Channel " << mName << " not in frame." << endl;
    }
    return -3;
}

//======================================  Fill adc data into the series.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  TSeries is not allocated.
//	 -7  Unsupported data type
//       -9  Invalid data (NaN, Inf or Unnormalized).
//
int
Channel::FillSeries(fradcdata_pointer adc, const Time& t0, 
		    Interval off, Interval dT)
{
    int rc = 0;

    //----------------------------------  Get data  pointers.
    if (adc->RefData().empty()) return -3;
    FrVectRef vect(adc->RefData().front());
    if (vect.getUnits() != adc->GetUnits()) {
	cerr << "Mismatched units in FrAdcData: " << adc->GetName() << endl;
    }

    //----------------------------------  Set the sample rate.
    Interval tSample(1./adc->GetSampleRate());

    //----------------------------------  Minute trend fix: Ugly Ugly Ugly
    //
    //   Most of the LIGO minute trend data have FrAdcData:sample == 1
    //   when it should be 1/60. Check for this case specifically and
    //   use the correct sample time if it's broken.
    //
    if (tSample == Interval(1.0) && vect.getDimDx(0) == 60.0) {
	rc = setSample(vect);
    } else {
    //   Other frames have Dx == 1 do use 1/sample rate by default.
	rc = setSample(tSample);
    }
    if (rc) return rc;

    int fix = 0;
    //----------------------------------  Fix S2 word order screwup
    //
    //  From Jan 6, 2003 through July 29, 2003 pairs of 16-bit words 
    //  were swapped in the raw data frames in channels read out by 
    //  the H1 suspension and pem ADCUs. These channels have channel
    //  IDs in the range 7805 - 7956 (inclusive) for the S2 runs. 
    //  The following code sets fix=1 if the data was taken during 
    //  S2 and the channel number is in the appropriate range.
    //
    if (t0 < Time(735000000) && t0 >= Time(729000000)) {
        int ichan = adc->GetChannelNumber();
	if (ichan >= 7805 && ichan < 7957) fix = 1;
    }

    //----------------------------------  Fill the series from the FrVect
    rc = FillSeries(vect, t0, off, dT, fix);
    if (!rc) {
        mType = kRaw;
	(*mUserPtr)->combineStatus(adc->GetDataValid() ? 1 : 0);
	(*mUserPtr)->setF0(adc->GetFShift());
	(*mUserPtr)->setUnits(adc->GetUnits());
    }
    return rc;
}

//======================================  Fill processed data into series.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  TSeries is not allocated.
//	 -7  Unsupported data type
//       -9  Invalid data (NaN, Inf or Unnormalized).
//
int
Channel::FillSeries(frprocdata_pointer proc, const Time& t0, 
		    Interval off, Interval dT)
{
    if (proc->RefData().empty()) return -3;
    FrVectRef vect = proc->RefData().front();
    Interval  pdoff = proc->GetTimeOffset();

    //-----------------------------------  Unpack a t-series channel.
    int rc(0);
    bool type_error(true);
    switch (proc->GetType()) {
    case FrProcData::UNKNOWN_TYPE:
	//  Unknown types are treated as TIME_SERIES until the Virgo h(t)
	//  has a valid type code.
    case FrProcData::TIME_SERIES:
	if (mType != kFSeries) {
	    rc = setSample(vect);
	    if (!rc) rc = FillSeries(vect, t0+pdoff, off-pdoff, dT, 0);
	    if (!rc) mType = kProcessed;
	    type_error = false;
	}
	break;
    case FrProcData::FREQUENCY_SERIES:
	if (mType == kFSeries) {
	    rc = FillFSeries(vect, t0+pdoff, proc->GetTRange());
	    if (!rc) mLast = t0 + off + dT;
	    type_error = false;
	}
	break;
    default:
	break;
    }

    //----------------------------------  Series type errors.
    if (type_error) {
	rc = -7;
	if (getDebug()) {
	    cout << "FrProcData channel " << getName() 
		 << " wrong series type." << endl;
	}
    }
    return rc;
}

//======================================  Fill sim data into the series.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  TSeries is not allocated.
//	 -7  Unsupported data type
//       -9  Invalid data (NaN, Inf or Unnormalized).
//
int
Channel::FillSeries(frsimdata_pointer sim, const Time& t0, 
		    Interval off, Interval dT)
{
    if (sim->RefData().empty()) return -3;
    FrVectRef vect = sim->RefData().front();
    int rc = setSample(vect);
    if (rc) return rc;
    rc = FillSeries( vect, t0, off, dT, 0);
    if (!rc) mType = kSimulated;
    return rc;
}

//====================================== Set sample from the FrVec dimension
int
Channel::setSample(const FrVectRef& vect) {
    if (vect.empty()) return -3;
    return setSample(Interval(vect.getDimDx(0)));
}

//====================================== Set the sample interval
int 
Channel::setSample(Interval dT) {
    if (dT <= Interval(0.0)) {
        if (getDebug()) cout << "Channel " << mName 
			     << " sample rate not valid." << endl;
	return -5;
    } else if (!mSample) {
        mSample = dT;
    } else if (mSample != dT) {
        if (mDebug) cout << "Channel " << mName << " sample rate changed from "
			 << mSample << " to " << dT << endl;
        return -2;
    }
    return 0;
}

//======================================  Fill FrVect into the series.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  TSeries is not allocated.
//	 -7  Unsupported data type
//       -9  Non-normal data
//
int
Channel::FillSeries(const FrVectRef& vect, const Time& t0, 
		    Interval off, Interval dT, int fix)
{
    if (fix == 1) {
        cerr << "Requested fix not implemented" << endl;
	throw std::runtime_error("Fix not implemented");
    }

    //-----------------------------------  Check arguments.
    if (!*mUserPtr) {
        if (getDebug()) cout << "TSeries is not allocated." << endl;
	return -6;
    }
    if (vect.empty()) return -3;

    //----------------------------------  Check the start time.
    size_type   first = vect.getIndex(0, off);
    size_type    last = vect.getIndex(0, off+dT);
    Time       tStart = t0 + vect.getDimX0(0) + double(first)*mSample;
    Interval reSample = mSample;

    //----------------------------------  Get raw FrVect data
    DVector* dv = vect.getDVector(first, last);
    int      rc = dv ? 0: -6;

    //----------------------------------  Check that data are valid.
    if (!rc && 
	!testFlag(kAllowNonNormal) && 
	!dv->normal() && 
	!getenv("DMT_IGNORE_NAN")
       ) {
	delete dv;
	rc = -9;
    }

    //----------------------------------  Perform optional decimation
    if (mDecim8 != 1 && !rc) {
	DVecType<double> dvd(*dv);
	delete dv;
	dv = 0;

	tStart  -= mSample * double(mNSample);
        reSample = mSample * double(mDecim8);

	size_type inx=0;
	size_type len=dvd.getLength();
        for (size_type i=0; i<len; ++i) {
	    mAccVal += dvd[i];
	    if (++mNSample >= mDecim8) {
	        dvd[inx++] = mAccVal / double(mNSample);
		mAccVal  = 0;
		mNSample = 0;
	    }
	}

	if (inx) dv = dvd.Extract(inx);
    }

    //----------------------------------  Append the (resampled) data vector
    if (!rc) {
        TSeries ts(tStart, reSample, dv);
	ts.setUnits(vect.getUnits());
	rc = (*mUserPtr)->Append(ts);
    }

    //----------------------------------  Clean up and go away.
    if (!rc) {
        mLast = t0 + double(last)*mSample;
    } else if (mDebug) {
        cout << "FillSeries: Error from TSeries, rc = " << rc << endl;
	cout << "   mLast=" << mLast << " mSample= " << mSample << endl;
	(*mUserPtr)->Dump(cout);
    }

    return rc;
}

//======================================  Fill FSeries from an FrVect.
//
//    Return codes from FillSeries
//        0  Successful completion
//	 -1  Frame start not contiguous to previous data
//	 -2  Sample rate isn't compatible with previous data.
//	 -3  Requested data not found in current frame
//	 -5  Frame data are not self-consistent.
//	 -6  Series is not allocated.
//	 -7  Unsupported data type
//
int
Channel::FillFSeries(const FrVectRef& vect, const Time& t0, Interval dT) {
    //-----------------------------------  Check arguments.
    int rc=0;
    if (!mFUserPtr || !*mFUserPtr) {
	rc = -6;
    } else if (vect.empty()) {
        rc = -3;


    //-----------------------------------  Fill FSeries
    } else {
        double f0 = vect.getDimX0(0);
	double dF = vect.getDimDx(0);
	DVector* dv = vect.getDVector();
	**mFUserPtr=FSeries(f0, dF, t0, dT, dv);
	if (!dv) rc = -7;
    }

    //-----------------------------------  Error handling
    if (rc && mDebug) {
        cout << "FillFSeries: Error from FSeries, rc = " << rc << endl;
	cout << "   mLast=" << mLast << " mSample= " << mSample << endl;
	(*mFUserPtr)->Dump(cout);
    }
    return rc;
}
