/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Time.hh"
#include "TrigBase.hh"
#include "Segment.hh"
#include "TrigClient.hh"
#include "SigFlag.hh"
#include "rndm.hh"
#include <string>
#include <cmath>
#include <cstdlib>
#include <stdexcept>
#include <unistd.h>
#include <time.h>

#define  PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Services/TrigMgr/TrigRndm.cc 6620 2012-03-13 00:07:24Z john.zweizig@LIGO.ORG $"
#define  PIDTITLE  "Random trigger generation utility"
#include "ProcIdent.hh"

using namespace std;

//======================================  TrigRndm class definition
class TrigRndm : public TrigClient {
public:
    TrigRndm(int argc, const char* argv[]);
    ~TrigRndm(void);
    int Application(void);
    void printSyntax(void);
private:
    int     mGenerate;
    double  mRate;
    std::string mTrigName;
    std::string mIfoName;
    int     mDispMask;
    bool    mSyntax;
    int     mNGener;
    int     mNError;
    double  mDuration;
    bool    mSegment;
    bool    mRealTime;
    SigFlag mTerm;
    Interval mFlush;
};

//======================================  Main function
int
main(int argc, const char* argv[]) {
    int rc = 0;
    try {
	TrigRndm app(argc, argv);
	rc = app.Application();
    } catch (exception& e) {
	cerr << "TrigRndm - Caught exception: " << e.what() << endl;
    }
    return rc;
}

//======================================  TrigRndm constructor (initializer)
TrigRndm::TrigRndm(int argc, const char* argv[]) 
  : TrigClient(trig::TrigWriter::kNone), mGenerate(0), mRate(1.0), 
    mDispMask(trig::d_all), mSyntax(false), mNGener(0), mNError(0), 
    mDuration(1.0), mSegment(false), mRealTime(true), mTerm(SIGINT),
    mFlush(0.0)
{
    string mode_str="mgr";
    string table;
    bool rt = false;
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
	if (argi == "-disp") {
	    mDispMask = strtol(argv[++i], 0, 0);
	} else if (argi == "-debug") {
	    if (++i < argc) setDebug(strtol(argv[i], 0, 0));
	    else            setDebug(1);
	} else if (argi == "-dt") {
	    mDuration = strtod(argv[++i], 0);
	} else if (argi == "--help") {
	    cerr << "TrigRndm generates random triggers and sends them to the"
		 << "  a file or to the TrigMgr via" << endl;
	    cerr << "the TrigClient interface."  << endl;
	    printSyntax();
	    throw runtime_error("Help query");
	} else if (argi == "-ifos") {
	    mIfoName = argv[++i];
	} else if (argi == "-flush") {
	    mFlush = strtod(argv[++i], 0);
	} else if (argi == "-mode") {
	    mode_str = argv[++i];
	} else if (argi == "-name") {
	    mTrigName = argv[++i];
	} else if (argi == "-ntrig") {
	    mGenerate = strtol(argv[++i], 0, 0);
	} else if (argi == "-o") {
	    table = argv[++i];
	} else if (argi == "-rate") {
	    mRate     = strtod(argv[++i], 0);
	} else if (argi == "-rt") {
	    rt = true;
	} else if (argi == "-segment") {
	    mSegment  = true;
	} else {
	    cerr << "Syntax error. Unidentified argument: " << argi << endl;
	    mSyntax = true;
	}
    }

    //------------------------------  Check arguments
    if (mTrigName.empty()) {
	cerr << "Syntax error. Name not specified." << endl;
	mSyntax = true;
    }

    trig::TrigWriter::trig_mode mode;
    if (mode_str == "mgr") {
	mode = trig::TrigWriter::kMgr;
    } else if (mode_str == "ldas") {
	mode = trig::TrigWriter::kWriter;
	mRealTime = rt;
    } else if (mode_str == "segdb") {
	mode = trig::TrigWriter::kSegWrt;
	mRealTime = rt;
    } else if (mode_str == "s6seg") {
	mode = trig::TrigWriter::kS6Seg;
	mRealTime = rt;
    } else {
	cerr << "Unrecognized output mode: " << mode_str << endl;
	mSyntax = true;
    }

    if (mSyntax) {
	printSyntax();
	throw std::runtime_error("Command syntax error");
    }
    mTerm.add(SIGTERM);

    //----------------------------------  Enroll with trigger manager 
    if (getDebug()) cout << "Enrolling trigger client..." << endl;
    int rc = enroll(mode);
    if (getDebug()) cout << "Registration complete, rc=" << rc << endl;
    if (rc) throw std::runtime_error("Unable to initialize trigger writer");
    if (!table.empty()) setTableFile(table.c_str());
}

void 
TrigRndm::printSyntax(void) {
    cerr << "TrigRndm command syntax: " << endl;
    cerr << "TrigRndm [-name <name>] [-ntrig <ntrig>] [-rate <rate>] "
	 << "[-ifos <ifo-name>]" << endl;
    cerr << "         [-disp <disp-mask>] [-dt <delta-t>] [-mode <db-mode>]" 
	 << endl;
    cerr << endl;
    cerr << "<db-mode>   TrigClient mode, i.e. mgr, ldass, segdb or s6seg" 
	 << endl;
}
 
//======================================  TrigRndm destructor (clean up).
TrigRndm::~TrigRndm(void) {
    if (!mSyntax) {
        cout << "TrigRndm completed. Trigger name: " << mTrigName
	     << " Triggers Generated: " << mNGener << " Total errors "
	     << mNError << endl;
    }
}

//======================================  Random trigger generation
int
TrigRndm::Application(void) {
    if (mSyntax) return 1;
    int rc = 0;
    Time now = Now();
    Time lastSeg(0), nextTrig(0);
    while (!mTerm && (!mGenerate || mNGener < mGenerate)) {

        //------------------------------  Wait a random time.
	if (nextTrig != Time(0)) {
	    if (mRealTime) {
		Interval dt = nextTrig - now;
		if (mFlush && dt > mFlush) dt = mFlush;
		long long nsec = (long long)(dt * 1E9);
		timespec wait = {long(nsec / 1000000000L), 
				 long(nsec % 1000000000L)};
		nanosleep (&wait, 0);
		now = Now();
	    } else if (nextTrig != Time(0)) {
		now = nextTrig;
	    }
	}

	int tErr(0);
	if (mSegment) {
	    //--------------------------  Generate intervening off segment
	    if (lastSeg != Time(0) && lastSeg < now) {
		if (getDebug()) cout << "Off segment: " << lastSeg.getS()
				     << "-" << now.getS() << endl;
		trig::Segment s(mTrigName.c_str(), 1, lastSeg, now);
		s.setActivity(0);
		s.setIfos(mIfoName.c_str());
		tErr = sendSegment(s);
		lastSeg = now;
	    }

	    //--------------------------  Generate and send a segment.
	    if (now >= nextTrig) {
		Time tEnd = now+mDuration;
		if (getDebug()) cout << "On segment: " << now.getS()
				     << "-" << tEnd.getS() << endl;
		trig::Segment s(mTrigName.c_str(), 1, now, tEnd);
		s.setIfos(mIfoName.c_str());
		tErr = sendSegment(s);
		lastSeg = tEnd;
	    }
	} else {
	    //--------------------------  Generate a trigger.
	    trig::TrigBase t("TrigRndm", mTrigName.c_str(), now, mDuration);
	    t.setDisposition(mDispMask);
	    tErr = sendTrigger(t);
	}

	//------------------------------  Bump counts - print errors
	mNGener++;
	if (tErr) {
	    cout << "Error " << tErr << " occurred in ";
	    if (mSegment) cout << "sendSegment" << endl;
	    else          cout << "sendTrigger" << endl;
	    mNError++;
	}

	//------------------------------  Generate next random delay
	Interval delay(-log(Rndm())/mRate);
	nextTrig = now + delay;
    }
    return rc;
}
