/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGCTLTABLE_HH
#define TRIGCTLTABLE_HH

#include "Time.hh"
#include "TrigBase.hh"
#include <string>
#include <iosfwd>
#include <list>

namespace trig {
  class Segment;
  class TrigProc;
}

namespace html {
  class table;
}

class TrigCtlEntry {
public:
    typedef unsigned long mask_t;

public:
    /**  Default constructor
      */
    TrigCtlEntry(void);

    /**  Construct and set a Control table entry.
      *  @memo Data constructor
      *  @param program Name of program(s) to be controlled by this entry.
      *  @param version Version of programs controlled by this entry
      *  @param trigid  ID(s) of triggers to be controlled by this entry.
      *  @param subid   SubID(s) of trigger to be controlled by this entry
      *  @param dest    Destination mask for controlled triggers.
      *  @param maxprio Maximum priority for controlled triggers.
      */
    TrigCtlEntry(const std::string& program, const std::string& version, 
		 const std::string& trigid,  const std::string& subid, 
		 mask_t dest=trig::d_all, int maxprio=trig::p_severe);

    /**
     *  Destroy a control table entry.
     *  \brief Destructor
     */
    ~TrigCtlEntry(void);

    /**
     *  Display a %TrigMgr control table entry.
     *  \brief Print control entry contents
     *  \param out Output stream
     *  \return Reference to the output stream
     */
    std::ostream& display(std::ostream& out) const;

    /** Compare Trigger ID to entry.
     */
    bool testKey(const std::string& program, const std::string& version, 
		 const std::string& trigid,  const std::string& subid) const;

    bool testKey(const TrigCtlEntry& t) const;

    /** Get the trigger count
     */
    int getCount(void) const;

    /** Get the destination mask
     */
    mask_t getDestMask(void) const;

    /** Get the trigger count
     */
    int getEpoch(void) const;

    /**  Return the highest allowed priority.
      */
    int getMaxPrty(void) const;

    /** Get the notify e-mail address
     */
    const char* getNotify(void) const;

    /** Get the program name
     */
    const char* getProgram(void) const;

    /**  Return the specified priority or the highest allowed priority 
      *  whichever is less.
      */
    int getPrty(int prio) const;

    /** Get the trigger sub-ID covered by this entry.
     */
    const char* getSubID(void) const;

    /**  Get the trigger rate threshold
      */
    double getThreshold(void) const;

    /**  Get the trigger ID
      */
    const char* getTrigger(void) const;

    /**  Get the program version
      */
    const char* getVersion(void) const;

    /**  Mask off any disallowed destination bits from the specified 
      *  destination specifier.
      */
    mask_t maskDest(mask_t dest) const;

    /** Set the destination mask
      */
    void setDestMask(mask_t dest);

    /** Set the maximum allowed priority for the specified trigger(s).
      */
    void setMaxPrty(int pri);

    /** Set the notify e-mail address.
      */
    void setNotify(const std::string& email);

    /** Set the notify e-mail address.
      */
    void setThreshold(double thresh);

    /**  Reset the statistics from this trigger.
      */
    void reset(void);

    /**  Reset the epoch count..
      */
    void epoch(void);

    /**  Increment the total and epoch counts.
     */
    int count(void) const;

private:
    /**  Name of program(s) covered by this entry.
      */
    std::string  mProgram;

    /**  Program version(s) covered by this entry.
      */
    std::string  mVersion;

    /**  Trigger ID(s) covered by this entry.
      */
    std::string  mTrigID;

    /**  Trigger sub-ID(s) covered by this entry.
      */
    std::string  mSubID;

    /**  Count of triggers covered by this entry.
      */
    mutable int mCount;

    /**  Count of triggers in the current epoch.
      */
    mutable int mEpoch;

    /**  Bit mask of allowed destinations.
      */
    mask_t mDestMask;

    /**  Maximum priority level.
      */
    int mMaxPrty;

    /** Rate threshold.
     */
    double mThreshold;

    /** Notify address.
     */
    std::string mNotify;
};

class TrigCtlTable {
private:
    typedef std::list<TrigCtlEntry> CtlList;
    typedef CtlList::const_iterator const_iter;
    typedef CtlList::iterator       list_iter;

public:
    TrigCtlTable(void);
    TrigCtlTable(const char* file);
    ~TrigCtlTable(void);
    const TrigCtlEntry* find(const trig::TrigProc& p, 
			     const trig::TrigBase& t) const;
    const TrigCtlEntry* find(const trig::TrigProc& p, 
			     const trig::Segment& t) const;
    const TrigCtlEntry* find(const std::string& program, 
			     const std::string& version, 
			     const std::string& trigID,  
			     const std::string& subID) const;
    const TrigCtlEntry* find(const TrigCtlEntry& t) const;
    list_iter findAfter(const TrigCtlEntry& t, const list_iter& p);
    void clear(void);
    void epoch(void);
    void readFile(const std::string& file, bool reset=true);
    void reset(void);

    /**  Write the trigger statistics to the specified output file.
     */
    void writeStats(std::ostream& out) const;

    /**  makeHtmlTable(void);
     */
    html::table makeHtmlTable(void) const;

private:
    CtlList mList;
    Time    mReset;
    Time    mEpoch;
};

//====================================== Inline accessors
inline int 
TrigCtlEntry::count(void) const {
    ++mEpoch;
    return ++mCount;
}

inline int 
TrigCtlEntry::getCount(void) const {
    return mCount;
}

inline TrigCtlEntry::mask_t 
TrigCtlEntry::getDestMask(void) const {
    return mDestMask;
}

inline int 
TrigCtlEntry::getEpoch(void) const {
    return mEpoch;
}

inline int 
TrigCtlEntry::getMaxPrty(void) const {
    return mMaxPrty;
}

inline const char* 
TrigCtlEntry::getNotify(void) const {
    return mNotify.c_str();
}

inline const char* 
TrigCtlEntry::getProgram(void) const {
    return mProgram.c_str();
}

inline int 
TrigCtlEntry::getPrty(int prio) const {
    return (prio < mMaxPrty) ? prio : mMaxPrty;
}

inline const char* 
TrigCtlEntry::getSubID(void) const {
    return mSubID.c_str();
}

inline double
TrigCtlEntry::getThreshold(void) const {
    return mThreshold;
}

inline const char* 
TrigCtlEntry::getTrigger(void) const {
    return mTrigID.c_str();
}

inline const char* 
TrigCtlEntry::getVersion(void) const {
    return mVersion.c_str();
}

inline TrigCtlEntry::mask_t 
TrigCtlEntry::maskDest(mask_t dest) const {
    return mDestMask & dest;
}

#endif  // TRIGCTLTABLE_HH
