/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "ProcTable.hh"
#include "TrigProc.hh"
#include <fstream>
#include <stdlib.h>

using namespace xsil;
using namespace std;
using namespace trig;

//======================================  Pack N decimal digits into N/2 bytes
void
ProcTable::packKey(int len, const char* in, unsigned char* out) {
    for (int j=0 ; j < 2*len ; j+=2) {
        *out++ = ((in[j] & 0x0f)<<4) + (in[j+1] & 0x0f);
    }
}

//======================================  Define the process table.
ProcTable::ProcTable(bool alphaPID) 
  : MetaTable("process"), proc_id_length(nProcessKeyBytes), 
    process_id(&proc_id_length, processID), altPID(alphaPID)
{
    setColumns();
}

//======================================  Define the columns in the MetaTable
void
ProcTable::setColumns(void) {
    MetaTable::clear();
    defineColumn("program",        &program);
    defineColumn("version",        &version);
    defineColumn("cvs_repository", &cvsname);    
    defineColumn("cvs_entry_time", &cvstime);    
    defineColumn("comment",        &comment);
    defineColumn("node",           &node);
    defineColumn("username",       &username);
    defineColumn("unix_procid",    &unix_procid);
    defineColumn("start_time",     &start_time);
    defineColumn("end_time",       &end_time);
    if (altPID) defineColumn("process_id", &PIDstr, true);
    else        defineColumn("process_id", &process_id);
    defineColumn("ifos",           &ifo);
}

//======================================  Destroy the process table
ProcTable::~ProcTable(void) {
    MetaTable::clear();
}

//======================================  Fill the process table.
int
ProcTable::addRow(const trig::TrigProc& p) {
    program     = p.getName();
    version     = p.getVersion();
    cvsname     = p.getSource();
    cvstime     = p.getModTime();
    comment     = p.getTitle();
    node        = p.getNode();
    username    = p.getAuthor();
    unix_procid = p.getProcID();
    start_time  = p.getStartTime();
    end_time    = p.getEndTime();
    if (altPID) PIDstr = p.getProcessID();
    else        packKey(nProcessKeyBytes, p.getProcessID(), processID);
    ifo         = p.getIFOs();
    int irow    = getNRow();
    putRow();
    return irow;
}

//======================================  Check that the pointers are OK
void
ProcTable::check(const char* title) const {
    MetaTable::check(title, sizeof(ProcTable));
}
