/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DQSegTable.hh"
#include "TrigBase.hh"
#include "TrigProc.hh"
#include "Segment.hh"
#include <fstream>
#include <stdlib.h>

using namespace xsil;
using namespace std;
using namespace trig;

//======================================  Null S5 Segment definer construtor
DQSegDef::DQSegDef(void) 
    : mVersion(0)
{}

//======================================  S5 Segment definer data construtor
DQSegDef::DQSegDef(const Segment& seg) {
    mProcID  = seg.getProcess();
    mIfos    = seg.getIfos();
    mName    = seg.getGroup();
    mVersion = seg.getVersion();
    mComment = seg.getComment();
}

//======================================  S5 Segment definer construtor
void 
DQSegDef::setDefinerID(const std::string& defid) {
    mDefID = defid;
}

//======================================  Define the MySql segment table
DQSegTable::DQSegTable(void) 
  : MetaTable("segment")
{
    defineColumn("segment_id", &segid, true);
    defineColumn("start_time", &start_time);
    defineColumn("end_time",   &end_time);
    defineColumn("active",     &active);
    defineColumn("process_id", &procid, true);
}

//======================================  Delete the trigger table
DQSegTable::~DQSegTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
DQSegTable::addRow(const trig::Segment& s) {
    start_time = s.getStartTime().getS();
    end_time   = s.getEndTime().getS();
    active     = s.getActivity();
    segid      = s.getSegID();
    procid     = s.getProcess();
    putRow();
}

//======================================  Check that the pointers are OK
void
DQSegTable::check(const char* title) const {
    MetaTable::check(title, sizeof(DQSegTable));
}

//======================================  MySql segment definer table
DQSegDefTable::DQSegDefTable(void) 
  : MetaTable("segment_definer")
{
    defineColumn("process_id",     &procid, true);
    defineColumn("segment_def_id", &defid, true);
    defineColumn("run",            &run);
    defineColumn("ifos",           &ifos);
    defineColumn("name",           &name);
    defineColumn("version",        &version);
    defineColumn("comment",        &comment);

    //----------------------------------  Fill in the run name
    run = "Sn";
    const char* home = getenv("HOME");
    if (!home)  home = ".";
    string file(home);
    file += "/pars/RunID";
    ifstream ridfile(file.c_str(), ios::in);
    if (!ridfile) return;
    ridfile >> run;
}

//======================================  Delete the trigger table
DQSegDefTable::~DQSegDefTable(void) {
    MetaTable::clear();
}

//======================================  Fill the trigger table.
void
DQSegDefTable::addRow(const DQSegDef& d) {
    procid  = d.refProcess();
    defid   = d.refDefinerID();
    ifos    = d.refIfos();
    name    = d.refName();
    version = d.getVersion();
    comment = d.refComment();
    putRow();
}

//======================================  Check that the pointers are OK
void
DQSegDefTable::check(const char* title) const {
    MetaTable::check(title, sizeof(DQSegDefTable));
}

//======================================  MySql segment definer table
DQSegMapTable::DQSegMapTable(void) 
  : MetaTable("segment_def_map")
{
    defineColumn("segment_id",     &segid,  true);
    defineColumn("segment_def_id", &defid,  true);
    defineColumn("process_id",     &procid, true);
}

//======================================  Delete the trigger table
DQSegMapTable::~DQSegMapTable(void) {
    MetaTable::clear();
}

//======================================  Fill the segment map
void
DQSegMapTable::addRow(const trig::Segment& s, const trig::DQSegDef& d) {
    addRow(s.getSegID(), d.refDefinerID(), s.getProcess());
}

void 
DQSegMapTable::addRow(const string& s, const string& d, const string& p) {
    segid  = s;
    defid  = d;
    procid = p;
    putRow();
}

//======================================  Check that the pointers are OK
void
DQSegMapTable::check(const char* title) const {
    MetaTable::check(title, sizeof(DQSegMapTable));
}
