#ifndef HOSTTABLE_HH
#define HOSTTABLE_HH
#include "lmsg/MsgAddr.hh"
#include <string>
#include <vector>
#include <list>
#include <iosfwd>

/**  HostEntry holds an IP address and network mask for a single host.
  *  This might be extended to contain a quality of service quantifier, 
  *  but for the moment we can get the same results from a careful 
  *  ordering of the entries.
  *  @memo Host address entry.
  *  @author John Zweizig
  *  @version 1.0; Modified June 22, 2000
  */
class HostEntry {
public:

  /**  Data type used to represent IP addresses.
    *  @memo IP address data type.
    */
  typedef lmsg::MsgAddr::ipaddr_t ip_addr;

  /**  Create a host table entry and zero the IP address and network mask.
    *  @memo Default constructor.
    */
  HostEntry(void);

  /**  Construct a host entry from an address string. The address string
    *  has the form "<ip-address>/<net-mask>". The "<ip-address>" and
    *  "<net-mask>" fields may be specified either as a 32-bit integer 
    *  field, or as four 8-bit integers separated by '.'s.
    */
  HostEntry(const std::string& in);

  /**  Construct a table entry and copy in the contents of the argument
    *  entry.
    *  @memo Copy COnstructor.
    */
  HostEntry(const HostEntry& x);

  /**  Destroy a host table entry.
    *  @memo Entry destructor.
    */
  ~HostEntry(void) {}

  /**  Compare two entries for equivalence.
    *  @memo Compare for equal.
    *  @return True if the argument entry is exactly equal to the lhs entry.
    *  @param x Entry to be compared to the current object.
    */
  bool operator==(const HostEntry& x) const throw();

  /**  Test whether the IP address of the host specified by this entry is 
    *  equal to that specified by the argument.
    *  @memo Compare to an IP address.
    *  @return true if current entry address is same as the specified address.
    *  @param ipaddr IP address to be compared to that of this entry.
    */
  bool isNode(ip_addr ipaddr) const throw();

  /**  Test whether the specified IP address is on the same network as the 
    *  host specified by this entry. The network mask of the current entry
    *  is used to find the network number of both operands.
    *  @memo Compare network fields.
    *  @return true if the argument network field equals that of this host.
    *  @param ipaddr IP address to be compared to that of this entry.
    */
  bool sameNet(ip_addr ipaddr) const throw();

  /**  Fetch the IP node address of this host.
    *  @memo Get the host IP node address.
    *  @return The IP node address of the host represented by this entry.
    */
  ip_addr getAddr(void) const throw();

  /**  Print the Host data to the specified output stream.
    *  @memo Print host data.
    *  @param out Output stream to which the data will be printed.
    */
  void dump(std::ostream& out) const;

public:
  /**  Format an IP address as "a.b.c.d" and spew it out to the specified 
    *  output stream..
    */
  static void dumpIP(std::ostream& out, ip_addr addr);

private:
  ip_addr mAddr;
  ip_addr mMask;
};

//======================================  Inline functions
inline HostEntry::ip_addr
HostEntry::getAddr(void) const throw() {
    return mAddr;
}

/**  The HostList class contains a list of IP address that refer to the 
  *  same node. Entries in the list may be referenced by sequence number
  *  or via a few general access commands.
  *  @memo List of aliases for a single node.
  *  @author J. Zweizig
  */
class HostList {
public:

  /**  Standard container used to hold the list of entries. For now we
    *  use a vector, but any STL container would work.
    *  @memo Host list container.
    */
  typedef std::vector<HostEntry> HostVector;
  typedef unsigned int size_type;

  /**  Construct an empty list.
    *  @memo Default constructor.
    */
  HostList(void);

  /**  Construct a list and fill it from a string. The string contains an
    *  arbitrary number of host entries separated by one or more blanks or
    *  tab characters.
    *  @memo Construct a host list from a string.
    *  @param in String containing the host list.
    */
  HostList(std::string& in);

  /**  Costruct a host list and fill it with entries from the argument list.
    *  @memo Copy constructor
    *  @param x Host list to be copied 
    */
  HostList(const HostList& x);

  /**  Empty a list and destroy it.
    *  @memo Destructor.
    */
  ~HostList(void);

  /**  An entry is added to the end of the list and the contents of the
    *  argument are copied into it.
    *  @memo Add an entry to the list.
    *  @param x Entry whose contents are used to initialize the new entry.
    */
  void addEntry(const HostEntry& x);

  /**  Test whether the specified IP address corresponds to one of the 
    *  IP addresses in the list.
    *  @memo Test whether the specified IP address is an alias of this node.
    */
  bool isNode(HostEntry::ip_addr x) const;

  /**  Find the alias in the list that corresponds to the network of the 
    *  specified IP Address. If no aliases are found on the specified 
    *  network, #netAlias# returns 0.
    *  @memo Find a reachable alias.
    *  @param x IP node for which a reachable address is to be found.
    *  @return Alias address reachable from the specified nod or 0.
    */
  HostEntry::ip_addr netAlias(HostEntry::ip_addr x) const;

  /**  Get the number of networks specified in this list.
    *  @memo Number of entries.
    *  @return Number of entries in the list.
    */
  size_type size(void) const;

  /**  Get the entry specified by the index.
    *  @memo get an entry.
    *  @return Constant reference to the specified entry.
    *  @param i Index spcifying which entry is to be returned.
    */
  const HostEntry& operator[](unsigned int i) const;

  /**  Print the Host data to the specified output stream.
    *  @memo Print host data.
    *  @param out Output stream to which the data will be printed.
    */
  void dump(std::ostream& out) const;

private:
  HostVector mHostVect;
};

/**  TunnelEntry defines a tunnel process, its server port and the 
  *  network(s) to which it connects.
  *  @memo Tunnel process entry.
  *  @author John Zweizig
  *  @version 1.0; Modified August 13, 2002
  */
class TunnelEntry {
public:

  /**  Data type used to represent IP addresses.
    *  @memo IP address data type.
    */
  typedef lmsg::MsgAddr::ipaddr_t ip_addr;

  /**  Create a host table entry and zero the IP address and network mask.
    *  @memo Default constructor.
    */
  TunnelEntry(void);

  /**  Construct a host entry from an address string. The address string
    *  has the form "<ip-address>/<net-mask>". The "<ip-address>" and
    *  "<net-mask>" fields may be specified either as a 32-bit integer 
    *  field, or as four 8-bit integers separated by '.'s.
    */
  TunnelEntry(ip_addr netip, ip_addr netmask, const lmsg::MsgAddr& addr);

  /**  Destroy a host table entry.
    *  @memo Entry destructor.
    */
  ~TunnelEntry(void) {}

  /**  Test whether the IP address of the spcified host is reachable via
    *  this tunnel.
    *  @memo Test if the specified address is reachable via this tunnel.
    *  @return true if tunnel reaches the specified address.
    *  @param ipaddr IP address to be resolved.
    */
  bool isAccessible(const lmsg::MsgAddr& addr) const throw();

  /**  Test whether the tunnel server port has the specified address.
    *  @memo Test the tunnel address.
    *  @return true if tunnel has the specified address.
    *  @param ipaddr Server address.
    */
  bool isTunnel(const lmsg::MsgAddr& addr) const throw();

  /**  Fetch the IP node address of this host.
    *  @memo Get the host IP node address.
    *  @return The IP node address of the host represented by this entry.
    */
  lmsg::MsgAddr getAddr(void) const throw();

private:
  ip_addr       mNetAddr;
  ip_addr       mNetMask;
  lmsg::MsgAddr mTunnel;
};

inline
TunnelEntry::TunnelEntry(TunnelEntry::ip_addr netip, 
			 TunnelEntry::ip_addr netmask, 
			 const lmsg::MsgAddr& addr) 
  : mNetAddr(netip), mNetMask(netmask), mTunnel(addr)
{}

inline
TunnelEntry::TunnelEntry(void) 
  : mNetAddr(0), mNetMask(0), mTunnel(0, 0, 0)
{}

inline bool
TunnelEntry::isAccessible(const lmsg::MsgAddr& addr) const throw() {
    return !( (addr.getIPAddr() ^ mNetAddr) & mNetMask);
}

inline bool
TunnelEntry::isTunnel(const lmsg::MsgAddr& addr) const throw() {
    return (addr == mTunnel);
}

inline lmsg::MsgAddr
TunnelEntry::getAddr(void) const throw() {
    return mTunnel;
}

/**  The Host table contains a list of all known multi-network hosts.
  *  The table may be used to select an alias for a given host which
  *  is reachable by noher specified host.
  *  @memo Table if known multi-net Hosts.
  *  @author J. Zweizig
  *  @version 1.0; Modified June 22, 2000
  */
class HostTable {
public:
  /**  Container used to hold the Host lists.
   */
  typedef std::list<HostList> HostVList;

  /**  Iterator over the host list container. The host iterator dereferences
    *  to a HostList.
    */
  typedef HostVList::const_iterator host_iter;

  /**  Container used to hold the Host lists.
   */
  typedef std::list<TunnelEntry> TunnelList;

  /**  Iterator over the host list container. The host iterator dereferences
    *  to a HostList.
    */
  typedef TunnelList::iterator       tunnel_iter;
  typedef TunnelList::const_iterator const_tunnel_iter;
  typedef HostList::size_type size_type;

  /**  Construct an empty host table.
    *  @memo default constructor.
    */
  HostTable(void);

  /**  Construct a host table and initialize it from the specified file.
    *  The data file format is as described in the setTable() function.
    *  @memo Construct a host table and initialize it from a file.
    */
  HostTable(const char* infile);

  /**  Construct a host table and initialize it from the specified stream.
    *  The data stream format is described in the #setTable()# documentation.
    *  @memo Construct a host table and initialize it from a stream.
    */
  HostTable(std::istream& in);

  /**  Construct a host table and initialize it from the specified stream.
    *  The data stream contains one node specification per line. Lines may
    *  be interspersed with comments starting with a pound sign ("#"). Lines
    *  terminated by a backslash ("\") are continued on the succeding line.
    *  @memo Initialize the host table from a stream.
    */
  void setTable(std::istream& in);

  /**  Find a host list for the specified node. If no list is found for the 
    *  specified host, the iterator returned is equal to end().
    *  @memo find alias list for a specified host.
    *  @return Iterator pointing to alias list of the specified host.
    *  @param addr  IP address of the host to be located.
    */
  host_iter find(const lmsg::MsgAddr& addr) const;

  /**  Host iterator pointing to the end of the host table. Invalid host
    *  iterators point to this (non)entry.
    *  @memo Get end of host table.
    *  @return Invalid iterator to end of host table.
    */
  host_iter end(void) const;

  /**  Get the number of networks that a known node is attached to. If the
    *  node is not specified in the host table, it is assumed to be connected
    *  to one network.
    *  @memo Number of network attachments for a specified node.
    *  @return Number of network attachments for the specified node.
    *  @param req Any known IP address of the requested node.
    */
  size_type getNNet(const lmsg::MsgAddr& req) const;

  /**  Find an alias for the target node that is on the same network as 
    *  the requestor node. If no target alias is found on the same network
    *  as the requestor, #getRoute()# returns zero.
    *  @memo Find a reachable alias.
    *  @return Target alias on the same network as the requestor or zero.
    *  @param IP address or alias for target node.
    *  @param IP address or alias for the requestor node.
    */
  lmsg::MsgAddr getRoute(const lmsg::MsgAddr& req, 
			 const lmsg::MsgAddr& target) const;

  /**  Add a tunnel to the tunnel list.
    *  @memo Add a Tunnel
    *  @param netip   Network IP address.
    *  @param netmask Network Mask
    *  @param addr    Tunel server address.
    */
  void addTunnel(TunnelEntry::ip_addr netip, TunnelEntry::ip_addr netmask,
		 const lmsg::MsgAddr& addr);

  /**  Remove a tunnel from the tunnel list.
    *  @memo Remove a Tunnel
    *  @param addr    Tunel server address.
    */
  void remTunnel(const lmsg::MsgAddr& addr);

  /**  Print the Host data to the specified output stream.
    *  @memo Print host data.
    *  @param out Output stream to which the data will be printed.
    */
  void dump(std::ostream& out) const;

  /**  Set the debug flags
    *  @memo set the debug flag.
    *  @param torf new debug mode flag
    */
  void setDebug(bool torf=true) {mDebug=torf;}

private:
  HostVList  mHostVList;
  TunnelList mTunnels;
  bool       mDebug;
};

inline HostTable::host_iter
HostTable::end(void) const {
    return mHostVList.end();
}

#endif   //  HOSTTABLE_HH

