/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    DPUSH is the application to push the data from the NDS into the 
//    shared memory partition in frame format. This source file contains
//    both the traditional (CDS Network data server) and multicast versions.
//    The multicast version is selected with the -DMULTICNDS compilation
//    flag.
//
//    Syntax:
//
//       Dpush [NDS <node>] [partition <pname>] [lbuf <nbytes>] [nbuf <n>]
//             [log <logfile>] [time <ticks>] [verbose <vfile>] [-nocheck]
//
//       node    Network data server node name or number.
//       pname   Shared memory partition name
//       nbytes  Length of shared memory buffers
//       n       Number of shared memory buffers
//       logfile Log file.
//       ticks   Watchdog (statistics summary) timeout interval
//
//    Notes:
//      1) Partitions aren't currently created by the program.
//
//    Revision History:
//
//      18-Feb-1999  J. Zweizig
//          Preliminary draft version. 
//
//------- A few definitions
//
//    DEFNDSIP the default NDS server IP address.
//    DEFPART  is the default partition name that is used if the "partition" 
//             command line argument is not specified.
//    CDSDATA  receive data in CDS proprietary format.
//
#if defined(sun) && !defined(__EXTENSIONS__)
#define __EXTENSIONS__
#endif

#ifdef MULTICNDS
#define DEFNDSIP "225.0.0.1"       /* multicast address */
#else
// #define CDSDATA 1
#define DEFNDSIP "131.215.115.67"  /* "albeireo.ligo.caltech.edu" */
#endif
#define DEFPART "LIGO_Online"

#include "PConfig.h"
//--> Include till you drop.
//    Note: unistd.h included for swab(). This didn't work in old versions
//          of linux (e.g. RH6.2), but seems to be OK with any modern version.
#include <iostream>
#include <fstream>
#include <netdb.h>
#include <netinet/in.h>
#include <cstdlib>
#include <cstdio>
#include <string>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include "SigFlag.hh"
#include "Time.hh"
#include "lsmp_prod.hh"
#include "checksum_crc32.hh"
#ifdef MULTICNDS
  #include "framexmit/framexmit.hh"
#else
  #include "DAQSocket.hh"
#endif

//--------------------------------------  Sun stuff needed to set R/T priority
#ifdef P__LINUX
  #define MAX_RTPRIO 99
  #include <sched.h>
#elif defined(P__SUN)
  #include <sys/priocntl.h>
  #include <sys/rtpriocntl.h>
  #include <sys/mman.h>
  #include <strings.h>
#endif
#include "swapframe.hh"
#include "stdio_fd.hh"
#include <stdexcept>
#include <cstring>
#ifdef P__DARWIN
#include "fdstream.hh"
#endif

using namespace std;

//------- Statistics class.
class Stats {
public:
    typedef unsigned long ulong_t;
    Stats();
    ~Stats();
    void endStats(ostream& out) const;
    void Error(int errcd) {totalErrors++;}
    void Log(ostream& out);
    void Reset(void);
    void Start(ulong_t Secs);
    bool Test(void);
    void Update(ulong_t FrameID, ulong_t GPS, ulong_t dt, ostream& out);
private:
    ulong_t nSecs;
    SigFlag watchdog;
    time_t  Stamp;
    ulong_t GPSfirst;
    ulong_t GPSstart;
    ulong_t GPSlast;
    ulong_t RecIDstart;
    ulong_t RecIDlast;
    ulong_t FrameCount;
    ulong_t FrameLost;
    ulong_t FrameUnsent;
    ulong_t totalFrames;
    ulong_t totalSec;
    ulong_t totalLost;
    ulong_t totalUnsent;
    ulong_t totalErrors;
};

//
//    NDS interface stuff
//
#ifdef MULTICNDS
static framexmit::frameRecv* NDS = 0;
#else
static DAQSocket* NDS = 0;
#endif

//---->  Data swapper
//static SwapFrame SF;

//---->  Shared memory output device
static LSMP_PROD* OPart=0;

//---->  Termination signal flag pointer
static SigFlag term;
static bool    active;
static const char* verbose(0);
static bool    debug(false);
static bool    datacheck(true);
static int rd(-1), vrd(-1);

//---->  Watchdog statistics timer.
static Stats wstat;

//---->  Verbose log stream
#ifdef P__DARWIN
static fd_ostream vstream(-1);
#else
static ofstream vstream;
#endif

//----> Internal entry points.
//
//    Dpsh_Init     Set up the input and output functions
//    Dpsh_Halt     Set down.
//    Dpsh_Read     Read routine.
//    Dpsh_Check    Check a frame is consistent.
//    stamp         Generate a time stamp.
static bool Dpsh_Init(int argc, const char **argv);
static void Dpsh_Halt(void);
static void Dpsh_Read(void);
static bool Dpsh_Check(unsigned int length, const char* addr);
static string stamp(time_t now);

//
//    main function. 
//      * Set up. This includes initializing the input and output 
//        files and the exception handling.
//      * Loop over records until the ceiling falls in on us
//      * Call Dpsh_Read to process a frame
//      * When the roof caves in set-down the application
//
int 
main(int argc, const char **argv) {

    //-----------------------------------  Fork a daemon
    if (argc > 1 && string(argv[1]) == "-daemon") {
	switch (fork()) {
	case -1:
	    exit(1);
	case 0:
	    break;
	default:
	    exit(0);
	}
    }

    //-----------------------------------  Loop over frames
    active = !Dpsh_Init(argc, argv);

    //-----------------------------------  Loop over frames
    while (active && !term) {
        if (wstat.Test()) wstat.Log(cout);
        Dpsh_Read();
    }
    Dpsh_Halt();
    return 0;
}

//
//    Dpsh_Init - set up the internal environment for pushing data.
//       * Parse any command line parameters.
//       * Create a shared memory partition for data distribution.
//       * Make a connection to the CDS Network Data Server. Register 
//         notification of all frames.
//       * Set up the termination signal handling.
//
static bool Dpsh_Init(int argc, const char **argv) {
    const char*   FileName = (const char*)0;
    int   nbuf = 1, lbuf = 1000000, Secs=60;

#ifdef P__SUN
    //---------------------------------  Don't run as root
    seteuid(getuid());
#endif

    //---------------------------------  Define default parameters
    const char*   ParName = getenv("LIGOSMPART");
    if (!ParName) ParName = DEFPART;

    std::string NDS_IP = DEFNDSIP;
    int         port   = 0;
#ifdef MULTICNDS
    if (getenv("LIGOMCASTIP")) NDS_IP  = getenv("LIGOMCASTIP");
    port = framexmit::frameXmitPort;
#else
    if (getenv("LIGONDSIP"))   NDS_IP  = getenv("LIGONDSIP");
#endif

    //---------------------------------  Parse the arguments
    for (int i=1 ; i<argc ; i++) {
        string argi = argv[i];
        if (argi == "NDS") {
	    NDS_IP   = argv[++i];
	} else if (argi == "-daemon") {
	    if (i != 1) {
		cerr << "Warning, -daemon is not the first argument" << endl;
	    }
	} else if (argi == "partition") {
	    ParName  = argv[++i];
	} else if (argi == "file") {
	    FileName = argv[++i];
	} else if (argi == "lbuf") {
	    lbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "nbuf") {
	    nbuf = strtol(argv[++i], 0, 0);
	} else if (argi == "port") {
	    port = strtol(argv[++i], 0, 0);
	} else if (argi == "time") {
	    Secs = strtol(argv[++i], 0, 0);
	} else if (argi == "verbose") {
	    verbose = argv[++i];
	} else if (argi == "nocheck") {
	    datacheck = false;
	} else if (argi == "debug") {
	    debug = true;
	} else if (argi == "log") {
	    int flags = O_WRONLY | O_CREAT | O_NONBLOCK | O_APPEND;
            int fd = open(argv[++i], flags, 0664);

	    //----------------------------------  Add an unused reader
	    stdio_fd file(fd);
	    if (file.is_pipe()) {
		rd = open(argv[i], O_RDONLY, 0664);
	    }
            close(STDOUT_FILENO); // stdout
            if (dup(fd) != STDOUT_FILENO) {
                cerr << "stdout not diverted!" << endl;
            }
            close(STDERR_FILENO); // stderr
            if (dup(fd) != STDERR_FILENO) {
                cout << "stderr not diverted!" << endl;
            }
            close(fd);
	    cout << stamp(0) << argv[0] << " Starting..." << endl;
	} else {
	    cerr << argv[0] << ": Invalid argument name " << argv[i] << endl;
	    cerr << "Command syntax:" << endl << endl;
	    cerr << argv[0] << " [NDS <server>] [partition <partition>] "
		 << "[file <input-file>] [debug] [nocheck] \\" << endl;
	    cerr << "       [nbuf <#buffers>] [lbuf <buffer-length>] "
		 << "[log <log-file>] [time <stat-timer>]" << endl;
	    return true;
	}
    }

    string::size_type pinx = NDS_IP.find(':');
    if (pinx != string::npos) {
	port = strtol(NDS_IP.c_str()+pinx+1, 0, 0);
	NDS_IP.erase(pinx);
    }

    //---------------------------------  Create a Frame output
    if (FileName) {
	cerr << stamp(0) << "File output is not supported by Dpush." << endl;
	return true;
    } else {
	OPart = new LSMP_PROD(ParName, nbuf, lbuf);
	if (!OPart->valid()) {
	    cerr << stamp(0) << "Unable to access partition " << ParName 
		 << ": " << OPart->Error() << "." << endl;
	    return true;
	}
	OPart->lock(true);
    }

    //---------------------------------  Set up the statistics task
    wstat.Start(Secs);

    //---------------------------------  Set up the verbose stream
    if (verbose) {
#ifndef P__DARWIN
	vstream.open(verbose, ios_base::app | ios_base::out);
	if (!vstream.is_open()) {
	    cerr << "Unable to open verbose log: " << verbose << endl;
	    verbose = 0;
	}
	stdio_fd file(vstream);
#else
	int flags = O_WRONLY | O_CREAT | O_NONBLOCK | O_APPEND;
	int fd = open(verbose, flags, 0664);
	*dynamic_cast<fd_streambuf*>(vstream.rdbuf()) = fd_streambuf(-1, fd);
	stdio_fd file(fd);
#endif
	//----------------------------------  Add an unused reader. Make a 
	//                                    non-blocking pipe
	if (file.is_pipe()) {
	    file.block(false);
	    vrd = open(verbose, O_RDONLY, 0664);
	}

    }


    //---------------------------------  Set up the termination signals
    term.add(SIGINT);
    term.add(SIGTERM);
    term.add(SIGHUP);

    //---------------------------------  Open the NDS connection.
#ifdef MULTICNDS
    const char* inet = getenv("LIGONDSIP");
    if (!inet)  inet = "10.4.0.0";
    char addr[16];
    struct hostent* ptr = gethostbyname(inet);
    if (ptr) {
        inet = (const char*) ptr->h_addr_list[0];
	sprintf(addr, "%i.%i.0.0", (unsigned char)inet[0], 
		                   (unsigned char)inet[1]);
	inet = addr;
    }
    framexmit::par.init();
    framexmit::par.display(cout);
    cout << stamp(0) << "Opening connection to frame multicaster: "
	 << NDS_IP << " on network with " << inet << endl;
    NDS = new framexmit::frameRecv();
    //  if (debug) NDS->setDebug(1); // not currently a method.
    if (!NDS->open(NDS_IP.c_str(), inet, port)) {
	int errnum = errno;
	cerr << stamp(0) << "Multicast receiver open failed: " 
	     << strerror(errnum) << endl;
	return true;
    }
    NDS->logging();
#else
    cout << "Opening connection to NDS at " << NDS_IP << endl;
    NDS = new DAQSocket(NDS_IP);
    NDS->setDebug(debug);
    NDS->AddChannel("all");
#ifndef CDSDATA
    NDS->RequestFrames();
    cout << stamp(0) << "Frame data requested." << endl;
#else
    NDS->RequestData();
    cout << stamp(0) << "CDS proprietary data requested" << endl;
#endif
#endif

#if defined(P__LINUX)
    sched_param param;
    param.sched_priority = MAX_RTPRIO;
    int sched_suxs = sched_setscheduler(0, SCHED_FIFO, &param);
    if (sched_suxs) perror("Error trying to set RT priority");
#elif defined(P__SUN)
    //---------------------------------  Establish RealTime priority
    int sveuid = geteuid();
    seteuid(0);
    if (!geteuid()) {
        pcinfo_t pci;
	pcparms_t pcp;
	rtparms_t rtp;

	cout << stamp(0) << "Setting RT priority." << endl;
	strcpy(pci.pc_clname, "RT");
	int ret = priocntl(P_PID, getpid(), PC_GETCID, (char*)&pci);
	if (ret < 0) {
	    int errnum = errno;
	    cerr << stamp(0) << "Error getting CID: " 
		 << strerror(errnum) << endl;
	} else {
	    pcp.pc_cid = pci.pc_cid;
	    rtp.rt_pri = *(short*)&pci.pc_clinfo[0];
	    rtp.rt_tqsecs = 0;
	    rtp.rt_tqnsecs = RT_TQINF;

	    memcpy(pcp.pc_clparms, &rtp, sizeof(rtparms_t));
	    ret = priocntl(P_PID, getpid(), PC_SETPARMS, (char*)&pcp);
	    if (ret < 0) {
		int errnum = errno;
	        cerr << stamp(0)<<"Error setting priority: " 
		     << strerror(errnum) << endl;
	    }
	}
	mlockall(MCL_CURRENT | MCL_FUTURE);
	seteuid(sveuid);
    }
#endif

    //---------------------------------  Done, quit.
    return false;
}

//
//    Dpsh_Halt - set down.
//       * Release the shared memory partition.
//       * Close the connection to the CDS Network Data Server.
//       * Cancel the termination signal handling.
//
static void Dpsh_Halt(void) {

    //---------------------------------  Disable the signal
    term.zero();

    //---------------------------------  Close the NDS connection.
    if (NDS) delete NDS;

    //---------------------------------  Delete the partition
    if (OPart) delete OPart;

    //---------------------------------  Final statistics
    wstat.Log(cout);
    wstat.endStats(cout);
#ifndef P__DARWIN
    if (verbose) vstream.close();
#endif
}

//
//    Dpsh_Read - read the data.
//      *  Wait for a frame to arrive.
//      *  Read in the data
//      *  Pack data into a frame
//      *  Write frame to shared memory.
//
static void Dpsh_Read(void) {
    //---------------------------------  Wait for a data message
#ifdef MULTICNDS
    // bool rc = NDS->poll();  // this just purges a buffer
    // if (!rc) return;
#else
    int rc = NDS->WaitforData(); // allow a signal before data is received.
    if (rc <= 0) return;
#endif // defined(MULTICNDS)

    //---------------------------------  Get a buffer
    char* buffer = OPart->get_buffer();
    if (!buffer) {
	if (verbose) {
	    vstream << Now().getS() << " unable to get an shm buffer" << endl;
	}
	cout << stamp(0) << "Unable to get an shm buffer." << endl;
        active = false;
	return;
    }
    size_t maxlen =  OPart->getBufferLength();
    memset(buffer, 0, maxlen);

    //---------------------------------  Copy the data.
    unsigned int Frame, GPS, duration;
#ifdef MULTICNDS
    int length = NDS->receive(buffer, maxlen, &Frame, &GPS, &duration);
    int errnum = errno;

    if (length <= 0) {
	wstat.Error(length);
        OPart->return_buffer();
	switch (length) {
	case 0:
	    perror("error in framerecv");
	    cerr << stamp(0) << "Framerecv error 0: " 
		 << strerror(errnum) << endl;
	    return;
	case -1:
	    cerr << stamp(0) << "Socket assignment failed: " 
		 << strerror(errnum) << endl;
	    break;
	case -2:
	    cerr << stamp(0) << "Receive buffer allocation failed: " 
		 << strerror(errnum) << endl;
	    break;
	case -3:
	    if (errno == EINTR) {
		cout << "read interrupted by signal" << endl;
		return;
	    }
	    cerr << stamp(0) << "Unable to get packet: " 
		 << strerror(errnum) << endl;
	    break;
	case -5:
	    cerr << stamp(0) << "Invalid data array" << endl;
	    break;
	case -6:
	    cerr << stamp(0) << "Retransmission request failed." << endl;
	    break;
	default:
	    cerr << stamp(0) << "Frame length (" << -length 
		 <<") too long for buffer."<< endl;
	    if (verbose) {
		vstream << Now().getS() << " get " << GPS << " " << -length 
			<< " too long for buffer." << endl;
		vstream.flush();
	    }
	}
	active = false;
	return;
    }
    if (debug) {
        cout << "Frame read: length/sequence/GPS/duration = " << length 
	     << "/" << Frame << "/" << GPS << "/" << duration << endl;
    }
    if (verbose) {
	vstream << Now().getS() << " get " << GPS << " " << length;
    }

#else // if defined(MULTICNDS)
    DAQDRecHdr hdr;
#ifndef CDSDATA
    int length = NDS->GetFrame(buffer, OPart->getBufferLength(), &hdr);
#else // if !defined(CDSDATA)
    int length = NDS->GetData(buffer, OPart->getBufferLength());
    hdr = *((DAQDRecHdr*)buffer);
#endif // if !defined(CDSDATA)
    if (length <= 0) {
        OPart->return_buffer();
	cerr << stamp(0) << "Receive error, rc =" << length << endl;
	active = false;
	return;
    }
    Frame    = hdr.SeqNum;
    GPS      = hdr.GPS;
    duration = hdr.Secs;
#endif // if defined(MULTICNDS)

    //try {
    //    SF.Swap(buffer, length);
    //} catch (runtime_error& e) {
    //    cout << "Exception in SwapFrame: " << e.what() << endl;
    //	  active = false;
    //	  return;
    //}

    //----------------------------------  Check data
    if (!Dpsh_Check(length, buffer) && datacheck) {
        OPart->return_buffer();
	cerr << stamp(0) << "Data check failed on frame " << Frame << endl;
#ifdef MULTICNDS
	cerr << stamp(0) << NDS->logmsg() << endl;
#endif
	wstat.Error(-9);
    } else {
	if (verbose) {
	    vstream << " OK." << endl;
	    vstream << Now().getS() << " put " << GPS << " " << length;
	}
        OPart->SetID(GPS);
        OPart->release(length);
#ifdef MULTICNDS
	NDS->clearlog();
#endif
	if (verbose) {
	    vstream << " OK." << endl;
	    vstream.flush();
	}
    }

    //----------------------------------  Check for discontinuities
    wstat.Update(Frame, GPS, duration, cout);
}

//=======================     Frame Check Code      =======================
#define MAX_ID 256
#define LSTRUCTNAME 32

inline void
ReadInc(const char*& addr, unsigned short& x, bool bs) {
    if (bs) swab(addr, &x, sizeof(x));
    else    memcpy(&x, addr, sizeof(x)); 
    addr += sizeof(x);
}

typedef long long int8_type;
struct IGWhdr {
    unsigned long len;
    short id;
    int inst;
};

static bool  DefID[MAX_ID];
static short IDEOFile;
static short IDEOFrame;
static unsigned int hdrSz;
static unsigned int cksSz;

static bool
procSH(const IGWhdr& h, const char* addr, bool bswap) {
    unsigned short lname, ID, lcomment;
    unsigned int vblen = h.len - hdrSz - 6 - cksSz;

    addr += hdrSz;

    ReadInc(addr, lname, bswap);
    if ((lname < 2) || (lname > vblen)) {
        cout << "Inconsistent name length in SH, l=" << lname << endl;
	return true;
    }
    const char* name  = addr;
    addr += lname;

    ReadInc(addr, ID, bswap);

    ReadInc(addr, lcomment, bswap);
    if (lcomment != vblen-lname) {
        cout << "Inconsistent comment length in SH, l=" << lname << endl;
	return true;
    }
    addr += cksSz;
    if (!ID || ID > MAX_ID) {
        cout << "Illegal structure ID(" << ID << ") defined" << endl;
	return true;
    }
    DefID[ID] = true;
    if (!strncmp(name, "FrEndOf", 7)) {
        if (!strcmp(name, "FrEndOfFile"))       IDEOFile  = ID;
	else if (!strcmp(name, "FrEndOfFrame")) IDEOFrame = ID;
    }
    return false;
}

//
//  Dpsh_Check  Checks the incoming frame data structure headers for 
//              consistency.
//
//  The structure headers must meet the following requirements
//  1) The length must be greater than 8 (the length of the header) and 
//     less than the length of the Frame.
//  2) The structure type must be 1 or 2 or have been defined in a previous
//     SH structure. The Structure ID must be in the range 0 < ID < MAX_ID 
//     where MAX_ID is arbitrarily set to 256 (for now).
//  3) the structure instance numbers must increase by one with each new 
//     instance and start at either 0 or 1.
//
//  In addition the SH structures are unpacked and must be internally 
//  consistent (see procSH).
//
//  Dpsh_Check returns true if the frame passes the requirements.
//
static bool
Dpsh_Check(unsigned int length, const char* addr) {
#ifndef CDSDATA

    const char*   In   = addr;
    unsigned long left = length;
    IGWhdr h;

    //---------------------------------  Check the IGWD specifier
    if (left < 40 || strncmp("IGWD", In, 4)) {
        cerr << "Invalid IGWD header" << endl;
        if (verbose) {
	    vstream << "Invalid IGWD header" << endl;
	    vstream.flush();
	}
	return false;
    }

    //---------------------------------  See if we can use a checksum
    if (In[39] == 1) {
	checksum_crc32 chk;
	uint32_t crc32;
	chk.add((const unsigned char*)In, length - 4);
	memcpy(&crc32, In+length-4, sizeof(crc32));
	if (crc32 != chk.result()) {
	    cerr << "Invalid frame checksum" << endl;
	    if (verbose) {
		vstream << "Invalid frame checksum" << endl;
		vstream.flush();
	    }
	    return false;
	}
	return true;
    }

    //---------------------------------  Frame isn't checksummed. 
    //                                   Check the structure header integrity.
    bool bSwap = *((const short*) (In+12)) != 0x1234;
    int FrVsn = In[5];
    if (FrVsn > 4) hdrSz = 14;
    else           hdrSz = 8;
    if (FrVsn < 8) cksSz = 0;
    else           cksSz = 4;
    In   += 40;
    left -= 40;

    //---------------------------------  Initialize the instance counts
    int strinst[MAX_ID];
    for (int i=0 ; i<MAX_ID ; i++) {
        strinst[i] = 0;
	DefID[i] = false;
    }
    DefID[1] = true;
    DefID[2] = true;

    //---------------------------------  Loop over structures
    while (1) {
        if (left < sizeof(h)) {
	    cerr << "Unexpected EOD." << endl;
	    return false;
	}

	//------------------------------  Unpack V4 headers
	if (FrVsn <=4) {
	    struct {
	      int l;
	      short s;
	      short i;
	    } h4;

	    if (bSwap) {
	        swab(In, (char*)&h4, sizeof(h4));
		short* sptr = reinterpret_cast<short*>(&h4);
		short  ts = sptr[0];
		sptr[0]  = sptr[1];
		sptr[1]  = ts;
	    } else {
	        memcpy(&h4, In, sizeof(h4));
	    }
	    h.len  = h4.l;
	    h.id   = h4.s;
	    h.inst = h4.i;

	} 


	//------------------------------  Unpack V5-7 headers
	else if (FrVsn < 8) {
	    union {
	      int8_type l;
	      short     s[4];
	    } hlen;

	    union {
	      int    i;
	      short  s[2];
	    } hinst;

	    short hid;

	    if (bSwap) {
	        short hw[7];
	        swab(In, (char*)&hw, sizeof(hw));
		hlen.s[0]  = hw[3];
		hlen.s[1]  = hw[2];
		hlen.s[2]  = hw[1];
		hlen.s[3]  = hw[0];
		hid        = hw[4];
	        hinst.s[0] = hw[6];
	        hinst.s[1] = hw[5];		
	    } else {
	        memcpy(hlen.s,  In   , sizeof(hlen.l));
	        memcpy(&hid,    In+ 8, sizeof(hid));
	        memcpy(hinst.s, In+10, sizeof(hinst.i));
	    }

	    h.len  = hlen.l;
	    h.id   = hid;
	    h.inst = hinst.i;
	}
	else {
	    union {
	      int8_type l;
	      short     s[4];
	    } hlen;

	    union {
	      int    i;
	      short  s[2];
	    } hinst;

	    short hid;

	    if (bSwap) {
	        short hw[7];
	        swab(In, (char*)&hw, sizeof(hw));
		hlen.s[0]  = hw[3];
		hlen.s[1]  = hw[2];
		hlen.s[2]  = hw[1];
		hlen.s[3]  = hw[0];
		hid        = *(In+9);
	        hinst.s[0] = hw[6];
	        hinst.s[1] = hw[5];		
	    } else {
	        memcpy(hlen.s,  In   , sizeof(hlen.l));
	        hid = *(In+9);
	        memcpy(hinst.s, In+10, sizeof(hinst.i));
	    }

	    h.len  = hlen.l;
	    h.id   = hid;
	    h.inst = hinst.i;
	}

	if (h.len < hdrSz || h.len > left) {
	    cerr << "Illegal structure length " << h.len 
		 << " at offset " << length-left << endl;
	    return false;
	}

	if (h.id >= MAX_ID || !DefID[h.id]) {
	    cerr << "Illegal or undefined structure ID " << h.id 
		 << " at offset " << length-left << endl;
	    return false;
	}

	if (h.inst != strinst[h.id]++) {
	    if (h.inst == 1 && strinst[h.id] == 1) {
	        strinst[h.id]++;
#ifdef CHECK_INSTANCE_NUMBERS
	    } else {
	        cerr << "Invalid structure instance " << h.inst 
		     << " of ID " << h.id << ", expected " 
		     << strinst[h.id]-1 << " at offset " 
		     << length-left << endl;
		return false;
#endif
	    }
	}

	if ((h.id == 1) && procSH(h, In, bSwap)) {
	    cerr << "Invalid SH structure at offset " << length - left << endl;
	    return false;
	}

	In   += h.len;
	left -= h.len;
#ifndef FRAMELIB
	if (h.id == IDEOFrame) {
	    if (FrVsn >= 4) {
	        if (FrVsn>=6) for (int i=3 ; i<MAX_ID ; i++) strinst[i] = 0;
		else          for (int i=1 ; i<MAX_ID ; i++) strinst[i] = 0;
	    }
	}
#endif
	if (h.id == IDEOFile) break;
    }
    if (left) {
        cerr << "Data left after end-of-frame." << endl;
	return false;
    }
#endif  // !def(CDSDATA)
    return true;
}

static string 
stamp(time_t now) {
    if (now == 0) now = time(0);
    string datel(::ctime(&now)+4, 15);
    datel += " ";
    return datel;
}

//--------------------------------------  Statistics class methods
Stats::Stats(void) 
    : GPSfirst(0), GPSlast(0), RecIDlast(0), 
      totalFrames(0), totalSec(0), totalLost(0), totalUnsent(0), totalErrors(0)
{
    Reset();
}

Stats::~Stats() {
}

bool 
Stats::Test() {
  return watchdog;
}

void 
Stats::Log(ostream& out) {
    time_t now = time(0);
    double Rate;
    if (now == Stamp) Rate = 1.0;
    else              Rate = double(GPSlast - GPSstart) / double(now-Stamp);
    if (FrameLost || FrameUnsent || Rate > 1.02 || Rate < 0.98) {
        out << stamp(now) << " Statistics - Frames read: " 
	    << FrameCount << " Lost: " << FrameLost << " Unsent: " 
	    << FrameUnsent << " Rate: " << Rate << endl;
    }
    if (debug) {
        out << "Lock delay count = " << OPart->getLockCount() << endl;
	OPart->clrLockCount();
    }
    Reset();
    watchdog.clear();
}


void 
Stats::endStats(ostream& out) const {
    out << stamp(0) << "Start of data:      " << GPSfirst    << endl;
    out << stamp(0) << "Total frames read:  " << totalFrames << endl;
    out << stamp(0) << "Total frame errors: " << totalErrors << endl;
    out << stamp(0) << "Total data seconds: " << totalSec   << endl;
    out << stamp(0) << "End data GPS:       " << GPSlast    << endl;
    out << stamp(0) << "Dpush terminating. Error log closed." << endl;
}

void 
Stats::Reset(void) {
    GPSstart   = GPSlast;
    RecIDstart = RecIDlast;
    FrameCount = 0;
    FrameLost  = 0;
    FrameUnsent = 0;
    ::time(&Stamp);
}

void 
Stats::Start(ulong_t Secs) {
    struct itimerval ticks;

    ticks.it_value.tv_sec  = Secs;
    ticks.it_value.tv_usec  = 0;
    ticks.it_interval.tv_sec  = Secs;
    ticks.it_interval.tv_usec  = 0;
    nSecs = Secs;
    setitimer(ITIMER_REAL, &ticks, 0);
    watchdog.setMode(SigFlag::kBlock);
    watchdog.add(SIGALRM);
    time(&Stamp);
}

void 
Stats::Update(ulong_t FrameID, ulong_t GPS, ulong_t dt, ostream& log) {
    ulong_t nLost   = 0;
    ulong_t nUnsent = 0;
    if (!GPSlast) {
        log << stamp(0) << "First frame read, GPS: " << GPS 
	    << " Frame: " << FrameID << endl;
        GPSlast   = GPS;
	GPSfirst  = GPS;
	RecIDlast = FrameID;
	Reset();
    } else if (FrameID == 1) {
        log << stamp(0) << "Frame broadcaster restarted, first GPS= " << GPS
	    << " previous= " << GPSlast << endl;
    } else if (FrameID <= RecIDlast) {
        log << stamp(0) << "Frame received out of order, Frame=" << FrameID
	    << " expected " << RecIDlast+1 << endl;
    } else if (GPS <= GPSlast) {
        log << stamp(0) << "Frame received out of order, GPS=" << GPS 
	    << " expected " << GPSlast+dt << endl;
    } else {
        if (GPS != GPSlast+dt) {
	    log << "Missing frame(s) " << GPSlast+dt;
	    if (GPS-GPSlast>2*dt) log << "-" << GPS-dt;
	    log << endl;
	}
	if (FrameID > RecIDlast + 1) nLost = FrameID - RecIDlast - 1;
	ulong_t nExp = 0;
	if (GPS > GPSlast)    nExp    = (GPS - GPSlast)/dt;
	if (nExp > nLost + 1) nUnsent = nExp - 1 - nLost;
    }
    FrameLost   += nLost;
    FrameUnsent += nUnsent;
    FrameCount++;
    GPSlast   = GPS;
    RecIDlast = FrameID;
    totalFrames++;
    totalSec    += dt;
    totalLost   += nLost;
    totalUnsent += nUnsent;
}
