#ifndef LMSG_APPSERVER_HH
#define LMSG_APPSERVER_HH

#include "lmsg/MsgAddr.hh"
#include "lmsg/MsgHeader.hh"
#include "NameData.hh"
#include <map>
#include <string>

namespace lmsg {
  class BufferPool;
  class MsgHeader;
  class MsgHandler;
  class Message;
#ifdef TCP_TRANSPORT
  class TransportTCP;
#else
  class TransportMsg;
#endif
  class NameClient;


  /**  AppServer provides a server API for the LIGO/DMT messaging system.
    *  @memo Application server class
    *  @author John Zweizig.
    *  @version 1.2; Modified May 17, 2002
    */
  class AppServer {
  public:

    ///  %AppServer (open) state flag enumerator
    enum openflags {
	o_none,       ///< Server is not odefined
	o_register=1, ///< Server is registered
	o_async=2     ///< Server accpting asynchronous requests
	};

    /**  Construct an message server and optionally register it with the
      *  local name server.
      *  @memo Construct a server.
      *  @param Name Server name.
      *  @param ofl  Open flags.
      *  @param Typ  Server type.
      */
    AppServer(const char* Name=0, long ofl=o_none, lmsg::NameProcs Typ=p_Any);

    /**  Destroy the server. Unregister its name and release the 
      *  socket and addresss.
      *  @memo destroy the server.
      */
    virtual ~AppServer(void);

    /**  Add a message type to the server handler list, with a pointer to
      *  the appropriate handler.
      *  @memo Specify a message handler.
      *  @param type   Message type to be handled.
      *  @param handle Message handler pointer.
      */
    void addHandler(MsgHeader::MsgType type, MsgHandler* handle);

    /**  Close the socket used by the server and unregister the server
      *  name if one has been registered.
      *  \brief Close the server socket.
      *  \return lmsg error code.
      */
    error_type close(void);

    /**  Read in a message and process it with the appropriate handler.
      *  @memo Read a message and handle it.
      *  @return Lmsg error code.
      */
    error_type handleMessage(void);

    /**  Open a socket for this server and bind it to an available IP 
      *  address. Register the server name if one has been specified and 
      *  the o_register flag was set. SIGIO exceptions are also enabled
      *  on the socket if the o_async flag is set.
      *  @memo Open the server socket.
      *  @return Lmsg error code.
      *  @param flags Bit mask indicating optional processing to be performed.
      */
    error_type open(long flags);

    /**  Receive a message. The message header is returned in hdr and the 
      *  message text is returned in data. If no message is received in 
      *  'timeout' seconds receive will return with a TimeOut error.
      *  @memo Receive a message.
      *  @return Lmsg standard error code.
      *  @param hdr     Buffer to receive the mesage header.
      *  @param data    Buffer to receive the message text.
      *  @param length  Length of the message text buffer in bytes.
      *  @param timeout Maximum wait time in seconds.
      */
    error_type receive(MsgHeader& hdr, void* data, size_type length, 
		    double timeout);

    /**  Receive a message. The message header is returned in hdr and the 
      *  message text is returned in data. If no message is received in 
      *  within the period specified with setTimeOut, receive will return 
      *  with a TimeOut error.
      *  @memo Receive a message.
      *  @return Lmsg standard error code.
      *  @param hdr     Buffer to receive the mesage header.
      *  @param data    Buffer to receive the message text.
      *  @param length  Length of the message text buffer in bytes.
      */
    error_type receive(MsgHeader& hdr, void* data, size_type length);

    /**  A reply message is sent to the requesting client address with the
      *  correct transaction identifier. The request message header 
      *  should be used as the 'to' argument for the reply.
      *  @memo Reply to a client request.
      *  @return lmsg standard error codes.
      *  @param to   Header of the request message.
      *  @param data Reply message.
      */
    error_type reply(const MsgHeader& to, const Message& data);

    /**  A reply message is sent to the requesting client address with the
      *  correct transaction identifier. The request message header  
      *  should be used as the 'to' argument for the reply.
      *  @memo Reply to a client request.
      *  @return lmsg standard error codes.
      *  @param to   Header of the request message.
      *  @param rhdr Reply message header.
      *  @param data Pointer to the reply message text.
      */
    error_type reply(const MsgHeader& to, const MsgHeader& rhdr, 
		  const void* data);

    /**  Send a message to a peer process. The send method transmits messages 
      *  from the server port and unlike the client request (see AppClient)
      *  doesn't wait for a message in reply.
      *  @memo Send a message to a peer.
      *  @return Lmsg standard error codes.
      *  @param hdr Header containing destination address of message.
      *  @param msg Message to be sent.
      */
    error_type send(const MsgHeader& hdr, const Message& msg);

    /**  Send a message to a peer process. The send method transmits messages 
      *  from the server port and unlike the client request (see AppClient)
      *  doesn't wait for a message in reply.
      *  @memo Send a message to a peer.
      *  @return Lmsg standard error codes.
      *  @param hdr  Header of message to be sent.
      *  @param data Text of message to be sent.
      */
    error_type send(const MsgHeader& hdr, const void* data);

    /**  Wait for a message to arrive. If a message arrives in the specified
      *  maximum wait time, %waitMsg() returns true. Otherwise false is 
      *  returned. The message is left unread in the system input buffer. 
      *  %waitMsg() will return false when a signal is caaught before the 
      *  time expires. The wait time is specified as for the default timeout
      *  period: a zero time results in an immediate return and a negative
      *  time results in waiting indefinitely until a message is received or
      *  a signal is caught.
      *  @memo Wait for a message to arrive.
      *  @return true if the message has arrived.
      *  @param time Maximum time to Wait for a message.
      */
    bool waitMsg(wtime_type time);

    /**  Test whether the server socket is open.
      *  @memo Test is socket is open.
      *  @return true if socket is open.
      */
    bool isOpen(void) const;

    /**  Test whether the server socket name has been registered with the
      *  name server.
      *  @memo Test is server name is registered.
      *  @return true if socket is registered.
      */
    bool isRegistered(void) const;

    /**  Get the default maximum wait time in seconds. 
      *  @memo Get the default wait time.
      *  @return the default wait time in seconds.
      */
    wtime_type getTimeOut(void) const;

    /**  Get the current debug level.
      *  @memo Debug level.
      *  @return Current debugging level.
      */
    index_type getDebug(void) const;

    /**  Register the server with the given name. If the server is already 
      *  registered with a different name, it is unregistered and 
      *  reregistered with the new name.
      *  \brief Register the server name.
      *  \param server name to register.
      *  \param type   Server type to be sent to the name server.
      *  \return Error code or zero.
      */
    error_type register_name(const char* server, NameProcs type=p_Any);

    /**  Set the pointer to a buffer pool.
      *  \brief Set buffer pool address
      *  \param pool Buffer pool address
      */
    void setBufferPool(BufferPool* pool);

    /**  Set the debug printout level. If the server is open for business,
      *  (\e i.e. if a TransportMsg instance has been defined) the transport
      *  instance debug level is set also. The debug level may also be set
      *  at runtime by setting APPSERVER_DEBUG to the desired print level. 
      *  \brief Set debug level
      *  \param debug Debug printout level 
      */
    void setDebug(index_type debug);

    /**  Specify the name of the local name server.
      *  \brief Set the domain name
      *  \param domain Domain name string.
      */
    void setDomainName(const char* domain);

    /**  Specify the server address.
      *  \brief Set the server address.
      *  \param addr Server address.
      */
    void setServerPort(const MsgAddr& addr);

    /**  Set the default maximum wait time in seconds. If the wait time
      *  is set to zero, all functions using the default time will return
      *  TimeOut if a message is not immediately available. If the 
      *  time is negative, the functions using the default will not return
      *  until a message is received or until the function is interrupted
      *  by a signel.
      *  @memo Get the default wait time.
      *  @param time he default wait time in seconds.
      */
    void setTimeOut(wtime_type time);

  private:
    error_type register_name(NameClient& nc);
    error_type unregister(void);

  private:
    std::string   mDomainName;
    std::string   mServerName;
    NameProcs     mServerType;
    bool          mRegistered;
    BufferPool*   mBuffPool;
#ifdef TCP_TRANSPORT
    TransportTCP* mServer;
#else
    TransportMsg* mServer;
#endif
    wtime_type    mTimeout;
    MsgAddr       mPort;
    index_type    mDebug;
    typedef std::map<MsgHeader::MsgType, MsgHandler*> HandlerMap;
    typedef HandlerMap::const_iterator Handler_iterator;
    HandlerMap    mHandler;
  };
} // namespace lmsg

inline double
lmsg::AppServer::getTimeOut(void) const {
    return mTimeout;
}

inline lmsg::index_type
lmsg::AppServer::getDebug(void) const {
    return mDebug;
}

inline bool
lmsg::AppServer::isRegistered(void) const {
    return mRegistered;
}

#endif // LMSG_APPSERVER_HH
