/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "endtimes.hh"
#include "ParseLine.hh"
#include "Segment.hh"
#include <sstream>
#include <iostream>
#include <stdexcept>

using namespace std;

#define PIDCVSHDR "$Header$"

#define PIDTITLE  "Generate pre-lockloss segments"

#include "ProcIdent.hh"

EXECDAT(EndTimes)

//======================================  End times constructor
EndTimes::EndTimes(int argc, const char* argv[])
: DatEnv(argc, argv), _oscl(getDacc())
{
   string cfile = "EndTimes.cfg";
   string ofile = "LockLoss.osc";
   for (int i=1; i<argc; i++) {
      string argi = argv[i];
      if (argi == "-conf") {
	 cfile = argv[++i];
      }
      else if (argi == "-osc") {
	 ofile = argv[++i];
      }
      else if (isDatEnvArg(argv[i])) {
	i++;
      }
      else {
	 cerr << "EndTimes: unrecognized command line argument: " << argi
	      << endl;
	 finish();
      }
   }

   //-----------------------------------  Read the configuration
   if (configure(cfile)) {
      finish();
      return;
   }
   getDacc().setIgnoreMissingChannel(true);

   //-----------------------------------  Read the OSC list
   _oscl.readConfig(ofile);
}

//======================================  End times destructor
EndTimes::~EndTimes(void) {
}

//======================================  Process a data stride
void
EndTimes::ProcessData(void) {
   Time tStart = getDacc().getFillTime();
   Time tEnd   = tStart +  getDacc().getStride();
   for (etime_iter i=_list.begin(); i != _list.end(); ++i) {
      i->test(tStart, tEnd, _oscl, static_cast<TrigClient&>(*this));
   }
}
   
//======================================  Configure the monitor
bool
EndTimes::configure(const std::string& file) {
   _list.reserve(16);
   ParseLine pl(file.c_str());
   if (!pl.isOpen()) {
      cerr << "EndTimes: Unable to open configuration file: " << file << endl;
      return true;
   }
   while (pl.getLine() >= 0) {
      int narg = pl.getCount();
      if (!narg) continue;
      if (narg < 2) {
	 cerr << "EndTimes: Insufficient arguments" << endl;
	 return true;
      }
      end_test et(pl[0], Interval(pl.getDouble(1)));
      if (narg > 2 && string(pl[2]) != "-") et.set_segname(pl[2]);
      _list.push_back(et);
   }
   return false;
}

//======================================  end_test constructor
EndTimes::end_test::end_test(const std::string& osc_name, Interval delay)
   : _osc_name(osc_name), _delay(delay), _last_state(false), _last_time(0),
     _proc_time(0)
{
   ostringstream sn;
   sn << osc_name << "_PRE_" << delay;
   _seg_name = sn.str();
}

//======================================  end_test destructor
EndTimes::end_test::~end_test(void) {
}

//======================================  Book condition for one osc
void 
EndTimes::end_test::book(OperStateCondList& oscl) {
   if (!oscl.defined(_osc_name)) {
      throw runtime_error(string("OSC condition: ") + _osc_name + " not defined");
   }
}

//======================================  Set segment name
void
EndTimes::end_test::set_segname(const std::string& sn) {
   _seg_name = sn;
}

//======================================  Test a condition
void 
EndTimes::end_test::test(const Time& tStart, const Time& tEnd,
			 OperStateCondList& oscl, TrigClient& tc)
{
   //------------------------------------  Give up immediately if osc undefined
   if (!oscl.defined(_osc_name)) {
      cerr << "EndTimes: Undefined condition: " << _osc_name << endl;
      _proc_time = Time(0);
      return;
   }

   //------------------------------------  Test the state, make up for unknowns
   int now_state = oscl.satisfied(_osc_name) ? 1 : 0;
   if (_proc_time != tStart) {
      _last_time = tStart;
      _last_state = now_state;
   }

   //-----------------------------------  if the state condition is satisfied 
   //                                     Generate a segment to the red-line
   Time segt_start = _last_time;
   Time segt_end;
   int seg_active(0);
   if (now_state) {
      segt_end = tEnd - _delay;
   }

   //-----------------------------------  if the state is off, generate a
   //                                     segment from last to now
   else {
      segt_end   = tEnd;
      seg_active = _last_state;
   }

   //-----------------------------------  Send the segment if defined
   if (segt_end > segt_start) {
      trig::Segment seg(_seg_name, 1, segt_start, segt_end);
      seg.setActivity(seg_active);
      int rc = tc.sendSegment(seg);
      if (rc) {
	 cerr << "EndTimes: Failed to record segment: " << _seg_name
	      << " time: " << segt_start.getS() << "-" << segt_end.getS()
	      << " (rc=" << rc << ")" << endl;
      }
      _last_time  = segt_end;
      _last_state = now_state;
   }
   
   //-----------------------------------  Update process time
   _proc_time = tEnd;
}
