// C++ class definition for IIR filter
//
// Edward Daw, 2nd September 2000
#ifndef IIRfilter_bank_HH
#define IIRfilter_bank_HH 
#include <string>

/**  The filter bank class stores a set of IIR filters, and allows 
 *   retrieval of an filter coefficient from any second order section
 *   of any of the filters in the bank. The main function of this class
 *   is to allow a large number of filters to be read in from a single
 *   text file.
 *   @author  Edward Daw
 *   @version 1.1; Last modified October 4th, 2000
 */

class IIRfilter_bank {

public:
  /** Constructor from a text file.
   *  @param bank_filename A text string of maximum length 30 characters
   *  giving the path to a text file containing all the coefficients for
   *  all the filters in the bank. The format of the file is as follows:
   *  (int) number of filters \n
   *  (text 30 char max) name of filter 1 \n
   *  (int) sampling rate (Hz) (power of 2) for filter 1 \n
   *  (int) number of second order sections in filter 1 \n
   *  (doubles:) bo \t b1 \t b2 \t a1 \t a2 \n
   *  (doubles:) bo \t b1 \t b2 \t a1 \t a2 \n
   *  .....
   *  (text 30 char max) name of filter 2 \n
   *  (int) sampling rate (Hz) (power of 2) for filter 2 \n
   *  (int) number of second order sections in filter 2 \n
   *  (doubles:) bo \t b1 \t b2 \t a1 \t a2 \n
   *  (doubles:) bo \t b1 \t b2 \t a1 \t a2 \n
   *  .....
   *  ...etc, until done with all (no_of_filters) filters.
   *  @memo an example of an input file, octave_set_1.txt, is
   *  included with this distribution.
   */
  IIRfilter_bank(const std::string& bank_filename);
  
  /** Get order of a filter in the bank
   * @param filter_index 0 for first filter, 1 for 2nd...etc.
   * @param porder pointer where filter order will be written.
   */
  int get_order( int filter_index, int* porder);
  
  /** Get filter coefficient b0 from any filter in the bank
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param pb0 memory location where b0 is written.
   */
  int get_b0( int filter_index, int section_index, double* pb0 );
  /** Get filter coefficient b1 from any filter in the bank
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param pb0 memory location where b1 is written.
   */
  int get_b1( int filter_index, int section_index, double* pb1 );
  /** Get filter coefficient b2 from any filter in the bank
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param pb0 memory location where b2 is written.
   */
  int get_b2( int filter_index, int section_index, double* pb2 );
  /** Get filter coefficient a1 from any filter in the bank
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param pb0 memory location where a1 is written.
   */
  int get_a1( int filter_index, int section_index, double* pa1 );
  /** Get filter coefficient a2 from any filter in the bank
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param pb0 memory location where a2 is written.
   */
  int get_a2( int filter_index, int section_index, double* pa2 );

  /** Get filter sampling rate
   * @param filter_index 0 for first filter, 1 for second...etc.
   * @param psampling_rate memory location where sampling_rate is written.
   */
  int get_sampling_rate( int filter_index, int* psampling_rate );

  /** Get name of filter
   * @param filter_index 0 for first filter, 1 for second...etc.
   */
  char* get_filter_name( int filter_index );

  /** Destructor - frees memory associated with filter bank.
   */
  ~IIRfilter_bank( );

  int no_of_filters;

private:

  int get_filter_specs_from_file(const std::string& bank_filename,
				 int* pno_of_filters);
  
  IIRfilter* filter_data;
  unsigned int _bank_loaded;

};

#endif // IIRfilter_bank_HH	  
