//#############################################################################
//#  PhotonCal Calibration Montoring Program    v0.0
//#  
//#  By:  Peter Kalmus,   Feb 2006 <peter.kalmus@ligo.org> 
//#       Szabolcs Marka, Feb 2006 <smarka@gmail.com>
//#
//#############################################################################

#include "PhotonCal.hh"
#include "ParseLine.hh"

#ifndef __CINT__

//#include "ProcIdent.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include <cstdio>
#include <time.h>
#include <cmath>
#include <complex>
#include "TSeries.hh"
#include "FSeries.hh"
#include "DaccAPI.hh"
#include <string.h>
#include <cstdlib>
#include "FixedLenTS.hh"
#include "FDCalibrate.hh"
#include "Time.hh"

EXECDAT(PhotonCal)
#endif               // !def(__CINT__)

using namespace std;

char strideStr[40];
double stride               = 1.0;
char numAvgStr[40];
int numAvg                  = 1;

char numCyclesStr[40];
int numCycles               = 1;
int cycleCtr                = 1;
double runOcalXPropagated   = 0.0;
double runOcalYPropagated   = 0.0;
int goodStateFlag           = 0;
double firstStartTime       = 0.0;
double runPDarmErrX         = 0.0;
double runPDarmErrY         = 0.0;
double runNoiseX            = 0.0;
double runNoiseY            = 0.0;
double runPEtmxCal          = 0.0;
double runPEtmyCal          = 0.0;

char decimateStr[40];
int    decimateStages       = 2;
string calRefFileStr;
double readbackMin;


char channelGW[255];
char chEtmxCal[255];
char chEtmyCal[255];
char chState[255];

char IFO[2]; 

char webDir[255]            = "${HOME}/PhotonCal/html/index.html";
char LogDir[255]            = "${GDSAPACHE}/monitor_reports/PhotonCal/log/log.txt";
char TmpDir[255]            = "/tmp/PhotonCal/H1/PhotonCalTmpData.dat";
char LXO[4]                 = "L?O";
char COMM[255]              = "date; unsetenv DISPLAY; echo UNIX command here; date";
char CLEAN[255]             = "xxxCLEAN";
char RES[255]               = "xxxRES";

char dataServer[80]         = "xxxdataServer";
char trendTitle[80]         = "xxxtrendTitle";
char trendPrefix[80]              = "xxxtrendPrefix";
char trendChan_PCALX[80]     = "xxxtrendChanPCalX";
char trendChan_PCALY[80]     = "xxxtrendChanPCalY";
char trendChan_OCALX[80]     = "xxxtrendChanOCalX";
char trendChan_OCALY[80]     = "xxxtrendChanOCalY";
char trendChan_DARM_ERR[80] = "xxxtrendChanDarmErr";
char trendChan_ETMX_CAL[80] = "xxxtrendChanelETMX";
char trendChan_ETMY_CAL[80] = "xxxtrendChannelETMY";
char trendChan_LSC[80]      = "xxx";
char trendChan_ERR[80]      = "xxx";
char trendChan_QAF[80]      = "xxx";
char trendName[80]          = "xxxtrendName";

bool doEpics                = false;
bool propagateFlag          = true;
bool subtractNoiseFlag      = true;
char epicsName[80]          = "xxxepicsName";
char CaPutDir[256]          = "/cvs/cds/epics/extensions/bin/solaris";
char CaPutCom[256]          = "caput";
char EpicsComm[256]         = "";
char TTAG_Ch[40]            = "H0:DMT-PHOTONCAL_TTAG";
char ALIVE_Ch[40]           = "H0:DMT-PHOTONCAL_ALIVE";
char Alive(0);

int  ErrCode                = 0;
int  QualCode               = 0;
char AlarmMsg[255]          = "xxxalarmMsg";
char resultsFileName[80]    = "";

int  Fs                     = 16384;
FDCalibrate *calibrator;
char memleakStr[40];
int memleak                 = 0;

//ETMX/Y conversion constants, ETMX/Y photon calibrator frequencies
double RefRMagX;
double RefRPhaseX;
double RefCMagX;
double RefCPhaseX;
double RefRMagY;
double RefRPhaseY;
double RefCMagY;
double RefCPhaseY;
double Kx;
double Ky;
double Fx;
double Fy;

static const Interval HalfSec(0.51);

// for logging
ofstream* logger = 0;
ofstream* resultsWriter = 0;

// PhotonCal  constructor
PhotonCal::PhotonCal(int argc, const char *argv[])              
  : DatEnv(argc, argv),
    MonServer("PhotonCal_TEST"),
    MaxFrame(999999999), 
    mTrendMinute("PhotonCalTrend", Trend::kMinute, 720),
    mAlarm("PHOTONCAL")
{

  strncpy(LXO, getenv("LIGOSITE"), 3); 
  if (!(strcmp(LXO,"LLO") || strcmp(LXO,"LHO"))) {
    cerr << LXO << " cannot be identified as LIGO site! Check the $LIGOSITE environmental variable" << endl;
    exit(0);
  }

    if (DEBUG) {
      cout << "LIGO Site identified : " << LXO << endl;
    }
    if (logger) *logger << "LIGO Site identified : " << LXO << endl;

  if (!strcmp(LXO,"LHO")) {
    //strcpy(webDir, "${GDSAPACHE}/monitor_reports/PhotonCal/index.html");
    //strcpy(LogDir, "${GDSAPACHE}/monitor_reports/PhotonCal/log/log.txt");
    strcpy(webDir, getenv("HOME"));
    strcat(webDir, "/PhotonCal/lho/html/index.html");
    strcpy(LogDir, getenv("HOME"));
    strcat(LogDir, "/PhotonCal/lho/log/log.txt");
  }  

    const char* cfile = "photoncal.conf";
    for (int i=1 ; i<argc ; i++) {
        if (!strcmp("-conf", argv[i])) {
            cfile = argv[++i];
        }
    }
    mCFile = cfile;

    // Read configuration and set up
    if (ReadConfig()) {
        finish();
	    return;
    }

    for (int i=1 ; i<argc ; i++) {
            if (!strcmp("-h", argv[i])) {
               cout << "Usage: PhotonCal \n" 
            << "Command line arguments override config file parameters. \n"
            << "        [optional -h provides this help ] \n"
            << "        [required -i <ifo> specifies ifo e.g. H1, H2, L1 ] \n"
            << "        [optional -s <stride in seconds> integration length ] \n"
            << "        [optional -cycles <number of stride cycles to average> ] \n"
            << "        [optional -av <number of PSD averages> ] \n"
            << "        [optional -r <results file name> name tag for ASCII results file ] \n"
            << "        [optional -config <config file, default: photoncal.conf> ] \n"
            << "        [optional -infile <list of gwf files (can use * wildcard) > ] \n"
            << "        [optional -inlist <text TOC file with one gwf file per line> ] \n"
            << "        [optional -decimate <stages> number of powers of 2 to decimate (default is 2) ] \n"
            << "        [optional -noPropagate do not propagate conventional response ] \n"
            << "        [optional -t <trend output channel name prefix> ] \n"
            << "        [optional -l <log output directory/file> ] \n"
            << "        [optional -e <epics name> switch ON epics communication ] \n"
            << "        [optional -n <N> maximum number of frames to read ]\n" 
            << "        [optional -w <web output directory/file> (not implemented) ] \n" 
            << "Example: PhotonCal -conf 'pcal.conf' -e H1:DMT-PC_ -t H1:DMT-PCAL_ -i H1 \n"
            << endl;
           exit(0);
        } else if (!strcmp("-s", argv[i])) {
            strcpy(strideStr, argv[++i]);
            stride = atof(strideStr);
            if (DEBUG) {
                //logInfo("Stride (seconds) loaded from command line : " + string(stride));
                cout << "Stride (seconds) read from command line: " << stride << endl;
            }
            if (logger) *logger << "Stride (seconds) read from command line: " << stride << endl;
        } else if (!strcmp("-memleak", argv[i])) {
            strcpy(memleakStr, argv[++i]);
            memleak = atoi(memleakStr);
            if (DEBUG) {
                cout << "memleak hook: " << memleak << endl;
            }
        } else if (!strcmp("-cycles", argv[i])) {
            strcpy(numCyclesStr, argv[++i]);
            numCycles = atoi(numCyclesStr);
            if (DEBUG) {
                cout << "number of cycles read from command line: " << numCycles << endl;
            }
            if (logger) *logger << "number of cycles read from command line: " << numCycles << endl;
        } else if (!strcmp("-av", argv[i])) {
            strcpy(numAvgStr, argv[++i]);
            numAvg = atoi(numAvgStr);
            if (DEBUG) {
                cout << "number of averages read from command line: " << numAvg << endl;
            }
            if (logger) *logger << "number of averages read from command line: " << numAvg << endl;
        } else if (!strcmp("-decimate", argv[i])) {
            strcpy(decimateStr, argv[++i]);
            decimateStages = atoi(decimateStr);
            if (DEBUG) {
                //logInfo("Stride (seconds) loaded from command line : " + string(stride));
                cout << "Decimate stages read from command line: " << decimateStages << endl;
            }
            if (logger) *logger << "Stride (seconds) read from command line: " << stride << endl;
        } else if (!strcmp("-e", argv[i])) {
          strcpy(epicsName, argv[++i]);
          doEpics = true;
            if (DEBUG) {
                  cout << "Epics communication is on. " << epicsName << endl;
            }
        }       
        else if (!strcmp("-t", argv[i])) {
            strcpy(trendPrefix, argv[++i]);
            if (DEBUG) {
                cout << "Output trend channel prefix loaded from command line : " << trendPrefix << endl;
            }
        }
        else if (!strcmp("-i", argv[i])) {
            strcpy(IFO, argv[++i]);
            if (DEBUG) {
                cout << "IFO name loaded from command line : " << IFO << endl;
            }
        }
        else if (!strcmp("-n", argv[i])) {
            MaxFrame = strtol(argv[++i], 0, 0);
            if (DEBUG) {
                cout << "Maximum number of frames loaded from command line : " <<  MaxFrame << endl;
            }
        } 
        else if (!strcmp("-w", argv[i])) {
            strcpy(webDir, argv[++i]);
            if (DEBUG) {
                cout << "Web output is going to : " << webDir  << endl;
            }
        } 
        else if (!strcmp("-l", argv[i])) {
            strcpy(LogDir, argv[++i]);
            if (DEBUG) {
                cout << "Log output is going to : " << LogDir  << endl;
            }
        }
        else if (!strcmp("-r", argv[i])) {
            strcpy(resultsFileName, argv[++i]);
            if (DEBUG) {
                cout << "ASCII results file specifier loaded from command line : " << resultsFileName << endl;
            }
        }
        else if (!strcmp("-conf", argv[i])) {
            ++i;
                //we have already processed it, just don't give syntax error.
        }
        else if (!strcmp("-infile", argv[i])) {
            ++i;
                //let superclass handle it w/o terminating
        }
        else if (!strcmp("-inlist", argv[i])) {
            ++i;
                //let superclass handle it w/o terminating
        } else if (!strcmp("-noNoiseSubtract", argv[i])) {
            subtractNoiseFlag = false;
            if (DEBUG) {
                cout << "noNoiseSubtract read from command line: will not subtract noise from DARM_ERR measurements" << endl;
            }
            if (logger) *logger << "noNoiseSubtract read from command line: will not subtract noise from DARM_ERR measurements" << endl;
        } else if (!strcmp("-noPropagate", argv[i])) {
            propagateFlag = false;
            if (DEBUG) {
                cout << "noPropagate read from command line: will not propagate conventional response function." << endl;
            }
            if (logger) *logger << "noPropagate read from command line: will not propagate conventional response function." << endl;
        } else {
            cerr << "Error in command. Syntax:" << endl;
            cerr << argv[0] << " [-conf <cfile>] [-ndsindex [<file>]] "  << endl;
            cerr << "unrecognized option: " << argv[i] << endl;
            finish();
            return;
        }
    }  

    // set channel strings and IFO-specific variables
    if (!strcmp("H1", IFO)) {
        string channelGWStr = mDict.getString("H1GW");
        const char* channelGWChar = channelGWStr.c_str();
        strcpy(channelGW, channelGWChar);

        string chEtmxCalStr = mDict.getString("H1Calx");
        const char* chEtmxCalChar = chEtmxCalStr.c_str();
        strcpy(chEtmxCal, chEtmxCalChar);

        string chEtmyCalStr = mDict.getString("H1Caly");
        const char* chEtmyCalChar = chEtmyCalStr.c_str();
        strcpy(chEtmyCal, chEtmyCalChar);

        string chStateStr = mDict.getString("H1State");
        const char* chStateChar = chStateStr.c_str();
        strcpy(chState, chStateChar);

        RefRMagX = mDict.getDouble("H1RefRMagX");
        RefRPhaseX = mDict.getDouble("H1RefRPhaseX");
        RefCMagX = mDict.getDouble("H1RefCMagX");
        RefCPhaseX = mDict.getDouble("H1RefCPhaseX");
        RefRMagY = mDict.getDouble("H1RefRMagY");
        RefRPhaseY = mDict.getDouble("H1RefRPhaseY");
        RefCMagY = mDict.getDouble("H1RefCMagY");
        RefCPhaseY = mDict.getDouble("H1RefCPhaseY");

        Kx = mDict.getDouble("H1Kx");
        Ky = mDict.getDouble("H1Ky");
        Fx = mDict.getDouble("H1Fx");                                                  
        Fy = mDict.getDouble("H1Fy");

        calRefFileStr = mDict.getString("H1CalRefFile");

    } else if (!strcmp("H2", IFO)) {
        string channelGWStr = mDict.getString("H2GW");
        const char* channelGWChar = channelGWStr.c_str();
        strcpy(channelGW, channelGWChar);

        string chEtmxCalStr = mDict.getString("H2Calx");
        const char* chEtmxCalChar = chEtmxCalStr.c_str();
        strcpy(chEtmxCal, chEtmxCalChar);

        string chEtmyCalStr = mDict.getString("H2Caly");
        const char* chEtmyCalChar = chEtmyCalStr.c_str();
        strcpy(chEtmyCal, chEtmyCalChar);

        string chStateStr = mDict.getString("H2State");
        const char* chStateChar = chStateStr.c_str();
        strcpy(chState, chStateChar);

        RefRMagX = mDict.getDouble("H2RefRMagX");
        RefRPhaseX = mDict.getDouble("H2RefRPhaseX");
        RefCMagX = mDict.getDouble("H2RefCMagX");
        RefCPhaseX = mDict.getDouble("H2RefCPhaseX");
        RefRMagY = mDict.getDouble("H2RefRMagY");
        RefRPhaseY = mDict.getDouble("H2RefRPhaseY");
        RefCMagY = mDict.getDouble("H2RefCMagY");
        RefCPhaseY = mDict.getDouble("H2RefCPhaseY");

        Kx = mDict.getDouble("H2Kx");
        Ky = mDict.getDouble("H2Ky");
        Fx = mDict.getDouble("H2Fx");
        Fy = mDict.getDouble("H2Fy");

        calRefFileStr = mDict.getString("H2CalRefFile");

    } else if (!strcmp("L1", IFO)) {
        string channelGWStr = mDict.getString("L1GW");
        const char* channelGWChar = channelGWStr.c_str();
        strcpy(channelGW, channelGWChar);

        string chEtmxCalStr = mDict.getString("L1Calx");
        const char* chEtmxCalChar = chEtmxCalStr.c_str();
        strcpy(chEtmxCal, chEtmxCalChar);

        string chEtmyCalStr = mDict.getString("L1Caly");
        const char* chEtmyCalChar = chEtmyCalStr.c_str();
        strcpy(chEtmyCal, chEtmyCalChar);

        string chStateStr = mDict.getString("L1State");
        const char* chStateChar = chStateStr.c_str();
        strcpy(chState, chStateChar);

        RefRMagX = mDict.getDouble("L1RefRMagX");
        RefRPhaseX = mDict.getDouble("L1RefRPhaseX");
        RefCMagX = mDict.getDouble("L1RefCMagX");
        RefCPhaseX = mDict.getDouble("L1RefCPhaseX");
        RefRMagY = mDict.getDouble("L1RefRMagY");
        RefRPhaseY = mDict.getDouble("L1RefRPhaseY");
        RefCMagY = mDict.getDouble("L1RefCMagY");
        RefCPhaseY = mDict.getDouble("L1RefCPhaseY");

        Kx = mDict.getDouble("L1Kx");
        Ky = mDict.getDouble("L1Ky");
        Fx = mDict.getDouble("L1Fx");
        Fy = mDict.getDouble("L1Fy");

        calRefFileStr = mDict.getString("L1CalRefFile");

    } else {
       cerr << "Invalid ifo: " << IFO << "; choose [H1 H2 L1]" << endl;
       finish();
       return;
    }

    //initialize directories
    sprintf(COMM,   "mkdir -p /tmp/PhotonCal/%s", IFO);
    system(COMM);
    sprintf(COMM,  "cd /tmp/PhotonCal/%s", IFO);
    system(COMM);
    sprintf(TmpDir, "/tmp/PhotonCal/%s/PhotonCalTmpData.dat",IFO);
    sprintf(RES,    "/tmp/PhotonCal/%s/PhotonCalResults.txt",IFO);
    sprintf(RES,  "/home/pkalmus/PhotonCal/%sresults.txt",IFO);

    //initialize trends
    sprintf(dataServer, "PhotonCal_DataServer_%s",IFO);
    setServerName(dataServer);   
    sprintf(trendName, "PhotonCal_%s",IFO);
    mTrendMinute.setName(trendName);
    mTrendMinute.setFrameCount(1);   
    mTrendMinute.setType(Trend::kMinute);
    mTrendMinute.setAutoUpdate(false);	  
    sprintf(trendChan_PCALX, "%sPCALX_VALUE", trendPrefix);
    mTrendMinute.addChannel(trendChan_PCALX);
    sprintf(trendChan_PCALY, "%sPCALY_VALUE", trendPrefix);
    mTrendMinute.addChannel(trendChan_PCALY);
    sprintf(trendChan_OCALX, "%sOCALX_VALUE", trendPrefix);
    mTrendMinute.addChannel(trendChan_OCALX);
    sprintf(trendChan_OCALY, "%sOCALY_VALUE", trendPrefix);
    mTrendMinute.addChannel(trendChan_OCALY);
    sprintf(trendChan_DARM_ERR, "%sLSC_DARM_ERROR_BLRMS", trendPrefix);
    mTrendMinute.addChannel(trendChan_DARM_ERR);
    sprintf(trendChan_ETMX_CAL, "%sLSC_ETMX_CALIBRATION", trendPrefix);
    mTrendMinute.addChannel(trendChan_ETMX_CAL);
    sprintf(trendChan_ETMY_CAL, "%sLSC_ETMY_CALIBRATION", trendPrefix);
    mTrendMinute.addChannel(trendChan_ETMY_CAL);
    sprintf(trendChan_LSC, "%sLSC_CALIBRATION", trendPrefix);
    mTrendMinute.addChannel(trendChan_LSC);
    sprintf(trendChan_ERR, "%sErrorFlag", trendPrefix);
    mTrendMinute.addChannel(trendChan_ERR);
    sprintf(trendChan_QAF, "%sSignalQualityFlag", trendPrefix);
    mTrendMinute.addChannel(trendChan_QAF);

    if (DEBUG) {
        cout << "PhotonCal GW channel watched : " << channelGW << endl;
    }
    if (logger) *logger << "PhotonCal GW channel watched : " << channelGW << endl;

    if (DEBUG) {
        cout << "PhotonCal State channel watched : " << chState << endl;
    }
    if (logger) *logger << "PhotonCal State channel watched : " << chState << endl;

    //initialize dacc
    getDacc().setStride(stride);   
    getDacc().addChannel(channelGW);                                  // Add PhotonCal Channel
    getDacc().addChannel(chEtmxCal);                                  // Add PhotonCal Channel
    getDacc().addChannel(chEtmyCal);                                  // Add PhotonCal Channel
    getDacc().addChannel(chState);                                    // Add PhotonCal Channel
    cout << "channels added to Dacc." << endl;
    
    sprintf(trendTitle, "%s-PhotonCal-Minute-Trend",IFO);
    serveData(trendChan_PCALX,  &(mTrendMinute.find(trendChan_PCALX).refAvgSeries()), trendTitle);
    serveData(trendChan_PCALY,  &(mTrendMinute.find(trendChan_PCALY).refAvgSeries()), trendTitle);
    serveData(trendChan_OCALX,  &(mTrendMinute.find(trendChan_OCALX).refAvgSeries()), trendTitle);
    serveData(trendChan_OCALY,  &(mTrendMinute.find(trendChan_OCALY).refAvgSeries()), trendTitle);
    serveData(trendChan_DARM_ERR,  &(mTrendMinute.find(trendChan_DARM_ERR).refAvgSeries()), trendTitle);
    serveData(trendChan_ETMX_CAL,  &(mTrendMinute.find(trendChan_ETMX_CAL).refAvgSeries()), trendTitle);
    serveData(trendChan_ETMY_CAL,  &(mTrendMinute.find(trendChan_ETMY_CAL).refAvgSeries()), trendTitle);
    serveData(trendChan_LSC,       &(mTrendMinute.find(trendChan_LSC).refAvgSeries()), trendTitle);
    serveData(trendChan_ERR,       &(mTrendMinute.find(trendChan_ERR).refAvgSeries()), trendTitle); 
    serveData(trendChan_QAF,       &(mTrendMinute.find(trendChan_QAF).refAvgSeries()), trendTitle);
    cout << "data served." << endl;
    
    //define alarms
    //absent photon calibrator - both readbacks beolw minimum
    const char* alarmDir = "../monitor_reports/PhotonCal/";
    sprintf(AlarmMsg, "%s-Absent_Photon_Cal_Line",IFO);
    AlarmData al1("PHOTONCAL", AlarmMsg, Interval(120), 3, "Photon Calibrator Absent", "");
    al1.setWebFile(alarmDir);
    al1.jamFlags(AlarmData::kReTrigger);
    mAlarm.defineAlarm(al1);

    //sprintf(AlarmMsg, "%s-PHOTONCAL_SIGNAL_IS_NOT_PRESENT",IFO);
    //AlarmData al2("PHOTONCAL", AlarmMsg, Interval(120), 3, "No PHOTONCAL signal", "");
    //al2.setWebFile(alarmDir);
    //al2.jamFlags(AlarmData::kReTrigger);
    //mAlarm.defineAlarm(al2);    
    cout << "alarms defined." << endl;

    //initialize EPICS
    if (getenv("EPICSBINDIR") != 0) {
        strcpy(CaPutDir, getenv("EPICSBINDIR"));
        cout << "epics stuff set up. CaPutDir: " << CaPutDir << endl;
    } else {
        cerr << "EPICS Warning:  no $EPICSBINDIR environment variable defined.  Using CaPutDir: " << CaPutDir << endl;
    }

    //initialize workhorses
    black = new Blackman;
    psd = new PSD(black, numAvg);

    //initialize results logger
    string resultsFileNameTmp = string("");
    resultsFileNameTmp += IFO;
    resultsFileNameTmp += "results_";
    resultsFileNameTmp += resultsFileName;
    resultsFileNameTmp += ".txt";
	resultsWriter = new ofstream(resultsFileNameTmp.c_str(), ios::out);

    if (resultsWriter) *resultsWriter << "%stride: " << stride << endl;
    if (resultsWriter) *resultsWriter << "%cycles: " << numCycles << endl;
    if (resultsWriter) *resultsWriter << "%total integration: " << stride*numCycles << endl;
    if (resultsWriter) *resultsWriter << "%GW channel: " << channelGW << endl;
    if (resultsWriter) *resultsWriter << "%Fx: " << Fx << "  Kx: " << Kx << endl;
    if (resultsWriter) *resultsWriter << "%Fy: " << Fy << "  Ky: " << Ky << endl;
    if (resultsWriter) *resultsWriter << "%GPS          State        CalX           oCalXprop       oCalXref         CalY         oCalYprop       oCalYref" << endl;
    if (resultsWriter) *resultsWriter << "%-------------------------------------------" << endl;

    //initialize calibrator
    if (propagateFlag || memleak)  {
        const char* calRefFile = calRefFileStr.c_str();
        cout << "calRefFileStr: " << calRefFileStr << endl;
        //getDacc().setStride(30); 
        DaccAPI *dacc = &getDacc();
        calibrator = new FDCalibrate(dacc, calRefFile, true, 60.0, 1.0, 2000);
        //getDacc().setStride(stride); 
    }

    //initialize decimators
    cout << "decimateStages: " << decimateStages << endl;
    // the if statement causes "pure virtual method" error
    //if (decimateStages != 0) {
        gwdecimator = new DecimateBy2(decimateStages);
        xdecimator = new DecimateBy2(decimateStages);
        ydecimator = new DecimateBy2(decimateStages);
    //}

    cout << "constructor finished." << "\n===========================\n\n" << endl;
}

PhotonCal::~PhotonCal() 
{
    if (logger) {
        logger->close();
        delete logger;
    }
    if (resultsWriter) {
        resultsWriter->close();
        delete resultsWriter;
    }
    if (calibrator) {
        delete calibrator;
    }
    cout << "PhotonCal is finished" << endl;
}

void PhotonCal::Attention(void) {
  MonServer::Attention();
}

void PhotonCal::ProcessData(void) {   
    logInfo("ProcessData");    
    
    typedef short VecType;
    double DarmERRrms = 0;
    double Result = 0; 
    double calX = 0 ;
    double calY = 0 ;
    bool linePresent = false;
    time_t TS;
    Time GT;
    long int NanoSec;
    int   sec=0,min=0,hour=0,day=0;
    char  da[4],ho[2],mi[2],se[2];
    char  UTCtime[64];

    ErrCode  = 0;
    QualCode =0;

    TSeries* tDarmErrRaw = getDacc().refData(channelGW);
    TSeries* tEtmxCalRaw = getDacc().refData(chEtmxCal);
    TSeries* tEtmyCalRaw = getDacc().refData(chEtmyCal);
    TSeries* tState      = getDacc().refData(chState);

    double startTimeRaw = tDarmErrRaw->getStartTime().totalS();
    cout << "tDarmErrRaw start time: " << printf("%9.0f", startTimeRaw) << endl;

    //decimate
    TSeries *tDarmErr = tDarmErrRaw;
    TSeries *tEtmxCal = tEtmxCalRaw;
    TSeries *tEtmyCal = tEtmyCalRaw;

    // pure virtual method if decimator->apply goes inside if block...
    if (!gwdecimator->isDataValid(*tDarmErrRaw)) {
        gwdecimator->reset();
    }
    TSeries gwDecimated = gwdecimator->apply(*tDarmErrRaw);
    
    if (!xdecimator->isDataValid(*tEtmxCalRaw)) {
        xdecimator->reset();
    }
    TSeries etmxDecimated = xdecimator->apply(*tEtmxCalRaw);

    if (!ydecimator->isDataValid(*tEtmyCalRaw)) {
        ydecimator->reset();
    }
    TSeries etmyDecimated = ydecimator->apply(*tEtmyCalRaw);

    if (decimateStages != 0) {
        tDarmErr = &gwDecimated; 
        if (tEtmxCalRaw->getNSample() == Fs * stride) {
            tEtmxCal = &etmxDecimated;
            tEtmyCal = &etmyDecimated;
        } else {
            cout << "not decimating ETMX_CAL and ETMY_CAL, rds use suspected (already decimated)" << endl;
        }
    } else {
        cout << "not decimating." << endl;
    }

    //statistics
    int    nDarmErr = tDarmErr->getNSample();      // Number of samples in the segment
    double tstep =  tDarmErr->getTStep().GetSecs();
    double startTime = tDarmErr->getStartTime().totalS();
    if (cycleCtr == 1) {
        firstStartTime = startTime;
    }
    Time startTimeObj = tDarmErr->getStartTime();

    double ADarmErr = tDarmErr->getAverage();      // Determine the zero offset
    double MDarmErr = tDarmErr->getMaximum();      // Determine the maximum
    double IDarmErr = tDarmErr->getMinimum();      // Determine the minimum
    //double RDarmErr = MDarmErr - IDarmErr;            

    //int    nEtmxCal = tEtmxCal->getNSample();    // Number of samples in the segment
    //double AEtmxCal = tEtmxCal->getAverage();    // Determine the zero offset
    //double MEtmxCal = tEtmxCal->getMaximum();    // Determine the maximum
    //double IEtmxCal = tEtmxCal->getMinimum();    // Determine the minimum
    //double REtmxCal = MEtmxCal - IEtmxCal;            

    //int    nEtmyCal = tEtmyCal->getNSample();    // Number of samples in the segment
    //double AEtmyCal = tEtmyCal->getAverage();    // Determine the zero offset
    //double MEtmyCal = tEtmyCal->getMaximum();    // Determine the maximum
    //double IEtmyCal = tEtmyCal->getMinimum();    // Determine the minimum
    //double REtmyCal = MEtmyCal - IEtmyCal;           
 
    double AState   = tState->getAverage();

    //Determines the midtime of the segment
    // Takes care of proper rounding of seconds	 
    GT= tDarmErr->getBinT((int)Fs/2);
    NanoSec = (long int) (GT.getN()/1000);
    if (NanoSec > 500000) {
      GT = GT + HalfSec;
      NanoSec -= 1000000;
    }
    
    // Converts to human readable time	 
    if (DEBUG) {	 
        cout << "NanoSecs: " << NanoSec << endl;
    }    

    TimeStr(GT, UTCtime, "%D:%H:%N:%S");
    TimeStr(GT, da, "%D");
    day = atoi(da);
    TimeStr(GT, ho, "%H");
    hour = atoi(ho);
    TimeStr(GT, mi, "%N");
    min = atoi(mi);
    TimeStr(GT, se, "%S");
    sec = atoi(se);
	       
    // Restores correct time 			 
    if (NanoSec > 500000) {
      GT = GT - HalfSec;
    }

    ofstream TmpOut(TmpDir, ios::out | ios::binary);    
    if (!TmpOut) {
        cerr << "Unable to open data output buffer file !";
	ErrCode += 1;
    }

    //for (int i=0; i<nDarmErr; i++) {
      // TmpOut.write((char *)(&gDarmErr[i]), 2);
    //}

    //for (int i=0; i<nEtmxCal; i++) {
      //TmpOut.write((char *)(&gEtmxCal[i]), 2);
    //}

    //for (int i=0; i<nEtmyCal; i++) {
      //TmpOut.write((char *)(&gEtmyCal[i]), 2);
    //}

    TmpOut.flush();
    TmpOut.close();

    if (DEBUG) {
        cout << "Print timestamp... "  << endl;
        system("date");
    }

    // ####################################################################
    // Propagate response magnitude.
    // a good candidate for a separate thread - dare I go down that route?
    
    // this method relies on reference files...
    //calibrator->UpdateAlphaBeta(startTimeObj);

    double ocalXPropagated = 0.0;
    double ocalYPropagated = 0.0;

    if (propagateFlag) {
        calibrator->UpdateBeta();
        calibrator->UpdateAlpha();
        double alpha = calibrator->GetAlpha();
        double beta = calibrator->GetBeta();

        cout << "alpha: " << alpha << "; beta: " << beta << endl;

        // now propagate:  R = (1+ gamma(RC - 1))/(gamma*C)
        double gamma = alpha*beta;

        complex<double> respX = polar(RefRMagX, RefRPhaseX);
        complex<double> senseX = polar(RefCMagX, RefCPhaseX);
        ocalXPropagated = abs((polar(1.0, 0.0) + gamma*(respX*senseX-polar(1.0,0.0)))/(gamma*senseX));

        complex<double> respY = polar(RefRMagY, RefRPhaseY);
        complex<double> senseY = polar(RefCMagY, RefCPhaseY);
        ocalYPropagated = abs((polar(1.0, 0.0) + gamma*(respY*senseY-polar(1.0,0.0)))/(gamma*senseY));
        
    } else if (memleak == 1) {
        cout << "memleak 1" << endl;
        calibrator->UpdateBeta();
    } else if (memleak == 2) {
        cout << "memleak 2" << endl;
        calibrator->UpdateBeta();
        calibrator->UpdateAlpha();
    } else if (memleak == 3) {
        cout << "memleak 3" << endl;
        calibrator->UpdateBeta();
        calibrator->UpdateAlpha();
        // double alpha = calibrator->GetAlpha();
    } else if (memleak == 4) {
        cout << "memleak 4  " << endl;
        calibrator->UpdateBeta();
        calibrator->UpdateAlpha();
        // double alpha = calibrator->GetAlpha();
	// double beta = calibrator->GetBeta();
    }

    runOcalXPropagated += ocalXPropagated;
    runOcalYPropagated += ocalYPropagated;
    ocalXPropagated = runOcalXPropagated / (double)cycleCtr;
    ocalYPropagated = runOcalYPropagated / (double)cycleCtr;

    cout << "ref X: " << RefRMagX << "; propagated X: " << ocalXPropagated << endl;
    cout << "ref Y: " << RefRMagY << "; propagated Y: " << ocalYPropagated << endl;

    // ####################################################################
    // Check state vector
    
    if (AState != 65535.0) {
        goodStateFlag = 0;
    } else {
        goodStateFlag = 1;
    }

    if (goodStateFlag) {
        cout << "State vector always in science mode." << endl;
    } else {
        cout << "State vector not always science.  AState: " << AState << endl;
    }


    // ####################################################################
    //---Estimate amplitude of GW lines using sum of power in bins around line freq.
    //      Procedure:
    //       1. Decimate time series data (from dacc, already done)
    //       2. Compute PSD with Blackman window, 1 average.
    //       3. Line power = power in bin containing line freq, 
    //          + power in neighboring bins (3 bins total).
    //          - noise power in counted bins,
    //            estimated as average power in nearby bins.
    //       4. Line ampl = [ 2*dF*Line power ]^(0.5)
    //          where dF = n_windows/T = frequency resolution.

    //---Calcuate PSD.
    psd->generate(fdata, tDarmErr);
    double bw = fdata.getFStep();

    //will factor out, but useful for debugging
    double norm = sqrt( 2.0 * bw);

    //---Estimate line power using central bin plus one on either side:
    double pDarmErrX = fdata.getSum(Fx - bw, 3.0 * bw );
    double pDarmErrY = fdata.getSum(Fy - bw, 3.0 * bw );
cout << "runPDarmErrX before: " << runPDarmErrX << endl;
    runPDarmErrX += pDarmErrX;
cout << "runPDarmErrX after: " << runPDarmErrX << endl;
cout << "pDarmErrX before: " << pDarmErrX << endl;
    pDarmErrX = runPDarmErrX / (double)cycleCtr;
cout << "pDarmErrX after: " << pDarmErrX << endl;

    runPDarmErrY += pDarmErrY;
    pDarmErrY = runPDarmErrY / (double)cycleCtr;

    if (subtractNoiseFlag) {
        //---Estimate noise power/bin using 10 bins on either side:
        //will probably want to have some kind of low S/N alarm defined

        double noiseX = fdata.getSum(Fx - 15.0 * bw, 10.0 * bw );
        noiseX += fdata.getSum(Fx + 5.0 * bw, 10.0 * bw );
        noiseX /= 20.0;
        runNoiseX += noiseX;
        noiseX = runNoiseX / (double)cycleCtr;

        if (DEBUG) {
            cout << "pDarmErrX: " << pDarmErrX << "; noiseX pwr: " << 3 * noiseX << endl;
        }
        pDarmErrX = pDarmErrX - 3.0 * noiseX;

        double noiseY = fdata.getSum(Fy - 15.0 * bw, 10.0 * bw );
        noiseY += fdata.getSum(Fy + 5.0 * bw, 10.0 * bw );
        noiseY /= 20.0;
        runNoiseY += noiseY;
        noiseY = runNoiseY / (double)cycleCtr;

        if (DEBUG) {
            cout << "pDarmErrY: " << pDarmErrY << "; noiseY pwr: " << 3 * noiseY << endl;
        }
        pDarmErrY = pDarmErrY - 3.0 * noiseY;
    }

    //---Now calculate amplitudes of readback lines, which have high S/N.
    psd->generate(fdata, tEtmxCal);
    double pEtmxCal = fdata.getSum(Fx - bw, 3.0 * bw );
    runPEtmxCal += pEtmxCal;
    pEtmxCal = runPEtmxCal / double(cycleCtr);

    psd->generate(fdata, tEtmyCal);
    double pEtmyCal = fdata.getSum(Fy - bw, 3.0 * bw );
    runPEtmyCal += pEtmyCal;
    pEtmyCal = runPEtmyCal / double(cycleCtr);

    double aDarmErrX = 0.0;
    double aDarmErrY = 0.0;
    double aEtmxCal = 0.0;
    double aEtmyCal = 0.0;

    if (pDarmErrX > 0) {
        aDarmErrX = norm * sqrt(pDarmErrX);
    }
    if (pDarmErrY > 0) {
        aDarmErrY = norm * sqrt(pDarmErrY);
    }
    if (pEtmxCal > 0) {
        aEtmxCal = norm * sqrt(pEtmxCal);
    }
    if (pEtmyCal > 0) {
        aEtmyCal = norm * sqrt(pEtmyCal);
    }

    if (aDarmErrX > 0) {
        calX = Kx *  aEtmxCal / aDarmErrX;
    } else {
        calX = 0.0;
    }

    if (aDarmErrY > 0) {
        calY = Ky *  aEtmyCal / aDarmErrY;
    } else {
        calY = 0.0;
    }

    if (aEtmxCal < readbackMin && aEtmyCal < readbackMin) {
        linePresent = false;
    }
    else {
        linePresent = true;
    }
    time(&TS);

    if (DEBUG) {
        cout << "---------------------------------------" << endl;
        cout << "stride: " << stride << endl;
        cout << "cycleCtr: " << cycleCtr << endl;
        cout << "channelGW: " << channelGW << endl;
        cout << "chEtmxCal: " << chEtmxCal << endl;
        cout << "chEtmyCal: " << chEtmyCal << endl;
        cout << "Fx: " << Fx << "  Fy: " << Fy << endl;
        cout << "Kx: " << Kx << "  Ky: " << Ky << endl;
        cout << "*******aDarmErrX: " << aDarmErrX << endl;
        cout << "*******aEtmxCal: " << aEtmxCal<< endl;
        cout << "*******aDarmErrY: " << aDarmErrY << endl;
        cout << "*******aEtmyCal: " << aEtmyCal<< endl;
        cout << "*************************************PhotonCal: calX: " << calX << endl;
        cout << "*************************************PhotonCal: calY: " << calY << endl;
        cout << "Line Present = " << linePresent << endl;
        cout << "channelGW stats: tstep: " << tstep << "; startTime: " << startTimeObj << "; length: " <<  nDarmErr << "; average: " << ADarmErr << "; max: " << MDarmErr << "; min: " << IDarmErr << endl;
    }
 
    time_t rawtime;
    struct tm * timeinfo;
    time ( &rawtime );
    timeinfo = localtime ( &rawtime );

    if (logger) *logger << "---------------------------------------" << endl;
    if (logger) *logger << "stride: " << stride << endl;
    if (logger) *logger << "channelGW: " << channelGW << endl;
    if (logger) *logger << "chEtmxCal: " << chEtmxCal << endl;
    if (logger) *logger << "chEtmyCal: " << chEtmyCal << endl;
    if (logger) *logger << "Fx: " << Fx << "  Fy: " << Fy << endl;
    if (logger) *logger << "Kx: " << Kx << "  Ky: " << Ky << endl;
    if (logger) *logger << "*******aDarmErrX: " << aDarmErrX << endl;
    if (logger) *logger << "*******aEtmxCal: " << aEtmxCal<< endl;
    if (logger) *logger << "*******aDarmErrY: " << aDarmErrY << endl;
    if (logger) *logger << "*******aEtmyCal: " << aEtmyCal<< endl;
    if (logger) *logger << "*****************PhotonCal: calX: " << calX << endl;
    if (logger) *logger << "*****************PhotonCal: calY: " << calY << endl;
    if (logger) *logger << "Line Present = " << linePresent << endl;
    if (logger) *logger << "channelGW stats: tstep: " << tstep << "; startTime: " << startTimeObj << "; length: " <<  nDarmErr << "; average: " << ADarmErr << "; max: " << MDarmErr << "; min: " << IDarmErr << endl;
 
    // ####################################################################

    // write html output
    if (DEBUG) {
        if (logger) *logger << "HTML file: " << webDir << endl;
    }
	ofstream webOut(webDir , ios::out );
    webOut  << "<!doctype html public \"-//w3c//dtd html 4.0 transitional//en\">\n<html>\n"
            << "<head> \n"
			<< "<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\"> \n"
			<< "<meta name=\"Author\" content=\"Peter Kalmus\"> \n"
			<< "<meta name=\"GENERATOR\" content=\"Mozilla/4.78 [en] (Win98; U) [Netscape]\"> \n"
			<< "<title>PhotonCal at LXO</title> \n"
			<< "<meta http-equiv=\"Refresh\" content=\"60; URL=index.html\">\n"
			<< "<META HTTP-EQUIV=\"Pragma\" CONTENT=\"no-cache\">"
			<< "</head>\n"
			<< "<body>\n" 
			<< "\n<b><font color=\"#FF6600\"><font size=+4>PhotonCal</font></font></b> \n"
            << "<br><font color=\"#66AAAA\"><font size=+2>[ " 
            << asctime(localtime(&TS)) << " ( <b>" << asctime(gmtime(&TS)) << "</b>UTC) ]<br>"
			<< "\n<br> GW Channel Watched: "  << channelGW << "\n"
            << "\n<br> stride: " << stride << "\n"
            << "\n<br> cal factor x: " << calX << " at " << Fx << " Hz\n"
            << "\n<br> cal factor y: " << calY << " at " << Fy << " Hz\n"
            << "</font></font>"
            << endl;

    if (!linePresent) {
        webOut  << "\n<br><b><blink><font color=\"#FF0000\"><font size=+4WebViewLink>&nbsp;&nbsp;ERROR.  No Photon Calibrator Line Present.</font></font></blink></b><br>" << endl;
	} 
    webOut  << "\n<br>\n<hr SIZE=1 WIDTH=\"100%\"> \n"
			<< "<br><font size=-1>This page was last modified : "
			<< asctime(localtime(&TS)) << " ( " << asctime(gmtime(&TS)) << "(UTC) )" 
			<< " by PhotonCal</font> \n<br><font size=-1>" 
			<< "Contact: <a href=\"mailto:peter.kalmus@ligo.org\">Peter Kalmus</a>.</font> \n"
			<< "<br><b><tt><font color=\"#000099\"><font size=+1></font></font></tt></b>&nbsp; \n"
            << "</body>\n</html>\n"
			<< endl;

	webOut.close();

    // Cleanup of files in /tmp/PhotonCal...
    //system(CLEAN);

    if (DEBUG) {
        //cout << "Output directory was cleaned... "  << endl;
        system("date");
    }

    //Write minute trend
    //mTrendMinute.trendData(trendPrefix, GT, Result[3]);

    mTrendMinute.trendData(trendChan_PCALX, GT, calX);
    mTrendMinute.trendData(trendChan_PCALY, GT, calY);
    mTrendMinute.trendData(trendChan_OCALX, GT, ocalXPropagated);
    mTrendMinute.trendData(trendChan_OCALY, GT, ocalYPropagated);  

    mTrendMinute.trendData(trendChan_DARM_ERR, GT, DarmERRrms);
    mTrendMinute.trendData(trendChan_ETMX_CAL, GT, calX);
    mTrendMinute.trendData(trendChan_ETMY_CAL, GT, calY);
    mTrendMinute.trendData(trendChan_LSC,      GT, Result);
    mTrendMinute.trendData(trendChan_ERR,      GT, ErrCode);
    mTrendMinute.trendData(trendChan_QAF,      GT, QualCode);

    if (DEBUG) {
          cout << "Trend: " << trendPrefix << Result << " | " << ErrCode << " | " << QualCode << endl;
    }     

    //Send data to EPICS
    if (doEpics) {
        // EPICS communication by the minute...

        Alive+=128;
        ostringstream Comm;
        sprintf(EpicsComm, "%s/caput %sKEEPALIVE %d >> /dev/null 2>&1 &", CaPutDir, epicsName, Alive);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %sTTAG \" `tconvert now` \" >> /dev/null 2>&1 &", CaPutDir, epicsName);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %sDErms %4.3g >> /dev/null 2>&1 &", CaPutDir, epicsName, DarmERRrms);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %scalX %4.3g >> /dev/null 2>&1 &", CaPutDir, epicsName, calX);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %scalY %4.3g >> /dev/null 2>&1 &", CaPutDir, epicsName, calY);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %sERR_BITS %d >> /dev/null 2>&1 &", CaPutDir, epicsName, ErrCode);
        Comm << EpicsComm << endl;
        sprintf(EpicsComm, "%s/caput %sQA_BITS %d >> /dev/null 2>&1 &", CaPutDir, epicsName, QualCode);
        Comm << EpicsComm << endl;

        system(Comm.str().c_str());

        if (DEBUG) {
            cout << "Sent to EPICS: " << Comm.str().c_str() << endl;
            cout << "Did EPICS communication..." << endl;
        }	
      }

      // Set alarms if necessary
      lmsg::error_t rc = 0;
      AlarmHandle han;

      if ( !linePresent ) {
        //ostringstream param;
        //param << LXO << "x photon calibration is off by: " << calX << ends;
        
        sprintf(AlarmMsg, "%s-Absent_Photon_Cal_Line",IFO);
        AlarmData al1("PHOTONCAL", AlarmMsg, Interval(120), 3, "Photon Calibrator Absent", "");
        rc = mAlarm.setAlarm(al1, han);
        if (rc) cerr << "[PhotonCal] Error sending alarm: " << rc << endl;
        if (DEBUG) {
            cout << "ALARM: " << channelGW << " NO PHOTON CALIBRATOR LINE"  << endl;
         }	
      }

    //update cycle
    if (cycleCtr == numCycles) {
        char resultsOutput[300] = "";
        sprintf(resultsOutput, "%9.0f\t%i\t%1.3e\t%1.3e\t%1.3e\t%1.3e\t%1.3e\t%1.3e\t", firstStartTime, goodStateFlag, calX, ocalXPropagated, RefRMagX, calY, ocalYPropagated, RefRMagY);
        if (resultsWriter) *resultsWriter << resultsOutput << endl;
        cycleCtr = 1;
        runOcalXPropagated   = 0.0;
        runOcalYPropagated   = 0.0;
        goodStateFlag           = 0;
        firstStartTime       = 0.0;
        runPDarmErrX         = 0.0;
        runPDarmErrY         = 0.0;
        runNoiseX            = 0.0;
        runNoiseY            = 0.0;
        runPEtmxCal          = 0.0;
        runPEtmyCal          = 0.0;
    } else {
        cycleCtr += 1;
    }
}


//======================================  Read in the configuration file.
bool
PhotonCal::ReadConfig(void) {
    //Reset();

    //----------------------------------  Define and initialize parameters.
    mDict.addPar("DEBUG", int(1));
    mDict.addPar("Stride", double(1.0));
    mDict.addPar("ReadbackMin", double(0.0));
    
    mDict.addPar("H1Calx", "");
    mDict.addPar("H1Caly", "");
    mDict.addPar("H1GW", "");
    mDict.addPar("H1State", "");

    mDict.addPar("H1RefRMagX", double(0.0));
    mDict.addPar("H1RefRPhaseX", double(0.0));
    mDict.addPar("H1RefCMagX", double(0.0));
    mDict.addPar("H1RefCPhaseX", double(0.0));
    mDict.addPar("H1RefRMagY", double(0.0));
    mDict.addPar("H1RefRPhaseY", double(0.0));
    mDict.addPar("H1RefCMagY", double(0.0));
    mDict.addPar("H1RefCPhaseY", double(0.0));

    mDict.addPar("H1Kx", double(0.0));
    mDict.addPar("H1Ky", double(0.0));
    mDict.addPar("H1Fx", double(0.0));
    mDict.addPar("H1Fy", double(0.0));
    mDict.addPar("H1CalRefFile", "");

    mDict.addPar("H2Calx", "");
    mDict.addPar("H2Caly", "");
    mDict.addPar("H2GW", "");
    mDict.addPar("H2State", "");

    mDict.addPar("H2RefRMagX", double(0.0));
    mDict.addPar("H2RefRPhaseX", double(0.0));
    mDict.addPar("H2RefCMagX", double(0.0));
    mDict.addPar("H2RefCPhaseX", double(0.0));
    mDict.addPar("H2RefRMagY", double(0.0));
    mDict.addPar("H2RefRPhaseY", double(0.0));
    mDict.addPar("H2RefCMagY", double(0.0));
    mDict.addPar("H2RefCPhaseY", double(0.0));

    mDict.addPar("H2Kx", double(0.0));
    mDict.addPar("H2Ky", double(0.0));
    mDict.addPar("H2Fx", double(0.0));
    mDict.addPar("H2Fy", double(0.0));
    mDict.addPar("H2CalRefFile", "");

    mDict.addPar("L1Calx", "");
    mDict.addPar("L1Caly", "");
    mDict.addPar("L1GW", "");
    mDict.addPar("L1State", "");

    mDict.addPar("L1RefRMagX", double(0.0));
    mDict.addPar("L1RefRPhaseX", double(0.0));
    mDict.addPar("L1RefCMagX", double(0.0));
    mDict.addPar("L1RefCPhaseX", double(0.0));
    mDict.addPar("L1RefRMagY", double(0.0));
    mDict.addPar("L1RefRPhaseY", double(0.0));
    mDict.addPar("L1RefCMagY", double(0.0));
    mDict.addPar("L1RefCPhaseY", double(0.0));

    mDict.addPar("L1Kx", double(0.0));
    mDict.addPar("L1Ky", double(0.0));
    mDict.addPar("L1Fx", double(0.0));
    mDict.addPar("L1Fy", double(0.0));
    mDict.addPar("L1CalRefFile", "");


    //----------------------------------  Set up the line parser
    ParseLine lp(mCFile.c_str());
    if (!lp.isOpen()) {
        cerr << "Unable to open configuration file " << mCFile << endl;
	return true;
    }


    string outfile;
    outfile=string("photoncal");
	outfile += ".log";
	logger = new ofstream(outfile.c_str(), ios::out);
	lp.setLog(*logger);
    //} else if (Debug()) {
    //    lp.setLog(cout);
    //}

    for (int nArg=lp.getLine() ; nArg>=0 ; nArg=lp.getLine()) {

        //------------------------------  Empty command line
        if (!lp[0]) {
	    continue;

        //------------------------------  Parse parameter line
        } else {
	    for (int i=0 ; i<nArg ; i+=2) {
	      if (!mDict.exists(lp[i])) {
		    cout << "No such parameter: " << lp[i] << endl;
		    if (logger) *logger << "**** No such parameter: " << lp[i] << endl;
		    break;
		  } else {
		    if (Debug()) cout << "Set parameter: " << lp[i] << " to: " << lp[i+1] << endl;
		    mDict.setPar(lp[i], lp[i+1]);
		    }
          }
        }
    }


    //----------------------------------  Act on the parameter values.
    stride = Interval(mDict.getDouble("Stride"));
    readbackMin = mDict.getDouble("ReadbackMin");
    DEBUG = mDict.getInt("DEBUG");

    return false;
}

//======================================  Clear all lists.
void
PhotonCal::Reset(void) {
//not used yet

}

//=====================================  Print to stdout and logfile
void
PhotonCal::logInfo(string s) {
    cout << "info: " << s << endl;
    if (logger) {
        *logger << "info: " << s << endl;
    } else {
        cerr << "No log out put to log to..." << endl;
    }
}

void
PhotonCal::logError(string s) {
    cout << "error: " << s << endl;
    if (logger) {
        *logger << "error: " << s << endl;
    } else {
        cerr << "No log out put to log to..." << endl;
    }
}
