#ifndef PSLOCHAN_HH
#define PSLOCHAN_HH

#include "FixedLenTS.hh"
#include "Time.hh"
#include <string>

class Trend;
class MonServer;

/*  OutChan holds an output channel that serves a trend and/or a dmtviewer
 *  object. The standard channel name convention is enforced.
 */
class OutChan {
public:
  //====================================  Statistics class
  class stats_type {
  public:
    //----------------------------------  Constructor
    stats_type(void);

    //----------------------------------  Get average value
    double getAvg(void) const;

    //----------------------------------  Get standard deviation
    double getSigma(void) const;

    //----------------------------------  Get the number of entries
    long   getN(void) const;

    //----------------------------------  Get the total live time.
    Interval getDt(void) const;

    //----------------------------------  Update the statistics
    void update(double data, Interval dt);

  private:
    double   _Sumx;
    double   _Sumxx;
    long     _SumN;
    Interval _Sumt;
  };

public:
  /** Null channel constructor
    */
  OutChan(void);

  /** Named channel constructor.
    */
  OutChan(Trend& tr, const std::string&  chan="", const std::string& prefix="",
	  const std::string& suffix="");

  /**  Destructor.
    */
  virtual ~OutChan(void);

  /**  Add the specified data point to the trend and viewer series.
    */
  void addTrendChannel(void);

  /**  Add the specified data point to the trend and viewer series.
    */
  void fillData(const Time& t, double data, Interval dt);

  /**  Return the trend channel name associated with this OutChan as a 
    *  constant character pointer.
    *  @memo Get trend channel name.
    */
  const char* getTrendChannel(void) const;

  /**  Return the trend channel name associated with this OutChan as a 
    *  constant character pointer.
    *  @memo Get trend channel name.
    */
  const char* getViewerChannel(void) const;

  /**  Return a constant reference to the time series.
   */
  const TSeries& refTSeries(void) const;

  /**  Define the viewer series.
    */
  void serveViewerSeries(MonServer& mon, const char* comm=0);

  /**  Set the trend and viewer channel names based on the specified input
    *  channel name and a suffix. The input channel name: <ifo>:<sys>-<chan>
    *  is transformed to <ifo>:DMT-<monitor>_<sys>_<chan>_<suffix>.
    *  @memo  Set the channel names to a standardized derivative.
    *  @param chan Input channel from which the output names are derived
    *  @param suffix Output channel name suffix.
    */
  void setChannel(const std::string& chan, const std::string& suffix);

  /**  Set the history time-series to the data saved in the trends.
    *  @memo Set history data.
    *  @param tEnd End time of history to be inserted.
    *  @param dT   History series time bin.
    */
  void setHistoryData(const Time& tEnd, Interval dT);

  /**  Set the length of the history time-series.
    *  @memo Set history length.
    *  @param lhist Duration of history.
    */
  void setHistoryLen(Interval lhist);

  /**  Set the end time and increment of the history time-series.
    *  @memo Set history length.
    *  @param t0 History series end time.
    *  @param dt History time bin.
    */
  void setHistoryTime(const Time& t0, Interval dt);

  /**  Set the channel prefix (monitor name).
    */
  void setPrefix(const std::string& pfx);

  /**  Set the trend channel name as specified.
   */
  void setTrendChannel(const std::string& chan);

  /**  Set the trend object address.
   */
  void setTrend(Trend& trnd);

  /**  Set the viewer channel name as specified.
   */
  void setViewerChannel(const std::string& chan);

  /**  Get the statistics structure.
    */
  const stats_type& refStats(void) const;

private:
  Trend*        mTrend;
  std::string   mPrefix;
  std::string   mTrendChannel;
  std::string   mViewerChannel;
 
  FixedLenTS    mHistory;
  stats_type    mStats;
};

//======================================  Inline methods.
inline const char*
OutChan::getTrendChannel(void) const {
    return mTrendChannel.c_str();
}

inline const char*
OutChan::getViewerChannel(void) const {
    return mViewerChannel.c_str();
}

inline const TSeries& 
OutChan::refTSeries(void) const {
    return mHistory;
}

inline const OutChan::stats_type& 
OutChan::refStats(void) const {
    return mStats;
}

inline double 
OutChan::stats_type::getAvg(void) const {
    if (!_SumN) return 0.0;
    return _Sumx/_SumN;
}

inline long
OutChan::stats_type::getN(void) const {
    return _SumN;
}

inline Interval
OutChan::stats_type::getDt(void) const {
    if (!_SumN) return 0.0;
    return _Sumt;
}

#endif  // !defined(PSLOCHAN_HH)
