#ifndef WPIPE_READDATA_HH
#define WPIPE_READDATA_HH

// WREADDATA Read multiple channels of data from frame files
//
// WREADDATA finds and retrieves multiple channels of time series data from a set
// of frame files.  The data are specified by the frame file type, channel name,
// start time, and duration or stop time.  The necessary frame files are located
// using a file name caching scheme.
//
// WREADDATA is build on top of the READFRAMEDATA set of high level functions
// for reading data from frame files.
//
// usage:
//
//   [data, sampleFrequencies] = 
//     wreaddata(frameCache, channelNames, frameTypes, startTime, stopTime, 
//               timeShifts, debugLevel);
//
//   frameCache           frame file cache structure from LOADFRAMECACHE
//   channelNames         cell array of channel names
//   frameTypes           cell array of frame types
//   startTime            gps start time of data to extract
//   stopTime             gps stop time (or duration) of data to extract
//   timeShifts           vector of time shifts [seconds]
//   debugLevel           verboseness of debug output
//
//   data                 cell array of extracted data
//   sampleFrequencies    vector of sample frequencies
//
// The data is returned as a cell array of row vectors in the same order as in
// the cell array of channel names.
//
// WREADDATA retrieves data from the requested start time up to, but not
// including, the requested stop time, such that stop minus start seconds are
// retrieved.  Alternatively, the desired duration in seconds may be specified
// instead of the GPS stop time parameter.
//
// The optional time shift argument should be a vector with one element per
// channel that specifies an additional time shift to apply to the start time for
// each detector.  By convention, a positive time shift corresponds to a delay of
// the corresponding time series.  If no time offsets are specified, a default
// time shift of zero is assumed for all detectors.
//
// To avoid the effects of round-off error, the requested start time, stop
// time, and time shifts should all be aligned with samples of the input data
// stream.
//
// If it is unable to load the requested data, WREADDATA returns empty result
// vectors and zero sample frequency for the failed channel as well as issuing a
// warning if (debugLevel is set to 1 or higher.  By default, a debugLevel of
// unity is assumed.
//
// See also READFRAMEDATA, LOADFRAMECACHE, CREATEFRAMECACHE, FRGETVECT,
// WSEARCH, and WEXAMPLE.

// Shourov K. Chatterji <shourov@ligo.caltech.edu>

#include "wtypes.hh"

namespace wpipe {

  class wframecache;

  /** single channel entry.
   */
  void 
  wreaddata(const wframecache& frameCache, const std::string& channelNames, 
	    const std::string& frameTypes, const Time& startTime, 
	    const Time& stopTime, double timeShifts, int debugLevel, 
	    tser_vect& data);

  /** Multiple channel entry.
   */
  void 
  wreaddata(const wframecache& frameCache, const str_vect& channelNames, 
	    const str_vect& frameTypes, const Time& startTime, 
	    const Time& stopTime, const dble_vect& timeShifts, int debugLevel, 
	    tser_vect& data);
}

#endif // !defined(WPIPE_READDATA_HH)
