/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "wblock.hh"
#include "wcondition.hh"
#include "weventlist.hh"
#include "woutput.hh"
#include "wparameters.hh"
#include "wreaddata.hh"
#include "wresample.hh"
#include "wtile.hh"
#include "wtransform.hh"
#include "matlab_fcs.hh"
#include "lcl_array.hh"
#include <iostream>
#include <sstream>
#include <iomanip>

using namespace std;

// constant inf
static const double dInf = 1.0/0.0;

// bugs fixed: Injection time shift and data factors incorrect if data 
// bounced by incorrect state.

namespace wpipe {

   //=====================================  Process a block of data.
   void
   wblock::process(const Time& blockStartTime, const wtile&  tiling, 
		   const Time& eventTime, const wparameters& pars, 
		   const wframecache& frameCache, woutput& outputFiles, 
		   int debugLevel, str_vect& channelNames) 
   {

      /////////////////////////////////////////////////////////////////////////
      //                               read parameters                       //
      /////////////////////////////////////////////////////////////////////////
      // extract parameters from pars structure
      channelNames = pars.channelNames;
      str_vect frameTypes = pars.frameTypes;

      double fSample = pars.sampleFrequency;
      dble_vect timeShifts = pars.timeShifts;
      str_vect  injectionNames = pars.injectionNames;
      str_vect  injectionTypes = pars.injectionTypes;
      dble_vect injectionTimeShifts = pars.injectionTimeShifts;
      size_t numberOfChannels = pars.numberOfChannels();

      //--------------------------------  Get block start, stop times.
      Time blockStopTime = blockStartTime + Interval(pars.blockDuration);

      //--------------------------------  Generate time stamp
      Interval dTrans = tiling.transientDuration();
      ostringstream timestamp;
      timestamp << (blockStartTime + dTrans).getS() << "-" 
		<< long(pars.blockDuration - floor(2.0 * dTrans));
      outputFiles.update(timestamp.str(),  wparameters::validFileTypes);

      //--------------------------------  Report status
      if (debugLevel >= 1) {
	 cout << fixed << setprecision(2) 
	      << "  block start time:        " << blockStartTime.totalS() 
	      << endl
	      << "  block stop time:         " << blockStopTime.totalS()
	      << endl;
      }

      /////////////////////////////////////////////////////////////////////////
      //                       transform sky coordinate                      //
      /////////////////////////////////////////////////////////////////////////
      dble_vect coordinate;

#ifndef NO_COHERENT
      // if (the skyPosition parameter is not empty, determine coordinate
      if (!pars.skyPosition.empty()) {

	 // gps center time of block
	 blockCenterTime = blockStartTime + (pars.blockDuration / 2);
	 if (debugLevel >= 2) {
	    cout << "  block center time:       " << blockCenterTime << endl;
	 }

	 // convert sky coordinate into geocentric coordinate
	 wconvertskycoordinates coordinate(pars.skyPosition, blockCenterTime, 
					   pars.skyCoordinateSystem, 
					   "geocentric");

	 if (debugLevel >= 1) {
	    cout <<  "  sky coordinate:          [" << coordinate[1] << " " 
		 << coordinate[2] << endl;
	 }
      }
#endif

      /////////////////////////////////////////////////////////////////////////
      //                             read detector state                     //
      /////////////////////////////////////////////////////////////////////////
      vector<int>  chanIdent;

      //--------------------------------  No detector states are requested
      if (pars.stateNames.empty()) {
	 wlog(debugLevel, 1, "  no detector state constraints");
	 chanIdent.resize(numberOfChannels);
	 for (size_t i=0; i<numberOfChannels; i++) chanIdent[i] = i;
      }

      //--------------------------------  Scan the state values
      else {
	 // report status
	 wlog(debugLevel, 1, "  reading detector state");

	 // read detector data
	 tser_vect stateData;
	 wreaddata(frameCache, pars.stateNames, pars.stateTypes, 
		   blockStartTime, blockStopTime, timeShifts, 
		   debugLevel, stateData);

	 // bool array for tracking valid channels
	 bool_vect validChannels(numberOfChannels, true);
  
	 // loop over channels
	 for (size_t chanNumber=0; chanNumber<numberOfChannels; chanNumber++) {

	    //--------------------------   check for missing data
	    int mask = int(pars.stateMasks[chanNumber]);
	    if (stateData[chanNumber].empty()) {
	       if (debugLevel >= 1) {
		  cout << "    " << pars.stateNames[chanNumber] 
		       << ": state read error: Empty time-series." << endl;
	       }
	       validChannels[chanNumber] = false;
	       if (pars.errorOnStateError) {
		  error("error reading detector state");
	       }
	       continue;
	    }

	    //--------------------------   check for missing data
	    int min_stat = stateData[chanNumber].getMinimum();
	    if (min_stat >= 0 && min_stat < mask) {
	       if (debugLevel > 0) {
		  cout << "    " << pars.stateNames[chanNumber] 
		       << ": detector not in requested state, min (" 
		       << min_stat << ") < mask" << endl;
	       }
	       validChannels[chanNumber] = false;
	    }

	    //--------------------------  loop over data words, check state
	    else {
	       int N    = stateData[chanNumber].getNSample();
	       lcl_array<int> state_chan(N);
	       stateData[chanNumber].getData(N, state_chan);
	       for (int i=0; i<N; i++) {
		  if ((state_chan[i] & mask) != mask) {
		     validChannels[chanNumber] = false;
		     break;
		  }
	       }
	       if (!validChannels[chanNumber] && debugLevel > 0) {
		  cout << "    " << pars.stateNames[chanNumber] 
		       << ":   detector not in requested state" << endl;
	       }
	    }
	 }

	 //-----------------------------------  Remove channels failing checks
	 numberOfChannels = 0;
	 for (size_t i=0; i<validChannels.size(); i++) {
	    if (!validChannels[i]) continue;
	    chanIdent.push_back(i);
	    if (i != numberOfChannels) {
	       channelNames[numberOfChannels]   = channelNames[i];
	       frameTypes[numberOfChannels]     = frameTypes[i];
	       timeShifts[numberOfChannels]     = timeShifts[i];
	       injectionNames[numberOfChannels] = injectionNames[i];
	       injectionTypes[numberOfChannels] = injectionTypes[i];
	       injectionTimeShifts[numberOfChannels] = injectionTimeShifts[i];
	    }
	    numberOfChannels++;
	 }

	 //----------------------------------- Erase unavailable channels.
	 channelNames.erase(channelNames.begin()+numberOfChannels,
			    channelNames.end());
	 frameTypes.erase(frameTypes.begin()+numberOfChannels, 
			  frameTypes.end());
	 timeShifts.erase(timeShifts.begin()+numberOfChannels, 
			  timeShifts.end());
	 injectionNames.erase(injectionNames.begin()+numberOfChannels,
			      injectionNames.end());
	 injectionTypes.erase(injectionTypes.begin()+numberOfChannels,
			      injectionTypes.end());
	 injectionTimeShifts.erase(injectionTimeShifts.begin()
				   +numberOfChannels,
				   injectionTimeShifts.end());
      }    // end test for requested detector state

      // error if (there is no data after state checks
      if (channelNames.empty()) {
	 wlog(debugLevel, 1, "    no valid data to analyze");
	 return;
      }
  
      /////////////////////////////////////////////////////////////////////////
      //                         read detector data                          //
      /////////////////////////////////////////////////////////////////////////

      // report status
      wlog(debugLevel, 1, "  reading detector data");

      // read detector data
  
      tser_vect data;
      wreaddata(frameCache, channelNames, frameTypes, blockStartTime, 
		blockStopTime, timeShifts, debugLevel, data);


      //-------------- pre-process data
      string networkString;
      for (size_t chanNumber=0; chanNumber < numberOfChannels; chanNumber++) {
	 int chanID = chanIdent[chanNumber];

	 // test for missing detector data
	 if (data[chanNumber].empty()) error("error reading detector data");

	 // standardize channel names
	 channelNames[chanNumber]=strrep(channelNames[chanNumber], ";", ":");

	 // apply multiplicative factor to data if (specified
	 if (pars.dataFactors[chanID] != 1.0) {
	    data[chanNumber] *= pars.dataFactors[chanNumber];
	 }

	 // report data loaded
	 if (debugLevel >= 1) {
	    cout <<  "    " << channelNames[chanNumber] << ":        loaded x "
		 <<  pars.dataFactors[chanID] << endl;
	 }

	 // record analyzed network
	 if (chanNumber != 0) networkString += ",";
	 networkString += channelNames[chanNumber].substr(0,2);
      }

      /////////////////////////////////////////////////////////////////////////
      //                          read injection data                        //
      /////////////////////////////////////////////////////////////////////////

      // if (any injections are requested
      if (pars.injectionChannels() != 0) {
	 wlog(debugLevel, 1, "  reading injection data");

	 // read injection data
	 tser_vect injectionData;
	 wreaddata(frameCache, injectionNames, injectionTypes, 
		   blockStartTime, blockStopTime, injectionTimeShifts, 
		   debugLevel, injectionData);

	 // test for missing or inconsistent injection data
	 for (size_t chanNumber=0; chanNumber<injectionData.size(); 
	      chanNumber++) {
	    int chanID = chanIdent[chanNumber];

	    if (injectionData[chanNumber].empty()) {
	       error("error reading injection data");
	    }

	    if (injectionData[chanNumber].getTStep() != 
		data[chanNumber].getTStep()) {
	       error("inconsistent detector and injection data");
	    }

	    // Scale injection data and add it in.
	    double fact = pars.injectionFactors[chanID];
	    injectionData[chanNumber] *= fact;
	    data[chanNumber] += injectionData[chanNumber];

	    // report injection data loaded
	    if (debugLevel >= 1) {
	       cout <<  "    " << injectionNames[chanNumber] 
		    << ":        loaded x " << pars.injectionFactors[chanID] 
		    << endl;
	    }
	 }
      }  // end test for requested injections

      /////////////////////////////////////////////////////////////////////////
      //                             resample data                           //
      /////////////////////////////////////////////////////////////////////////

      wlog(debugLevel, 1, "  resampling data");

      // resample data
      tser_vect resdata = resample.wresample(data, fSample);

      /////////////////////////////////////////////////////////////////////////
      //                             condition data                          //
      /////////////////////////////////////////////////////////////////////////

      // report status
      wlog(debugLevel, 1, "  conditioning data");

      // condition data
      wcondition conditioner(resdata, tiling, pars.doubleWhiten);

      /////////////////////////////////////////////////////////////////////////
      //                         transform and threshold                     //
      /////////////////////////////////////////////////////////////////////////

      Time thresholdReferenceTime = blockStartTime + pars.blockDuration * 0.5;
      dble_vect thresholdTimeRange;
      dble_vect thresholdFrequencyRange;
      dble_vect thresholdQRange;

      // The original version combined the transformation and triggering.
      // I don't understand why this would be necessary except for memory 
      // management. this could be done in c++ but it certainly isn't a 
      // critical problem. If it should prove to be important at a later 
      // time, the channel-by-channel transform+thresholding can be 
      // implemented using the single channel transform and thresholding 
      // classes.
      wlog(debugLevel, 1, "  transforming data");
      dft_vect condDft, coeffs;
      conditioner.whitenedDFT(condDft);
      conditioner.coefficientDFT(coeffs);

      wtransform transforms(condDft, tiling, pars.outlierFactor, 
			    pars.analysisMode, channelNames, coeffs, 
			    coordinate);

      wlog(debugLevel, 1, "  thresholding data");

      //  Select tiles using event (and veto) threshold(s) calculated to 
      //  give a specified rate for white noise.
      //
      //  The threshold is set to yield the specified false event rate when 
      //  applied to all available frequencies and Qs of a white noise signal.
      //  It is not modified to account for restricted f, Q ranges. It is also
      //  only a rough estimate, and the result false event rate may vary 
      //  significantly depending on the quality of the data.
      //
      weventstack triggers, triggersThresholded, triggersDownselected;
      double eventThreshold = pars.eventThreshold;
      if (!eventThreshold) {
	 eventThreshold = tiling.threshold_from_rate(pars.falseEventRate);
      }

      double vetoThreshold = dInf;
      if (pars.vetoThreshold > 0) {
 	 vetoThreshold = pars.vetoThreshold;
      } else if (pars.falseVetoRate > 0) {
	 vetoThreshold = tiling.threshold_from_rate(pars.falseVetoRate);
      }

      //--------------------------------  Apply thresholds
      triggersThresholded.wthreshold(transforms, tiling, 
				     eventThreshold, thresholdReferenceTime, 
				     thresholdTimeRange, 
				     thresholdFrequencyRange, 
				     thresholdQRange, 
				     pars.maximumSignificants, 
				     pars.analysisMode, vetoThreshold, 
				     pars.uncertaintyFactor, 
				     pars.correlationFactor, debugLevel);

      // report significant tile numbers
      if (debugLevel >= 1) {
	 // triggersThresholded[0][0].dump(cout);
	 triggersThresholded.status(cout);
      }

      /////////////////////////////////////////////////////////////////////////
      //                        select significant triggers                  //
      /////////////////////////////////////////////////////////////////////////

      wlog(debugLevel, 1, "  selecting significant triggers");

      // excluding overlapping significant tiles
      triggersDownselected.wselect(triggersThresholded, pars.durationInflation,
				   pars.bandwidthInflation, 
				   pars.maximumTriggers, 
				   debugLevel);

      // replace triggers
      triggers.swap(triggersDownselected);

      // report downselected tile numbers
      if (debugLevel >= 1) {
	 // triggers[0][0].dump(cout);
	 triggers.status(cout);
      }

      /////////////////////////////////////////////////////////////////////////
      //                       write single detector triggers                //
      /////////////////////////////////////////////////////////////////////////

      if (outputFiles.enabled("DOWNSELECT")) {
      
	 // report status
	 wlog(debugLevel, 1, "  writing selected triggers");

	 // write triggers
	 triggers.writeEvents(outputFiles["DOWNSELECT"], pars.triggerFields, 
			      pars.triggerFormat);
    
      }

      /////////////////////////////////////////////////////////////////////////
      //                        veto inconsistent triggers                   //
      /////////////////////////////////////////////////////////////////////////
#ifndef NO_COHERENT
      // if veto is requested and analysis mode is coherent
      if (pars.applyVeto && pars.analysisMode == "coherent") {

	 // report status
	 wlog(debugLevel, 1, "  applying veto");

	 // apply null stream veto
	 eventstack triggersVetoed;
	 triggersVetoed.wveto(triggers, pars.durationInflation, 
			      pars.bandwidthInflation, 
			      vetoDurationFactor, vetoBandwidthFactor, 
			      maximumConsistents, debugLevel);
  
	 // replace triggers
	 triggers = triggersVetoed;

	 // report consistent tile numbers
	 if (debugLevel >= 1) {
	    triggersVetoed.status(cout);
	 }
      } // end test for apply veto
#endif // !defined(NO_COHERENT)

      /////////////////////////////////////////////////////////////////////////
      //                            cluster triggers                         //
      /////////////////////////////////////////////////////////////////////////
      weventstack clusters;
      if (pars.applyClustering || pars.bayesian) {

	 // report status
	 wlog(debugLevel, 1, "  applying clustering");

	 // apply clusterings
	 weventstack triggersClustered = triggers;

	 //----------------------------------  Density clustering
	 string method = pars.clusterMethod;
	 if (method == "density") {
	    clusters.wcluster(triggersClustered, pars.clusterRadius, 
			      pars.clusterDensity, pars.clusterSingles, 
			      pars.distanceMetric, pars.durationInflation, 
			      pars.bandwidthInflation, debugLevel);
	 }


	 //----------------------------------  Density clustering
	 else if (method == "hierarchical") {
	    clusters.wcluster(triggersClustered, pars.clusterLinkage, 
			      pars.clusterCriterion, pars.clusterThreshold, 
			      pars.distanceMetric, pars.durationInflation, 
			      pars.bandwidthInflation, debugLevel);
	 }

	 //----------------------------------  Density clustering
	 else {
	    error(string("Unknown cluster method: ")+method);
	 }

	 // replace triggers
	 triggers = triggersClustered;

	 // report cluster numbers
	 if (debugLevel >= 1) {
	    // clusters[0][0].dump(cout);
	    clusters.status(cout);
	 }

	 //////////////////////////////////////////////////////////////////////
	 //                         write cluster triggers                   //
	 //////////////////////////////////////////////////////////////////////

	 if (outputFiles.enabled("CLUSTER")) {
	    wlog(debugLevel, 1, "  writing cluster triggers");

	    // write triggers or clusters
	    if (pars.writeClusters) {
	       clusters.tsort();
	       clusters.writeEvents(outputFiles["CLUSTER"], pars.triggerFields,
				    pars.triggerFormat);
	    } else {
	       triggersClustered.tsort();
	       triggersClustered.writeEvents(outputFiles["CLUSTER"], 
					     pars.triggerFields, 
					     pars.triggerFormat);
	    }
	 }
      } // end test for apply clustering

      /////////////////////////////////////////////////////////////////////////
      //                         coincide triggers                           //
      /////////////////////////////////////////////////////////////////////////

#ifndef NO_COINCIDENCE
      bool coincidentsFound = false;

      // apply coincidence if (requested
      if ((pars.coincidenceNumber > 1) && (numberOfChannels > 1)) {

	 // report status
	 wlog(debugLevel, 1, "  coinciding triggers");

	 // apply coincidence
	 weventstack triggersCoincident;
	 triggersCoincident.wcoincide(triggers, 
				      pars.coincidenceNumber, 
				      pars.durationInflation,
				      pars.bandwidthInflation, 
				      pars.maximumCoincidents, 
				      debugLevel);

	 // replace triggers
	 triggers = triggersCoincident;
  
	 // cull coincident clusters based on coincident triggers
	 if (!clusters.empty()) {
	    weventstack clustersCoincident;
	    clustersCoincident.wcoincidecluster(triggersCoincident, clusters, 
						debugLevel);
      
	    // replace clusters
	    clusters = clustersCoincident;
	 }

	 // note if (any coincidents were found
	 if (!triggersCoincident.empty()) {
	    coincidentsFound = true;
	 }

	 // report coincidence numbers
	 if (debugLevel >= 1) {
	    for (size_t chanNumber=0; chanNumber < triggersCoincident.size();
		 chanNumber++) {
	       cout <<  "    " << triggersCoincident[chanNumber].channelName 
		    << ": " << triggersCoincident[chanNumber].size() 
		    << " coincidents" << endl;
	    }
	 }

	 //////////////////////////////////////////////////////////////////////
	 //                   write coincident triggers                      //
	 //////////////////////////////////////////////////////////////////////
  
	 if (outputFiles.enabled("COINCIDE")) {

	    // report status
	    wlog(debugLevel, 1, "  writing coincident triggers");

	    // write triggers or clusters
	    if (pars.applyClustering && pars.writeClusters) {
	       wwriteevents(clustersCoincident, outputFiles.COINCIDE, 
			    pars.triggerFields, pars.triggerFormat);
	    }
	    else {
	       wwriteevents(triggersCoincident, outputFiles.COINCIDE, 
			    pars.triggerFields, pars.triggerFormat);
	    }
	 }
      }    // end test for apply coincidence

      /////////////////////////////////////////////////////////////////////////
      // EVENT FOLLOWUP
      /////////////////////////////////////////////////////////////////////////

      // apply followup if (requested and coincident tiles and clusters have 
      // been found
      if (coincidentsFound) {

	 //////////////////////////////////////////////////////////////////////
	 // create the output event structure
	 //////////////////////////////////////////////////////////////////////
	 event.id = "Discrete Q-transform event structure";
	 event.network = networkString;
	 event.blockStartTime = blockStartTime;
	 event.blockStopTime = blockStopTime;
	 event.livetime = pars.blockDuration - 2 * tiling.transientDuration;
  
	 event.outputFields = {"network", "blockStartTime", "blockStopTime", 
			       "livetime"};

	 //////////////////////////////////////////////////////////////////////
	 // find candidate event cluster
	 //////////////////////////////////////////////////////////////////////

	 // find candidate cluster
	 wlog(debugLevel, 1, "  finding candidate cluster");

	 wfindcandidate candidateCluster(clustersCoincident, triggersClustered,
					 eventTime, pars, debugLevel);

	 event.time = candidateCluster.clusterTime;
	 event.frequency = candidateCluster.clusterFrequency;
	 event.duration = candidateCluster.clusterDuration;
	 event.bandwidth = candidateCluster.clusterBandwidth;

	 event.outputFields = {event.outputFields{:}, "time", "frequency", 
			       "duration", "bandwidth"};

	 //////////////////////////////////////////////////////////////////////
	 // FOLLOWUP ANALYSES
	 //////////////////////////////////////////////////////////////////////
#ifndef NO_FOLLOWUP
	 if (pars.bayesian || pars.xCoherentCheck || pars.maxEnt){
	    wlog(debugLevel, 1, "  Event followups:");

	    [event, skymap] = 
	       wfollowups(event, data, coefficients, tiling, 
			  blockStartTime, coordinate, candidateCluster, 
			  channelNames, pars, outputFiles, 
			  debugLevel);
	 }
#endif // !defined( NO_FOLLOWUP)

	 //////////////////////////////////////////////////////////////////////
	 // write event
	 //////////////////////////////////////////////////////////////////////
    
	 wlog(debugLevel, 1, "  writing event");

	 wwriteevents(event, outputFiles.EVENTS, event.outputFields, 
		      pars.triggerFormat);
      }
#endif // !defined(NO_COINCIDENCE)
   }


} // namespace wpipe
