/***************************************************************************
    File        : WindowingFilter Design.cpp
    Description : Creates a filter based on the Windowing method
 ---------------------------------------------------------------------------
    Begin       : Thu Dec 11 2003
    Author(s)   : Roberto Grosso
 ***************************************************************************/


// Project files
#include "WindowFilterDesign.h"



// Method: filter
//   Generates a filter using the default window which is Hamming
void
gwd::WindowFilterDesign::ComputeFilter(const unsigned int order,const std::vector<Complex>& freq,Vector& flt)
{
  // Create the default window
  Vector wnd;
  Window window;
  window.Hanning(order+1,wnd);

  // Create the filter
  ComputeFilter(wnd,freq,flt);
}

// Method: filter
//   Generates a filter using a window. The
//   window has to be specified
void
gwd::WindowFilterDesign::ComputeFilter(const WindowType wt,const unsigned int order,const std::vector<Complex>& freq,Vector& flt)
{
  // Create the Window
  Vector wnd;
  Window window;
  window.window(wt,order+1,wnd);
  // Create the filter
  ComputeFilter(wnd,freq,flt);
}

// Method: filter
//   Generates a filter using a window. The window
//   is specified as input parameter
void
gwd::WindowFilterDesign::ComputeFilter(const Vector& window,const std::vector<Complex>& freq,Vector& flt)
{
  // Set sizes
  unsigned int wSize = (unsigned int)window.size();
  unsigned int fSize = (unsigned int)freq.size();
  
  // The frequency response
  std::vector<Complex> ct(wSize/2+1);
  std::fill(ct.begin(),ct.end(),double(0));
  if (fSize <= (wSize/2+1))
    std::copy(freq.begin(),freq.end(),ct.begin());
  else
    std::copy(freq.begin(),freq.begin()+(wSize/2+1),ct.begin());
  
  // Time-shift: causality
  double dt = 0.5*wSize;
  double w0 = 2*gwd::PI/static_cast<double>(wSize/2+1);
  for (unsigned int nn = 0; nn < (unsigned int)ct.size(); nn++)
  {
    double val = w0*dt*nn;
    ct[nn] *= Complex(cos(val),-sin(val));
  }

  // Generate time-domain filter
  Fourier dft;
  dft.dft(ct,flt);
  // Normalize inverse fourier transform
  const double factor = 1./static_cast<double>(flt.size());
  gwd::Multiply<double>(flt,factor);

  // Apply window
  gwd::Multiply<double>(flt,window);
}


