/***************************************************************************
    File        : NormalDistribution.h
    Description : Normal distributed random number generator.
                  Adaption of the code by: non-uniform random number generators
                  by Agner Fog, sources stocc.zip at www.agner.org/random/
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/


/***************************************************************************
* A copy rigth note from the author of the non-uniforma number generator   *
* Agner Fog.                                                               *
*                                                                          *
*  2002, 2004 Agner Fog.                                                  *
* GNU General Public License www.gnu.org/copyleft/gpl.html                 *
****************************************************************************/

#ifndef __NORMALDISTRIBUTION_H
#define __NORMALDISTRIBUTION_H

/*! \file NormalDistribution.h
 *  Implemtes a Normal distributed random numbers. This class uses a
 *  random number generator of type Mersenne twister. The implementation is
 *  based on the the C++ library of non-uniform random number generators
 *  by Agner Fog, sources stocc.zip at www.agner.org/random/
 *   2001 - 2004 A. Fog.
 *  GNU General Public License www.gnu.org/copyleft/gpl.html
 *  \brief Implements a Gaussian random number generator.
 */

// Libs
#include <cmath>

// Project files
#include "RandomMersenne.h"

namespace gwd {

  class NormalDistribution {
  public:
    /** Default Constructor */
    NormalDistribution() : mNormal(0),mNormalValid(false) {}

    /** Destructor */
    ~NormalDistribution() {}

    // Methods
    /*!
    ** Normal distribution with mean m and standard deviation s
    ** @param m mean value
    ** @param s standard deviation
    ** @return double  normal distributed
    **/
    inline double Normal(const double m,const double s);

  private:
    RandomMersenne mRandom;
    double mNormal;
    bool mNormalValid;
  };


  // INLINE FUNTIONCS
  inline double
  NormalDistribution::Normal(const double m,const double s)
  {
    // normal distribution with mean m and standard deviation s
    double normal;          // first random coordinate (normal_x2 is member of class)
    double w;               // radius
    if (mNormalValid)
    { // we have a valid result from last call
      mNormalValid = false;
      return mNormal * s + m;
    }

    // make two normally distributed variates by Box-Muller transformation
    do {
      normal  = 2. * mRandom.Random() - 1.;
      mNormal = 2. * mRandom.Random() - 1.;
      w = normal*normal + mNormal*mNormal;
    }
    while (w >= 1. || w < 1E-30);

    w = std::sqrt(log(w)*(-2./w));

    // normal and mNormal are independent normally distributed variates
    normal *= w;  mNormal *= w;

    // save mNormal for next call
    mNormalValid = true;

    return normal * s + m;
  }

} // namespace gwd
#endif // __NORMALDISTRIBUTION_H
