// $Id: InspiralMon.cc 5580 2009-03-24 18:22:05Z john.zweizig $
#include "TSeries.hh"
#include "TriggerFiles.hh"
#include "InspiralMon.hh"
#include "xsil/MetaIO.hh"
#include <iostream>
#include <cstring>
#include <glob.h>

EXECSVR(InspiralMon)

using namespace std;

//======================================  Construct the server program
InspiralMon::InspiralMon(int argc, const char* argv[]) 
  : DMTServer(argc,argv), mDelay(0), mMinSnr(0.0), mMaxSnr(30.0), 
    mMaxChiSq(20.0), mChisqBins(1.0), mChisqDeltaSq(0.0)
{
  string mPath;

  // parse the command line arguments
  bool syntax = false;
  for ( int i = 1; i < argc; ++i )
  {
    string argi = argv[i];
    if (argi == "-path")
    {
      mPath = argv[++i];
    } 
    else if (argi == "-ifo")
    {
      mIFOName = std::string(argv[++i]);
    }
    else if (argi == "-delay")
    {
      mDelay = strtod(argv[++i], 0);
    }
    else if (argi == "-minsnr")
    {
      mMinSnr = strtod(argv[++i], 0);
    }
    else if (argi == "-maxsnr")
    {
      mMaxSnr = strtod(argv[++i], 0);
    }
    else if (argi == "-maxchisq")
    {
      mMaxChiSq = strtod(argv[++i], 0);
    }
    else if (argi == "-chisqdeltasq")
    {
      mChisqDeltaSq = strtod(argv[++i], 0);
    }
    else if (argi == "-numchisqbins")
    {
      mChisqBins = strtod(argv[++i], 0);
    }
    else if (isServerArg(argv[i]))
    {
      ++i;
    } 
    else
    {
      cerr << "Unrecognized command line argument: " << argi << endl;
      syntax = true;
    }
  }

  //---------------------------------  Check the command line syntax
  if ( mPath.empty() || mIFOName.empty() ) syntax = true;
  if ( syntax )
  {
    cerr << "Syntax error on command line" << endl;
    finish();
  }

  //---------------------------------  Set the server name
  string mInspiralMonName = "InspiralMon_" + mIFOName;
  InspiralMon::setServerName( mInspiralMonName.c_str() );
  cerr << "Set server name to " << mInspiralMonName << endl;

  //---------------------------------  Initialize updates
  cerr << "Delay from present time " << mDelay << " seconds" << endl;
  mStep = 60.0;
  mLength = 12.0 * 60.0 * 60.0;
  mLastUpdate = Now() - mDelay - mLength;

  //---------------------------------  Create the fixed length time series
  int tSeriesLength(int(mLength/mStep));
  
  float *zeroData = new float[tSeriesLength];
  memset( zeroData, 0, tSeriesLength * sizeof(float) );

  mTS_A.setMaxLen(mLength);
  mTS_N.setMaxLen(mLength);
  mTS_D.setMaxLen(mLength);
  mTS_T_mean.setMaxLen(mLength);
  mTS_T_min.setMaxLen(mLength);
  mTS_T_max.setMaxLen(mLength);

  mTS_A.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_N.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_D.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_T_mean.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_T_min.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_T_max.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );

  delete[] zeroData;

  //---------------------------------  Initialize the trigger files
  mTrigFiles = TriggerFiles(mPath, mLastUpdate.getS());

  //---------------------------------  Serve the time series
  serveData((mIFOName + "_Analyzed_Data").c_str(),      &mTS_A);
  serveData((mIFOName + "_Number_of_Triggers").c_str(), &mTS_N);
  serveData((mIFOName + "_Optimal_Distance").c_str(),   &mTS_D);
  serveData((mIFOName + "_Mean_SNR").c_str(),           &mTS_T_mean);
  serveData((mIFOName + "_Max_SNR").c_str(),            &mTS_T_max);
  serveData((mIFOName + "_Min_SNR").c_str(),            &mTS_T_min);

  //---------------------------------  Serve the histograms
  serveData((mIFOName + "_SnrSq_Hist_2hr").c_str(),  &mSHist2 );
  serveData((mIFOName + "_SnrSq_Hist_6hr").c_str(),  &mSHist6 );
  serveData((mIFOName + "_SnrSq_Hist_12hr").c_str(), &mSHist12 );
  serveData((mIFOName + "_ChiSq_Hist_2hr").c_str(),  &mCHist2 );
  serveData((mIFOName + "_ChiSq_Hist_6hr").c_str(),  &mCHist6 );
  serveData((mIFOName + "_ChiSq_Hist_12hr").c_str(), &mCHist12 );

}

//======================================  Termination entry
InspiralMon::~InspiralMon(void)
{
}

void 
InspiralMon::ProcessData(const Time& t)
{
  Time t0 = t - mDelay;
  if ( mDelay ) mTrigFiles.setEndTime(t0.getS());

  if (Debug()) cerr << "Starting cycle at " << t << endl;

  //--------------------------------  This should be parsed from the args
  string mTriggerTable = "sngl_inspiral";
  string mTriggerTime = "end_time";
  string mTriggerAmplitude = "snr";
  string mTriggerConsistency = "chisq";

  //---------------------------------  Extend the time series to now
  mTS_A.padUntil(t0, mStep);
  mTS_N.padUntil(t0, mStep);
  mTS_D.padUntil(t0, mStep);
  mTS_T_mean.padUntil(t0, mStep);
  mTS_T_min.padUntil(t0, mStep);
  mTS_T_max.padUntil(t0, mStep);

  if (Debug()) cerr << "Coverage is " << mTS_A.getStartTime() << " to " 
		    << mTS_A.getEndTime() << endl;

  //---------------------------------  Create the ts data pointers
  float* mTS_A_data = (float *) mTS_A.refData();
  float* mTS_N_data = (float *) mTS_N.refData();
  float* mTS_D_data = (float *) mTS_D.refData();
  float* mTS_T_mean_data = (float *) mTS_T_mean.refData();
  float* mTS_T_max_data = (float *) mTS_T_max.refData();
  float* mTS_T_min_data = (float *) mTS_T_min.refData();

  //---------------------------------  Update list of trigger files
  vector <string> mFileList;
  try {
    mFileList = mTrigFiles.newFiles();
  } catch (std::exception &e) {
    cerr << "Exception in TrigFiles::newFiles: " << e.what() << endl;
  }

  //---------------------------------  Process the new files
  xsil::MetaIO mio;
  for ( vector< string >::const_iterator iFile = mFileList.begin(); 
      iFile != mFileList.end(); ++iFile )
  {
    string mFile = *iFile;

    // read the search summary table (mandatory)
    mio.open(mFile.c_str(), "search_summary");

    if ( ! mio.is_open() )
    {
      cerr << "Unable to open search_summary table in file " << mFile << endl;
      finish();
    }

    while ( mio.getRow() )
    {
      Time tSummStart = Time(mio.getInt("out_start_time"));
      Time tSummEnd = Time(mio.getInt("out_end_time"));
      Time tNow;

      for ( Time tNow = tSummStart; tNow < tSummEnd; tNow += mStep )
        mTS_A_data[long(mTS_A.getBin(tNow))] = 1.0;
    }

    mio.close();

    // read the summ value table (optional)
    mio.open(mFile.c_str(), "summ_value");
    if ( mio.is_open() )
    {
      while ( mio.getRow() )
      {
        Time tSummStart = Time(mio.getInt("start_time"));
        Time tSummEnd = Time(mio.getInt("end_time"));

        for ( Time tNow = tSummStart; tNow < tSummEnd; tNow += mStep )
          mTS_D_data[long(mTS_A.getBin(tNow))] = mio.getFloat("value");
      }

      mio.close();
    }

    // read in the triggers (if they exist)
    mio.open(mFile.c_str(), mTriggerTable.c_str());
    if ( mio.is_open() )
    {
      while( mio.getRow() )
      {
        Time trigTime = Time(mio.getInt(mTriggerTime.c_str()));
        float snr = mio.getFloat(mTriggerAmplitude.c_str());
        float chisq = mio.getFloat(mTriggerConsistency.c_str());
        long bin = long(mTS_N.getBin(trigTime));

        // add the triggers to the histogram storage
        mSnrSqData.insert( mmsnr_t::value_type( trigTime.getS(), snr * snr ) );
        mChiSqData.insert( mmsnr_t::value_type( trigTime.getS(), 
              chisq / (mChisqBins + mChisqDeltaSq * snr * snr) ) );

        // count the triggers
        ++mTS_N_data[bin];

        // compute the mean
        if ( mTS_T_mean_data[bin] )
        {
          mTS_T_mean_data[bin] = (mTS_T_mean_data[bin] + snr) / 2.0;
        }
        else
        {
          mTS_T_mean_data[bin] = snr;
        }

        // store the max snr
        if ( snr > mTS_T_max_data[bin] ) mTS_T_max_data[bin] = snr;

        // store the min snr
        if ( ! mTS_T_min_data[bin] )
        {
          mTS_T_min_data[bin] = snr;
        }
        else
        {
          if ( snr < mTS_T_min_data[bin] ) mTS_T_min_data[bin] = snr;
        }
      }

      mio.close();
    }

  }

  //---------------------------------  Create the trigger histograms
  mmsnr_t::iterator iter;
  
  // erase all triggers older than 12 hours
  mmsnr_t::const_iterator sub = mSnrSqData.upper_bound((t0-mLength).getS());
  for ( iter = mSnrSqData.begin(); iter != sub; ++iter ) mSnrSqData.erase(iter);

  mmsnr_t::const_iterator cub = mChiSqData.upper_bound((t0-mLength).getS());
  for ( iter = mChiSqData.begin(); iter != cub; ++iter ) mChiSqData.erase(iter);

  // construct the three histograms
  double minSnrSq = mMinSnr * mMinSnr;
  double maxSnrSq = mMaxSnr * mMaxSnr;
  Histogram1 hS2("Signal to Noise Ratio 2hr", 500, minSnrSq, maxSnrSq );
  Histogram1 hS6("Signal to Noise Ratio 6hr", 500, minSnrSq, maxSnrSq );
  Histogram1 hS12("Signal to Noise Ratio 12hr", 500, minSnrSq, maxSnrSq );
  Histogram1 hC2("Chi Squared 2hr", 500, 0, mMaxChiSq );
  Histogram1 hC6("Chi Squared 6hr", 500, 0, mMaxChiSq );
  Histogram1 hC12("Chi Squared 12hr", 500, 0, mMaxChiSq );

  const unsigned long t2 = (t0 - Interval(2*60*60.0)).getS();
  const unsigned long t6 = (t0 - Interval(6*60*60.0)).getS();

  mmsnr_t::const_iterator siend = mSnrSqData.end();
  for ( iter = mSnrSqData.begin(); iter != siend; ++iter )
  {
    hS12.Fill(iter->second);
    if ( iter->first > t6 )
    {
      hS6.Fill(iter->second);
    }
    if ( iter->first > t2 )
    {
      hS2.Fill(iter->second);
    }
  }

  mmsnr_t::const_iterator ciend = mChiSqData.end();
  for ( iter = mChiSqData.begin(); iter != ciend; ++iter )
  {
    hC12.Fill(iter->second);
    if ( iter->first > t6 )
    {
      hC6.Fill(iter->second);
    }
    if ( iter->first > t2 )
    {
      hC2.Fill(iter->second);
    }
  }


  // timestamp the histograms
  hS2.SetTime(t0);
  hS6.SetTime(t0);
  hS12.SetTime(t0);
  hC2.SetTime(t0);
  hC6.SetTime(t0);
  hC12.SetTime(t0);

  // serve to the viewer
  mSHist2 = hS2;
  mSHist6 = hS6;
  mSHist12 = hS12;
  mCHist2 = hC2;
  mCHist6 = hC6;
  mCHist12 = hC12;

  //---------------------------------  Discard files older than 12 hours
  mTrigFiles.flushOld((t0-mLength).getS());

  if (Debug()) cerr << "Finished cycle at " << Now() << endl;
}
