/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef XSIL_STREAM_HH
#define XSIL_STREAM_HH

#include "xsil/xobj.hh"
#include <string>
#ifdef __GNU_STDC_OLD
#include <gnusstream.h>
#else
#include <sstream>
#endif

//  If USEILWDCHAR is set the MetaTables will generate columns with
//  type=ilwd:char instead of type=lstring. The stream will also escape
//  spaces in strings.
#define USEILWDCHAR

namespace xsil {

    /**  Unsigned character vector to implement ilwd:char data type. It 
      *  contains pointers to the data section and to a length word.
      *  @memo Unsigned character string.
      *  @ingroup IO_xsil
      */
    struct UCVec {
	/**  Default constructor.
	  */
	UCVec(void);

	/**  Copy constructor.
	  *  \param uv %UCVec to be copied.
	  */
	UCVec(const UCVec& uc);

	/**  Construct an unsigned character vector.
	  *  \brief Data constructor.
	  *  \param pN Pointer to the string length.
	  *  \param pD Pointer to the unsigned character string.
	  */
	UCVec(int* pN, unsigned char* pD);

	/**  Construct an unsigned character vector and allocate pspecified 
	  *  space.
	  *  \brief Data constructor.
	  *  \param N Pointer to the string length.
	  */
	UCVec(int N);

	/**  Destructor.
	  */
	virtual ~UCVec(void);

	/// Ownership flag
	bool own;

	/// Pointer to the number of bytes in the string.
	int* pNBytes;

	///  Pointer to the data string.
	unsigned char* pData;
	
    };

    inline
    UCVec::UCVec(int* pN, unsigned char* pD) 
	: own(false), pNBytes(pN), pData(pD) 
    {}

/**  An %XSIL stream may be an internal list of data or it may point to a
  *  local file or remote data.
  *  @memo %XSIL data stream class.
  *  @author J. Zweizig
  *  @version 1.1; Modified December 13, 1999
  *  @ingroup IO_xsil
  */
class Stream : public xobj {
public:

  /**  Construct an unnamed empty stream.
    *  @memo %XSIL Stream default Constructor.
    */
  Stream(void);

  /**  Construct an XML %Stream with the specified name and type attributes 
    *  and optionally fill it with the specified character string. 
    *  By default an empty stream is produced.
    *  @memo %XSIL Stream Constructor.
    *  \param Name %Stream name attribute.
    *  \param Type %Stream type attribute.
    *  \param Data Option starting data.
    */
  Stream(const char* Name, const char* Type=0, const char* Data=0);

  /**  Construct a %Stream that is an identical copy of the argument.
    *  \brief Copy constructor.
    *  \param x %Stream to be copied.
    */
  Stream(const Stream& x);

  /**  Destroy this stream and release all allocated data storage.
    *  \brief %Stream destructor.
    */
  ~Stream();

  /**  Copy the contents of the argument %Stream to this object.
    *  \brief Assignment operator.
    *  \param x Stream to be copied.
    *  \return reference to this %Stream.
    */
  Stream& operator=(const Stream& x);

  /**  Write the %Stream structure and data to an XML document.
    *  \brief Write an xml document.
    *  \param xout Xwriter to which stream data are to be written.
    */
  void Spew(Xwriter& xout) const;

  /**  Append a string to the current local stream contents. Each string is
    *  assumed to be a lexically complete token and is separated by the 
    *  current delimiter character from the preceding data. If the string 
    *  pushes the current line length beyond the maximum length, a new-line
    *  character is added and the white space is inserted so that the string
    *  will start at the current indent level.
    *  @memo Add a string to the current local stream.
    *  @param s Lexically complete token to be added to the stream.
    */
  void append(const std::string& s);

  /**  Create a identical copy of the current Stream.
    *  @memo Clone a stream.
    *  @return Pointer to the copy.
    */
  Stream* Clone(void) const;

  /**  Test for data in stream.
    *  @return true if the stream is empty.
    */
  bool empty(void) const;

  /**  Get the list delimiter string.
    *  @memo Get delimiter.
    *  @return List delimiter character.
    */
  const char* getDelimit(void) const;

  /**  Get binary data encoding type, \e e.g. "BigEndian" / "LittleEndian", 
    *  "base64".
    *  @memo Get binary encoding.
    *  \return Encoding type string pointer.
    */
  const char* getEncode(void) const;

  /**  Get the object type name string for the xobj interface. This is set 
    *  to "Stream" for the stream objects.
    *  \brief Object type name.
    *  \return Object type name string.
    */
  const char* getObjType(void) const;

  /**  Set the delimiter character.
    *  The delimiter character must be unique.
    *  @memo Set the delimiter character.
    *  @param Delim New delimiter character.
    */
  void delimit(char Delim);

  /**  Append a floating point entry to the stream string.
    *  @memo Append a float.
    *  @param x float point number to be added to the stream.
    */
  void Add(float x);

  /**  Append a string to the stream string.
    *  @memo Append a string.
    *  @param x     Character string to be appended to the stream.
    *  @param escsp Escape blanks.
    */
  void Add(const std::string& x, bool escsp=false);

  /**  Append a floating point entry to the stream string.
    *  @memo Append an integer.
    *  @param x Integer number to be added to the stream.
    */
  void Add(int x);

  /**  Append a floating point entry to the stream string.
    *  @memo Add a double float.
    *  @param x Double floating point number to be added to the stream.
    */
  void Add(double x);

  /**  Append an unsigned byte string.
    *  @memo Add a byte string.
    *  @param x reference to the byte string descriptor.
    */
  void Add(const UCVec& x);

  /**  Clear out the stream data.
    *  @memo Clear the stream data.
    */
  void Clear(void);

  /**  Fill stream from a floating point array.
    *  @memo Fill from floats.
    *  @param N Number of strings to be copied to the stream.
    *  @param x float point array to be converted and copied to the stream.
    *  @param perLine Number of data values to print per line.
    */
  void Fill(int N, const float* x, int perLine=0);

  /**  Fill stream from a double floating point array.
    *  @memo Fill from doubles.
    *  @param N Number of strings to be copied to the stream.
    *  @param x double array to be converted and copied to the stream.
    *  @param perLine Number of data values to print per line.
    */
  void Fill(int N, const double* x, int perLine=0);

  /**  Fill stream from an integer array.
    *  @memo Fill with integers.
    *  @param N Number of strings to be copied to the stream.
    *  @param x Integer array to be converted and copied to the stream.
    *  @param perLine Number of data values to print per line.
    */
  void Fill(int N, const int* x, int perLine=0);

  /**  Fill stream from a string array.
    *  @memo Fill from strings.
    *  @param N Number of strings to be copied to the stream.
    *  @param x String array to be copied to the stream.
    *  @param perLine Number of data values to print per line.
    */
  void Fill(int N, const std::string x[], int perLine=0);

  /**  Decode and copy base64 encoded data from the stream into a data array.
    *  The data are swapped if the specified byte ordering is different 
    *  than that of the node. The byte-swapping granularity is specified 
    *  by the \a dLen argument.
    *  @memo Fill from stream.
    *  @param data String array to be encoded into the stream.
    *  @param dLen Data granularity for use in byte swapping.
    *  @param N    Length of data to be encoded into the stream.
    *  @return Read error if true
    */
  bool decode64Data(void* data, int dLen, int N);

  /**  Fill stream with base64 encoded data.
    *  @memo Fill from string.
    *  @param data String array to be encoded into the stream.
    *  @param N    length of data to be encoded into the stream.
    */
  void encode64Data(const void* data, int N);

  /**  Open the stream for input.
   */
  void open(void);

  /**  Insert a line break at the current position.
    *  @memo Insert a line break.
    */
  void lineBreak(void);

  /**  Read \a N double float data words from the input stream.
    *  @brief Read \a N data words.
    *  @param data Double vector to receive requested data.
    *  @param N    Number of data words to read.
    *  @return     True if the requested data were read.
    */
  bool read(double      data[], int N);

  /**  Read \a N float data words from the input stream.
    *  @brief Read \a N data words.
    *  @param data Float vector to receive requested data.
    *  @param N    Number of data words to read.
    *  @return     True if the requested data were read.
    */
  bool read(float       data[], int N);

  /**  Read \a N integer data words from the input stream.
    *  @brief Read \a N data words.
    *  @param data Integer vector to receive requested data.
    *  @param N    Number of data words to read.
    *  @return True if the requested data were read.
    */
  bool read(int         data[], int N);

  /**  Read \a N character strings from the input stream.
    *  @brief Read \a N data strings.
    *  @param data String vector to receive requested data.
    *  @param N    Number of data strings to read.
    *  @return True if the requested data were read.
    */
  bool read(std::string data[], int N);


  /**  Make the stream a remote stream and define the file name.
    *  @memo Define file for remote stream.
    *  @param File File name or web address from which data are to be read.
    */
  void setRemote(const char* File);

  /**  Set the input data to the specified character string.
    *  @memo set stream data.
    *  @param Data pointer to a data string.
    */
  void setData(const char* Data);

  /**  Specify the encoding technique used for binary data.
    *  @memo  Specify the encoding technique.
    *  @param Code pointer to a string describing the encoding technique.
    */
  void setEncode(const char* Code);

  /**  Set the Maximum line size.
    *  @memo set line size.
    *  @param Size Maximum line size to be used when printing the stream.
    */
  void setSize(int Size);

private:
  /**  Expand the local stream capacity to the specified estimated length.
    */
  void estLength(std::string::size_type len);

  /**  Expand local stream capacity to allow for a current estimate of the 
    *  stream length. The estimate is arrived at by specifying a fraction
    *  complete and dividing the current stream data length by the estimated 
    *  fraction completed.
    *  @param frac Estimated fraction of data already written to the stream.
    */
  void estDone(double frac);

private:
  std::string mData;
  std::string mRemote;
  int    mLineSz;
  int    mMaxSz;
  std::string mDelimit;
  bool   mEndLine;
  std::string mEncoding;
  std::istream* mInput;
};   // xsil::Stream

}    //  namespace xsil

//----------------------------------------  Test if stream is empty
inline bool
xsil::Stream::empty(void) const {
    return mData.empty();
}

//----------------------------------------  Get the encoding technique
inline const char*
xsil::Stream::getEncode(void) const {
    return mEncoding.c_str();
}

//----------------------------------------  Get the delimiter character
inline const char*
xsil::Stream::getDelimit(void) const {
    return mDelimit.c_str();
}

#endif  // !defined(XSIL_STREAM_HH)
