/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: xsilFSpectrum   					*/
/*                                                         		*/
/* Module Description: Classes for LIGO-LW input/output to DMT objects.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* (testing-- R. Rahkola)       Initial revision                        */
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: none 						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Rauha Rahkola (509) 372-8141  (509) 372-8137  rrahkola@darkwing.     */
/*                                                        uoregon.edu   */
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 2001.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _GDS_XSILFSPECTRUM_HH
#define _GDS_XSILFSPECTRUM_HH

#include <complex>
#include <string>
#include <iosfwd>
//=============  xml common header file
#include "xml/Xsil.hh"
//=============  DMT C++ header files
#include "FSpectrum.hh"
#include "FSeries.hh"
#include "Complex.hh"

namespace xml {

/** \ingroup xml_handlers
    xsilFSpectrum converts a DMT FSpectrum object or DMT FSeries objecet
    to an output stream in LIGO_LW format as described in LIGO-T000067.
    @memo Writes FSpectrum in LIGO_LW format as described in LIGO-TOOOO67
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
    \{
  *************************************************************************/
   class xsilFSpectrum {
   protected:
   /// Indent level
      int              fLevel;
   /// Pointer to FSeries object
      const FSeries   *fData0;
   /// Pointer to FSpectrum object
      const FSpectrum *fData1;
   /// Subtype (FSpectrum or FSeries)
      const int        fSubtype;
   /// Write the FSpectrum data to output stream
      std::ostream& writeFSpectrum (std::ostream &os) const;
   /// Write the FSeries data to output stream
      std::ostream& writeFSeries (std::ostream &os) const;
   
   public:
   /// Constructor (no FSpectrum)
      xsilFSpectrum(int level = 1): fLevel(level),fData0(0),fData1(0),
      fSubtype(-1) {
      }
   /// Constructor for FSpectrum
      xsilFSpectrum(const FSpectrum* fs, int level = 1):
      fLevel(level),fData0(0),fData1(fs),fSubtype(1) {
      }
   /// Constructor for FSpectrum
      xsilFSpectrum(const FSpectrum& fs, int level = 1):
      fLevel(level),fData0(0),fData1(&fs),fSubtype(1) {
      }
   /// Constructor for FSeries
      xsilFSpectrum(const FSeries* fs, int level = 1):
      fLevel(level),fData0(fs),fData1(0),fSubtype(0) {
      }
   /// Constructor for FSeries
      xsilFSpectrum(const FSeries& fs, int level = 1):
      fLevel(level),fData0(&fs),fData1(0),fSubtype(0) {
      }
   /// Write the data to output stream
      std::ostream& write (std::ostream &os) const;
   };

   typedef xsilFSpectrum xsilFSeries;

/** Write FSpectrum or FSeries to an output stream.
    @param os output stream
    @param fs xsilFSpectrum object
    @memo XML output of FSpectrum
    @return output stream
  *************************************************************************/
   inline std::ostream& operator<< (std::ostream &os, const xsilFSpectrum &fs) {
      return fs.write(os); }

/** Handler for FSpectrum data types
    @memo FSpectrum datatype handler
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
  *************************************************************************/
   class xsilHandlerFSpectrum: public xsilHandler {
   protected:
    /// Pointer to DMT Object vector class
      std::vector<FSpectrum>     *fSpectrums;
    /// Pointer to DMT Object vector class
      std::vector<FSeries>       *fSeries;
    /// Name of data object
      std::string                 fName;
    /// Data subtype
      int                         fSubtype;
    /// Time (sec)
      unsigned long               fSec;
    /// Time (nsec)
      unsigned long               fNsec;
    /// Time interval (sec)
      double                      fDt;
    /// Minimum frequency (Hz)
      float                       fF0;
    /// Frequency step (Hz)
      double                      fDf;
    /// Number of averages
      int                         fAverages;
    /// Bandwidth
      double                      fBW;
    /// Flag for complex data
      bool                        fComplex;
    /// Data pointer
      float*                      fData;
    /// Data dimension 1 (length of each column)
      int                         fDim1;
    /// Data dimension 2 (number of columns)
      int                         fDim2;
   
   public:
    /// Constructor
      explicit xsilHandlerFSpectrum (std::vector<FSpectrum> *specs, 
                        std::vector<FSeries> *sers,
                        const attrlist *attr=0, bool ignore=false);
    /// Destructor
      virtual ~xsilHandlerFSpectrum();
    /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
    /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
    /// short parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
    /// int parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, const int& p, int N=1);
   #ifndef __CINT__
    /// long parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const long long& p, int N = 1) {
         return false; }
   #endif
    /// float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const float& p, int N = 1) {
         return false; }
    /// double parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, 
                        const double& p, int N=1);
    /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
    /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
    /// string parameter callback (must return true if handled)
      virtual bool HandleParameter(const std::string& name, 
                        const attrlist& attr, const std::string& p);
    /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name, const attrlist& attr,
                        unsigned long sec, unsigned long nsec);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, float* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, std::complex<float>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0);
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, double* x, 
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }			     
    /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name, std::complex<double>* x,
                        int dim1, int dim2=0, int dim3=0, int dim4=0) {
         return false; }
   };

   typedef xsilHandlerFSpectrum xsilHandlerFSeries;

/** Xsil FSpectrum handler query class.
    The query will return a handler if the data object type is FSpectrum.
    @memo Xsil FSpectrum handler query
    @author Written September 2001 by Rauha Rahkola
    @version 0.1
  *************************************************************************/
   class xsilHandlerQueryFSpectrum: public xsilHandlerQuery {
   protected:
    /// FSpectrum vector pointer
      std::vector<FSpectrum>     *fSpectrum;
    /// FSeries vector pointer
      std::vector<FSeries>       *fSeries;
   
   public:
    /// Constructor for FSeries
      explicit xsilHandlerQueryFSpectrum(std::vector<FSeries> &sers)
      : fSpectrum(NULL),fSeries(&sers) {
      }
    /// Constructor for FSpectrum
      explicit xsilHandlerQueryFSpectrum (std::vector<FSpectrum> &specs)
      : fSpectrum(&specs), fSeries(NULL) {
      }
    /// Constructor for both FSpectrum and FSeries
      xsilHandlerQueryFSpectrum (std::vector<FSpectrum> &specs, 
                        std::vector<FSeries> &sers)
      : fSpectrum(&specs), fSeries(&sers) {
      }
    /// Destructor
      ~xsilHandlerQueryFSpectrum (void) {
      }
    /// returns a handler for the specified object (or 0 if not)
      virtual xsilHandler* GetHandler(const attrlist& attr);
   };

   typedef xsilHandlerQueryFSpectrum xsilHandlerQueryFSeries;

  /** \} */

} // namespace xml

#endif // define GDS_XSILFSPECTRUM_HH




