#ifndef _LIGO_XSIL_H
#define _LIGO_XSIL_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Xsil							*/
/*                                                         		*/
/* Module Description: Utility classes for XML/XSIL input/output.	*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 10Jun00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: TLGXsil.html						*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include <iosfwd>
#include <iomanip>
#include <complex>
#include <string>
#include <map>
#include <vector>
#include "Time.hh"

#if defined(__SUNPRO_CC) && defined(G__DICTIONARY)
   namespace xml {}
   using namespace xml;
#endif

/** \defgroup IO_xml XML Package
  * \ingroup IO

    This package exports input and output functions for reading and
    writing LIGO light-weight data. The approach taken with this library 
    is to provide output manipulators which allow easy XML formatting of
    data and parameter objects on standard output streams. It is the
    callers responsibility to cycle through all the data objects
    which have to be written. A simple example may look like:
    \verbatim
    double	x[3] = {1, 3.2, 5.1};
    float	y[6] = {1.1, 2.2, 3.3, 4.4, 5.5, 6.6};
    cout << xsilHeader() << endl;
    cout << xsilDataBegin ("Result", "Spectrum") << endl;
    cout << xsilParameter<int> ("Averages", 12) << endl;
    cout << xsilParameter<bool> ("online", true) << endl;
    cout << xsilParameter<const char*> ("Channel", "H2:PEM-LVEA_SEISX") << endl;
    cout << xsilParameter<double> ("scanPoints", *x, 3) << endl;
    cout << xsilDataEnd<float> (6, y) << endl;
    cout << xsilTrailer() << endl;
    \endverbatim
    which produced the following output:
    \verbatim
    <?xml version="1.0"?>
    <!DOCTYPE LIGO_LW SYSTEM "http://www.cacr.caltech.edu/projects/ligo_lw.dtd">
    <LIGO_LW>
      <LIGO_LW Name="Result" Type="Spectrum">
        <Param Name="Averages" Type="int">12</Param>
        <Param Name="online" Type="boolean">1</Param>
        <Param Name="Channel" Type="string">H2:PEM-LVEA_SEISX</Param>
        <Param Name="scanPoints" Type="double" Dim="3">1 3.2 5.1</Param>
        <Array Type="float">
          <Dim>6</Dim>
          <Stream Encoding="BigEndian,base64">
    P4zMzUAMzM1AUzMzQIzMzUCwAABA0zMz
          </Stream>
        </Array>
      </LIGO_LW>
    </LIGO_LW>
    \endverbatim

    Input from a LIGO-LW file is handled differently. The user has
    to contstruct a XML parser object with a input stream and a set
    if data object callback routines. Whenever a new data object is
    encountered in the stream it is successively handled to the
    callback routines until a handler is found which will parse
    the object. This handler will return a 'callback' object which
    has methods to handle parameters, and the data associated with
    the object. The parser will call the parameter handling method
    for every new parameter belonging to the current data object and
    it will call the data handling method with the actual data 
    read from the stream.

    The classes are further divided into the following groups
    - \ref xml_constants  Constants
    - \ref xml_utils      Utility operators and manipulators.
    - \ref xml_ostream    Output stream manipulators.
    - \ref xml_parse      Xml parsing classes.
    - \ref xml_handlers   Handlers for DMT classes.

    \{
*/

/**  Expat based XSIL I/O functions.
  *  @brief Read/write Ligo light-weight data.
  *  @author Written June 2000 by Daniel Sigg
  *  @version 1.0
  */
namespace xml {

/**  \defgroup xml_constants Common xml constants.
  *  Constants used in the text generation and parsing.
  *  \brief Common xml constants.
  *  @author Written June 2000 by Daniel Sigg
  *  @version 1.0
  *  \{
  */

  /// Document header version.
   const char* const xmlVersion = "<?xml version=\"1.0\"?>";

  /// Document definition text
   const char* const xmlDocType = 
   "<!DOCTYPE LIGO_LW [\n"
   "<!ELEMENT LIGO_LW ((LIGO_LW|Comment|Param|Time|Table|Array|Stream)*)>\n"
   "<!ATTLIST LIGO_LW Name CDATA #IMPLIED Type CDATA #IMPLIED>\n"
   "<!ELEMENT Comment (#PCDATA)>\n"
   "<!ELEMENT Param (#PCDATA)>\n"
   "<!ATTLIST Param Name CDATA #IMPLIED Type CDATA #IMPLIED Dim CDATA #IMPLIED\n"
   "                Unit CDATA #IMPLIED>\n"
   "<!ELEMENT Table (Comment?,Column*,Stream?)>\n"
   "<!ATTLIST Table Name CDATA #IMPLIED Type CDATA #IMPLIED>\n"
   "<!ELEMENT Column EMPTY>\n"
   "<!ATTLIST Column Name CDATA #IMPLIED Type CDATA #IMPLIED Unit CDATA #IMPLIED>\n"
   "<!ELEMENT Array (Dim*,Stream?)>\n"
   "<!ATTLIST Array Name CDATA #IMPLIED Type CDATA #IMPLIED>\n"
   "<!ELEMENT Dim (#PCDATA)>\n"
   "<!ATTLIST Dim Name CDATA #IMPLIED>\n"
   "<!ELEMENT Stream (#PCDATA)>\n"
   "<!ATTLIST Stream Name CDATA #IMPLIED Type (Remote|Local) \"Local\"\n"
   "          Delimiter CDATA \",\" Encoding CDATA #IMPLIED Content CDATA #IMPLIED>\n"
   "<!ELEMENT Time (#PCDATA)>\n"
   "<!ATTLIST Time Name CDATA #IMPLIED Type (GPS|Unix|ISO-8601) \"ISO-8601\">\n"
   "]>";
   // "<!DOCTYPE LIGO_LW SYSTEM "
   // "\"http://www.cacr.caltech.edu/projects/ligo_lw.dtd\">";

   /// Ligo light-weight name
   const char* const xmlLigoLW = "LIGO_LW";

   /// XML tag: data container
   const char* const xmlContainer = "LIGO_LW";
   /// XML tag: table
   const char* const xmlTable = "Table";
   /// XML tag: time
   const char* const xmlTime = "Time";
   /// XML tag: parameter
   const char* const xmlParam = "Param";
   /// XML tag: table column
   const char* const xmlColumn = "Column";
   /// XML tag: array
   const char* const xmlArray = "Array";
   /// XML tag: array dimension
   const char* const xmlDim = "Dim";
   /// XML tag: array stream
   const char* const xmlStream = "Stream";

   /// XML flag: Name
   const char* const xmlName = "Name";
   /// XML flag: Type
   const char* const xmlType = "Type";
   /// XML flag: Flag
   const char* const xmlFlag = "Flag";
   /// XML flag: Comment
   const char* const xmlComment = "Comment";
   /// XML flag: Unit
   const char* const xmlUnit = "Unit";
   /// XML flag: Delimiter
   const char* const xmlDelimiter = "Delimiter";
   /// XML flag: Link
   const char* const xmlLink = "Link";
   /// XML flag: Reference
   const char* const xmlRef = "Ref";
   /// XML flag: Encoding
   const char* const xmlEncode = "Encoding";
   /// XML flag: Encoding scheme
   const char* const xmlEncodeStd = "BigEndian,base64";
   /// XML flag: Encoding scheme, big endian
   const char* const xmlEncodeBE = "BigEndian,base64";
   /// XML flag: Encoding scheme, little endian
   const char* const xmlEncodeLE = "LittleEndian,base64";

   /// XML object type: Test paramete
   const char* const xmlObjTypeTestParameters = "TestParameters";
   /// XML object type: Time series
   const char* const xmlObjTypeTimeSeries = "TimeSeries";
   /// XML object type: Frequency series
   const char* const xmlObjTypeFrequencySeries = "FrequencySeries";
   /// XML object type: Spectrum
   const char* const xmlObjTypeSpectrum = "Spectrum";
   /// XML object type: Spectrum
   const char* const xmlObjTypeTransferFunction = "TransferFunction";
   /// XML object type: Spectrum
   const char* const xmlObjTypeCoefficients = "Coefficients";
   /// XML object type: Histogram
   const char* const xmlObjTypeHistogram = "Histogram";
   /// XML object type: Parameter settings
   const char* const xmlObjTypeSettings = "Settings";
   /// XML object type: Results
   const char* const xmlObjTypeResult = "Result";

   /// XML Tag close character
   const char* const xmlTagClosing = ">";

  /// XML name attribute text
   const char* const xmlAttrName = " Name=\"";

  /// XML type attribute text
   const char* const xmlAttrType = " Type=\"";

  /// XML flag attribute text
   const char* const xmlAttrFlag = " Flag=\"";

  /// XML unit attribute text
   const char* const xmlAttrUnit = " Unit=\"";

  /// XML dimension attribute text
   const char* const xmlAttrDim = " Dim=\"";

  /// XML encoding attribute text
   const char* const xmlAttrEncoding = " Encoding=\"";

  /// XML comment attribute text
   const char* const xmlAttrComment = " Comment=\"";

  /// XML close attribute text
   const char* const xmlAttrClosing = "\"";


   /// Data type enumeration.
   enum gdsDataType {
   /// void, unknown
   gds_void = 0,
   /// 8 bit integer
   gds_int8 = 1, 
   /// 16 bit integer
   gds_int16 = 2, 
   /// 32 bit integer
   gds_int32 = 3, 
   /// 64 bit integer
   gds_int64 = 4, 
   /// single precision floating point
   gds_float32 = 5, 
   /// double precision floating point
   gds_float64 = 6,
   /// complex number with single precision floating point
   gds_complex32 = 7, 
   /// complex number with double precision floating point
   gds_complex64 = 8, 
   /// string
   gds_string = 9,
   /// channel name
   gds_channel = 10,
   /// bool
   gds_bool = 11,
   /// time
   gds_time = 12,
   /// time
   gds_table = 13
   };

/** \}  */


/** \defgroup xml_utils Utility operators and manipulators.

    Internal use.
   
    @memo Helper manipulators to write XML formated streams
    @author Written June 2000 by Daniel Sigg
    @version 1.0
    \{
*/

/** Xsil indent manipulator.
    This manipulator adds an indent to the output stream. Example:
    \verbatim
    os << xsilIndent(1) << ... << endl;
    \endverbatim
    This manipulator is typically used internally. 

    @memo Xsil indent manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilIndent {
   protected:
      /// Indent level
      int		fLevel;
   public: 
      /// Constructor. Takes a indent level as argument.
      explicit xsilIndent (int level = 0) : fLevel (level) {
      }
      /// Write indent to output stream
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil start tag manipulator.
    This manipulator adds a start tag to the output stream. Example:
    \verbatim
    os << xsilTagBegin (xmlParam) << xmlTypeAttr << 
          xsilDataTypename<T>() << xmlTypeClosing <<
          xmlDimType (N) << xmlTagClosing;
    \endverbatim
    Typically used internally.

    @memo Xsil start tag manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTagBegin {
   protected:
      /// Pointer to start tag
      const char*	fTag;
   public: 
      /// Constructor with tag name
      explicit xsilTagBegin (const char* tag = 0) : fTag (tag) {
      }
      /// Write XML start tag
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil end tag manipulator.
    This manipulator adds an end tag to the output stream. Example:
    \verbatim
    os << xsilTagEnd(xmlParam) << endl;
    \endverbatim
    Typically used internally.

    @memo Xsil end tag manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTagEnd {
   protected:
      /// Pointer to tag name
      const char*	fTag;
   public: 
      /// Constructor with tag name
      explicit xsilTagEnd (const char* tag = 0) : fTag (tag) {
      }
      /// Write XML end tag
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil dimension attribute manipulator.
    This manipulator adds a dimension attributeto the output stream. 
    Example:
    \verbatim
    os << xsilTagBegin (xmlParam) << xmlTypeAttr << 
          xsilDataTypename<T>() << xmlTypeClosing <<
          xmlDimType (N) << xmlTagClosing;
    \endverbatim
    Typically used internally.

    @memo Xsil dimension attribute manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/   
   class xsilDimAttr {
   protected:
      /// Dimension
      int		fDim;
   public:
      /// Constructor with dimesnion
      explicit xsilDimAttr (int dim = 0) : fDim (dim) {
      }
      /// Write dimension attribute
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil encoding attribute manipulator.
    This manipulator adds a encoding attributeto the output stream. 
    Example:
    \verbatim
    os << xsilTagBegin (xmlStream) << xmlTypeAttr << 
          xsilDataTypename<T>() << xmlTypeClosing <<
          xsilEncodingAttr() << xmlTagClosing;
    \endverbatim
    Typically used internally.

    @memo Xsil encoding attribute manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/     
   class xsilEncodingAttr {
   public:
      /// Constructor
      xsilEncodingAttr () {
      }
      /// Write encoding attribute
      std::ostream& write (std::ostream& os) const;
   };


/** XML type name manipulator.
    This manipulator adds the name of a data type to the output stream. 
    Example:
    \verbatim
    os << xsilTagBegin (xmlStream) << xmlTypeAttr << 
          xsilDataTypename<T>() << xmlTypeClosing <<
          xsilEncodingAttrAttr() << xmlTagClosing;
    \endverbatim
    Typically used internally.

    @memo XML type name manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilDataTypename {
   public:
      /// Constructor
      explicit xsilDataTypename () {
      }
   };


/** Xsil data encoding manipulator.
    This manipulator adds the name of a data type to the output stream. 
    Example:
    \verbatim
    os << xsilIndent (fLevel+1) << xsilTagBegin (xmlArray) << 
          xmlTypeAttr << xsilDataTypename<T>() << xmlAttrClosing <<
          xmlTagClosing << std::endl;
    if (fDim1 > 0) {
       os << xsilIndent (fLevel+2) << xsilTagBegin (xmlDim) << 
             xmlTagClosing << fDim1 << xsilTagEnd (xmlDim) << std::endl;
    }
    if (fDim2 > 0) {
       os << xsilIndent (fLevel+2) << xsilTagBegin (xmlDim) << 
             xmlTagClosing << fDim2 << xsilTagEnd (xmlDim) << std::endl;
    }
    os << xsilIndent (fLevel+2) << xsilTagBegin (xmlStream) << 
          xsilEncodingAttrAttr() << xmlTagClosing << std::endl;
    xsilBase64<T> (fData, fN).write (os);
    os << xsilIndent (fLevel+2) << xsilTagEnd (xmlStream) << std::endl;
    os << xsilIndent (fLevel+1) << xsilTagEnd (xmlArray) << std::endl;
    \endverbatim
    Typically used internally.

    @memo Xsil data encoding manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilBase64 {
   protected:
      /// Data array pointer
      const T*		fData;
      /// Length of data array
      int		fN;
   public:
      /// Constructor with data array and length
      xsilBase64 (const T* data = 0, int N = 0)
      : fData (data), fN (N) {
      }
      /// Write data array to output stream (base64 encoded)
      std::ostream& write (std::ostream& os) const;
   };


   /// Encode a data array in base64
   std::ostream& base64encode (std::ostream& os, const char* p, 
                     int num, int elsize);
   /// Decode a data array from base64 (ctype=0) or from uuencoding (ctype=1)
   bool base64decode (const char* code, int codelen,
                     char* p, int len, int ctype = 0);

/** Xsil base attributes manipulator.
    This manipulator adds a name, unit and comments attributes to the 
    output stream. Example:
    \verbatim
    os << xsilTagBegin (xmlParam) << xsil_base ("Points") <<
          xmlDimType (N) << xmlTagClosing;
    \endverbatim
    Typically used internally.
   
    @memo Xsil base attributes manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
   ************************************************************************/   
   class xsil_base {
   protected:
      /// Pointer to name
      const char* 	fName;
      /// Pointer to unit
      const char*	fUnit;
      /// Pointer to comment
      const char*	fComment;
      /// Indent level
      int		fLevel;
   public:
      /// Costructor with name and indent level
      explicit xsil_base (const char* name = 0, int level = 0)
      : fName (name), fUnit (0), fComment (0), fLevel(level) {
      }
      /// Costructor with name, unit and indent level
      explicit xsil_base (const char* name, const char* unit, 
                        int level = 0)
      : fName (name), fUnit (unit), fComment (0), fLevel(level) {
      }
      /// Set unit
      void SetUnit (const char* unit) {
         fUnit = unit; }
      /// Set comment
      void SetComment (const char* comment) {
         fComment = comment; }
      /// Write base attributes to output stream
      std::ostream& write (std::ostream& os) const;
   };


   /// Write indent to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilIndent& i) {
      return i.write (os); }
   /// Write XML start tag
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilTagBegin& t) {
      return t.write (os); }
   /// Write XML end tag
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilTagEnd& t) {
      return t.write (os); }
   /// Write dimension attribute
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDimAttr& d) {
      return d.write (os); }
   /// Write encoding attribute
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilEncodingAttr& e) {
      return e.write (os); }
   /// Write base attributes to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsil_base& b) {
      return b.write (os); }
   /// Write boolean type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<bool>& t) {
      return (os << "boolean"); }
   /// Write byte type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<char>& t) {
      return (os << "byte"); }
   /// Write byte short to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<short>& t) {
      return (os << "short"); }
   /// Write int type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<int>& t) {
      return (os << "int"); }
#ifndef __CINT__
   /// Write long type to output stream
   inline  std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<long long>& t) {
      return (os << "long"); }
#endif
   /// Write float type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<float>& t) {
      return (os << "float"); }
   /// Write double type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<double>& t) {
      return (os << "double"); }
   /// Write complex<float> type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<std::complex<float> >& t) {
      return (os << "floatComplex"); }
   /// Write complex<double> type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<std::complex<double> >& t) {
      return (os << "doubleComplex"); }
   /// Write string type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<const char*>& t) {
      return (os << "string"); }
   /// Write string type to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<std::string>& t) {
      return (os << "string"); }
   /// Write Time type to output stream (only used for tables)
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<Time>& t) {
      return (os << "time"); }
   /// Dummy class for writing the table typename
   struct xsil_table {
   };
   /// Write table type to output stream (only used for tables)
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilDataTypename<xml::xsil_table>& t) {
      return (os << "table"); }

  /** \} */


/** \defgroup xml_ostream Output stream manipulators.
    These are the main XSIL output manipulators and operators.
   
    @memo Manipulators to write XML formatted streams
    @author Written June 2000 by Daniel Sigg
    @version 1.0
    \{
 ************************************************************************/

/** Xsil header manipulator.
    This manipulator should be called exactly once at the very beginning
    of the stream. Example:
    \verbatim
    os << xsilHeader() << endl;
    \endverbatim

    @memo Xsil header manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHeader {
   public: 
     /// Constructor
      xsilHeader () {
      }
      /// Write header to output stream
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil trailer manipulator.
    This manipulator should be called exactly once at the very end
    of the stream. Example:
    \verbatim
    os << xsilTrailer() << endl;
    \endverbatim

    @memo Xsil trailer manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTrailer {
   public:
      /// Constructor
      xsilTrailer () {
      }
      /// Write trailer to output stream
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil comment manipulator.
    This manipulator adds a comment object to the output stream.
    Example:
    \verbatim
    os << xsilComment ("This is a comment") << endl;
    \endverbatim

    @memo Xsil comment manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilComment : public xsil_base {
   public:
      /// Constructor with name, data, length and indent level
      xsilComment (const char* comment, int level = 2)
      : xsil_base (comment, level) {
      }
      /// Constructor with name, data, length and indent level
      xsilComment (const std::string& comment, int level = 2)
      : xsil_base (comment.c_str(), level) {
      }
      /**  Write comment to output stream 
        *  \brief Write comment
	*  \param os STL output stream
	*  \return Reference to the output stream.
        */
      std::ostream& write (std::ostream& os) const;
   };

/** Xsil time manipulator.
    This manipulator adds a time object to the output stream.
    Example:
    \verbatim
    os << xsilTime ("Time", sec, nsec) << endl;
    \endverbatim

    @memo Xsil time manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTime : public xsil_base {
   public:
      /// time format
      typedef unsigned long gdstime_t;
   
   protected:
      /// GPS sec
      gdstime_t 	fSec;
      /// GPS nsec
      gdstime_t 	fNSec;
   public:
      /// Constructor with name, time and indent level
      xsilTime (const char* name, const Time& time, 
               int level = 2)
      : xsil_base (name, level), fSec (time.getS()), fNSec (time.getN()) {
      }
      /// Constructor with name, time and indent level
      xsilTime (const char* name, gdstime_t sec, gdstime_t nsec, 
               int level = 2)
      : xsil_base (name, level), fSec (sec), fNSec (nsec) {
      }

      /**  Write a time to the output stream in the specified format.
        *  \brief Write time to output stream 
	*  \param os     STL output stream to receive XML structure.
	*  \param format Format in which time is to be written: 0 - gps, 
	*                1 - gps and utc
	*  \return Reference to output stream.
	*/
      std::ostream& write (std::ostream& os, int format = 1) const;
   };

/** Xsil parameter manipulator.
    This manipulator adds a parameter object to the output stream.
    Example:
    \verbatim
    os << xsilParameter<int>("Averages", 10) << endl;
    \endverbatim

    @memo Xsil parameter manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilParameter : public xsil_base {
   protected:
      /// Data pointer
      const T*		fData;
      /// Length of data array
      int		fN;
   public:
      xsilParameter() : fData (0), fN (0) {
      }
      /// Constructor with name, data, length and indent level
      xsilParameter (const char* name, const T& x, int N = 1, int level = 2)
      : xsil_base (name, level), fData (&x), fN (N) {
      }
      /// Constructor with name, unit, data, length and indent level
      xsilParameter (const char* name, const char* unit, const T& x, 
                    int N = 1, int level = 2)
      : xsil_base (name, unit, level), fData (&x), fN (N) {
      }
      /// Write parameter to output stream
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil array manipulator.
    This manipulator adds an array object to an output stream. Example:
    \verbatim
    float y[200];
    os << xsilArray<float> ("Error", 200, y) << endl;
    \endverbatim

    @memo Xsil array manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilArray : public xsil_base {
   protected:
      /// Dimensions
      int		fDim[4];
      /// Data array
      const T*		fData;
   
   public:
      xsilArray () : fData (0) {
         fDim[0] = 0; fDim[1] = 0; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 1 dim data
      xsilArray (const char* name, int dim1, 
                const T* data, int level = 2)
      : xsil_base (name, level), fData (data) {
         fDim[0] = dim1; fDim[1] = 0; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 2 dim data
      xsilArray (const char* name, 
                int dim1, int dim2, const T* data, int level = 2)
      : xsil_base (name, level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 3 dim data
      xsilArray (const char* name, int dim1, int dim2, int dim3, 
                const T* data, int level = 2)
      : xsil_base (name, level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = dim3; fDim[3] = 0; }
      /// Constructor for 4 dim data
      xsilArray (const char* name, 
                int dim1, int dim2, int dim3, int dim4,
                const T* data, int level = 2)
      : xsil_base (name, level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = dim3; fDim[3] = dim4; }
      /// Write array of data object to output stream
      std::ostream& write (std::ostream& os, bool term = true) const;
   };


/** Xsil data begin manipulator.
    This manipulator adds the start tag of a data object to an
    output stream. Example:
    \verbatim
    os << xsilDataBegin ("Result", "TimeSeries") << endl;
    os << xsilParameter<int> ("Averages", 12) << endl;
    os << xsilDataEnd<float>() << endl;
    \endverbatim

    @memo Xsil data begin manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilDataBegin : public xsil_base {
   protected:
      /// Object type pointer
      const char*	fType;
      /// Object flag pointer
      const char*	fFlag;
   public:
      /// Constructor with name, type and indent level
      xsilDataBegin (const char* name, const char* type, int level = 1) 
      : xsil_base (name, level), fType (type), fFlag (0) {
      }
      /// Constructor with name, type and indent level
      xsilDataBegin (const char* name, const char* type, 
                    const char* flag, int level = 1) 
      : xsil_base (name, level), fType (type), fFlag (flag) {
      }
      /// Write data object head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil data end manipulator.
    This manipulator adds the data of a data object to an
    output stream. It also adds the end tag. Example:
    \verbatim
    os << xsilDataBegin ("Result", "TimeSeries") << endl;
    os << xsilParameter<int> ("Averages", 12) << endl;
    os << xsilDataEnd<float>() << endl;
    \endverbatim

    @memo Xsil data end manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilDataEnd  {
   protected:
      /// Indent level
      int		fLevel;
      /// Dimensions
      int		fDim[4];
      /// Data array
      const T*		fData;
   public:
      /// Constructor (no data)
      xsilDataEnd (int level = 1) 
      : fLevel (level), fData (0) {
         fDim[0] = 0; fDim[1] = 0; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 1 dim data
      xsilDataEnd (int dim1, const T* data, int level = 1)
      : fLevel (level), fData (data) {
         fDim[0] = dim1; fDim[1] = 0; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 2 dim data
      xsilDataEnd (int dim1, int dim2, const T* data, int level = 1)
      : fLevel (level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = 0; fDim[3] = 0; }
      /// Constructor for 3 dim data
      xsilDataEnd (int dim1, int dim2, int dim3,
                  const T* data, int level = 1)
      : fLevel (level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = dim3; fDim[3] = 0; }
      /// Constructor for 4 dim data
      xsilDataEnd (int dim1, int dim2, int dim3, int dim4,
                  const T* data, int level = 1)
      : fLevel (level), fData (data) {
         fDim[0] = dim1; fDim[1] = dim2; fDim[2] = dim3; fDim[3] = dim4; }
      /// Write data of data object to output stream
      std::ostream& write (std::ostream& os, bool term = true) const;
   };


/** Xsil table begin manipulator.
    This manipulator adds the start tag of a table object to an
    output stream. Example:
    \verbatim
    os << xsilTableBegin ("Events", "gds") << endl;
    os << xsilComment ("GDS EVENTS") << endl;
    os << xsilTableColumn<double> ("Amplitude") << endl;
    os << xsilTableColumn<double> ("Noise") << endl;
    os << xsilTableColumn<int> ("Time") << endl;
    os << xsilTableColumn<int> ("TimeN") << endl;
    os << xsilTableDataBegin() << endl;
    for (int r = 0; r < rows; ++r) {
       os << xsilTableEntry<double> (event[r].fAmplitude) 
          << xsilTableEntryDelimiter()
          << xsilTableEntry<double> (event[r].fNoise)
          << xsilTableEntryDelimiter()
          << xsilTableEntry<int> (event[r].fTime.GetS())
          << xsilTableEntryDelimiter()
          << xsilTableEntry<int> (event[r].fTime.GetN());
       if (r + 1 < rows) os <<  xsilTableEntryDelimiter();
       os << endl;
    }
    os << xsilTableDataEnd() << endl;
    os << xsilTableEnd() << endl;
    \endverbatim

    @memo Xsil table begin manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTableBegin : public xsil_base {
   protected:
      /// Object type pointer
      const char*	fType;
   public:
      /// Constructor with name, type and indent level
      xsilTableBegin (const char* name, const char* type = 0, int level = 1) 
      : xsil_base (name, level), fType (type) {
      }
      /// Write table object head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil table end manipulator.
    This manipulator adds the end tag of a table object to an
    output stream.

    @memo Xsil table end manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTableEnd {
   protected:
      /// Indent level
      int		fLevel;
   public:
      /// Constructor with indent level
      xsilTableEnd (int level = 1) : fLevel (level) {
      }
      /// Write table object head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil table column manipulator.
    This manipulator adds a table column to an output stream.

    @memo Xsil table column manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <class T>
   class xsilTableColumn : public xsil_base {
   public:
      /// Constructor with name and indent level
      xsilTableColumn (const char* name = 0, int level = 2)
      : xsil_base (name, level) {
      }
      /// Write table column object head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil table data begin manipulator.
    This manipulator adds the start tag of a table stream to an
    output stream.

    @memo Xsil table data begin manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTableDataBegin : public xsil_base {
   public:
      /// Constructor with name, type and indent level
      xsilTableDataBegin (const char* name = 0, int level = 2) 
      : xsil_base (name, level) {
      }
      /// Write table stream data head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil table data end manipulator.
    This manipulator adds the end tag of a table stream to an
    output stream.

    @memo Xsil table data end manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTableDataEnd {
   protected:
      /// Indent level
      int		fLevel;
   public:
      /// Constructor with indent level
      xsilTableDataEnd (int level = 2) : fLevel (level) {
      }
      /// Write table object head
      std::ostream& write (std::ostream& os) const;
   };


/** Xsil table entry manipulator.
    This manipulator adds a table entry to an output stream.

    @memo Xsil table entry manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
  template <class T>
  class xsilTableEntry {
  protected:
    /// Value
    const T*		fValue;
  public:
    ///  Default constructor.
    xsilTableEntry() 
      : fValue (0) 
    {}
   
    /// Constructor with value
    xsilTableEntry (const T& value)
      : fValue (&value)
    {}

    ///  Write table entry object head
    ///  \brief write object tag.
    ///  \param os output stream
    ///  \return stream reference
    ///
    std::ostream& write (std::ostream& os) const;
  };

/** Xsil table entry manipulator (complex<double> specialization).
  * This manipulator adds a table entry to an output stream.
  *
  * \brief Xsil table entry manipulator.
  * \param os Output stream.
  * \return Reference to the output stream.
  * \author Written June 2000 by Daniel Sigg
  * \version 1.0
  ************************************************************************/
template <>
   std::ostream& xsilTableEntry<std::complex<double> >::write (std::ostream& os) const;


/** Xsil table entry manipulator (Time specialization).
  * This manipulator adds a table entry to an output stream.
  *
  * \brief Xsil table entry manipulator.
  * \param os Output stream.
  * \return Reference to the output stream.
  * \author Written June 2000 by Daniel Sigg
  * \version 1.0
  ************************************************************************/
template <>
   std::ostream& xsilTableEntry<Time>::write (std::ostream& os) const;

/** Xsil table entry manipulator (string specialization).
    This manipulator adds a table entry to an output stream.

    @memo Xsil table entry manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
template <>
   class xsilTableEntry<const char*> {
   protected:
      /// Value
      const char*	fValue;
      /// Length
      int		fLength;
   public:
      /// Constructor with value
      xsilTableEntry<const char*> (const char* value = 0)
      : fValue (value), fLength (-1) {
      }
      /// Constructor with value and length
      xsilTableEntry<const char*> (const char* value, int len)
      : fValue (value), fLength (len) {
      }
      /// Write table entry object head
      std::ostream& write (std::ostream& os) const;
   };

   /// Escape special characters <>&"'
   std::string xsilStringEscape (const char* p);
   /// Escape a string so it can be stored as a table cell
   std::string xsilEscape (const char* p, int len);
   /// Unescape a string from a table cell
   std::string xsilUnescape (const char* p);


/** Xsil table delimiter manipulator.
    This manipulator adds a table delimiter to an output stream.

    @memo Xsil table delimiter manipulator.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilTableEntryDelimiter {
   public:
      /// Constructor
      xsilTableEntryDelimiter() {
      }
      /// Write table entry object head
      std::ostream& write (std::ostream& os) const;
   };


   /// Write header to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilHeader& h) {
      return h.write(os); }
   /// Write trailer to output stream
   inline std::ostream& operator<< (std::ostream& os,
                     const xsilTrailer& t) {
      return t.write(os); }

   /// Write parameter to output stream
template <class T>
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<T>& p) {
      return p.write (os); }

   /// Write comment to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilComment& t) {
      return t.write (os); }

   /// Write time to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTime& t) {
      return t.write (os); }

   /// Write array of data object
template <class T>
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilArray<T>& arr) {
      return arr.write (os); }

  /// Write data object head to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataBegin& d) {
      return d.write (os); }

   /// Write data of data object plus end of data object tag
template <class T>
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataEnd<T>& d) {
      return d.write (os); }

  /// Write table begin object to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableBegin& d) {
      return d.write (os); }

  /// Write table end object to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEnd& d) {
      return d.write (os); }

  /// Write table column to output stream
template <class T>
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<T>& d) {
      return d.write (os); }

  /// Write table data begin object to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableDataBegin& d) {
      return d.write (os); }

  /// Write table data end object to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableDataEnd& d) {
      return d.write (os); }

  /// Write table data entry object to output stream
template <class T>
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<T>& d) {
      return d.write (os); }

  /// Write table delimiter to output stream
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntryDelimiter& d) {
      return d.write (os); }


#ifdef __CINT__
//    inline std::ostream& operator<< (std::ostream& os, 
//                       const xsilParameter<bool>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<char>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<short>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<int>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<float>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<double>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<std::complex<float> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<std::complex<double> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilParameter<const char*>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilArray<float>& arr);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilArray<double>& arr);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilArray<std::complex<float> >& arr);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilArray<std::complex<double> >& arr);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataEnd<float>& d);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataEnd<double>& d);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataEnd<std::complex<float> >& d);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilDataEnd<std::complex<double> >& d);
//   inline std::ostream& operator<< (std::ostream& os, 
//                       const xsilTableColumn<bool>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<char>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<short>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<int>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<float>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<double>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<std::complex<float> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<std::complex<double> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<Time>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableColumn<const char*>& p);
//   inline std::ostream& operator<< (std::ostream& os, 
//                       const xsilTableEntry<bool>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<char>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<short>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<int>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<float>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<double>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<std::complex<float> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<std::complex<double> >& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<Time>& p);
   inline std::ostream& operator<< (std::ostream& os, 
                     const xsilTableEntry<const char*>& p);
#endif

  /** \} */

/** \defgroup xml_parse Input parsing classes.
       
    @memo Classes to read XML formated streams
    @author Written June 2000 by Daniel Sigg
    @version 1.0

     \{
 ************************************************************************/

/** This is the callback handler which is constructed by a handler 
    query object of the corresponding xsil data object. The user
    has to define a xsilHandler class derived from this one for
    each type of xsil object intended for parsing. The following 
    automatic type conversions are supported for parameters and
    table entries:
    \verbatim
    char -> short -> int -> long long
    short -> int -> long long
    int -> long long
    long long -> int
    float -> double
    double -> float
    complex float -> complex double
    complex double -> complex float
    \endverbatim
    The native type is always tried first and the conversion process
    stops with the first handler returning true.

    For tables the column handler is called for each column definition
    using a column index starting at zero. The column handler is then
    called again with a column index of -1 and an empty name to indicate
    that the last column defintion has been read. Subsequently, the 
    table entry handler is called for each table cell. The table entry
    handler has the column and row index as an argument.

    @memo Xsil callback handler.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandler {
   protected:
      /// True if nested data objects should be ignored
      bool		fIgnoreNested;
      /// Comment string
      std::string	fComment;
      /// Unit string	
      std::string	fUnit;
   public:
      /// Attribute list
      typedef std::map<std::string, std::string> attrlist;
   
      /// Constructor
      explicit xsilHandler (bool ignore = false) :
      fIgnoreNested (ignore) {
      }
      /// Destructor
      virtual ~xsilHandler() {
      }
      /// comment callback
      virtual bool CommentHandler (const std::string& comment);
   
      /// parameter callback (will call correct type)
      virtual bool ParameterHandler (int type, void* x, int size,
                        const attrlist& attr);
      /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1) {
         return false; }
      /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const char& p, int N = 1) {
         return false; }
      /// short parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const short& p, int N = 1) {
         return false; }
      /// int parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const int& p, int N = 1) {
         return false; }
   #ifndef __CINT__
      /// long parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const long long& p, int N = 1) {
         return false; }
   #endif
      /// float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const float& p, int N = 1) {
         return false; }
      /// double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const double& p, int N = 1) {
         return false; }
      /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1) {
         return false; }
      /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1) {
         return false; }
      /// string parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::string& p) {
         return false; }
   
      /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name,
                        const attrlist& attr,
                        unsigned long sec, unsigned long nsec) {
         return false; }
   
      /// data callback (must return true if data was adopted)
      virtual bool DataHandler (const std::string& name,
                        int type, void* x, int size, 
                        int dim1, int dim2 = 0, int dim3 = 0,
                        int dim4 = 0);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        float* x, int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0) {
         return false; }
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        std::complex<float>* x, 
                        int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0) {
         return false; }
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        double* x, int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0) {
         return false; }
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        std::complex<double>* x, 
                        int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0) {
         return false; }
   
      /// table column callback
      virtual bool HandleTableColumn (int col, const std::string& name,
                        int type, const attrlist& attr) {
         return false; }
   
      /// table entry callback (will call correct type)
      virtual bool TableEntryHandler (int row, int col, int type, 
                        void* x);
      /// bool table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, bool p) {
         return false; }
      /// byte table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, char p) {
         return false; }
      /// short table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, short p) {
         return false; }
      /// int table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, int p) {
         return false; }
   #ifndef __CINT__
      /// long table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, long long p) {
         return false; }
   #endif
      /// float table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, float p) {
         return false; }
      /// double table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, double p) {
         return false; }
      /// complex float table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::complex<float>& p) {
         return false; }
      /// complex double table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::complex<double>& p) {
         return false; }
      /// Time table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col, const Time& p) {
         return false; }
      /// string table entry callback (must return true if handled)
      virtual bool HandleTableEntry (int row, int col,
                        const std::string& p) {
         return false; }
   
      /// handler for nested data objects (0 for none)
      virtual xsilHandler* GetHandler (const attrlist& attr) {
         return fIgnoreNested ? new xsilHandler (true) : 0; }
      /// handler for nested table (0 for none)
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return fIgnoreNested ? new xsilHandler (true) : 0; }
   };


/** Xsil handler query class.
    This is the callback handler query class which is called by the
    parser when a new data object is encountered. If the query
    recognizes the data object it should return a new callback 
    handler which will be used by the parser for parameters, time
    and nested data objects. Typically every type of xsil data object
    has its own handler callback class and its own calllback handler
    query class. This is an abstract class.

    @memo Xsil callback handler query.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandlerQuery {
   public:
      /// Attribute list
      typedef std::map<std::string, std::string> attrlist;
      xsilHandlerQuery() {
      }
      /// Destructor
      virtual ~xsilHandlerQuery() {
      }
      /// returns a handler for the specified object (or 0 if not)
      virtual xsilHandler* GetHandler (const attrlist& attr) = 0;
      /// returns a handler for the specified table (or 0 if not)
      virtual xsilHandler* GetTableHandler (const attrlist& attr) {
         return 0; }
   };


/** Handler for unknown data objects. Data object gets written to 
    stream.
   
    @memo Unknown data objects handler
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandlerUnknown : public xsilHandler {
      xsilHandlerUnknown (const xsilHandlerUnknown&);
      xsilHandlerUnknown& operator= (const xsilHandlerUnknown&);
   
   public:
      /// Constructor
      explicit xsilHandlerUnknown (std::ostream& os, 
                        const attrlist* attr = 0, bool ignore = false);
      /// Destructor
      virtual ~xsilHandlerUnknown();
   
      /// comment callback
      virtual bool CommentHandler (const std::string& comment);
   
      /// bool parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const bool& p, int N = 1);
      /// byte parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const char& p, int N = 1);
      /// short parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const short& p, int N = 1);
      /// int parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const int& p, int N = 1);
   #ifndef __CINT__
      /// long parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const long long& p, int N = 1);
   #endif
      /// float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const float& p, int N = 1);
      /// double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const double& p, int N = 1);
      /// complex float parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<float>& p, int N = 1);
      /// complex double parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr,
                        const std::complex<double>& p, int N = 1);
      /// string parameter callback (must return true if handled)
      virtual bool HandleParameter (const std::string& name,
                        const attrlist& attr, const std::string& p);
      /// time callback (must return true if handled)
      virtual bool HandleTime (const std::string& name,
                        const attrlist& attr,
                        unsigned long sec, unsigned long nsec);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        float* x, int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        std::complex<float>* x, 
                        int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        double* x, int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0);
      /// data callback (must return true if data is adopted)
      virtual bool HandleData (const std::string& name,
                        std::complex<double>* x, 
                        int dim1, int dim2 = 0, 
                        int dim3 = 0, int dim4 = 0);
   
   protected:
      /// Output string for unrecognized data object
      std::ostream*	fOs;
      /// Data header/trailer?
      bool		fTrailer;
      /// dimensions
      int		fDim[4];
      /// complex data?
      bool		fComplex;
      /// double precision data?
      bool		fDouble;
      /// data pointer
      void*		fData;
   
   };


/** Xsil unknown object handler query class.
    The query will always return the unknown handler.

    @memo Xsil unknown object handler query.
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilHandlerQueryUnknown : public xml::xsilHandlerQuery {
   public:
      /// Constructor
      xsilHandlerQueryUnknown (std::ostream* os) : fOs (os) {
      }
      /// returns a handler for the specified object (or 0 if not)
      virtual xsilHandler* GetHandler (const attrlist& attr);
   
   protected:
      /// string stream
      std::ostream*		fOs;
   };

  /**  Auto pointer for handler
    *  @brief Autopointer for handler.
    *  @author Daniel Sigg
    *  @version 1.0; Last major Modification January 2002.
    */
   class xsilHandlerPtr {
      mutable xsilHandler* ptr;
   public:

      /**  Construct an xsilHandlerPtr and optionally set it to the 
        *  argument xsilHandler pointer.
        *  \brief Default constructor
	*  \param x Pointer to target handler.
	*/
      explicit xsilHandlerPtr (xsilHandler* x = 0) : ptr (x) {
      }

     /**  Construct a copy of the argument pointer. The new pointer retains 
       *  ownership of the target.
       *  \brief Copy constructor.
       *  \param p %xsilHandlerPtr to be copied.
       */
      xsilHandlerPtr (const xsilHandlerPtr& p) : ptr (0) {
         *this = p; }

     /**  Destroy the pointer and free the referenced xsilHandler.
       *  \brief Destructor
       */
      ~xsilHandlerPtr() {
         if (ptr) delete ptr; }

     /**  Assign the contents of one pointer to another. The argument 
       *  pointer is invalidated. 
       *  \note The pointer is mutable so the const attribute is not meaningful.
       *  \brief Assignment operator
       *  \param p %xsilHandlerPtr to be copied.
       *  \return Reference to current object.
       */
      xsilHandlerPtr& operator= (const xsilHandlerPtr& p) {
         if (this != &p) {
            ptr = p.ptr; p.ptr = 0;
         }
         return *this; }

     /**  %xsilHandlerPtr pointer dereference operator.
       *  \brief Dereference operator
       *  \return Reference to the target xsilHandler.
       */
      xsilHandler& operator* () const {
         return *ptr; }

     /**  Get a pointer to the referenced handler.
       *  \brief Get the pointer.
       *  \return Pointer to the referenced xsilHandler.
       */
      xsilHandler* operator-> () const {
         return ptr; }

     /**  Test if equal to the specified pointer.
       *  \brief Test for equal pointer.
       *  \param p Pointer to test against
       *  \return true if pointers are equal.
       */
      bool operator== (const xsilHandler* p) {
         return (ptr == p); }

     /** Test if not equal to the specified pointer.
       *  \brief Test for unequal pointer.
       *  \param p Pointer to test against
       *  \return true if pointers are not equal.
       */
      bool operator!= (const xsilHandler* p) {
         return (ptr != p); }
   };


   /// Temporary handler class
   class xsilHandlerTemp {
   
   public:
      /// handler status
      enum status {
      // normal
      normal = 0,
      // parameter
      param = 1,
      // time
      time = 2,
      // column
      column = 3,
      // stream
      stream = 4,
      // array
      array = 5,
      // array dim
      arraydim = 6,
      // array stream 
      arraystream = 7,
      // comment
      comment = 8
      };
      /// Attribute list
      typedef std::map<std::string, std::string> attrlist;
      /// Column list 
      typedef std::vector <int> columnlist;
   
      /// Is table?
      bool		fIsTable;
      /// Handler status
      status		fStatus;
      /// Pointer to handler
      xsilHandlerPtr	fHandler;
      /// comment text
      std::string	fComment;
      /// temp text
      std::string	fText;
      /// Data type
      std::string	fDataName;
      /// Data type
      int		fDataType;
      /// Pointer to data
      char*		fData;
      /// Size of data
      int		fSize;
      /// dimensions of data
      std::vector<int>	fDim;
      /// Encoding type
      int		fEncoding;
      /// Parameter attributes
      attrlist		fPrmAttr;
      /// Parameter name
      std::string	fPrmName;
      /// Parameter type
      int		fPrmType;
      /// Length of parameter data
      int		fPrmSize;
      /// Column type
      columnlist	fColumns;
      /// Column index
      int		fColumn;
      /// Row index
      int		fRow;
      /// Ignore text till after next comma (table only)
      bool		fIgnoreNextComma;
   
      /// Constructor
      xsilHandlerTemp(xsilHandler* handler, bool istable = false)
      : fIsTable (istable), fStatus  (normal), fHandler (handler), 
      fData (0), fColumn (0), fRow (0), fIgnoreNextComma (false) {
      }
      /// Destructor
      ~xsilHandlerTemp() {
         SetData (0); } 
      /// Set data
      void SetData (char* dat) {
         if (fData) delete [] fData; 
         fData = dat; }
      /// Flush table data (until last comma if comma is true)
      bool FlushTableEntries (bool comma = false);
      /// Setup a table-in-a-table object
      bool SetTableEntry();
   
   };

/** This is the input parser for reading data formatted in the LIGO
    light weight data format. The parser works on the premise of 
    callback functions. The user feeds data to the parser which in
    return will invoke data and parameter methods of a handler object
    whenever a LIGO LW object is encountered in the input stream. 
    Before feeding data the user has to supply at least one handler
    object. The 'IsHandler' method of the handler objects are called
    in turn for every encounterd LIGO-LW data object. The first 
    handler returning true will obtain all parameters associated
    with the data object. Again, parameters are passed to the handler
    by invoking the HandleParameter methods. Since there are 
    many different parameter types, this method is overloaded. If a
    parameter handler recognizes the parameter it has to return true,
    otherwise the parser will try calling other parameter handler 
    methods after converting the parameter data into the new type
    (this automatic type conversion follows essentially the C++
    rules, but will always try the string handler last).
    The handler object can also establish a callback for nested
    data objects; if not, nested data objects are treated like 
    stand-alone.

    Example:
    \verbatim
    xsilHandlerQueryFile fileQ (filename, os);
    xsilParser parser;
    parser.AddHandler (fileQ);
    parser.Parse (filename);
    \endverbatim

    @memo Xsil input parser
    @author Written June 2000 by Daniel Sigg
    @version 1.0
 ************************************************************************/
   class xsilParser {
   public:
      /// Attribute list
      typedef std::map<std::string, std::string> attrlist;
      /// Handler list
      typedef std::vector<xsilHandlerQuery*> handlerquerylist;
      /// work list of handlers
      typedef std::vector<xsilHandlerTemp> worklist;
   
      /// Default handler (ignores current data object)
      static xsilHandlerQuery& DefaultHandler();
      /// Ignore handler (ignores current data object and all nested)
      static xsilHandlerQuery& IgnoreAllHandler();
   
   protected:
      /// list of data object handlers
      handlerquerylist	fHandler;
      /// Level of ignore
      int		fIgnore;
      /// New line flag 
      bool 		fNewLine;
      /// True if end of XML data has been reached
      bool		fEndXML;
      /// stack of nested handlers
      worklist		fWork;
      /// XML parser
      void*		fXML;
   
   public:
      /// Constructor
      xsilParser ();
      /// Destructor
      ~xsilParser();
      /// Parsing init
      void Init();
      /// Parsing done
      void Done();
   
      /** Add a handler (handler query must stay valid during parsing!)
        * The handler query object is not adopted
	* \brief add a handler
	* \param handler Reference to the handler.
	*/
      void AddHandler (xsilHandlerQuery& handler);
      /// Parse an array of characters
      bool Parse (const char* p, int len);
      /// Parse a string
      bool Parse (const std::string& s) {
         return Parse (s.c_str(), s.size()); }
      /// Parse an input stream
      bool Parse (std::istream& is);
      /// Parse a file
      bool ParseFile (const char* filename);
   
   public:
      /// define a start element
      void Startelement (const char* name, const attrlist& attr);
      /// handle text
      void Texthandler (const char* text, int len);
      /// define an end element
      void Endelement (const char* name);
   };
  /**  \} 
   */

#ifndef __CINT__
/** \cond ignore
  */
// Template defintions: Due to a strange SUN/C++ bug (generating 
// undefined symbols for string constants) these definitions
// have to be in the header and have to be declared inline.

// xsilBase64: XML base64 object 
template <class T>
inline std::ostream& xsilBase64<T>::write (std::ostream& os) const 
{
  return base64encode (os, (const char*) fData, fN, sizeof (T));
}

// xsilParameter: XML parameter object
template <class T>
   inline std::ostream& xsilParameter<T>::write (std::ostream& os) const 
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlParam);
      this->xsil_base::write (os);
      os << xmlAttrType << xsilDataTypename<T>() << xmlAttrClosing <<
         xsilDimAttr (fN) << xmlTagClosing;
      for (int i = 0; i < fN; i++) {
         os << (i > 0 ? " " : "") << fData[i];
      }
      return (os << xsilTagEnd (xmlParam));
   }

// xsilParameter: XML parameter object (string specialization)
template <>
   inline std::ostream& xsilParameter<const char*>::write (std::ostream& os) const 
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlParam);
      this->xsil_base::write (os);
      os << xmlAttrType << xsilDataTypename<const char*>() << xmlAttrClosing 
         << xsilDimAttr (fN) << xmlTagClosing;
      for (int i = 0; i < fN; i++) {
         os << (i > 0 ? "\n" : "") << xsilStringEscape (fData[i]);
      }
      return (os << xsilTagEnd (xmlParam));
   }

// xsilArray: XML array object
template <class T>
   inline std::ostream& xsilArray<T>::write (std::ostream& os, 
                     bool term) const 
   {
      int fN = 1;
      bool nValid = false;
      for (int i = 0; i < 4; ++i) {
	 if (fDim[i] > 0) {
	    fN *= fDim[i];
	    nValid = true;
	 }
      }
      if (!nValid) fN = 0;
      if ((fData != 0) && (fN > 0)) {
         os << xsilIndent (fLevel) << xsilTagBegin (xmlArray); 
         xsil_base::write (os);
         os << xmlAttrType << xsilDataTypename<T>() << xmlAttrClosing <<
            xmlTagClosing << std::endl;
         for (int i = 0; i < 4; ++i) {
            if (fDim[i] > 0) {
               os << xsilIndent (fLevel+1) << xsilTagBegin (xmlDim) << 
                  xmlTagClosing << fDim[i] << xsilTagEnd (xmlDim) << 
                  std::endl;
            }
         }
         os << xsilIndent (fLevel+1) << xsilTagBegin (xmlStream) << 
            xsilEncodingAttr() << xmlTagClosing << std::endl;
         xsilBase64<T> (fData, fN).write (os);
         os << xsilIndent (fLevel+1) << xsilTagEnd (xmlStream) << 
            std::endl;
         os << xsilIndent (fLevel) << xsilTagEnd (xmlArray);
      }
      return os; 
   }

// xsilDataEnd: XML data end object
template <class T>
   inline std::ostream& xsilDataEnd<T>::write (std::ostream& os, 
                     bool term) const 
   {
      os << xsilArray<T> (0, fDim[0], fDim[1], fDim[2], fDim[3],
                         fData, fLevel + 1) << std::endl; 
      if (term) 
         os << xsilIndent (fLevel) << xsilTagEnd (xmlContainer);
      return os; 
   }

// xsilTableColumn: XML table column object
template <class T>
   inline std::ostream& xsilTableColumn<T>::write (std::ostream& os) const
   {
      os << xsilIndent (fLevel) << xsilTagBegin (xmlColumn);
      this->xsil_base::write (os);
      os << xmlAttrType << xsilDataTypename<T>() << xmlAttrClosing
         << "/" << xmlTagClosing;
      return os;
   }

// xsilTableEntry: XML table entry object
template <class T>
   inline std::ostream& xsilTableEntry<T>::write (std::ostream& os) const
   {
      if (fValue) os << *fValue;
      return os;
   }

  /// write a table entry.
  template<>
  inline std::ostream& 
  xsilTableEntry<std::complex<double> >::write (std::ostream& os) const {
      if (fValue) os << fValue->real() << " " << fValue->imag();
      return os;
  }

  template<> 
  /// xsilTableEntry: XML table entry object (Time specialization)
  inline std::ostream& 
  xsilTableEntry<Time>::write (std::ostream& os) const
   {
      char buf[64];
      TimeStr (*fValue, buf, "%s.%n");
      os << buf;
      return os;
   }

  // xsilTableEntry: XML table entry object (string specialization)
  inline std::ostream& 
  xsilTableEntry<const char*>::write (std::ostream& os) const
   {
      if (fValue) os << "\"" << xsilEscape (fValue, fLength) << "\"";
      return os;
   }

  /*******************************************************************
   *  Finish up....
   *  \endcond
   */

#endif // __CINT__

  /** \} */

}

#endif // _LIGO_XSIL_H
