/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LVSHM_API_H
#define LVSHM_API_H

#ifdef __cplusplus
extern "C" {
#endif

  /**
   * \file lvshmapi.h Ligo Shared Memory C API
   * \ingroup IO_lsmp Ligo Shared Memory C API
   *  The functions included here for a C language binding for the DMT 
   *  shared memory class.
   *  \author John Zweizig (john.zweizig@ligo.org)
   *  \version $Id: lvshmapi.h 6541 2011-12-07 20:00:33Z john.zweizig@LIGO.ORG $
   */

  /**
   *  Opaque shared memory object handle
   */
  typedef void* lvshm_handle;

  /**
   *  Shared memory data type select mask.
   */
  typedef unsigned int lvshm_mask;

  /**
   *  Shared memory data id (GPS) type
   */
  typedef unsigned long lvshm_dataID;

  /**
   *  No wait function modifier flag for use with.
   */
  #define LVSHM_NOWAIT 1

  /**
   *  Mask enabling all triggers
   */
  #define LVSHM_ALLTRIGGERS (~0)

  /**
   *  Initialize a shared memory consumer and open it for input. The control
   *  section of the named memory partition is attached and mapped into 
   *  memory and the process is registered as a data consumer.
   *  \brief Initialize share memory handle.
   *  \param name   Pointer to the share memory partition name string.
   *  \param mask   Share memory data type mask;
   *  \return Shared memory consumer handle or \c NULL.
   */  
  lvshm_handle lvshm_init(const char* name, lvshm_mask mask);

  /**
   *  Get the next buffer that is as yet unseen by this consumer. The first 
   *  unseen buffer is marked as being in use, and its address is returned.
   *  Buffers are kept in the order in which they were inserted. If the 
   *  -c LVSHM_NOWAIT flag is specified, the function will return \c NULL 
   *  if there are no currently available unseen buffers. If -c LVSHM_NOWAIT
   *  is not set, the function will block until a buffer is available, the
   *  maximum wait time is exceeded or the call is interrupted by a signal.
   *  \brief Get the next unseen buffer.
   *  \param handle Handle for the partition to be accessed.
   *  \param flags  Modifier flags.
   *  \return pointer to the first byte of data in the requested buffer.
   */
  const char* lvshm_getNextBuffer(lvshm_handle handle, int flags);

  /**
   *  Get the buffer with the specified data ID (GPS start time). If the 
   *  requested buffer is found, it is marked as in use and the data 
   *  address is returned. It must be released with lvshm_releaseDataBuffer()
   *  when the user is done with it. If the specified specified buffer is not 
   *  available in the partition, no buffer is reserved and NULL is returned.
   *  \brief Get the buffer with the specified start GPS.
   *  \param handle Handle for the partition to be accessed.
   *  \param gps    Data ID (start GPS) of the desired buffer
   *  \return Buffer data address or \c  NULL.
   */
  const char* lvshm_getBufferByGPS(lvshm_handle handle, lvshm_dataID gps);

  /**
   *  Get the address of the start of data in the buffer currently in use by 
   *  this consumer.
   *  \brief Get the data address.
   *  \param handle Handle for the partition to be accessed.
   *  \return Data address of the current buffer or \c NULL.
   */
  const char* lvshm_dataAddress(lvshm_handle handle);

  /**
   *  Get the length of the data in the buffer currently in use by this
   *  consumer.
   *  \brief Get the data length.
   *  \param handle Handle for the partition to be accessed.
   *  \return length of data, or 0 if no buffer is in use by this consumer.
   */
  int lvshm_dataLength(lvshm_handle handle);

  /**
   *  Get the data ID (start GPS) of the buffer currently in use by this
   *  consumer.
   *  \brief Start GPS of current buffer.
   *  \param handle Handle for the partition to be accessed.
   *  \return Data ID (start GPS), or 0 if no buffer is in use by this consumer.
   */
   lvshm_dataID lvshm_bufferGPS(lvshm_handle handle);

  /**
   *  Get the data ID (GPS) of the buffer most recently added to the shared
   *  memory.
   *  \brief Get most recently added GPS
   *  \param handle Handle for the partition to be accessed.
   *  \return Data ID (GPS) of most recent buffer.
   */
   lvshm_dataID lvshm_latestGPS(lvshm_handle handle);

  /**
   *  Release the buffer currently in use by this consumer.
   *  \brief Release the current buffer.
   *  \param handle Handle for the partition to be accessed.
   */
  void lvshm_releaseDataBuffer(lvshm_handle handle);

  /**
   *  Release the process consumer ID. Release and unmap the shared memory
   *  partition control area. Destroy the opaque data structure (handle).
   *  \brief Deaccess the shared memory partition.
   *  \param handle Handle for the shared partition to be accessed
   */
  void lvshm_deaccess(lvshm_handle handle);

  /**
   *  Mark all buffers currently in shared memory as having been seen by 
   *  this consumer.
   *  \brief Flush all current buffers.
   *  \param handle Handle for the shared partition to be accessed
   */
  void lvshm_flush(lvshm_handle handle);

  /**
   *  Set the maximum wait time for a new buffer. The wait time is specified 
   *  in seconds and may have the following values:
   *  <ul>
   *    <li> \a wait > 0 Maximum wait time in seconds.
   *    <li> \a wait == 0 Never wait
   *    <li> \a wait < 0 Wait indefinitely.
   *  </ul>
   *
   *  \brief Set maximum wait time.
   *  \param handle Handle for the shared partition to be accessed
   *  \param wait Maximum wait time in seconds.
   */
  void lvshm_setWaitTime(lvshm_handle handle, double wait);

#ifdef __cplusplus
}
#endif

#endif /* !defined(LVSHM_API_H) */
