#include "lmsg/Buffer.hh"
#include "lmsg/BufferPool.hh"
#include "lmsg/Message.hh"
#include "lmsg/ErrorList.hh"
#include "lmsg/TCPSocket.hh"
#include "lmsg/SocketPool.hh"
#include "lmsg/TransportTCP.hh"
#include "lmsg/TransInput.hh"
#include "lmsg/TransOutput.hh"
#include <stdio.h>
#include <iostream>

using namespace lmsg;
using namespace std;

//======================================  Destructor
TransportTCP::~TransportTCP(void) {
    close();
}

//======================================  Constructor
TransportTCP::TransportTCP(void) 
  : mFlags(0), mSocket(0), mCSocket(0),
    mBuffPool(&defaultLMsgBufferPool), mDebug(0) 
{
}

//======================================  Get socket address
MsgAddr 
TransportTCP::getAddr(void) const {
    if (isOpen()) return mSocket->getAddr();
    else          return MsgAddr();
}

//======================================  Is socket connected?
bool
TransportTCP::isConnected(void) const {
    if (!isOpen()) return false;
    if (mCSocket) return true;
    return mSocket->isConnected();
}

//======================================  Close the socket
void
TransportTCP::close(void) {
    if (mCSocket) {
        disconnect();
    }

    if (mSocket) {
        disconnect();
	mSocket->async(false);
        mSocket->Return();
	mSocket = 0;
    }
}

//======================================  Kill the socket?
void 
TransportTCP::Purge(void) {
    close();
}

//======================================  Open the socket
error_type 
TransportTCP::open(mask_type flags, const MsgAddr* addr) {
    mFlags  = flags;
    mSocket = new TCPSocket(addr);
    if (!isOpen()) return NotOpen;
    mSocket->setDebug(mDebug);
    if ((flags & o_async)) mSocket->async(true);
    mSocket->setPool(mBuffPool);
    if (!isClient()) mSocket->listen(4);
    error_type rc = mSocket->setSendBuf(1<<20);
    rc = mSocket->setRecvBuf(1<<20);
    return rc;
}

//======================================  Send a message
error_type 
TransportTCP::send(const MsgHeader& hdr, const char* b) {
    size_type msgLen = hdr.getMsgLength();
    Buffer* buf;
    if (mBuffPool && mBuffPool->getDataLength() > msgLen) {
        buf = mBuffPool->getBuffer();
    } else {
        buf = new Buffer(msgLen);
    }
    if (!buf) return NoBuffer;
    buf->getHeader() = hdr;
    memcpy(buf->getDataAddr(), b, msgLen);
    buf->setDataLength(msgLen);
    return send(buf);
}

//======================================  Send a message
error_type 
TransportTCP::send(const MsgHeader& hdr, const Message& msg) {

    //----------------------------------  Get the encoded message length
    TransOutput out;
    msg.getData(out);
    size_type msgLen = out.getNBytes() + sizeof(hdr);

    //----------------------------------  Get an appropriate buffer.
    Buffer* buf;
    if (mBuffPool && mBuffPool->getDataLength() >= msgLen) {
        buf = mBuffPool->getBuffer();
    } else {
        buf = new Buffer(msgLen);
    }
    if (!buf) return NoBuffer;
    buf->getHeader() = hdr;
    out.setBuffer(*buf);

    //----------------------------------  Fill the buffer.
    try {
        msg.getData(out);
    } catch (exception& e) {
        if (getDebug()) {
	    cout << "Exception: " << e.what() << " in TransportTCP::send" 
		 << endl;
	    cout << "Output buffer length: " << buf->getSize() << endl;
	}
	buf->Return();
	return SizeError;
    }
    index_type len = out.getNBytes();
    buf->setDataLength(len);
    buf->getHeader().setMsgLength(len);
    buf->getHeader().setMsgType(msg.getType());
    return send(buf);
}

//======================================  Send message contained in a buffer
error_type 
TransportTCP::send(Buffer* b) {
    if (!isOpen()) return NotOpen;
    b->getHeader().setSource(getAddr());
    if (getDebug() > 1) b->getHeader().Dump(cout);
    b->getHeader().Export();
    error_type rc;
    if (isClient()) rc = mSocket->send(b->getHeader().getDest(), b);
    else            rc = mCSocket->send(b->getHeader().getDest(), b);
    if (rc) b->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportTCP::receive(MsgHeader& hdr, char* Data, size_type maxlen, 
			    wtime_type timeout) {
    Buffer* buf;
    index_type rc = receive(&buf, timeout);
    if (rc) return rc;
    hdr = buf->getHeader();
    size_type len = hdr.getMsgLength();
    if (len > maxlen) rc = SizeError;
    else              memcpy(Data, buf->getDataAddr(), len);
    buf->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportTCP::receive(MsgHeader& hdr, Message& msg, wtime_type timeout) {
    Buffer* buf;
    index_type rc = receive(&buf, timeout);
    if (rc) return rc;
    hdr = buf->getHeader();
    if (hdr.getMsgType() != msg.getType()) {
        if (getDebug()) {
	    cout << "Message type: " << hdr.getMsgType() << " not expected (" 
		 << msg.getType() << ")" << endl;
	}
        rc = Invalid;
    } else {
        TransInput in(*buf);
	try {
	    msg.setData(in);
	} catch (exception& e) {
	    if (getDebug()) {
	        cout << "Exception: " << e.what() 
		     << " in TransportTCP::receive" << endl;
	    }
	    rc = SizeError;
	}
    }
    buf->Return();
    return rc;
}

//======================================  Receive a message
error_type 
TransportTCP::receive(Buffer** Data, wtime_type timeout) {
    MsgHeader hdr;
    error_type rc;

    //---------------------------------- Get the message header
    if (!isOpen()) return NotOpen;
    if (isClient()) rc = mSocket->read(reinterpret_cast<char*>(&hdr), sizeof(hdr));
    else            rc = mCSocket->read(reinterpret_cast<char*>(&hdr), sizeof(hdr));
    if (rc) return rc;
    hdr.Import();
    hdr.getSource() = mSocket->getPartner();
    MsgAddr* s = &hdr.getSource();
    if (s->getIPAddr() == 0) s->setIPAddr(mSocket->getPartner().getIPAddr());
    if (getDebug() > 1) hdr.Dump(cout);

    //---------------------------------- Get an appropriate buffer.
    size_type msgLen = hdr.getMsgLength();
    if (mBuffPool && mBuffPool->getLength() >= msgLen+sizeof(MsgHeader)) {
        *Data = mBuffPool->getBuffer();
    } else {
        *Data = new Buffer(msgLen+sizeof(MsgHeader));
    }
    if (!*Data) return NoBuffer;
    (*Data)->getHeader() = hdr;
    if (isClient()) rc =  mSocket->read((*Data)->getDataAddr(), msgLen);
    else            rc = mCSocket->read((*Data)->getDataAddr(), msgLen);
    if (rc) (*Data)->Return();
    return rc;
}

//======================================  Wait for a message to arrive
error_type 
TransportTCP::waitMsg(wtime_type timeout) {
    if (!mSocket)   return NotOpen;
    if (isClient()) return mSocket->wait(timeout);
    if (!mCSocket) {
        error_type rc = mSocket->waitConnect(timeout, &mCSocket);
	if (rc) return rc;
	mCSocket->async();
	return rc;
    }
    return mCSocket->wait(timeout);
}

//======================================  Connect to a partner.
error_type 
TransportTCP::connect(const MsgAddr& addr) {
    if (!isOpen()) return NotOpen;
    error_type rc = mSocket->connect(addr);
    if (rc == SystemError && mDebug) {
        ::perror("Error in connect:");
    } else if (rc && mDebug) {
        cerr << "Error in connect: " << rc << endl;
    } else if (mDebug > 1) {
        cout << "Made connection to " << addr << endl;
    }
    return rc;
}

//======================================  Break connection.
error_type 
TransportTCP::disconnect(void) {
    error_type rc;
    if (!isOpen()) return NotOpen;
    if (isClient()) {
        rc = mSocket->disconnect();
    } else if (mCSocket) {
        rc = mCSocket->disconnect();
        mCSocket->Return();
	mCSocket = 0;
    }
    return OK;
}

void 
TransportTCP::setBufferPool(BufferPool* pool) {
    mBuffPool = pool;
}

//======================================  Set the debug level
void 
TransportTCP::setDebug(index_type level) {
    mDebug = level;
    if (isOpen()) mSocket->setDebug(level);
}
