#ifndef LMSG_MESSAGE_HH
#define LMSG_MESSAGE_HH

#include "lmsg/MsgHeader.hh"
#include <stdexcept>

namespace lmsg {

  class TransInput;
  class TransOutput;

  /**  Message defines the APi for a data message passed over the 
    *  Ligo Message Interface.
    *  @memo Message API.
    *  @author John Zweizig
    *  @version 1.1; Modified March 30, 2000
    *  @ingroup IO_lmsg
    */
  class Message {
  public:
    /**  Null default constructor.
      */
    Message(void) {}

    /**  Null default destructor.
      */
    virtual ~Message(void) {}

    /**  Get the length of the message.
      *  @memo Get message length.
      */
    virtual size_type size(void) const = 0;

    /**  Copy data from the message to an lmsg::TransOutput stream.
      *  @memo  Serialize the message.
      *  @param out Stream to which the message data are copied.
      */
    virtual void getData(TransOutput& out) const = 0;

    /**  Function to poll the message type.
      *  @memo Message type.
      *  @return Type code of the message.
      */
    virtual MsgHeader::MsgType getType(void) const = 0;

    /**  Copy data from an lmsg::TransOutput stream into the message.
      *  @memo  Deserialize the message.
      *  @param in Stream from which the message data are copied.
      */
    virtual void setData(TransInput& in) throw(std::runtime_error) = 0;

  };   // class lmsg::Message


  /**  Template for messages without text data.
    *  @memo Dataless message template.
    *  @author John Zweizig
    *  @version 1.1; Modified March 30, 2000
    */
  template<unsigned int MsgID>
  class HdrOnlyMsg : public Message {
  public:

    /**  Construct a message.
      *  @memo Default constructor.
      */
    HdrOnlyMsg(void) {}

    /**  Destroy a message.
      *  @memo Default destructor.
      */
    ~HdrOnlyMsg(void) {}

    /**  Get the message text length.
      *  @memo Text length.
      *  @return Always zero for dataless messages.
      */
    size_type size(void) const {return 0;}

    /**  No data to copy - null routine.
      *  @memo Get message data.
      */
    void getData(TransOutput& out) const {}

    /**  Get the message type.
      *  @memo Message type.
      *  @return As specified by template argument.
      */
    MsgHeader::MsgType getType(void) const {return MsgID;}

    /**  No data to copy - Null routine.
      *  @memo Set message data.
      */
    void setData(TransInput& in) throw(std::runtime_error) {}
  private:
  };

}      // namespace lmsg

#endif //  def(LMSG_MESSAGE_HH)
