/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef LMSG_BUFFER_HH
#define LMSG_BUFFER_HH
#include "lmsg/MsgHeader.hh"

namespace lmsg {
    class BufferPool;
    class MsgHeader;

    /**  Preallocated uffer used in csocket I/O.
      *  @brief Socket I/O buffer.
      *  @author J. Zweizig
      *  @version 1.0; Last modified February 26, 2008
      *  @ingroup IO_lmsg
      */
    class Buffer {
    public:
	/**  Construct an empty buffer of a specified size. If the data address
	  *  is null the buffer is marked as local and a data segment of the
	  *  specified length is allocated. The allocated buffer is released
	  *  when the buffer is deleted.
	  *  @brief Constructor.
	  *  @param size Length of buffer
	  *  @param data Pointer to buffer to be used.
	  */
	Buffer(size_type size, char* data=0);

	/**  Destroy a buffer and free the data storage if it is local.
	  *  @brief Destructor.
	  */
	~Buffer(void);

	//------------------------------ Accessors

	/**  Get a constant pointer to the start of the buffer.
	  *  @brief Get buffer pointer.
	  *  @return Constant pointer to the buffer.
	  */
	const char* getBuffer(void) const;

	/**  Get a constant pointer to the start of the message data.
	  *  @brief Get data pointer.
	  *  @return Constant pointer to the message data.
	  */
	const char* getDataAddr(void) const;

	/**  Get a constant reference to the header of the message stored in 
	  *  the buffer.
	  *  @brief Get a header reference.
	  *  @return Constant reference to the message header.
	  */
	const MsgHeader& getHeader(void) const;

	/**  Get total message (data + header) length.
	  *  @brief Get message length.
	  *  @return Total message length.
	  */
	size_type   getLength(void) const;

	/**  Get the buffer length.
	  *  @brief Buffer length.
	  *  @return The buffer length
	  */
	size_type   getSize(void) const;

	/**  Get maximum allowable length of the message data.
	  *  @brief Get maximum data length.
	  *  @return Maximum allowable data length.
	  */
	size_type   getDataSize(void) const;

	/**  Get the current length of the data portion of the message.
	  *  @brief Current data length.
	  *  @return Current length of data portion of message.
	  */
	size_type   getDataLength(void) const;

	//------------------------------ Mutators
	/**  Get a writable pointer to the start of the buffer.
	  *  @brief Get buffer pointer.
	  *  @return Pointer to the buffer.
	  */
	char* getBuffer(void);

	/**  Get a writable pointer to the start of the message data.
	  *  @brief Get data pointer.
	  *  @return Pointer to the message data.
	  */
	char* getDataAddr(void);


	/**  Get a reference to the header of the message stored in 
	  *  the buffer.
	  *  @brief Get a header reference.
	  *  @return Reference to the message header.
	  */
	MsgHeader& getHeader(void);

	/**  Return the buffer to its pool.
	  *  @brief Return the buffer to pool.
	  */
	void Return(void);

	/**  Set the total message length.
	  *  @brief Set total message length
	  *  @param length Message length.
	  */
	void setLength(size_type length);

	/**  Set the message data length. Note that the header length is 
	  *  added to the data length to give the total message length.
	  *  @brief Set the data length
	  *  @param length Message data length.
	  */
	void setDataLength(size_type length);

	/**  Specify the pool to which the buffer will be returned.
	  *  @brief Set the buffer pool address.
	  *  @param pool reference to pool the buffer is to be returned to.
	  */
	void setPool(BufferPool& pool);
  private:
    size_type   mSize;
    size_type   mLength;
    BufferPool* mPool;
    char*       mData;
    bool        mLocalData;
  }; // Buffer class
} // namespace lmsg

inline const char* 
lmsg::Buffer::getBuffer(void) const {
    return mData;
}

inline char*
lmsg::Buffer::getBuffer(void) {
    return mData;
}

inline const char* 
lmsg::Buffer::getDataAddr(void) const {
    return mData + sizeof(MsgHeader);
}

inline char*
lmsg::Buffer::getDataAddr(void) {
    return mData + sizeof(MsgHeader);
}

inline lmsg::size_type
lmsg::Buffer::getDataLength(void) const {
    return mLength - sizeof(MsgHeader);
}

inline lmsg::size_type
lmsg::Buffer::getDataSize(void) const {
    return mSize - sizeof(MsgHeader);
}

inline const lmsg::MsgHeader& 
lmsg::Buffer::getHeader(void) const {
    return *(const MsgHeader*)mData;
}

inline lmsg::MsgHeader& 
lmsg::Buffer::getHeader(void) {
    return *(MsgHeader*)mData;
}

inline lmsg::size_type
lmsg::Buffer::getLength(void) const {
    return mLength;
}

inline lmsg::size_type
lmsg::Buffer::getSize(void) const {
    return mSize;
}

#endif // LMSG_BUFFER_HH
