/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef HTML_TABLE_HH
#define HTML_TABLE_HH

#include "html/compound.hh"
#include "html/Attrib.hh"
#include <vector>
#include <string>
#include <stdexcept>

namespace html {
   class align;

   /**  The HTML cell class is a compound object used inside a table.
     *  @memo Table cell class.
     *  @author J, Zweizig
     *  @version 1.0; Modified January 25, 2001
     *  @ingroup IO_html
     */
   class cell : public compound {
   public:
      /**  Construct an empty cell.
        *  @memo Default constructor.
	*/
      cell(void);

      /**  Destroy a cell.
        *  @memo Cell destructor
	*/
      virtual ~cell(void);

      /**  Create an exact copy of the cell. This implements the virtual
        *  html::object function.
	*  @return Pointer to a new cell object.
	*/
      cell* clone(void) const;

      /**  Return the html tag associated with a cell in an html table.
        *  @memo return the Cell tag ("td").
	*  @return Constant pointer to a string containing "td"
	*/
      const char* getTag(void) const {return "td";}

      /**  Set the alignment attribute for this cell.
        *  @memo Set the alignment attribute.
	*  @param a Alignment attribute value.
	*/
      void setAlign(const std::string& a);

      /**  Add the binary save attribute with the specified value.
        *  @memo set the save attribute.
	*  @param on Initial value for the save attribute.
	*/
      void setSave(bool on);
   };


   /**  A %header_cell is just like a %cell except it is printed in a \c \<th\>
     *  tag block. It is implemented by basing the %header_cell class on the
     *  %cell class and oerriiding the \c getTag method
     *  @author J, Zweizig
     *  @version 1.0; Modified January 25, 2001
     *  @ingroup IO_html
     */
   class header_cell : public cell {
   public:
      /**  Construct an empty cell.
        *  @memo Default constructor.
	*/
      header_cell(const std::string& txt="");

      /**  Destroy a cell.
        *  @memo Cell destructor
	*/
      virtual ~header_cell(void);

      /**  Create an exact copy of the cell. This implements the virtual
        *  html::object function.
	*  @return Pointer to a new cell object.
	*/
      header_cell* clone(void) const;

      /**  Return the html tag associated with a cell in an html table.
        *  @memo return the Cell tag ("td").
	*  @return Constant pointer to a string containing "th"
	*/
      const char* getTag(void) const {return "th";}
   };

   /**  The html table class stores a representation of an html table. The
     *  table contains an arbitrary number of rows and columns with each 
     *  entry containing a pointer to an html cell object. Each column has
     *  an optional header containing a title and attributes which are 
     *  inherited by all cells in the column.
     *  @memo Html Table object.
     *  @author J. Zweizig
     *  @version 1.1; Modified May 2, 2001
     *  @ingroup IO_html
     */
   class table : public object {
   public:
      /**  Construct an empty table. 
        *  @memo Default constructor.
	*/
      table(void);

      /**  Construct an empty html table and set the title.
        *  \brief Construct a table with a title
	*  \param title Initial title string
	*/
      table(const std::string& title);

      /**  Construct an exact copy of the specified table.
        *  \brief Copy constructor.
	*  \param x Table object to be copied.
	*/
      table(const table& x);

      /**  Destroy the table.
        *  @memo Table destructor.
	*/
      virtual ~table(void);

      /**  Create an exact copy of the table. This implements the virtual
        *  html::object function.
	*  @memo Clone a table.
	*  @return Pointer to a new table object.
	*/
      table* clone(void) const;

      /**  Set the table contents to that of the argument table.
        *  \brief Table assignment operator.
	*  \param x Table whose contents are to be copied.
	*  \return Reference to the modified table.
	*/
      table& operator=(const table& x);

      /**  Write the table to an HTML document stream.
        *  \brief Write the table to the HTML document.
	*  \param out Reference to an html writer.
	*/
      void write(writer& out) const;

      /**  Add a column to the table. All needed columns should be added
        *  to the table before adding rows or filling in cells.
	*  \brief Add a column.
	*  \param title Column title
	*  \return Number of columns.
	*/
      int  addColumn(const std::string& title);

      /**  Add an empty row to the end of the table. The row number is 
        *  returned.
	*  \brief Add a row to the table
	*  \return Row number.
	*/
      int  addRow(void);

      /**  Erase the specified row(s) from the table. All valid cells in 
        *  the deleted rows are deleted.
	*  @memo Erase one or more rows.
	*  @param row Number of first row to be erased
	*  @param count Number of rows to be erased
	*/
      void eraseRow(int row, int count=1) throw(std::runtime_error);

      /**  Insert floating point numeric data into the cell at the specified 
        *  row and column. If a cell exists at the specified location, its
	*  contents are replaced. Otherwise a new cell is created.
	*  @brief Replace cell contents with formatted number.
	*  @param row Row number of cell
	*  @param col Column number of cell
	*  @param x   Numeric value to be placed in cell.
	*  @param wid Field width for integer.
	*  @param prec Number of places after the decimal point.
	*/
      void insertData(int row, int col, double x, int wid=-1, int prec=-1) 
      throw(std::runtime_error);

      /**  Insert integer data with an arbitrary radix into the cell at the 
        *  specified row and column. If a cell exists at the specified 
	*  location, its contents are replaced. Otherwise a new cell is 
	*  created.
	*  @brief Replace cell contents with formatted integer.
	*  @param row Row number of cell
	*  @param col Column number of cell
	*  @param x   Integer value to be placed in cell.
	*  @param wid Field width for integer.
	*  @param radix Radix for integer.
	*/
      void insertData(int row, int col, long x, int wid=-1, int radix=10) 
      throw(std::runtime_error);

      /**  Insert text data into the cell at the specified row and column. 
        *  If a cell exists at the specified location, its contents are 
	*  replaced. Otherwise a new cell is created.
	*  @brief Replace cell contents with specified string.
	*  @param row Row number of cell
	*  @param col Column number of cell
	*  @param txt String to replace cell contents.
	*/
      void insertData(int row, int col, const std::string& txt) 
      throw(std::runtime_error);

      /**  Insert an arbitrary html object into the cell at the specified 
        *  row and column. If a cell exists at the specified location, its
	*  contents are replaced. Otherwise a new cell is created.
	*  @brief Replace cell contents with specified data.
	*  @param row Row number of cell
	*  @param col Column number of cell
	*  @param obj Html object to replace cell contents.
	*  @exception runtime_error 
	*/
      void insertData(int row, int col, const object& obj) 
      throw(std::runtime_error);

      /**  Insert an empty row in front of the specified row number. The
        *  new row will have number \c row and all existing rows numbered
	*  <tt>i >= row</tt> will have their row number incremented. If \c row
	*  is not valid (<tt>row < 0</tt> or <tt>row > nRow</tt>).
	*  @brief Add a row to the table.
	*  @exception runtime_error exception will be thrown after creating 
	*  a new row at the end of the table.
	*  @param row Position to insert row.
	*/
      void insertRow(int row) throw(std::runtime_error);

      /**  Get the current number of rows in the table.
        *  @memo Get the number of rows.
	*  @return Number of rows in table.
	*/
      int getNRow(void) const;

      /**  Get the number of columns defined in the table.
        *  @memo Get the number of columns.
	*  @return Number of columns in table.
	*/
      int getNColumn(void) const;

      /**  Get a reference to the header for the specified column.
        *  @memo Get a column header reference.
	*  @param col Column number.
	*  @return Reference to header cell of specified column.
	*/
      cell& refHeader(int col) throw(std::runtime_error);

      /**  Get a constant reference to the header for the specified column.
        *  @memo Get a column header reference.
	*  @param col Column number.
	*  @return Constant reference to header cell of specified column.
	*/
      const cell& refHeader(int col) const throw(std::runtime_error);

      /**  Get a reference to the cell at a specified location within the 
        *  table.
	*  \brief Get a specified cell reference.
	*  \param row Row number
	*  \param col Column number
	*  \return Reference to specified cell.
	*/
      cell& refCell(int row, int col) throw(std::runtime_error);

      /**  Get a constant reference to the cell at a specified location 
        *  within the table.
	*  \brief Get a specified cell reference.
	*  \param row Row number
	*  \param col Column number
	*  \return Constant reference to specified cell.
	*/
      const cell& refCell(int row, int col) const throw(std::runtime_error);

      /**  Set the table border attribute. If the border attribute is set, 
        *  a border is drawn around each cell of the table.
	*  \brief Enable/disable cell borders.
	*  \param On Draw cell borders if true.
	*/
      void setBorder(bool On=true);

   private:
      /**  Internal function to return the absolute index for the cell at
       *  the given row and column position.
       */
      int inxrc(int row, int col) const throw(std::runtime_error);

   private:
      std::string mTitle;
      int mRow;
      typedef std::vector<header_cell> ColHdr;
      ColHdr    mCTitle;
      attList   mAttrib;
      typedef std::vector<cell*> TableData;
      TableData mData;
   };

}

//======================================  Inline functions
inline int 
html::table::getNColumn(void) const {
   return mCTitle.size();
}

inline int 
html::table::getNRow(void) const {
   return mRow;
}

inline int
html::table::inxrc(int row, int col) const throw(std::runtime_error) {
   if (row >= mRow || row < 0) throw std::runtime_error("Invalid Row.");
   int nCol = getNColumn();
   if (col >= nCol || col < 0) throw std::runtime_error("Invalid Column.");
   return row*nCol + col;
}

inline const html::cell& 
html::table::refCell(int row, int col) const throw(std::runtime_error) {
   return *mData[inxrc(row, col)];
}

inline html::cell& 
html::table::refCell(int row, int col) throw(std::runtime_error) {
   return *mData[inxrc(row, col)];
}

inline const html::cell& 
html::table::refHeader(int col) const throw(std::runtime_error) {
   if (col<0 || col>=getNColumn()) throw std::runtime_error("Invalid Column.");
   return mCTitle[col];
}

inline html::cell& 
html::table::refHeader(int col) throw(std::runtime_error) {
   if (col<0 || col>=getNColumn()) throw std::runtime_error("Invalid Column.");
   return mCTitle[col];
}

#endif  // HTML_TABLE_HH
