/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: frameio							*/
/*                                                         		*/
/* Module Description: Frame IO routines				*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 0.1	 26Sep00  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: framefast.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#ifndef _LIGO_FRAMEIO_H
#define _LIGO_FRAMEIO_H


#include <time.h>
#include <string>


namespace framefast {


/** @name frame input/output
    This header exports frame input/output routines.
   
    @memo Fast frame input/output
    @author Written September 2000 by Daniel Sigg
    @version 1.0
    @ingroup IO_framefast
 ************************************************************************/

//@{


   /// Abstract helper object to store a frame
   class basic_frame_storage {
   public:
      /// Default constructor
      basic_frame_storage () {
      }
      /// Destructor (Overwrite! Call your own release funtion)
      virtual ~basic_frame_storage() {
      }
      /// Release the frame (protect against multiple calls!)
      virtual void reset() = 0;
      /// Return a pointer to the data (return zero if invalid)
      virtual const char* data() const = 0;
      /// Return the length of the data array
      virtual int size() const = 0;
      /// Returns true if container is empty
      virtual bool empty() const {
         return data() != 0; }
   
      /// Get name
      const char* name() const {
         return fName.c_str(); }
      /// Set name
      void setname (const char* n) {
         fName = n ? n : ""; }
   protected:
      /// filename
      std::string	fName;
   
   private:
      basic_frame_storage (const basic_frame_storage&);
      basic_frame_storage& operator= (const basic_frame_storage&);
   };


   /// Frame stored in memory
   class memory_frame_storage : public basic_frame_storage {
   public:
      /// Default constructor
      memory_frame_storage (const char* data, int len, bool own = true) 
      : fOwn (own), fData (data), fLength (len) {
      }
      /// Destructor (Overwrite! Call your own release funtion)
      virtual ~memory_frame_storage() {
         reset(); }
      /// Release the frame (protect against multiple calls!)
      virtual void reset();
      /// Return a pointer to the data (return zero if invalid)
      virtual const char* data() const {
         return fData; }
      /// Return the length of the data array
      virtual int size() const {
         return fLength; }
   
   protected:
      /// Own data?
      bool		fOwn;
      /// Data pointer
      const char* 	fData;
      /// Size of data
      int		fLength;
   };


   /// Frame file storage (read file into memory)
   class file_frame_storage : public memory_frame_storage {
   public:
      /// Default constructor
      explicit file_frame_storage (const char* filename = 0) 
      : memory_frame_storage (0, 0) {
         if (filename) load (filename); }
     /// Load a file
      bool load (const char* filename);
   };


   /// Memory mapped files for frames
   class mmap_frame_storage : public basic_frame_storage {
   public:
      /// Default constructor
      explicit mmap_frame_storage (const char* filename = 0) 
      : fAddr (0), fLength (0) {
         if (filename) map (filename); }
      /// Destructor (Overwrite! Call your own release funtion)
      virtual ~mmap_frame_storage() {
         reset(); }
      /// Release the frame (protect against multiple calls!)
      virtual void reset();
      /// Return a pointer to the data (return zero if invalid)
      virtual const char* data() const {
         return fAddr; }
      /// Return the length of the data array
      virtual int size() const {
         return fLength; }
     /// Map a file
      bool map (const char* filename);
   
   protected:
      /// Data pointer
      const char* 	fAddr;
      /// Size of data
      int		fLength;
   };


   /// Frame storage auto pointer
   class frame_storage_ptr {
   public:
      /// Default constructor
      frame_storage_ptr () : fFrame (0) {
      }
      /// Constructor (adopts the frame storage object)
      explicit frame_storage_ptr (basic_frame_storage* frame) : 
      fFrame (frame) {
      }
      /// Copy constructor
      frame_storage_ptr (const frame_storage_ptr& fptr) {
         fFrame = fptr.release(); }
      /// Copy operator
      frame_storage_ptr& operator= (const frame_storage_ptr& fptr) {
         if (this != &fptr) {
            reset(); fFrame = fptr.release(); }
         return *this; }
      /// Destructor
      ~frame_storage_ptr () {
         reset(); }
      /// Set a new frame
      void set (basic_frame_storage* frame) {
         reset(); fFrame = frame; }
      /// Reset
      void reset () {
         if (fFrame) delete fFrame; fFrame = 0; }
      /// Release
      basic_frame_storage* release () const {
         basic_frame_storage* frame = fFrame; fFrame = 0; 
         return frame; }
      /// Data pointer
      const char* data () const {
         return fFrame ? fFrame->data() : 0; }
      /// Data size 
      int size () const {
         return fFrame ? fFrame->size() : 0; }
     /// Original name
      const char* name () const {
         return fFrame ? fFrame->name() : 0; }
   
   protected:
      /// Frame storage
      mutable basic_frame_storage*	fFrame;
   };



   /// Abstract helper object to write a frame
   class basic_frameout {
   public:
      /// address and length of a data block
      struct src_dest_t {
         /// address
         const char* 	fAddr;
         /// length
         int 		fLen;
      };
   
      /// Constructor
      basic_frameout() : fSofar (0) {
      }
      /// Denstructor
      virtual ~basic_frameout() {
      }
      /** Open method.
          @param len Length of frame
          @return true if successful
       ******************************************************************/
      virtual bool open (int len = 0) = 0;
      /** Close method.
       ******************************************************************/
      virtual void close() = 0;
      /** Output method.
          @param p Pointer to data block
          @param len Length of data block
          @return true if successful
       ******************************************************************/
      virtual bool write (const char* p, int len) = 0;
      /** Output method. Scatter/gather write method.
          @param s Pointer to an array of data block addresses
          @param slen Length of array of data block addresses
          @return true if successful
       ******************************************************************/
      virtual bool write (const src_dest_t* s, int slen) {
         for (int i = 0; i < slen; i++) {
            if (!write (s[i].fAddr, s[i].fLen))
               return false; }
         return true; }
      /** Output size method. 
          @return Number of bytes written so far
       ******************************************************************/
      virtual int length() const {
         return fSofar; }
   
   protected:
      /// written so far
      int		fSofar;
   };


/** Writes a frame to memory.
    @memo Write a frame to memory.
    @ingroup IO_framefast
 ************************************************************************/
   class memory_out : public basic_frameout, public memory_frame_storage {
   public:
      /** Constructor.
          @param data    Data pointer
	  @param maxsize Length of free memory buffer.
	  @param own     Buffer owned by object of true.
       ******************************************************************/
      memory_out (char* data = 0, int maxsize = 0, bool own = false)
      : memory_frame_storage (data, maxsize, own) {
      }
      /** Destructor.
       ******************************************************************/
      virtual ~memory_out () {
         close (); }
      /** Open method.
          @param len Length of frame
          @return true if successful
       ******************************************************************/
      virtual bool open (int len = 0);
      /** Close method.
       ******************************************************************/
      virtual void close() {
      }
      /** Output method.
          @param p Pointer to data block
          @param len Length of data block
          @return true if successful
       ******************************************************************/
      virtual bool write (const char* p, int len);
      /** Output method. Scatter/gather write method.
          @param s Pointer to an array of data block addresses
          @param slen Length of array of data block addresses
          @return true if successful
       ******************************************************************/
      virtual bool write (const src_dest_t* s, int slen) {
         return basic_frameout::write (s, slen); }
   };


/** Adds a frame to an already open file descriptor.
    @memo Write a frame to a file descriptor.
    @ingroup IO_framefast
 ************************************************************************/
   class desc_out : public basic_frameout {
   public:
      /** Constructor.
          @param filedesc File descriptor
       ******************************************************************/
      desc_out (int filedesc) : fd (filedesc) {
      }
      /** Open method.
          @param len Length of frame
          @return true if successful
       ******************************************************************/
      virtual bool open (int len = 0) {
         return true; }
      /** Close method.
       ******************************************************************/
      virtual void close() {
      }
      /** Output method.
          @param p Pointer to data block
          @param len Length of data block
          @return true if successful
       ******************************************************************/
      virtual bool write (const char* p, int len);
      /** Output method.
          @param s Pointer to an array of data block addresses
          @param slen Length of array of data block addresses
          @return true if successful
       ******************************************************************/
      virtual bool write (const src_dest_t* s, int slen);
   
   protected:
      /// file descriptor
      int		fd;
   };


/** Writes a frame to a file.
    @memo Write a frame to a file.
    @ingroup IO_framefast
 ************************************************************************/
   class file_out : public desc_out {
   public:
      /** Constructor.
          @param filename Name of file
       ******************************************************************/
      file_out (const char* filename) 
      : desc_out (-1), fFilename (filename) {
      }
      /** Desctructor.
       ******************************************************************/
      virtual ~file_out () {
         close (); }
      /** Open method.
          @param len Length of frame
          @return true if successful
       ******************************************************************/
      virtual bool open (int len = 0);
      /** Close method.
       ******************************************************************/
      virtual void close();
   
   protected:
      /// Filename
      std::string	fFilename;
   };


//@}

}

#endif  // _LIGO_FRAMEIO_H


