/* version $Id: foton.cc 7389 2015-06-11 18:51:56Z james.batch@LIGO.ORG $ */
/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <TROOT.h>
#include <TApplication.h>
#include <TSystem.h>
#include "foton.hh"
#include "filterwiz/TLGFilterWizard.hh"
#include "goption.hh"

// Allow output of debug info to a file, /tmp/fotonDebug.txt JCB
fstream debugLog ;

   // ROOT object
   TROOT root ("GUI", "Foton");

   using namespace std;
   using namespace filterwiz;


   const char* const help_text =
   "Usage: foton [options] [filename]\n"
   " where options may be the following:\n"
   "       -w             read-write mode (readonly default)\n"
#ifdef LEGACY_WRITE
   "       -l             legacy write mode\n"
#endif
   "       -a             Allow any sample rate\n"
   "       -p 'path'      path for filter files\n"
   "       -d 'display'   X windows display\n"
   "       -c             Read and write the file without opening the GUI\n"
   "       -o 'outfile'   Used with the -c option, specify a filename to write the filters\n"
   "       -m 'mergefile' Merge matlab designs into existing filter file\n"
   "       -s 'filt:sec'  filter module/section selection\n"
   "       -h             this help\n"
   "\nFoton version " VERSION "\n$Id: foton.cc 7389 2015-06-11 18:51:56Z james.batch@LIGO.ORG $\n";


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// main                                                                 //
//                                                                      //
// Main Program                                                         //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
extern "C"
   int main (int argc, char** argv)
   {
      // JCB - for non-GUI processing of matlab merge file.
      int merge_mode = 0 ;
      int debug_enable = 1 ;
      int batch_mode = 0 ;
      int outfile_used = 0 ;

#ifdef LEGACY_WRITE
      gdsbase::option_string opt (argc, argv, "Llcvwm:o:p:d:s:h");
#else
      gdsbase::option_string opt (argc, argv, "Lacvwm:o:p:d:s:h");
#endif
      // help
      if (opt.help() || (opt.argnum() > 1)) {
         cout << help_text << endl;
         exit (0);
      }
      // This sends deep debugging stuff to a log file. JCB
      if (opt.opt('L'))
      {
	 debugLog.open("/tmp/fotonDebug.txt", fstream::out | fstream::trunc) ;
      }
      // If the file didn't open or the option wasn't selected, send the output to /dev/null.
      if (!debugLog.is_open())
      {
	 debugLog.open("/dev/null") ;
      }

      // Set the batch mode flag to read, write, and exit.  No GUI option,
      // this will allow a filter file to be generated as if it had been
      // saved without altering from the GUI.  Useful when the filter file
      // contains errors such as orphaned filter modules or no design strings.
      if (opt.opt('c'))
      {
	 batch_mode = 1 ;
      }

      // silence output if not verbose
      if (!opt.opt ('v')) {
         int i = ::open ("/dev/null", 2);
         (void) dup2 (i, 2); // stderr, cerr  
	 debug_enable = 0 ;
      }

      // Filename rules:
      // The file that is opened depends on the -p <path> option as well as the filename given
      // on the command line.
      //
      // If an absolute path to the file is given as part of the filename, as in
      // "foton -p /ligo/home/controls /opt/rtcds/lho/h1/chans/H1PEMMX.txt"
      // the file is opened from the directory given.  If no -o <outpath> is used, the file 
      // will be written to the same file and path.  A -p <path> option will be ignored in this case.
      //
      // If a relative path to the file is given as part of the filename, as in
      // "foton -p /ligo/home/controls jcb/H1PEMMX.txt
      // the path to the file will be created by prepending the path given in the -p option (or "./" if
      // the -p option isn't used) with the path part of the filename (if any) and the filename.  In the example
      // above, the full pathname is /ligo/home/controls/jcb/H1PEMMX.txt.
      //
      // If no relative path is given as part of the filename, if the -p <path> option is used
      // the file will be found in <path>/filename.

      // filename
      string filename;
      string shortname;
      string filepath;
      if (opt.argnum() == 1) {
         filename = *opt.argbegin();
         if (filename.find ("/") != string::npos) {
            filepath = gSystem->DirName (filename.c_str());
         }
         shortname = gSystem->BaseName (filename.c_str());
      }

      // Output file
      string outfile ;
      string outfileshort ;
      string outfilepath ;
      if (opt.getOpt ('o', outfile)) {
	 if (outfile.empty()) {
	    // Error, need output file name.
	 }
	 else {
	    outfile_used = 1 ;
	    if (outfile.find("/") != string::npos) {
	       outfilepath = gSystem->DirName(outfile.c_str()) ;
	    }
	    outfileshort = gSystem->BaseName(outfile.c_str()) ;
	 }
      }

      // path
      string path;
      string outpath ;
      opt.getOpt ('p', path);
      outpath = path ;
      if ((!filepath.empty() && (filepath[0] == '/')) || path.empty()) {
         path = filepath;
      }
      else if (!filepath.empty()) {
         path += "/" + filepath;
      }
      if (outfile_used) {
	 if ((!outfilepath.empty() && (outfilepath[0] == '/')) || outpath.empty()) {
	    outpath = outfilepath ;
	 }
	 else if (!outfilepath.empty()) {
	    outpath += '/' + outfilepath ;
	 }
      }

      // display
      string display;
      int rargc = 1;
      char* rargv[10];
      rargv[0] = argv[0];
      if (opt.getOpt ('d', display)) {
         rargv[1] = new char [12];
         strcpy (rargv[1], "-display");
         rargv[2] = new char [display.size() + 2];
         strcpy (rargv[2], display.c_str());
         rargc = 3;
      }
      // read-write mode
      bool rw_mode = opt.opt ('w');
      
#ifdef LEGACY_WRITE
      // Legacy or aLIGO write mode. - JCB
      bool legacy_mode = opt.opt('l') ;
#else
      bool legacy_mode = 0 ;
#endif
      
      // Allow any sample rate option.
      bool any_sample = opt.opt('a') ;
   
      // Batch Matlab merge option. Requires a matlab file
      // and a filter file, output file is optional and if
      // not specifies the filter file will be used.
      string matlab_file ;
      if (opt.getOpt('m', matlab_file))
      {
         if (matlab_file.empty() || shortname.empty())
         {
            cerr << "foton error: -m requires a " ;
            if (matlab_file.empty())
            {
               cerr << "matlab design file " ;
               if (shortname.empty())
               {
                  cerr << "and a" ;
               }
            }
            if (shortname.empty())
            {
               cerr << "filter file " ;
            }
            cerr << "to be specified on the command line.\n" ;
            return 1 ;
         }
         merge_mode = 1 ;
      }

      // For the batch mode, just create a new FilterFile, read the 
      // input file, and write the file.  Return 0 when done if successful.
      if (batch_mode)
      {
	 // realname is the name of the actual input file, in case it's
	 // a symbolic link.
	 string realname ;
	 string realoutname ;

	 // Create a new FilterFile object.
	 FilterFile  filter_file ;

	 // Make sure there's an input file.
	 if (shortname.empty())
	 {
	    cout << "Error: no filter file specified." << endl ;
	    return -1 ;
	 }

         // Find the real file name in case it's a link. That way the write
         // function won't overwrite the symbolic link.
         // If no path was specified, give getRealFilename a "." to work with.
         if (path.empty())
            path = "." ;
         realname = filter_file.getRealFilename(shortname, path) ;
         if (realname.empty())
         {
            cout << "Error: Could not determine path to file " << shortname << endl ;
            return -1 ;
         }

	 // Find the real file name of the output file in case it's a link.
	 if (outfile_used) 
	 {
	    if (outpath.empty())
	       outpath = "." ;
	    realoutname = filter_file.getRealFilename(outfileshort, outpath) ;
	    if (realoutname.empty())
	    {
	       cout << "Error: Could not determine path to file " << outfileshort << endl ;
	       return -1 ;
	    }
	 }


	 // Read the filter file.
	 // The act of reading the filter file will correct most errors.  The only 
	 // error that might occur is if a filter can't be created from the 
	 // coefficients present in the filter file.  
	 if (!filter_file.read(realname.c_str()))
	 {
	    cout << "Error: File " << realname << " could not be read." << endl ;
	    return -1 ;
	 }

	 // If there were errors, print them.
	 if (!filter_file.errorsEmpty())
	 {
	    filter_file.printFileErrors() ;
	 }

	 if (outfile_used)
	 {
	    if (!filter_file.write(realoutname.c_str()))
	    {
	       cerr << "Write failed." << endl ;
	       return -1 ;
	    }
	 }
	 else
	 {
	    // Write the filter file.
	    if (!filter_file.write(realname.c_str()))
	    {
	       cerr << "Write failed." << endl ;
	       return -1 ;
	    }
	 }
	 // Writing the file should have cleared the errors from reading.  
	 // If there's new errors, they are from writing.  Print them.
	 if (!filter_file.errorsEmpty())
	 {
	    cout << "Error(s) from writing file:" << endl ;
	    filter_file.printFileErrors() ;
	 }

	 return 0 ;

      }

      // For the matlab merge, just create a new FilterFile,
      // read the input file, merge the matlab file, and write the results.
      // Return 0 when done if successful.
      if (merge_mode)
      {
	 // realname is the name of the actual input file, in case it's
	 // a symbolic link.
	 string realname ;

	 // Create a new FilterFile object.
	 FilterFile  filter_file ;

	 // Sanity checks. Make sure there's input files to work with.
	 if (filename.empty())
	 {
	    cout << "Error: no filter file specified." << endl ;
	    return -1 ;
	 }
	 if (matlab_file.empty())
	 {
	    cout << "Error: no matlab merge file specified." << endl ;
	    return -1 ;
	 }

	 // Find the real file name in case it's a link. That way the write
	 // function won't overwrite the symbolic link.
	 // If no path was spcified, give getRealFilename a "." to work with.
	 if (filepath.empty())
	    filepath = "." ;
	 realname = filter_file.getRealFilename(filename, filepath) ;
	 if (realname.empty())
	 {
	    cout << "Error: Could not determine path to file " << filename << endl ;
	    return -1 ;
	 }

	 // Set the debug level for the merge functions, this allows errors to
	 // be printed if it fails.
	 filter_file.setMergeDebug(1);

	 // Read the filter file.
	 (void) filter_file.read(realname.c_str()) ;
	 if ((filter_file.errorsEmpty()))
	 {
	    cerr << "Read of " << realname << " succeeded!" << endl ;
	    // Merge the matlab file. (0 return for success)
	    if (filter_file.merge(matlab_file.c_str()) == 0)
	    {
	       cerr << "Merge of " << matlab_file << " successful!" << endl ;
	       // Write the merged filter file.
	       if (!filter_file.write(realname.c_str()))
	       {
		  cerr << "Write failed." << endl ;
		  return -1 ;
	       }
	       else
		  cerr << "Write succeeded!" << endl ;
	    }
	    else
	    {
	       // Error, print the merge errors.
	       cerr << "Merge failed!" << endl ;
	       filter_file.printMergeErrors() ;
	       return -1 ;
	    }
	 }
	 else
	 {
	    // Couldn't read the filter file, print the errors.
	    cerr << "Read Failed!" << endl ;
	    filter_file.printFileErrors() ;
	    return -1 ;
	 }
	 // Successful, so return a 0.
	 return 0 ;
      }

      // silence output if not verbose. cerr done above.
      if (!opt.opt ('v')) {
         int i = ::open ("/dev/null", 2);
         (void) dup2 (i, 1); // stdout, cout
	 debug_enable = 0 ;
      }

      // Options for preselecting a filter module and section
      string filter_select ;
      string fmodule = "" ;
      int fsection = 0 ;
      if (opt.getOpt('s', filter_select))
      {
	 string::size_type pos ;
	 pos = filter_select.find(":") ;
	 fmodule = filter_select.substr(0, pos) ;
	 if (pos != string::npos)
	 {
	    fsection = atoi(filter_select.substr(pos+1).c_str()) ;
	 }
	 //cerr << "-s option: fmodule = " << fmodule << ", fsection = " << fsection << endl ;
      }

      // If this wasn't merge mode, continue with the interactive setup.

      // initializing root environment
      TApplication theApp ("Foton", &rargc, rargv);
   
      // create main window
      bool modal = false;
      std::string fdesign;
      TLGFilterWizard mainWindow (gClient->GetRoot(), modal, "Foton", fdesign,
                           path.empty() ? 0 : path.c_str(), 
                           shortname.empty() ? 0 : shortname.c_str(),
			   fmodule.empty() ? 0 : fmodule.c_str(), fsection,
			   any_sample);

      // mainWindow.Setup() always needs to be called regardless
      // of mode.  It causes the filter file to be read (default or 
      // the one specified on the command line).
      mainWindow.Setup() ; // TLGMainWindow::Setup() gets called.

      // Set the merge debug if verbose was specified.
      if (debug_enable)
         mainWindow.setMergeDebug(1) ;

      mainWindow.SetReadOnly (!rw_mode);
      mainWindow.SetLegacyWrite(legacy_mode) ;	// JCB

      // run the GUI
      theApp.Run();
   
      debugLog << "Foton exit." << endl ;
      debugLog.close() ;

      return 0;
   }

