/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_GAUSSBURST_HH
#define GENERATOR_GAUSSBURST_HH

#include "DataSource.hh"

namespace generator {

    /**  GaussBurst generates a burst of gaussian noise inside a gaussian
      *  amplitude envelope. The generated waveform is:
      *  \f[  x(t) = A \exp{-(t-to)^2/(2 \sigma^2)} * grn() \f]
      *
      *  The full list of parameters is:
      *  <table>
      *  <tr><td> Parameter </td>
      *      <td> Default </td>
      *      <td> Meaning </td>
      *  </tr>
      *  <tr><td>\c A </td>
      *      <td> - </td>
      *      <td> Gaussian envelope amplitude at \c t=to </td>
      *  </tr>
      *  <tr><td>\c Sigma </td>
      *      <td> - </td>
      *      <td>Gaussian envelope sigma in seconds</td>
      *  </tr>
      *  <tr><td>\c Width </td>
      *      <td> 6 </td>
      *      <td> Width specifies the half-width of the waveform to be
      *           generated and is specified in \f$ \sigma \f$. At the 
      *           default width value of 6.0, the gaussian envelope has 
      *           decreased to \f$ 1.5 \times 10^{-8}\f$ of its central 
      *           amplitude</td>
      *  </tr>
      *  </table> 
      *
      *  @memo Gaussian noise burst
      *  @author John Zweizig (john.zweizig@ligo.org)
      *  @version $Id: GaussBurst.hh 7277 2015-01-12 07:21:33Z john.zweizig@LIGO.ORG $
      */
    class GaussBurst : public DataSource {
    public:
	/**  Default source constructor.
	  *  @memo Default constructor.
	  */
	GaussBurst(void);

	/**  Parameterized GaussBurst source constructor.
	  *  @memo Parameterized source contructor.
	  *  @param Amp   Gaussian noise sigma at peak of envelope.
	  *  @param Sigma Sigma of gaussian envelope.
	  *  @param wid   Event duation in sigma units.
	  */
	GaussBurst(double Amp, double Sigma=0.0, double wid=6.0);

	/**  Destroy a GaussBurst source.
	  *  @memo Destructor.
	  */
	~GaussBurst(void);

	/**  Clone a gaussian burst source.
	  *  @memo Clone an object.
	  *  @return Pointer to newly created clone.
	  */
	GaussBurst* clone(void) const;

	/**  Data source type name string ("GaussBurst").
	  *  @memo Get source type.
	  *  @return pointer to data source type string.
	  */
	const char* getDataType(void) const;

	/**  Get the duration of generated signal.
	  *  @memo Generated signal diration.
	  *  @return Signal duration in seconds.
	  */
	Interval getDuration(void) const;

    private:
	/**  Get the digitized signal time series for the specified time 
	  *  epoch.
	  *  @memo Get the time series.
	  *  @param t0   Epoch start time.
	  *  @param dT   Epoch duration
	  *  @param N    Number of data.
	  *  @param data Output data buffer.
	  */
	void getSeries(const Time& t0, Interval dT, int N, gen_sample_type* data);
    };

    inline const char*
    GaussBurst::getDataType(void) const {
	return "GaussBurst";
    }

} // namespace generator

#endif  // GENERATOR_GAUSSBURST_HH
