/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "generator/DataSource.hh"
#include "generator/DampedSine.hh"
#include "generator/FrameData.hh"
#include "generator/GaussBurst.hh"
#include "generator/Metastable.hh"
#include "generator/Sinc.hh"
#include "generator/SineData.hh"
#include "generator/SinGauss.hh"
#include "generator/WhiteNoise.hh"
#include <iostream>

using namespace std;
using namespace generator;

//======================================  Dictionary initialization.
void
DataSource::init_dict(SourceDict& dict) {
    dict.insert(dict_node("SinGauss",   new SinGauss));
    dict.insert(dict_node("WhiteNoise", new WhiteNoise));
    dict.insert(dict_node("DampedSine", new DampedSine));
    dict.insert(dict_node("FrameData",  new FrameData));
    dict.insert(dict_node("Sine",       new SineData));
    dict.insert(dict_node("SineData",   new SineData));
    dict.insert(dict_node("Sinc",       new Sinc));
    dict.insert(dict_node("GaussBurst", new GaussBurst));
    dict.insert(dict_node("Metastable", new Metastable));
}

//======================================  Dictionary initialization.
void
DataSource::kill_dict(SourceDict& dict) {
    for (dict_iter i=dict.begin(); i!=dict.end(); i++) {
	delete i->second;
	i->second = 0;
    }
    dict.clear();
}

//======================================  Static name parser
DataSource*
DataSource::parse(const std::string& id, const SourceDict& dict) {
    enum {
      kInName,
      kInOp,
      kFndEq,
      kCparen
    } state(kInName);
    DataSource* rc(0);

    bool inquote(false), syntax(false);//, fnop(false);
    int argop(0);
    string SrcName, argName, argValue;
    const char* ptr = id.c_str();
    while (*ptr && !syntax) {
        char c = *ptr++;

	//------------------------------  Process quoted string
	if (inquote) {
	    argValue += c;
	    inquote = (c != '\'' && c != '\"');
	} 

	//------------------------------  Parse string.
	else {
	    switch (c) {
	    case '(':
	        if (state == kInName) {
		    const_dict_iter iter = dict.find(SrcName);
		    if (iter != dict.end()) {
		        rc = iter->second->clone();
		    }
		    syntax = !rc;
		    state = kInOp;
		} else if (state == kFndEq) {
		    argValue += c;
		    ++argop;
		    //fnop = true;
		} else {
		    syntax = true;
		}
		break;

	    case '=':
	        if (argop) argValue += c;
	        else       state = kFndEq;
	        break;

	    case ')':
	    case ',':
	        if (argop) {
		    argValue += c;
		    if (c == ')') argop--;
	        } else if (state == kFndEq) {
		    rc->setParameter(argName, argValue);
		    argName.clear();
		    argValue.clear();
		    //fnop = false;
		    if (c == ',') state = kInOp;
		    else          state = kCparen;
		} else {
		    syntax = true;
		}
	        break;

	    case '\'':
	    case '\"':
	        argValue += c;
	        if ( state == kFndEq ) inquote = true;
		else                   syntax  = true;  
		break;

	    default:
	        switch (state) {
	        case kInName:
		    SrcName  += c;
		    break;
		case kInOp:
		    argName  += c;
		    break;
		case kFndEq:
		    argValue += c;
		    break;
		default:
		    syntax = true;
		}
	    }
	}
    }

    //----------------------------------  Check for syntax error
    if (argop || state == kInOp) {
        cerr << "Mismatched parentheses in " << id << endl;
	syntax = true;
    } else if (inquote) {
        cerr << "Mismatched quotes in " << id << endl;
	syntax = true;
    } else if (syntax) {
        cerr << "Syntax error in " << id << endl;
	syntax = true;
    }
    if (syntax && rc) {
        delete rc;
	rc = 0;
    }

    //-----------------------------------  Record definition
    if (rc) {
        rc->mSeries.setName(id.c_str());
    }
    return rc;
}
