/* -*-  mode: c++; c-basic-offset: 4; -*- */
#ifndef EVAL_STACK_HH
#define EVAL_STACK_HH

#include <iosfwd>
#include <string>
#include <vector>

/**  The %eval_token class is an abstract API for tokens used on the eval 
  *  stack.
  *  \brief Token API for calculation engine.
  */
class eval_token {
public:
    /**  Evaluation data type representation.
     */
    typedef unsigned int datype;

public:
    /**  Virtual destructor.
     */
    virtual ~eval_token(void) {}

    /**  Make an identical copy of the token.
      *  \brief clone a token.
      *  \return pointer to the cloned token.
      */
    virtual eval_token* clone(void) const = 0;

    /**  Get the data type of this token.
      *  \brief get token data type.
      *  \return token type identifier.
      */
    virtual datype getType() const = 0;
    /**  Cast the token data to a boolean value.
      *  \brief Get token boolean value.
      *  \return Token boolean value.
      */ 
    virtual bool getLogical(void) const;

    /**  Cast the token to a numeric value.
      *  \brief Numeric value of token.
      *  \return Token numeric value.
      */
    virtual double getNumeric(void) const;

    /**  Cast the token to a string value.
      *  \brief String value of token.
      *  \return Token string value.
      */
    virtual std::string getString() const = 0;

    /**  Set the token to a boolean value.
      *  \brief Set to a boolean value.
      *  \param yorn Value to which token is to be set.
      */
    virtual void setLogical(bool yorn);

    /**  Set the token to a numeric value.
      *  \brief Set to a numeric value.
      *  \param val Value to which token is to be set.
      */
    virtual void setNumeric(double val);

    /**  Set the token to a string value.
      *  \brief Set to a string value.
      *  \param s Value to which token is to be set.
      */
    virtual void setString(const std::string& s) = 0;
};

/**  The eval_stack is an abstract API for the evaluation stack use by the 
  *  calculation engine. The evaluation stack holds currently active 
  *  intermediate results. The stack items are store as pointers to eval_token 
  *  instances.
  *  @memo Evaluation stack API for the calculation engine.
  *  @author John Zweizig
  *  @version 1.3; Last modified March 14, 2006
  */
class eval_stack {

public:
    /** Operation code enumeration.
      */
    enum OpsEnum {
	kNoop,        ///< No operation.
	kPushNumer,   ///< push a numeric value.
	kPushConst,   ///< Push a constant
	kPushSymbol,  ///< Push symbol contents
	kCallFc,      ///< Call a function
	kAdd,         ///< Add top elements of the stack
	kSub,         ///< Subtract second element from top
	kMpy,         ///< Multiply second element by top
	kDiv,         ///< Divide second element by top
	kOr,          ///< Boolean or top elements of the stack
	kAnd,         ///< Boolean and top elements of the stack
	kLogAnd,      ///< Boolean "and" logical values of top elements of stack
	kLogOr,       ///< Boolean "or" logical values of top elements of stack
	kCmpEq,       ///< Compare top elements, true if equal
	kCmpNE,       ///< Compare top elements, true if not equal
	kCmpLt,       ///< Compare top elements, true if top is less
	kCmpLE,       ///< Compare top elements, true if top is less or equal
	kCmpGt,       ///< Compare top elements, true if top is greater
	kCmpGE,       ///< Compare top elements, true if top is greater or equal
	kModulo,      ///< Modulo top element by second
	//-----------   kNOps (number of operators) must be last entry.
	kNOps         ///< Number of defined operations
    };

    /**  Operation level representation data type.
     */
    typedef int op_level;

public:
    /**  Evaluation stack constructor.
      *  \brief Default constructor.
      */
    eval_stack(void);

    /**  Evaluation stack destructor. 
      *  \brief Destructor.
      */
    virtual ~eval_stack(void);

    /**  Clear the evaluation stack.
      *  \brief Clear the stack.
      */
    void clear(void);

    /**  Print out the contents of the evaluation stack.
      *  \brief Print the stack contents.
      *  \param out Output stream to receive the stack dump.
      */
    virtual void dump(std::ostream& out) const;

    /**  Test whether there are any data on the stack.
      *  \brief Test for empty
      *  \return true if the stack is empty.
      */
    bool empty(void) const;

    /**  Perform an operation on the top stack entries.
      *  \brief evaluate operation on top stack elements.
      *  \param op Enumerated operation to be evaluated.
      */
    virtual void evaluate(OpsEnum op);

    /**  Return the enumerated operator ID for the specified operator string.
      *  \brief  Get the operator ID.
      *  \param op Operator string.
      *  \return Enumerated operation ID.
      */
    virtual OpsEnum getOpID(const std::string& op) const;

    /**  Get the operator level (priority).
      *  \brief Get operator priority.
      *  \param op Enumerated operation ID.
      *  \return Operator priority level.
      */
    virtual op_level getOpLevel(OpsEnum op) const;

    /**  Get the operator symbol.
      *  \brief Get operator string.
      *  \param op Enumerated operation ID.
      *  \return Constant pointer to the operator string.
      */
    virtual const char* getOpSym(OpsEnum op) const;

    /**  Look at a datum on the stack. The datum position is specified 
      *  relative to the top of the stack, with \c inx=1 indicating the 
      *  top entry, \c inx=2 indicates the next to the top, etc.
      *  \brief Get a reference to the specified token.
      *  \param i Stack position of the required token.
      *  \return Constant reference to the requested token.
      */
    const eval_token& peek(int i=1) const;

    /**  Clone a token and push it onto the top of the stack.
      *  \brief Push a token.
      *  \param tkn Constant reference to the token to be cloned and pushed.
      */
    virtual void push(const eval_token& tkn);

    /**  Push a token pointer onto the top of the stack. Note that ownership
      *  of the specified token is assumed by the stack.
      *  \brief Push a token.
      *  \param tkn Pointer to the token to be cloned and pushed.
      */
    virtual void push(eval_token* tkn);

    /**  Construct ta token from a string and push it onto the stack.
      *  \brief Push a string token.
      *  \param str Constant string to be converted to a token and pushed.
      */
    virtual void push(const std::string& str);

    /**  Push a numeric value. The double precision float value is converted
      *  to a string before being pushed on to the stack.
      *  \brief Push a numeric token.
      *  \param val Numeric value to be tokenized and pushed onto the stack.
      */
    virtual void push(double val);

    /**  Push a boolean value onto the stack.
      *  \brief Push a boolean token.
      *  \param val Boolean value to be tokenized and pushed onto the stack.
     */
    virtual void push(bool val);

    /**  Remove the top token off the stack and delete it.
      *  \brief Remove the top token.
     */
    void remove(void);

    /**  Get the stack size.
      *  \brief Stack size
      *  \return Stack size
      */
    unsigned int size(void) const;

    /**  Pop the top value off the stack and convert it to a boolean value.
      *  Non-zero numeric values and the string #"true"# are considered to
      *  be true, and all other values are considered to be false.
      *  \brief Pop logical value
      *  \return Logical value of top element.
      */
    virtual bool pop_logical(void);

    /**  Pop the top value off the stack and convert it to a numeric value.
      *  \brief Pop numeric value
      *  \return Numeric value of top element.
      */
    virtual double pop_numeric(void);

    /**  Pop the top value off the stack and return the string value.
      *  \brief Pop string value
      *  \return String value of top element.
      */
    virtual std::string pop_string(void);

    /**  Pop the top value off the stack and return the string value.
      *  \brief Pop a token from the stack
      *  \return Point to the popped token.
      */
    virtual eval_token* pop_token(void);

    /**  Get a reference to the top element of the stack. The top element
      *  is not removed from the stack.
      *  \brief Get a reference to the top token
      *  \return Reference to the top token.
      */
    virtual eval_token& top(void);

    /**  Get a constant reference to the top element of the stack. The top 
      *  element is not removed from the stack.
      *  \brief Get a constant reference to the top token
      *  \return Constant reference to the top token.
      */
    virtual const eval_token& top(void) const;

    /**  Get the data type of the specified stack entry. Note that the top
      *  stack entry is \c off=1.
      *  \brief Get top data type.
      *  \param off Stack position of the requested token.
      *  \return Data type of the specified token.
      */
    eval_token::datype type(int off=1) const;

private:
    op_level mOpLevel[kNOps];
    const char* mOpStr[kNOps];

private:
    typedef std::vector< eval_token* > eval_vect;
    eval_vect mVect;
};

//======================================  Inline methods
inline bool 
eval_stack::empty(void) const {
    return mVect.empty();
}

inline eval_token*
eval_stack::pop_token(void) {
    eval_token* p = mVect.back();
    mVect.pop_back();
    return p;
}

inline void
eval_stack::remove(void) {
    delete pop_token();
}

inline void
eval_stack::clear(void) {
    while (!empty()) remove();
}

inline unsigned int
eval_stack::size(void) const {
    return mVect.size();
}

inline const eval_token&
eval_stack::top(void) const {
    return *mVect.back();
}

inline eval_token&
eval_stack::top(void) {
    return *mVect.back();
}

#endif // !defined(EVAL_STACK_HH)
