/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TRIGENV_HH
#define TRIGENV_HH

#ifndef __CINT__
#include "Dacc.hh"
#include "SigFlag.hh"
#include "TrigBase.hh"
#include "framedir.hh"
#include "xsil/MetaIO.hh"
#endif //  !defined(__CINT__)

/**  <h2> Introduction</h2>
  *  The triggered data processing base class defines the basic routines 
  *  necessary for processing data selected by a trigger table. Each trigger 
  *  is passed to a user method for selection and detemination of the time 
  *  epoch. The data from the specified epoch are then passed to a second 
  *  user method containing the data processing code. The %TrigEnv class 
  *  provides an environment similar to that supplied by DatEnv. 
  *  The functions provided by the %TrigEnv are:
  *  <ul>
  *    <li> Parse common command line arguments.
  *    <li> read and parse a trigger table.
  *    <li> Read frames selected by the triggers.
  *    <li> Copy requested data into time series (TSeries) containers.
  *    <li> Handling attention interrupts.
  *    <li> Clean up on termination.
  *  </ul>
  *
  *  <h2> Writing a %TrigEnv Monitor</h2>
  *  The strategy for writing a monitor program is to create a class 
  *  (\c MyClass) based on %TrigEnv including a constructor, destructor and
  *  ProcessData() method. An attention handler may also be provided if
  *  desired. The operations to be performed by each of these methods
  *  are:
  *
  *  <b> Constructor (MyClass(int argc, const char* argv[])) </b>
  *  <ul>
  *    <li> %Process user specific arguments
  *    <li> Define parameters (see TrigPar class).
  *    <li> Configure the process.
  *    <li> Request data channels with Dacc::addChannel().
  *  </ul>
  *
  *  <b> Destructor (~MyClass(void)) </b>
  *  <ul>
  *    <li> Save modified parameters
  *    <li> Print results and statistics.
  *  </ul>
  *
  *  <b> Trigger processing (bool ProcessTrigger(const trig::TrigBase\& x))</b>
  *  <ul>
  *    <li> Interpret trigger data.
  *    <li> Determine whether the triggered data are to be processed.
  *    <li> Defined the trigger epoch.
  *  </ul>
  *
  *  <b> Epoch processing (void ProcessData(void))</b>
  *  <ul>
  *    <li> %Process data from one epoch.
  *    <li> Generate trigger if necessary.
  *    <li> Keep statistics.
  *  </ul>
  *
  * <b> Interrupt handler (void Attention(void)) </b>
  *  <ul>
  *    <li> Monitor dependent interrupt processing.
  *  </ul>
  *
  *  The \c EXECDAT(myclass) macro is used to generate the main
  *  function. Definition and implementation templates are available 
  *  from "TrigTemplate.hh" and "TrigTemplate.cc".
  *
  *  The \c EXEROOT(myclass) macro is used to generate a main function
  *  with \c root graphics. Note that the "TROOT.h" and "TApplication.h"
  *  header files must be included in your monitor implementation file 
  *  if \c EXEROOT is used.
  *
  *  <b> Compiling the monitor </b>
  *  The monitor class can be compiled and linked using the default 
  *  makefile which can be found in 
  *  \c \$DMTHOME/\$DMTVERSION/doc/DMTBase/Makefile.default
  *
  *  <b> Running the Monitor</b>
  *  The base class constructor recognizes the following command line 
  *  arguments: 
  *  <table>
  *  <tr><td>-debug \<lvl\>  </td>
  *      <td>Set debug level to \<lvl\> </td></tr>
  *  <tr><td>-infile \<file\></td>
  *      <td>Frame data files (names may contain '*'s). </td></tr>
  *  <tr><td>-inlist \<list\></td>
  *      <td>Frame data file list (names may contain '*'s). </td></tr>
  *  <tr><td>-stride         </td>
  *      <td>Default stride time (in seconds). </td></tr>
  *  <tr><td>-toffset        </td>
  *      <td>Start of epoch relative to trigger time. </td></tr>
  *  <tr><td>-tepoch         </td>
  *      <td>Length of epoch (in seconds). </td></tr>
  *  <tr><td>-ttable         </td>
  *      <td>Ligo_lw table containing triggers. </td></tr>
  *  </table>
  *  If no input option is specified, the frames are read from the partition
  *  specified by the \c LIGOSMPART environment variable or, if this is not 
  *  defined, from "LIGO_Online".
  *
  *  When running under ROOT, the base class definition and implementation 
  *  files must be loaded before the monitor can be run. If these files 
  *  aren't provided automatically by the start-up script, they may be 
  *  loaded with:
  *  \verbatim
        gROOT->LoadMacro("TrigEnv.hh");
        gROOT->LoadMacro("TrigEnv.cc"); \endverbatim
  *  The monitor is run with the \c .x command or by loading it and calling 
  *  it as a function. See the note entitled "Online monitoring of LIGO Data
  *  Using ROOT" for more details.
  *  \brief Triggered data processing base class.
  *  @author John G. Zweizig
  *  @version 1.4; Last Modified March 4, 2003
  */
class TrigEnv {
  public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  \brief Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
    TrigEnv(int argc, const char *argv[]);

    /**  Close the input stream and go away.
      *  \brief Class destructor.
      */
    virtual ~TrigEnv(void);

    /**  MainLoop reads in data and invokes ProcessData for each data 
      *  epoch successfully read in. MainLoop exits when a SIGINT or 
      *  SIGTERM signal is caught, or when finish is called by a subsidiary 
      *  function (usually by ProcessData). This function is provided by the 
      *  base class.
      *  \brief Loop over data epochs.
      */
    void MainLoop(void);

    /**  ProcessData performs the monitor specific processing data from
      *  a single stride. It must be defined as part of the derived monitor 
      *  class implementation.
      *  \brief %Process one epoch (Supplied by user).
      */
    virtual void ProcessData(void)=0;

    /**  ProcessTrigger performs the monitor specific processing of the
      *  trigger used to specify an epoch. ProcessTrigger is invoked 
      *  after reading and parseing the trigger from the trigger table, but 
      *  before locating or reading the data from the trigger epoch. This
      *  allows the monitor designer to set the time parameters for the 
      *  epoch or to bypass the processing of the trigger epoch completely. 
      *  The epoch is skipped if ProcessTrigger returns false. The default
      *  ProcessTrigger method returns true for all triggers.
      *  \brief %Process a Trigger (Supplied by user).
      *  @return  true if data from trigger epoch are to be processed.
      *  @param t Trigger to be processed.
      */
    virtual bool ProcessTrigger(const trig::TrigBase& t);

    /**  Attention performs the monitor specific attention interrupt 
      *  processing. Its implementation is optional. Execution of 
      *  the attention handler is deferred until the processing of the 
      *  current epoch has completed. This means that there may be a 
      *  significant delay between the delivery of the SIGUSR1 signal and
      *  the invocation of the handler. The ROOT version uses the ROOT 
      *  event handler as a source of When used with ROOT, the default 
      *  action is to terminate the monitor. When used in the background, 
      *  it defaults to no action. The execution of the attention handler 
      *  is synchronized to the 
      *  \brief Attention interrupt handler (Supplied by user).
      */
    virtual void Attention(void);

    /**  finish stops processing by setting a flag that breaks the main 
      *  loop. Note that finish returns to the calling function (generally 
      *  the user's ProcessData) which must exit normally for the process 
      *  to terminate.
      *  \brief Stop data processing and go away cleanly.
      */
    void finish(void) {mActive = false;}

    /**  Debug returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  \brief Get the debug level.
      *  \return Debug level.
      */
    int Debug(void) const {return mDebug;}

    /**  Return a constant reference to the data accessor.
      *  \brief Get the data accessor.
      *  \return Constant reference to data accesor for this monitor.
      */
    const Dacc& getDacc(void) const {return mIn;}

    /**  Get the length of the epoch.
      *  \brief Get the epoch length.
      *  \return Epoch length
      */
    const Interval getEpochLength(void) const {return mEpoch;}

    /**  Get the time offset from the specified trigger time to the 
      *  start of the epoch to be processed. If the offset is negative,
      *  the spoch starts before the trigger time.
      *  \brief Get the epoch length.
      *  @return %Time offset from the trigger time to the data epoch start.
      */
    const Interval getEpochOffset(void) const {return mOffset;}

    /**  Get the data stride. The data stride is the length of data to be 
      *  read for each call to ProcessData.
      *  \brief Get the data stride.
      *  @return The data stride interval.
      */
    const Interval getStride(void) const {return mStride;}

    /**  Get the current trigger block.
      *  \brief Get the current trigger.
      *  @return Reference to the current trigger.
      */
    const trig::TrigBase& getTrigger(void) const {return mTrigger;}

    /**  Test for a %TrigEnv command line argument. All %TrigEnv command line 
      *  arguments take a value in the following argument position.
      *  \brief Test if string is a %TrigEnv command line argument.
      *  \param arg Argument strin to be tested.
      *  @return true if the specified string is a %TrigEnv argument.
      */
    bool isTrigEnvArg(const char* arg) const;

    /**  Set the length of the epoch.
      *  \brief Set the epoch length.
      *  \param t Epoch length
      */
    void setEpochLength(const Interval& t);

    /**  Set the time offset from the specified trigger time to the 
      *  start of the epoch to be processed. If the offset is negative,
      *  the epoch starts before the trigger time.
      *  \brief Set the epoch offset.
      *  \param t Epoch Offset
      *  \return %Time offset from the trigger time to the data epoch start.
      */
    void setEpochOffset(const Interval& t);

    /**  Get the length of the epoch.
      *  \brief Get the data stride.
      *  \param t Stride length
      */
    void setStride(const Interval& t);

  //------------------------------------  Private methods
  private:
    /**  Read in a trigger.
      */
    int readTrigger(void);

    /**  Test if a termination interrupt occurred.
      *  \brief Test for termination.
      */
    bool testTerm(void) const;

    /**  Test if an attention interrupt occurred.
      *  \brief Test for attention.
      */
    bool testAttn(void) const;

  protected:
    /**  Return a reference to the data accessor.
      *  \brief Get the data accessor.
      *  \return Reference to data accessor for this monitor.
      */
    Dacc& getDacc(void) {return mIn;}

    ///  Frame Input object
    Dacc  mIn;

  private:
    ///  Internal run flag
    bool     mActive;

    ///  Length of epoch.
    Interval mEpoch;

    ///  Offset of epoch from trigger time.
    Interval mOffset;

    ///  Length of stride.
    Interval mStride;

    ///  Current trigger
    trig::TrigBase mTrigger;

#ifndef __CINT__
    ///  Termination signal handling object
    SigFlag mTerm;

    ///  Attention signal handling object
    SigFlag mAttn;
#endif // !def(__CINT__)
  
    ///  Debug Level
    int  mDebug;
  
    ///  Number of triggers read
    int  mTrigRead;
  
    ///  Number of triggers processed
    int  mTrigProc;
  
    ///  Number of processed strides
    int  mProcStride;

    ///  metaio package environment structure.
    xsil::MetaIO mMetaIO;

    /// Frame directory for all infiles.
    FrameDir mFrDir;
};

//
//   The EXECDAT macro generates the main function to execute a monitor
//   based on %TrigEnv. The argument is whatever name you have chosen
//   for your class.
//
#define EXECDAT(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

//
//   The EXEROOT macro generates the main function to execute a monitor
//   based on %TrigEnv. This macro differs from EXECDAT in that it also 
//   sets up the root object structure which allows the derived class to 
//   use any root classes. The argument is whatever name you have chosen
//   for your class.
//
#define EXEROOT(myclass) \
TROOT root("TrigEnv", "Graphical background monitor"); \
int main(int argc, const char* argv[]) { \
    int bc(2); const char* bv[3] = {0, "-n", 0}; bv[0]=argv[0]; \
    TApplication theApp("App", &bc, const_cast<char**>(bv)); \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

#endif     //  TrigEnv_HH
